/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(C) 2025 Marvell.
 */

#ifndef __TEST_CRYPTODEV_ML_KEM_TEST_VECTORS_H__
#define __TEST_CRYPTODEV_ML_KEM_TEST_VECTORS_H__

#include "rte_crypto_asym.h"

struct crypto_testsuite_mlkem_params {
	const char *name;
	enum rte_crypto_ml_kem_type type;
	rte_crypto_param d;
	rte_crypto_param z;
	rte_crypto_param dk;
	rte_crypto_param ek;
	rte_crypto_param sk;
	rte_crypto_param message;
	rte_crypto_param cipher;
};

/* Seed */
uint8_t mlkem_512_d[] = {
	0x4d, 0x3f, 0xb3, 0x37, 0xd3, 0xec, 0x8f, 0x29,
	0xf9, 0x7b, 0x0e, 0xd6, 0x5b, 0x85, 0x78, 0xc5,
	0x31, 0x37, 0x21, 0x9a, 0x14, 0x64, 0xc6, 0x1c,
	0xc7, 0xd7, 0x45, 0x57, 0x06, 0x5b, 0x45, 0x96,
};

uint8_t mlkem_512_z[] = {
	0xdf, 0x29, 0x92, 0x4a, 0x81, 0x76, 0x4f, 0x4d,
	0x5a, 0x64, 0x79, 0xf2, 0x8b, 0xc8, 0x3b, 0xc9,
	0xf3, 0x35, 0xc7, 0x18, 0xcb, 0xb1, 0xa8, 0xd2,
	0x24, 0xdb, 0x4d, 0x6a, 0x6a, 0xf1, 0xeb, 0xd2,
};

/* Private key */
uint8_t mlkem_512_dk[] = {
	0xd8, 0xc8, 0x4d, 0x74, 0x57, 0x6d, 0x43, 0x65,
	0x84, 0xe0, 0x82, 0x87, 0x2d, 0xd0, 0x0b, 0x39,
	0x08, 0x41, 0xc8, 0x85, 0x15, 0x26, 0xfb, 0x85,
	0x68, 0xe1, 0x30, 0x98, 0xf0, 0xb1, 0x3d, 0xea,
	0xa5, 0x4c, 0xc4, 0x59, 0x81, 0xfb, 0x33, 0xed,
	0x20, 0x3a, 0x7f, 0x10, 0x82, 0x3d, 0x85, 0x18,
	0x69, 0x56, 0x6b, 0x5b, 0x7c, 0x92, 0x0f, 0x57,
	0x37, 0xf2, 0x93, 0x38, 0x24, 0xd3, 0x1e, 0xe1,
	0x84, 0x16, 0x39, 0x9a, 0x0d, 0xbd, 0xba, 0x22,
	0xaa, 0xc8, 0xa1, 0xd9, 0xd8, 0x2a, 0xc7, 0xc1,
	0x78, 0x21, 0x01, 0x85, 0x38, 0x54, 0x39, 0x34,
	0x25, 0x8c, 0xce, 0xda, 0xa3, 0x0c, 0x66, 0x99,
	0x39, 0xf7, 0x63, 0xb6, 0x86, 0x80, 0xe0, 0xe4,
	0x64, 0x2a, 0xfc, 0xc2, 0xe0, 0x71, 0x6c, 0x59,
	0x7c, 0x9a, 0x03, 0xac, 0x6f, 0x16, 0x08, 0x64,
	0x3d, 0x4b, 0xa5, 0x83, 0xc5, 0x34, 0x27, 0x2a,
	0x20, 0x18, 0xb8, 0x2d, 0x28, 0x61, 0x2a, 0x94,
	0x55, 0xc7, 0xc3, 0x93, 0x91, 0x71, 0x43, 0x7f,
	0x74, 0xb2, 0x6b, 0x55, 0xb1, 0x9d, 0xc8, 0x08,
	0x53, 0xda, 0xb6, 0x19, 0x46, 0x5c, 0x20, 0xea,
	0x31, 0x31, 0xfc, 0x41, 0x02, 0xe6, 0x3c, 0x3e,
	0xd8, 0x68, 0x6f, 0xa6, 0x66, 0x1e, 0xc7, 0xc6,
	0x0a, 0xf8, 0x02, 0x1e, 0x1e, 0xfa, 0x9a, 0x30,
	0x18, 0x64, 0xdc, 0x30, 0x20, 0x9a, 0xb6, 0x6e,
	0x85, 0xe8, 0xb0, 0xb3, 0xe1, 0x24, 0x4e, 0xf0,
	0x4a, 0x9c, 0xb3, 0x0e, 0xe0, 0xd2, 0x6a, 0x6e,
	0xc7, 0xc3, 0x15, 0x35, 0x49, 0x77, 0x07, 0x7e,
	0x33, 0x7c, 0x97, 0x6e, 0xeb, 0xb3, 0x7c, 0x86,
	0xb2, 0xa0, 0xe3, 0x8a, 0x13, 0x81, 0x99, 0xc6,
	0x75, 0x41, 0xa6, 0x2a, 0x6f, 0x16, 0xe2, 0x4e,
	0x52, 0x14, 0x0d, 0xcb, 0x74, 0x8d, 0xb4, 0x89,
	0xcf, 0xa8, 0x4c, 0x57, 0xaa, 0x18, 0x7d, 0xd7,
	0x19, 0x5c, 0x11, 0x04, 0xc6, 0xe2, 0x2c, 0x09,
	0x4d, 0x78, 0x7e, 0x9e, 0x1a, 0xa2, 0xab, 0x60,
	0x59, 0xe4, 0x30, 0xa5, 0x6d, 0x11, 0x63, 0xf9,
	0xb1, 0xa0, 0x98, 0xe2, 0x81, 0x34, 0xa6, 0x57,
	0xf7, 0x18, 0x3e, 0x4c, 0x36, 0x02, 0x4d, 0xda,
	0x6e, 0xac, 0x39, 0x6c, 0x29, 0x0b, 0x8c, 0xd7,
	0x32, 0x8e, 0x45, 0x67, 0x36, 0x41, 0x6c, 0xab,
	0xbb, 0x04, 0x2e, 0xae, 0xd3, 0x9f, 0xee, 0xfa,
	0x5f, 0x4a, 0x04, 0x85, 0xf7, 0xfb, 0x97, 0x95,
	0x78, 0xc4, 0x08, 0x39, 0xb8, 0xc5, 0x17, 0x9d,
	0x2e, 0x6c, 0x14, 0xcd, 0xf2, 0xc1, 0xc5, 0x05,
	0x9d, 0x34, 0x12, 0xcd, 0x86, 0xbc, 0x9e, 0x1c,
	0xf9, 0x1c, 0x05, 0xc2, 0xc3, 0x29, 0xf2, 0x9d,
	0xf4, 0xa9, 0xcc, 0x0d, 0x38, 0xce, 0x6f, 0xb4,
	0x06, 0x68, 0xc0, 0xcd, 0xb2, 0xb6, 0x5d, 0xdf,
	0x89, 0x58, 0x99, 0x70, 0xbe, 0x37, 0xa8, 0xb2,
	0xc3, 0x6c, 0x2c, 0x13, 0x8c, 0xa7, 0x8a, 0x38,
	0x36, 0x36, 0xe0, 0x77, 0x79, 0xf6, 0x30, 0xc5,
	0x9c, 0x8e, 0x10, 0xc0, 0x1b, 0xb4, 0xe0, 0x74,
	0x36, 0x6c, 0x9f, 0x80, 0x73, 0x3e, 0x57, 0x69,
	0x5a, 0x79, 0x6a, 0x44, 0xb8, 0xe8, 0x25, 0xbe,
	0x96, 0xae, 0xd7, 0x87, 0x30, 0x7e, 0x76, 0x8e,
	0x1c, 0x09, 0x99, 0xd9, 0x4c, 0xb3, 0xf7, 0x45,
	0x92, 0xfa, 0x77, 0x9e, 0x18, 0x72, 0x1a, 0x23,
	0x9c, 0x4e, 0x5e, 0xd9, 0x22, 0x2e, 0xb0, 0x16,
	0x9d, 0x03, 0x6a, 0x0e, 0x49, 0x94, 0x97, 0xa3,
	0x19, 0x94, 0xa1, 0x7e, 0x9d, 0x7b, 0x87, 0xc1,
	0x6a, 0x12, 0x42, 0xc9, 0x22, 0xbe, 0xeb, 0x38,
	0xbe, 0xfa, 0x0b, 0xac, 0x25, 0xb9, 0x78, 0xec,
	0x33, 0xdf, 0x86, 0xcb, 0xce, 0xe1, 0xa6, 0x69,
	0x28, 0x39, 0xd5, 0x68, 0x73, 0xcb, 0x8a, 0xcc,
	0x3d, 0xfa, 0x0e, 0x79, 0x71, 0x3e, 0x44, 0x19,
	0x9a, 0xac, 0xf4, 0x43, 0xa9, 0xba, 0x10, 0x7e,
	0x49, 0xc8, 0x8f, 0xf9, 0xac, 0xd2, 0x07, 0xb1,
	0x4b, 0x2b, 0x15, 0x65, 0x05, 0x12, 0xc3, 0xcc,
	0x36, 0x69, 0x17, 0x78, 0xa3, 0x00, 0x6d, 0xb7,
	0xe9, 0x03, 0x78, 0xb3, 0x33, 0xe5, 0xcc, 0x22,
	0xc3, 0x91, 0x8f, 0x47, 0x79, 0xaf, 0x2e, 0x19,
	0x91, 0x05, 0xab, 0x5e, 0x3e, 0x43, 0xa4, 0x34,
	0x0b, 0x8c, 0x89, 0x75, 0xc0, 0x08, 0x44, 0x07,
	0xdd, 0x17, 0x83, 0xc6, 0x48, 0x09, 0x12, 0xe6,
	0x25, 0xde, 0x99, 0x00, 0x51, 0xc1, 0x65, 0x8a,
	0x85, 0x02, 0xa8, 0xd5, 0x72, 0x41, 0x67, 0x66,
	0x08, 0xe9, 0x10, 0xa3, 0xa2, 0x07, 0xc8, 0x06,
	0xc6, 0x0c, 0xc6, 0x47, 0xf7, 0x35, 0x20, 0x40,
	0x69, 0x04, 0x3c, 0xba, 0x4b, 0xd3, 0x78, 0xa2,
	0xa4, 0x71, 0xc7, 0x83, 0x2b, 0x69, 0xf9, 0xa2,
	0x8b, 0xf5, 0xa6, 0x87, 0x46, 0xb3, 0x9b, 0x7a,
	0x35, 0x97, 0x0b, 0xf4, 0x68, 0x15, 0x4a, 0x87,
	0xeb, 0x0b, 0x2b, 0xa3, 0xba, 0xcd, 0xb3, 0x28,
	0x80, 0x38, 0x19, 0x17, 0x1f, 0x44, 0x6f, 0x7f,
	0x50, 0x75, 0x93, 0x00, 0x4c, 0x36, 0x07, 0x07,
	0x11, 0xec, 0x91, 0xc4, 0x99, 0x5d, 0xf8, 0xe1,
	0x53, 0x84, 0x01, 0xca, 0x6a, 0x64, 0x8c, 0xef,
	0x17, 0xce, 0x81, 0xf5, 0xce, 0xaa, 0x80, 0xc1,
	0x85, 0xc6, 0x6b, 0xe2, 0xc9, 0x43, 0xc9, 0xf5,
	0xc3, 0x3a, 0xdb, 0xcf, 0xb9, 0x29, 0x64, 0x8d,
	0xe7, 0x98, 0x0f, 0x8b, 0xb0, 0xd2, 0x84, 0x25,
	0x37, 0xd1, 0x80, 0x38, 0x81, 0xb9, 0xe8, 0xb4,
	0xa7, 0xc8, 0x3a, 0x27, 0x77, 0x9b, 0x81, 0xbb,
	0x61, 0xae, 0x59, 0x06, 0x57, 0x40, 0x69, 0x11,
	0xb2, 0x41, 0x68, 0x2d, 0x26, 0x62, 0x93, 0x40,
	0x4d, 0x97, 0x1c, 0xb4, 0x63, 0x65, 0x04, 0xf8,
	0x8c, 0x2a, 0x01, 0xb9, 0x6d, 0x3d, 0x87, 0x30,
	0x28, 0x70, 0x80, 0x1d, 0xb2, 0x24, 0xb0, 0x5c,
	0x90, 0x95, 0xa3, 0x6c, 0xdc, 0xa6, 0x66, 0x15,
	0x33, 0x37, 0x8c, 0xa8, 0x9f, 0x24, 0xa3, 0x52,
	0x21, 0x06, 0xb6, 0xaf, 0x51, 0xc0, 0x1d, 0x42,
	0xa6, 0x55, 0x93, 0x6b, 0x4a, 0x32, 0x4e, 0xec,
	0x46, 0xbe, 0xfa, 0x4a, 0x13, 0x5a, 0xca, 0x11,
	0x39, 0xa1, 0x81, 0x8b, 0xc4, 0x07, 0x1b, 0x43,
	0x1e, 0xa0, 0x5b, 0x34, 0x2d, 0x10, 0x6c, 0xdd,
	0x3b, 0x18, 0x78, 0xe1, 0x0d, 0x5b, 0x61, 0x3d,
	0xfd, 0x51, 0x6a, 0x78, 0x88, 0x2f, 0xa7, 0x84,
	0xa3, 0x4f, 0x30, 0x3e, 0xb0, 0xdc, 0x96, 0xa8,
	0xd7, 0xbb, 0x4c, 0xa6, 0x95, 0xb9, 0x27, 0x3f,
	0x70, 0x46, 0xa3, 0xd8, 0x36, 0x22, 0x3b, 0x4b,
	0x91, 0xa4, 0xe6, 0x31, 0x0d, 0xbb, 0x27, 0x2f,
	0x15, 0x2c, 0xa7, 0x44, 0x19, 0xa9, 0xc1, 0xc8,
	0xb2, 0x77, 0x1a, 0xd5, 0x14, 0x29, 0xa1, 0x76,
	0xcc, 0x81, 0x88, 0x47, 0x3c, 0xd9, 0x9e, 0xba,
	0x99, 0x26, 0xbb, 0xf6, 0x5e, 0xbc, 0x2a, 0x8e,
	0xa0, 0x32, 0x32, 0x2e, 0x24, 0xa1, 0xc3, 0x07,
	0xa5, 0xd4, 0xbb, 0x95, 0x03, 0x29, 0xb5, 0x59,
	0x5a, 0x2b, 0x6c, 0x95, 0x15, 0xba, 0xb0, 0xaf,
	0x27, 0xa2, 0xcb, 0xef, 0x46, 0xb6, 0xb2, 0x25,
	0x21, 0x23, 0xf8, 0x18, 0x1c, 0xb0, 0x54, 0xad,
	0x48, 0x38, 0xf6, 0x33, 0xa2, 0x55, 0xc8, 0x63,
	0xce, 0x56, 0x34, 0x86, 0x6a, 0x61, 0x75, 0x75,
	0x7f, 0x75, 0x49, 0x2a, 0x57, 0x72, 0x76, 0x3e,
	0xa7, 0x08, 0x56, 0x28, 0x0b, 0x44, 0xa9, 0x00,
	0xc2, 0x88, 0x28, 0x02, 0x55, 0x33, 0x55, 0x43,
	0x97, 0x97, 0x42, 0xa4, 0xfe, 0xb1, 0x95, 0xee,
	0x9b, 0x5d, 0xa9, 0x18, 0xc1, 0x90, 0xa5, 0x65,
	0x20, 0x99, 0x39, 0xd2, 0xea, 0x08, 0x1c, 0xda,
	0xa8, 0xfa, 0xa3, 0x3b, 0x56, 0xf0, 0x59, 0x17,
	0x10, 0x0b, 0x45, 0x4a, 0x86, 0x88, 0x96, 0x49,
	0x68, 0x98, 0x26, 0x55, 0x40, 0xb7, 0x5a, 0x29,
	0x82, 0x93, 0xa1, 0x0e, 0x7b, 0x4b, 0xa2, 0x49,
	0x61, 0x2b, 0xaf, 0xb4, 0x2d, 0xa1, 0xdc, 0xae,
	0x79, 0x44, 0x66, 0x20, 0x72, 0xc5, 0x01, 0x36,
	0x10, 0xd8, 0x81, 0x36, 0xb1, 0xea, 0xb1, 0x0d,
	0xc2, 0x05, 0x75, 0xd8, 0xad, 0xe0, 0x73, 0xc6,
	0x82, 0x8c, 0x7e, 0x97, 0x2b, 0x87, 0xb5, 0xc9,
	0x1e, 0xac, 0xb0, 0x2a, 0x16, 0x20, 0x88, 0xb9,
	0x41, 0x2f, 0xc8, 0x4b, 0xce, 0xd8, 0xc4, 0xa8,
	0xe2, 0x68, 0x35, 0xd9, 0xd0, 0xc4, 0x7a, 0xf2,
	0x25, 0xca, 0xb8, 0x06, 0x42, 0xcb, 0xc9, 0x8f,
	0x0a, 0x60, 0xde, 0x19, 0xa8, 0xcb, 0x90, 0x85,
	0x74, 0x42, 0x03, 0x89, 0xf4, 0x97, 0x8e, 0x41,
	0xc5, 0x72, 0xe3, 0x52, 0x55, 0x81, 0x73, 0xe1,
	0x71, 0x22, 0xa2, 0xfb, 0x8a, 0x36, 0xea, 0xb3,
	0xc3, 0x33, 0x4f, 0x23, 0x55, 0x03, 0x1c, 0xa1,
	0x84, 0x74, 0xe1, 0x69, 0x4d, 0xb3, 0xc2, 0x1f,
	0xbc, 0x62, 0xeb, 0xf0, 0x50, 0x45, 0xb7, 0x83,
	0xd3, 0x28, 0x37, 0x46, 0x35, 0x79, 0x04, 0x38,
	0x16, 0xbd, 0x72, 0x2f, 0xbc, 0xf9, 0x7d, 0xb8,
	0x20, 0x12, 0x87, 0xa7, 0xb0, 0x82, 0x7c, 0x3e,
	0x4d, 0xa7, 0x18, 0x04, 0x54, 0x75, 0x19, 0x52,
	0x61, 0x80, 0x57, 0xbe, 0x06, 0x37, 0x9c, 0xde,
	0xb0, 0x93, 0x70, 0x54, 0x3f, 0x15, 0x83, 0xb2,
	0xd3, 0x88, 0xa6, 0xd8, 0x56, 0x9b, 0x91, 0xf3,
	0x00, 0x01, 0x00, 0xcd, 0xcb, 0x97, 0x16, 0x27,
	0xc7, 0xaf, 0xb1, 0xd5, 0xba, 0x90, 0xa1, 0x24,
	0x33, 0x26, 0x95, 0xba, 0xcb, 0x63, 0xa1, 0x56,
	0xb0, 0x23, 0x09, 0x4e, 0xa8, 0x0a, 0x75, 0x02,
	0xd8, 0x3b, 0xc1, 0x05, 0xa0, 0x73, 0x25, 0x26,
	0x3e, 0xb8, 0x5b, 0xca, 0x6a, 0xae, 0x41, 0x22,
	0xc5, 0x9d, 0x88, 0x3f, 0x37, 0x8c, 0x50, 0xcd,
	0x09, 0x32, 0x34, 0x81, 0x98, 0xc1, 0x17, 0xa5,
	0x62, 0x94, 0x8d, 0xcd, 0xd0, 0xc3, 0x08, 0x07,
	0x2f, 0x8d, 0x52, 0x7b, 0xee, 0x81, 0xc5, 0x95,
	0x2c, 0x8b, 0x46, 0x45, 0x5d, 0xf2, 0x62, 0x4d,
	0x2e, 0x22, 0x7f, 0x0d, 0xa5, 0xc8, 0xba, 0x72,
	0xa0, 0x8f, 0xd0, 0x21, 0x45, 0x28, 0xb9, 0xad,
	0xac, 0x96, 0xa6, 0xc7, 0x33, 0x6d, 0xf3, 0x17,
	0xcb, 0x7a, 0x32, 0x4e, 0x4c, 0x83, 0xa7, 0x15,
	0x55, 0x6a, 0xca, 0xc1, 0xff, 0xd8, 0x90, 0xab,
	0xaa, 0xa4, 0x65, 0xb0, 0x25, 0xf7, 0x47, 0x00,
	0x2d, 0x20, 0x19, 0x23, 0xb5, 0x02, 0x39, 0x5a,
	0x69, 0x1c, 0xc4, 0xbc, 0xa3, 0x80, 0xcc, 0x82,
	0xd8, 0x66, 0x63, 0xc5, 0x7e, 0xa3, 0x6c, 0x2a,
	0xb0, 0x7a, 0x02, 0xc6, 0xb3, 0xb7, 0xb0, 0x29,
	0x3f, 0xa3, 0x99, 0xbd, 0x28, 0xf2, 0x8e, 0xd6,
	0x7b, 0xb3, 0xd5, 0x24, 0x12, 0xb9, 0xbb, 0xa5,
	0x68, 0xba, 0x9d, 0xe3, 0x44, 0xa3, 0xeb, 0x64,
	0x13, 0xb8, 0x8a, 0x5e, 0x72, 0x60, 0xad, 0x95,
	0xcb, 0x9d, 0xfb, 0x54, 0x6e, 0x0e, 0x66, 0xb5,
	0xf4, 0x68, 0x25, 0x91, 0xc5, 0x1a, 0x4f, 0x71,
	0x61, 0x50, 0xc8, 0xcb, 0x34, 0xc5, 0xc2, 0x84,
	0xbc, 0x4e, 0x5f, 0x28, 0x6f, 0xbf, 0xbb, 0xc8,
	0xb1, 0xc4, 0x4a, 0x4b, 0xb7, 0x1a, 0x00, 0xa0,
	0x2d, 0xe9, 0x07, 0x39, 0x93, 0xe9, 0x81, 0x51,
	0xe1, 0x34, 0x83, 0xf5, 0x82, 0x9d, 0xc7, 0x47,
	0xf1, 0x61, 0xc8, 0x29, 0x52, 0x4d, 0x51, 0xb7,
	0xa1, 0xeb, 0x49, 0x50, 0x6e, 0xda, 0x72, 0x5d,
	0x69, 0x27, 0x83, 0xca, 0x4c, 0x3b, 0xd4, 0xcc,
	0x7f, 0x26, 0x44, 0x2b, 0x76, 0x41, 0xe1, 0x50,
	0xce, 0x0f, 0x40, 0x0d, 0x8f, 0x00, 0x6e, 0x27,
	0x95, 0x6e, 0x92, 0x1b, 0xaf, 0xe4, 0x9b, 0x5e,
	0x96, 0xec, 0xc7, 0xbf, 0x64, 0x77, 0x75, 0x6d,
	0xda, 0xa8, 0x9c, 0x48, 0xf3, 0x79, 0x80, 0x4e,
	0xed, 0x6f, 0xe0, 0x20, 0x1a, 0x2b, 0x7c, 0xa1,
	0xac, 0x74, 0x42, 0x49, 0xd7, 0xe7, 0x7f, 0x57,
	0x5a, 0xdb, 0x69, 0xb5, 0xe2, 0x7e, 0x81, 0x8d,
	0xa9, 0x22, 0x5c, 0xda, 0xf3, 0xad, 0xbc, 0x3e,
	0x0b, 0xd4, 0x9d, 0x63, 0xef, 0x40, 0x34, 0x66,
	0x80, 0x8c, 0x2d, 0x31, 0x13, 0xdb, 0x31, 0x03,
	0xdf, 0x29, 0x92, 0x4a, 0x81, 0x76, 0x4f, 0x4d,
	0x5a, 0x64, 0x79, 0xf2, 0x8b, 0xc8, 0x3b, 0xc9,
	0xf3, 0x35, 0xc7, 0x18, 0xcb, 0xb1, 0xa8, 0xd2,
	0x24, 0xdb, 0x4d, 0x6a, 0x6a, 0xf1, 0xeb, 0xd2,
};

/* Public key */
uint8_t mlkem_512_ek[] = {
	0x28, 0x70, 0x80, 0x1d, 0xb2, 0x24, 0xb0, 0x5c,
	0x90, 0x95, 0xa3, 0x6c, 0xdc, 0xa6, 0x66, 0x15,
	0x33, 0x37, 0x8c, 0xa8, 0x9f, 0x24, 0xa3, 0x52,
	0x21, 0x06, 0xb6, 0xaf, 0x51, 0xc0, 0x1d, 0x42,
	0xa6, 0x55, 0x93, 0x6b, 0x4a, 0x32, 0x4e, 0xec,
	0x46, 0xbe, 0xfa, 0x4a, 0x13, 0x5a, 0xca, 0x11,
	0x39, 0xa1, 0x81, 0x8b, 0xc4, 0x07, 0x1b, 0x43,
	0x1e, 0xa0, 0x5b, 0x34, 0x2d, 0x10, 0x6c, 0xdd,
	0x3b, 0x18, 0x78, 0xe1, 0x0d, 0x5b, 0x61, 0x3d,
	0xfd, 0x51, 0x6a, 0x78, 0x88, 0x2f, 0xa7, 0x84,
	0xa3, 0x4f, 0x30, 0x3e, 0xb0, 0xdc, 0x96, 0xa8,
	0xd7, 0xbb, 0x4c, 0xa6, 0x95, 0xb9, 0x27, 0x3f,
	0x70, 0x46, 0xa3, 0xd8, 0x36, 0x22, 0x3b, 0x4b,
	0x91, 0xa4, 0xe6, 0x31, 0x0d, 0xbb, 0x27, 0x2f,
	0x15, 0x2c, 0xa7, 0x44, 0x19, 0xa9, 0xc1, 0xc8,
	0xb2, 0x77, 0x1a, 0xd5, 0x14, 0x29, 0xa1, 0x76,
	0xcc, 0x81, 0x88, 0x47, 0x3c, 0xd9, 0x9e, 0xba,
	0x99, 0x26, 0xbb, 0xf6, 0x5e, 0xbc, 0x2a, 0x8e,
	0xa0, 0x32, 0x32, 0x2e, 0x24, 0xa1, 0xc3, 0x07,
	0xa5, 0xd4, 0xbb, 0x95, 0x03, 0x29, 0xb5, 0x59,
	0x5a, 0x2b, 0x6c, 0x95, 0x15, 0xba, 0xb0, 0xaf,
	0x27, 0xa2, 0xcb, 0xef, 0x46, 0xb6, 0xb2, 0x25,
	0x21, 0x23, 0xf8, 0x18, 0x1c, 0xb0, 0x54, 0xad,
	0x48, 0x38, 0xf6, 0x33, 0xa2, 0x55, 0xc8, 0x63,
	0xce, 0x56, 0x34, 0x86, 0x6a, 0x61, 0x75, 0x75,
	0x7f, 0x75, 0x49, 0x2a, 0x57, 0x72, 0x76, 0x3e,
	0xa7, 0x08, 0x56, 0x28, 0x0b, 0x44, 0xa9, 0x00,
	0xc2, 0x88, 0x28, 0x02, 0x55, 0x33, 0x55, 0x43,
	0x97, 0x97, 0x42, 0xa4, 0xfe, 0xb1, 0x95, 0xee,
	0x9b, 0x5d, 0xa9, 0x18, 0xc1, 0x90, 0xa5, 0x65,
	0x20, 0x99, 0x39, 0xd2, 0xea, 0x08, 0x1c, 0xda,
	0xa8, 0xfa, 0xa3, 0x3b, 0x56, 0xf0, 0x59, 0x17,
	0x10, 0x0b, 0x45, 0x4a, 0x86, 0x88, 0x96, 0x49,
	0x68, 0x98, 0x26, 0x55, 0x40, 0xb7, 0x5a, 0x29,
	0x82, 0x93, 0xa1, 0x0e, 0x7b, 0x4b, 0xa2, 0x49,
	0x61, 0x2b, 0xaf, 0xb4, 0x2d, 0xa1, 0xdc, 0xae,
	0x79, 0x44, 0x66, 0x20, 0x72, 0xc5, 0x01, 0x36,
	0x10, 0xd8, 0x81, 0x36, 0xb1, 0xea, 0xb1, 0x0d,
	0xc2, 0x05, 0x75, 0xd8, 0xad, 0xe0, 0x73, 0xc6,
	0x82, 0x8c, 0x7e, 0x97, 0x2b, 0x87, 0xb5, 0xc9,
	0x1e, 0xac, 0xb0, 0x2a, 0x16, 0x20, 0x88, 0xb9,
	0x41, 0x2f, 0xc8, 0x4b, 0xce, 0xd8, 0xc4, 0xa8,
	0xe2, 0x68, 0x35, 0xd9, 0xd0, 0xc4, 0x7a, 0xf2,
	0x25, 0xca, 0xb8, 0x06, 0x42, 0xcb, 0xc9, 0x8f,
	0x0a, 0x60, 0xde, 0x19, 0xa8, 0xcb, 0x90, 0x85,
	0x74, 0x42, 0x03, 0x89, 0xf4, 0x97, 0x8e, 0x41,
	0xc5, 0x72, 0xe3, 0x52, 0x55, 0x81, 0x73, 0xe1,
	0x71, 0x22, 0xa2, 0xfb, 0x8a, 0x36, 0xea, 0xb3,
	0xc3, 0x33, 0x4f, 0x23, 0x55, 0x03, 0x1c, 0xa1,
	0x84, 0x74, 0xe1, 0x69, 0x4d, 0xb3, 0xc2, 0x1f,
	0xbc, 0x62, 0xeb, 0xf0, 0x50, 0x45, 0xb7, 0x83,
	0xd3, 0x28, 0x37, 0x46, 0x35, 0x79, 0x04, 0x38,
	0x16, 0xbd, 0x72, 0x2f, 0xbc, 0xf9, 0x7d, 0xb8,
	0x20, 0x12, 0x87, 0xa7, 0xb0, 0x82, 0x7c, 0x3e,
	0x4d, 0xa7, 0x18, 0x04, 0x54, 0x75, 0x19, 0x52,
	0x61, 0x80, 0x57, 0xbe, 0x06, 0x37, 0x9c, 0xde,
	0xb0, 0x93, 0x70, 0x54, 0x3f, 0x15, 0x83, 0xb2,
	0xd3, 0x88, 0xa6, 0xd8, 0x56, 0x9b, 0x91, 0xf3,
	0x00, 0x01, 0x00, 0xcd, 0xcb, 0x97, 0x16, 0x27,
	0xc7, 0xaf, 0xb1, 0xd5, 0xba, 0x90, 0xa1, 0x24,
	0x33, 0x26, 0x95, 0xba, 0xcb, 0x63, 0xa1, 0x56,
	0xb0, 0x23, 0x09, 0x4e, 0xa8, 0x0a, 0x75, 0x02,
	0xd8, 0x3b, 0xc1, 0x05, 0xa0, 0x73, 0x25, 0x26,
	0x3e, 0xb8, 0x5b, 0xca, 0x6a, 0xae, 0x41, 0x22,
	0xc5, 0x9d, 0x88, 0x3f, 0x37, 0x8c, 0x50, 0xcd,
	0x09, 0x32, 0x34, 0x81, 0x98, 0xc1, 0x17, 0xa5,
	0x62, 0x94, 0x8d, 0xcd, 0xd0, 0xc3, 0x08, 0x07,
	0x2f, 0x8d, 0x52, 0x7b, 0xee, 0x81, 0xc5, 0x95,
	0x2c, 0x8b, 0x46, 0x45, 0x5d, 0xf2, 0x62, 0x4d,
	0x2e, 0x22, 0x7f, 0x0d, 0xa5, 0xc8, 0xba, 0x72,
	0xa0, 0x8f, 0xd0, 0x21, 0x45, 0x28, 0xb9, 0xad,
	0xac, 0x96, 0xa6, 0xc7, 0x33, 0x6d, 0xf3, 0x17,
	0xcb, 0x7a, 0x32, 0x4e, 0x4c, 0x83, 0xa7, 0x15,
	0x55, 0x6a, 0xca, 0xc1, 0xff, 0xd8, 0x90, 0xab,
	0xaa, 0xa4, 0x65, 0xb0, 0x25, 0xf7, 0x47, 0x00,
	0x2d, 0x20, 0x19, 0x23, 0xb5, 0x02, 0x39, 0x5a,
	0x69, 0x1c, 0xc4, 0xbc, 0xa3, 0x80, 0xcc, 0x82,
	0xd8, 0x66, 0x63, 0xc5, 0x7e, 0xa3, 0x6c, 0x2a,
	0xb0, 0x7a, 0x02, 0xc6, 0xb3, 0xb7, 0xb0, 0x29,
	0x3f, 0xa3, 0x99, 0xbd, 0x28, 0xf2, 0x8e, 0xd6,
	0x7b, 0xb3, 0xd5, 0x24, 0x12, 0xb9, 0xbb, 0xa5,
	0x68, 0xba, 0x9d, 0xe3, 0x44, 0xa3, 0xeb, 0x64,
	0x13, 0xb8, 0x8a, 0x5e, 0x72, 0x60, 0xad, 0x95,
	0xcb, 0x9d, 0xfb, 0x54, 0x6e, 0x0e, 0x66, 0xb5,
	0xf4, 0x68, 0x25, 0x91, 0xc5, 0x1a, 0x4f, 0x71,
	0x61, 0x50, 0xc8, 0xcb, 0x34, 0xc5, 0xc2, 0x84,
	0xbc, 0x4e, 0x5f, 0x28, 0x6f, 0xbf, 0xbb, 0xc8,
	0xb1, 0xc4, 0x4a, 0x4b, 0xb7, 0x1a, 0x00, 0xa0,
	0x2d, 0xe9, 0x07, 0x39, 0x93, 0xe9, 0x81, 0x51,
	0xe1, 0x34, 0x83, 0xf5, 0x82, 0x9d, 0xc7, 0x47,
	0xf1, 0x61, 0xc8, 0x29, 0x52, 0x4d, 0x51, 0xb7,
	0xa1, 0xeb, 0x49, 0x50, 0x6e, 0xda, 0x72, 0x5d,
	0x69, 0x27, 0x83, 0xca, 0x4c, 0x3b, 0xd4, 0xcc,
	0x7f, 0x26, 0x44, 0x2b, 0x76, 0x41, 0xe1, 0x50,
	0xce, 0x0f, 0x40, 0x0d, 0x8f, 0x00, 0x6e, 0x27,
	0x95, 0x6e, 0x92, 0x1b, 0xaf, 0xe4, 0x9b, 0x5e,
	0x96, 0xec, 0xc7, 0xbf, 0x64, 0x77, 0x75, 0x6d,
	0xda, 0xa8, 0x9c, 0x48, 0xf3, 0x79, 0x80, 0x4e,
	0xed, 0x6f, 0xe0, 0x20, 0x1a, 0x2b, 0x7c, 0xa1,
	0xac, 0x74, 0x42, 0x49, 0xd7, 0xe7, 0x7f, 0x57,
};

/* Plain text */
uint8_t mlkem_512_message[] = {
	0x4d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20,
	0x66, 0x72, 0x6f, 0x6d, 0x20, 0x4d, 0x4c, 0x2d,
	0x4b, 0x45, 0x4d, 0x20, 0x74, 0x65, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x63, 0x74, 0x6f, 0x72, 0x73,
	0x2e
};

/* Cipher text */
uint8_t mlkem_512_cipher[] = {
	0x92, 0x98, 0x11, 0x65, 0x10, 0x78, 0xE3, 0xA4,
	0x34, 0x94, 0xD1, 0x22, 0x56, 0xD6, 0xB1, 0xF6,
	0xF7, 0x51, 0x2A, 0xCA, 0x8E, 0x04, 0x1D, 0x89,
	0xBB, 0xA7, 0x84, 0xEF, 0xAA, 0x70, 0x11, 0x75,
	0xFC, 0x48, 0xA1, 0x7E, 0x4C, 0x99, 0xBD, 0xB8,
	0x05, 0x01, 0xA6, 0xF0, 0x0E, 0x18, 0x0D, 0x75,
	0xF0, 0xBB, 0xBD, 0x6F, 0x28, 0xF4, 0xDC, 0x3A,
	0x58, 0x3D, 0x83, 0x1D, 0x78, 0xF9, 0x0E, 0x06,
	0x5C, 0x0B, 0x77, 0x4C, 0x28, 0x82, 0xE1, 0xA4,
	0x77, 0xAD, 0x14, 0x71, 0x3D, 0x5D, 0x53, 0x41,
	0xCE, 0x4F, 0x36, 0x32, 0x24, 0xD7, 0x09, 0xF0,
	0xB8, 0xF2, 0x95, 0x68, 0x84, 0x5C, 0x62, 0xC3,
	0x77, 0x9A, 0x74, 0xD3, 0x29, 0xC1, 0x88, 0x8B,
	0xBF, 0x05, 0x73, 0x43, 0x30, 0x20, 0xB4, 0xB0,
	0xB3, 0xE1, 0xB5, 0x91, 0xA8, 0xBD, 0x4B, 0x7F,
	0x2E, 0x53, 0x20, 0x55, 0xDE, 0x4D, 0xDB, 0x91,
	0x73, 0xF9, 0x2F, 0x21, 0xF6, 0xAE, 0x6D, 0xEE,
	0x67, 0x37, 0x82, 0x11, 0xE3, 0x0A, 0x5A, 0xB8,
	0x68, 0xC7, 0x22, 0x34, 0x2A, 0xD4, 0x37, 0xAF,
	0x50, 0xE0, 0xDB, 0xED, 0x89, 0x6D, 0x97, 0x0C,
	0xF6, 0x0D, 0x6D, 0x9B, 0x1D, 0x5B, 0xE1, 0xF6,
	0x45, 0x48, 0xF0, 0x75, 0x38, 0xE0, 0x2E, 0x11,
	0x88, 0xEC, 0x8B, 0x51, 0x47, 0x21, 0xF6, 0x6F,
	0x13, 0x3E, 0xA4, 0x46, 0xFE, 0xB0, 0x15, 0x81,
	0x68, 0x9C, 0x0F, 0x15, 0xCB, 0x64, 0x6D, 0x0B,
	0x04, 0x63, 0xBA, 0x67, 0x6C, 0x86, 0xE8, 0xAC,
	0xB3, 0xE5, 0x47, 0x88, 0x27, 0x4A, 0xBC, 0x58,
	0x13, 0x4D, 0x6F, 0xA6, 0x2C, 0xEC, 0x83, 0xB0,
	0x6E, 0x93, 0x7C, 0xC7, 0x6E, 0xBB, 0xBF, 0xE1,
	0x9F, 0xD6, 0x6E, 0xD8, 0x7A, 0xC5, 0xA2, 0x4D,
	0xA2, 0x1C, 0xDA, 0x55, 0x1E, 0xF2, 0x1C, 0x71,
	0x3E, 0x82, 0xD7, 0x92, 0x6F, 0x89, 0x1E, 0x53,
	0x48, 0x00, 0x9E, 0x93, 0xCB, 0xE3, 0x79, 0xB5,
	0x87, 0xDE, 0xF7, 0x84, 0xD6, 0x25, 0x57, 0x07,
	0x47, 0xEE, 0xDB, 0xA8, 0xC2, 0x7E, 0xE8, 0xF9,
	0x56, 0x22, 0x34, 0xE9, 0xD0, 0xC8, 0x3F, 0xF2,
	0xCB, 0xE7, 0x92, 0xE1, 0x03, 0xF4, 0x38, 0x88,
	0xE4, 0xC6, 0xB2, 0x42, 0x6F, 0x08, 0x61, 0x77,
	0x7F, 0x8A, 0xCD, 0x9D, 0x21, 0x3E, 0xE2, 0x18,
	0x52, 0x15, 0x02, 0x62, 0x2F, 0xCA, 0xEE, 0x44,
	0x60, 0x15, 0x6B, 0x76, 0x7D, 0x12, 0xC7, 0x0F,
	0xA0, 0x6C, 0xA1, 0x7B, 0xAE, 0xD6, 0x40, 0x1B,
	0x61, 0x46, 0xB4, 0x47, 0x97, 0x6C, 0xE1, 0x4C,
	0x47, 0x35, 0x29, 0x02, 0x6D, 0x69, 0x5E, 0x32,
	0xF7, 0x7F, 0x9C, 0xFA, 0x87, 0xD3, 0x7D, 0x33,
	0x85, 0x53, 0xF3, 0x8F, 0x0A, 0x41, 0x3C, 0x84,
	0x00, 0x14, 0x04, 0x63, 0xA8, 0xC3, 0xCB, 0xEB,
	0xAF, 0xCB, 0xE1, 0x78, 0xB2, 0x30, 0xC0, 0xA6,
	0x9D, 0x7A, 0x28, 0x0A, 0x60, 0x5F, 0x45, 0x74,
	0x24, 0xBE, 0xD8, 0x0D, 0xCB, 0x3A, 0x1B, 0xD1,
	0x9C, 0x2A, 0x4E, 0x9A, 0xD1, 0x27, 0xCB, 0xAB,
	0x08, 0x72, 0x89, 0xD9, 0xBA, 0x15, 0x01, 0x92,
	0x4F, 0xFA, 0x7A, 0xCA, 0x32, 0xD3, 0x7E, 0xB1,
	0xEB, 0xE7, 0xDF, 0x21, 0x33, 0x4F, 0xBD, 0x5F,
	0xCF, 0x64, 0x5D, 0x36, 0x21, 0x63, 0x34, 0x68,
	0x90, 0x81, 0x15, 0xC7, 0xF7, 0x28, 0x46, 0x42,
	0xC9, 0x78, 0xC0, 0xA3, 0x5F, 0xF8, 0x86, 0x01,
	0xF4, 0x1C, 0x6A, 0x65, 0xA2, 0x0E, 0x01, 0x5B,
	0x29, 0xF4, 0xF8, 0x9E, 0xA5, 0x5C, 0xC0, 0x4B,
	0x74, 0x2C, 0x88, 0xA3, 0x6B, 0xC2, 0xC4, 0xDA,
	0xEF, 0xF2, 0x02, 0xCD, 0x89, 0x0C, 0x42, 0xDA,
	0xE6, 0xBB, 0x67, 0x51, 0x29, 0x4A, 0x32, 0x99,
	0xAC, 0xCC, 0xDC, 0xB6, 0x35, 0x8B, 0xCC, 0xE8,
	0x04, 0x78, 0x69, 0xA8, 0x19, 0x0E, 0xDC, 0x1E,
	0x6C, 0xFF, 0x46, 0xDF, 0x4E, 0x7A, 0x51, 0xC5,
	0x9F, 0x39, 0x6F, 0xC7, 0x96, 0xFE, 0x54, 0xF4,
	0xE2, 0xD8, 0xEB, 0x5C, 0xA1, 0x28, 0x84, 0xB4,
	0xEA, 0x34, 0xA8, 0x18, 0x86, 0x93, 0x8B, 0xE9,
	0xA7, 0xAB, 0x5E, 0xD9, 0x1E, 0x3E, 0x89, 0x8F,
	0x83, 0xB4, 0xEF, 0x8E, 0x21, 0x30, 0xB4, 0xAD,
	0x07, 0xFE, 0x8A, 0x3B, 0x34, 0xAE, 0xDE, 0x5C,
	0xCA, 0xBA, 0x9E, 0xFB, 0x1B, 0x81, 0xEC, 0xD3,
	0x0E, 0x40, 0xED, 0x65, 0x77, 0xF3, 0xC0, 0x56,
	0xDC, 0xD8, 0x46, 0xDB, 0x4F, 0x51, 0x4B, 0x24,
	0x42, 0x3B, 0xC7, 0xE2, 0x32, 0xBA, 0xA7, 0xC0,
	0xA4, 0xFA, 0x14, 0xC4, 0xB7, 0x1B, 0x84, 0x0D,
	0xC9, 0xD1, 0x8E, 0x1C, 0x50, 0xF1, 0xBB, 0xA6,
	0x14, 0xD4, 0x88, 0xD8, 0x6C, 0xCE, 0x44, 0x76,
	0x3A, 0x29, 0xF8, 0x69, 0xBD, 0x5B, 0x53, 0x4B,
	0x65, 0xB8, 0x76, 0x5F, 0xD2, 0x01, 0xE9, 0x55,
	0x83, 0x47, 0x2C, 0xD3, 0x9B, 0xCA, 0x85, 0x75,
	0x60, 0x14, 0x1B, 0x5D, 0x48, 0x97, 0x67, 0x18,
	0x9F, 0x8E, 0x8C, 0x86, 0x31, 0xCF, 0x07, 0x7D,
	0x9E, 0xEB, 0xDE, 0x06, 0x14, 0xFE, 0x00, 0x3F,
	0x9C, 0x09, 0xDF, 0x2B, 0x99, 0x08, 0x7A, 0x4C,
	0xD9, 0xC8, 0x3B, 0x54, 0xF0, 0x34, 0x07, 0x19,
	0xC5, 0x4A, 0x34, 0xBF, 0xB8, 0x31, 0xBF, 0x1C,
	0x6F, 0x5C, 0x07, 0x37, 0x76, 0xD9, 0xBD, 0x3B,
	0xB6, 0xB6, 0x8D, 0x6A, 0x1A, 0xE6, 0xE9, 0xCB,
	0x4D, 0xC3, 0xDC, 0x76, 0x91, 0xE7, 0x6F, 0x11,
	0x66, 0xF7, 0x76, 0xB3, 0x40, 0xAA, 0x51, 0x09,
	0x7C, 0xFE, 0xA2, 0x37, 0xBF, 0xC3, 0x92, 0xFA,
	0x75, 0x44, 0x76, 0xC7, 0x54, 0xC8, 0x91, 0x89,
	0x65, 0x84, 0x03, 0x3E, 0x46, 0x92, 0x1E, 0x67,
	0x8D, 0x8F, 0x52, 0x0D, 0x06, 0x22, 0x18, 0xFC,
	0x0B, 0x92, 0xCD, 0x94, 0xC2, 0x0A, 0x3F, 0x41,
};

/* Shared secret */
uint8_t mlkem_512_sk[] = {
	0x26, 0xCD, 0x28, 0x15, 0xEB, 0x3E, 0x16, 0x56,
	0x84, 0x2D, 0x15, 0xAC, 0x32, 0x33, 0xDA, 0x01,
	0x71, 0x21, 0x82, 0x1B, 0xC7, 0xE3, 0x44, 0xF9,
	0x5E, 0x7A, 0xB9, 0x3A, 0x40, 0xAD, 0x38, 0x6A
};

uint8_t mlkem_512_dk_noseed[] = {
	0x87, 0xCA, 0x19, 0x93, 0xB6, 0x4D, 0x89, 0x32,
	0xAE, 0x3B, 0x22, 0x52, 0x82, 0xA1, 0xB3, 0xC1,
	0x37, 0x65, 0xDC, 0xC1, 0x22, 0x4C, 0x43, 0x77,
	0x33, 0x0A, 0x04, 0xEC, 0xEC, 0x0B, 0x25, 0x05,
	0x40, 0x07, 0x53, 0x82, 0xBE, 0x37, 0x52, 0x53,
	0x12, 0x87, 0x7D, 0x77, 0x69, 0xFD, 0x59, 0x4F,
	0xBD, 0x16, 0x42, 0x82, 0x58, 0x9D, 0xEE, 0x5C,
	0x0F, 0x2C, 0x14, 0x7F, 0xC6, 0x2A, 0x95, 0x42,
	0x10, 0x3C, 0x08, 0xBC, 0xCC, 0xA0, 0x05, 0x82,
	0xE9, 0xC5, 0x26, 0x81, 0xFA, 0xB8, 0x79, 0x78,
	0x5B, 0x3E, 0x79, 0x49, 0x68, 0x44, 0xB3, 0x7B,
	0xF5, 0x26, 0x62, 0x7A, 0x8A, 0x3C, 0xD8, 0x82,
	0x1F, 0x16, 0x1D, 0x92, 0x99, 0xAC, 0xC4, 0xA9,
	0xB9, 0x30, 0x32, 0x6B, 0x6B, 0x67, 0x3D, 0x16,
	0x13, 0x1D, 0xF0, 0x98, 0x94, 0x42, 0x90, 0x68,
	0xFC, 0x65, 0xA3, 0xE5, 0x16, 0x22, 0x09, 0x64,
	0xC0, 0x7D, 0x54, 0x03, 0x47, 0x89, 0xBE, 0xAE,
	0x61, 0x4B, 0x13, 0xA1, 0xCD, 0xAD, 0xBC, 0x20,
	0x5E, 0x36, 0x36, 0x34, 0x41, 0x1D, 0x5B, 0x3A,
	0x26, 0x91, 0x80, 0x75, 0xE0, 0x63, 0x9D, 0xD6,
	0x35, 0xC9, 0x28, 0x81, 0xA4, 0x6E, 0xFB, 0x95,
	0x01, 0x11, 0x8F, 0xCC, 0x18, 0xB3, 0x9A, 0x91,
	0x66, 0xA6, 0x37, 0x6C, 0xEB, 0x71, 0x42, 0x29,
	0xEC, 0x71, 0xA4, 0x99, 0x6D, 0x92, 0x97, 0x9D,
	0x94, 0x64, 0x6E, 0xC3, 0xF0, 0x5D, 0xA5, 0x49,
	0x8F, 0x66, 0xA5, 0x0A, 0x9A, 0xB9, 0xCF, 0x85,
	0x20, 0xA7, 0x28, 0xE1, 0xC2, 0x10, 0x08, 0x72,
	0x58, 0x71, 0x56, 0x3E, 0x7B, 0x47, 0x46, 0x81,
	0x7D, 0x74, 0xFA, 0xB2, 0xB6, 0xF2, 0xA0, 0xE3,
	0x4A, 0x0A, 0x5E, 0x95, 0x2B, 0x32, 0xF1, 0x07,
	0x2C, 0x30, 0x5E, 0x81, 0x84, 0x58, 0xF3, 0x42,
	0x34, 0xDB, 0xAA, 0xC4, 0x06, 0xCB, 0x63, 0x72,
	0xFA, 0x01, 0x86, 0xE8, 0xCA, 0xD7, 0x73, 0x1D,
	0xAC, 0xD8, 0x64, 0x60, 0x66, 0xB1, 0x19, 0xB1,
	0x59, 0xAC, 0x78, 0x21, 0xAB, 0x9A, 0x62, 0xBF,
	0xE8, 0x1B, 0xD1, 0xDC, 0x75, 0x08, 0xA4, 0x9E,
	0x22, 0x54, 0xA8, 0x36, 0x68, 0x4B, 0xB7, 0x22,
	0xA3, 0xBC, 0x04, 0x09, 0xE2, 0xE9, 0x4D, 0xCE,
	0xF5, 0x46, 0x69, 0x1A, 0x47, 0x80, 0xB2, 0xA5,
	0xA0, 0x24, 0xCF, 0x0D, 0x60, 0x95, 0x99, 0x33,
	0xAD, 0x6A, 0x58, 0x7A, 0x56, 0x53, 0x86, 0x44,
	0xA8, 0x3C, 0x1F, 0x92, 0x55, 0x3F, 0x3A, 0x3B,
	0x5F, 0x81, 0xAA, 0x0C, 0xC4, 0x4B, 0x1A, 0xE3,
	0x61, 0x8A, 0xD0, 0x5D, 0x29, 0x87, 0xB6, 0x7D,
	0x1C, 0x85, 0xA5, 0x14, 0xB0, 0xDE, 0x1C, 0x8D,
	0x5C, 0xC1, 0x5C, 0x04, 0xFC, 0x77, 0xAD, 0x03,
	0x55, 0x96, 0xE0, 0xA7, 0x43, 0xB5, 0x95, 0x9A,
	0xD5, 0x22, 0xEF, 0x13, 0x5F, 0x14, 0x9C, 0x0E,
	0x7E, 0x56, 0x89, 0x37, 0x33, 0xC9, 0x00, 0x54,
	0x7E, 0xEA, 0x32, 0x4F, 0x02, 0x85, 0xCD, 0xE5,
	0x9C, 0x25, 0x00, 0x3B, 0xAD, 0xC2, 0x72, 0x3A,
	0x38, 0x66, 0x95, 0xF9, 0xF4, 0x22, 0x1F, 0x50,
	0x19, 0x20, 0x6A, 0x31, 0x03, 0xF8, 0x97, 0x91,
	0xF0, 0x42, 0xBB, 0xC0, 0x86, 0xDE, 0x56, 0x93,
	0xF2, 0x78, 0x9B, 0xC9, 0x98, 0x16, 0x47, 0x83,
	0x67, 0x55, 0x92, 0x4E, 0x5A, 0xFA, 0x5D, 0x88,
	0xF9, 0xC8, 0xEF, 0xA8, 0x21, 0x34, 0x58, 0x5F,
	0xCA, 0xBB, 0x52, 0x98, 0xBC, 0x5C, 0xF5, 0xA7,
	0xAE, 0xC5, 0xFC, 0x78, 0xF9, 0xA5, 0x30, 0x16,
	0x68, 0x9D, 0x62, 0x17, 0x41, 0x7A, 0x95, 0xCF,
	0x27, 0xB6, 0x6D, 0xAE, 0x58, 0xA7, 0x28, 0x8C,
	0x8F, 0xC3, 0x28, 0xAC, 0x06, 0x79, 0x9D, 0x94,
	0xC4, 0x9D, 0xED, 0xB2, 0x61, 0xF4, 0x44, 0x86,
	0xEC, 0x12, 0xC3, 0x13, 0x97, 0xA7, 0x8B, 0x9A,
	0x63, 0x2E, 0xF1, 0x66, 0x08, 0x84, 0x32, 0xF6,
	0x15, 0x3B, 0x91, 0xCA, 0xCE, 0xF7, 0x40, 0x53,
	0xA9, 0x28, 0x11, 0x63, 0xA0, 0x23, 0x2E, 0xC4,
	0x44, 0x28, 0x05, 0x01, 0x02, 0x74, 0xCF, 0x9C,
	0x3A, 0x1A, 0xBC, 0x93, 0x8A, 0x2C, 0xE8, 0x9A,
	0xCA, 0xE1, 0x74, 0x62, 0x03, 0x88, 0xC7, 0x12,
	0x20, 0x96, 0x3C, 0x4D, 0x10, 0x79, 0x28, 0x6F,
	0x7B, 0xA8, 0x1B, 0xFB, 0x5E, 0x57, 0x17, 0xCC,
	0x6D, 0xD0, 0x72, 0x8A, 0xB2, 0x70, 0xA3, 0x0A,
	0x88, 0x03, 0x5B, 0x88, 0x5D, 0x35, 0x12, 0x8E,
	0xAC, 0xC1, 0x81, 0xCA, 0xB7, 0x2B, 0xB1, 0x96,
	0xF6, 0x35, 0xCE, 0xBB, 0x75, 0xAD, 0x0D, 0xD0,
	0xBA, 0x4E, 0x43, 0x5B, 0x31, 0x08, 0x93, 0x32,
	0x72, 0x63, 0x58, 0xC5, 0xA2, 0x70, 0x95, 0x12,
	0x8F, 0xF2, 0xC1, 0x61, 0xEB, 0x22, 0xE9, 0x4A,
	0x65, 0xA5, 0x48, 0x5D, 0x4C, 0x11, 0x78, 0x69,
	0x0B, 0x1F, 0x39, 0x2B, 0x7F, 0x63, 0x77, 0xAD,
	0x96, 0x6B, 0x67, 0x80, 0x90, 0x70, 0x57, 0x2B,
	0xBC, 0x68, 0x9C, 0xAA, 0xB2, 0xD8, 0x3C, 0xBF,
	0xD6, 0x4F, 0xC4, 0x28, 0x65, 0x74, 0x54, 0x84,
	0x43, 0x9A, 0x1C, 0x96, 0x50, 0x02, 0x97, 0xC6,
	0xCF, 0xB0, 0xB1, 0x1D, 0x98, 0x9A, 0x32, 0x94,
	0x73, 0x88, 0x94, 0x78, 0x2D, 0x5F, 0x25, 0x05,
	0x5F, 0xA6, 0x96, 0x7A, 0xC8, 0x3A, 0xDF, 0xA8,
	0x19, 0xB2, 0x53, 0x53, 0x05, 0xF9, 0x31, 0xDC,
	0x58, 0x6C, 0xD1, 0x3A, 0x9B, 0x47, 0x3B, 0x7D,
	0x87, 0xE5, 0xB4, 0xB2, 0xD9, 0x96, 0x2A, 0x59,
	0x90, 0x3C, 0xCC, 0xAD, 0xDC, 0xA2, 0x57, 0x87,
	0x71, 0xC6, 0x7E, 0x5A, 0x49, 0x98, 0xC8, 0x94,
	0x29, 0x30, 0x7B, 0x0E, 0x01, 0x97, 0x53, 0x18,
	0x32, 0x50, 0x73, 0x9E, 0x14, 0x47, 0x97, 0xBD,
	0xCC, 0x22, 0xAB, 0x02, 0x95, 0xD7, 0xC5, 0x32,
	0xDF, 0x17, 0x84, 0x86, 0x77, 0x41, 0x6E, 0x95,
	0x4D, 0x66, 0xF9, 0xB0, 0x9E, 0x12, 0x81, 0x53,
	0x2A, 0x2E, 0x8F, 0x0C, 0x6A, 0xBE, 0x00, 0x37,
	0xE7, 0xE8, 0x11, 0x90, 0x97, 0xC9, 0xEC, 0x84,
	0x5A, 0xA0, 0x69, 0x85, 0xC0, 0x88, 0x55, 0x2C,
	0x41, 0xB6, 0x15, 0x17, 0x36, 0x42, 0xC1, 0x02,
	0x51, 0xC0, 0x6E, 0x91, 0xA2, 0x5C, 0x24, 0x3C,
	0x02, 0x63, 0xB6, 0x75, 0xC7, 0x20, 0x7D, 0x58,
	0x70, 0x1D, 0x13, 0xA5, 0x2E, 0xAB, 0x92, 0x56,
	0x5E, 0xF1, 0xA1, 0xDE, 0xFB, 0xAE, 0xFE, 0x4C,
	0x0B, 0x03, 0xF5, 0x04, 0x44, 0xA3, 0xBE, 0x20,
	0xB1, 0x71, 0x31, 0x0B, 0xA3, 0xF2, 0x08, 0x52,
	0xF3, 0xA2, 0xA1, 0x8E, 0x72, 0x29, 0x40, 0x70,
	0x15, 0x64, 0x60, 0x52, 0xD2, 0xE7, 0x3A, 0xBE,
	0xE3, 0x18, 0xD7, 0x55, 0x89, 0x9C, 0x78, 0x4A,
	0x6F, 0xB5, 0xAA, 0xB1, 0x7C, 0x90, 0x2C, 0xB6,
	0x8A, 0xFA, 0x36, 0x57, 0xB9, 0x01, 0x2E, 0x1B,
	0xB8, 0xB6, 0xAA, 0xC2, 0x68, 0x9A, 0xA4, 0x8C,
	0x2D, 0x42, 0x62, 0xBB, 0x29, 0x63, 0xB0, 0x6A,
	0x24, 0x22, 0x82, 0x54, 0x1C, 0xF6, 0x14, 0x19,
	0xD9, 0x0E, 0x2B, 0xD3, 0x66, 0xBE, 0xE7, 0xA4,
	0x36, 0x9B, 0x72, 0x41, 0xA6, 0x6B, 0xF2, 0x64,
	0x5D, 0x8E, 0x90, 0x4C, 0x3F, 0x7B, 0x30, 0x0C,
	0x04, 0x1D, 0x56, 0x87, 0x39, 0x3D, 0x2C, 0xA9,
	0x29, 0x75, 0xCB, 0xFD, 0xC0, 0x18, 0xDD, 0xC6,
	0x0A, 0xF5, 0x62, 0x00, 0x9E, 0xB0, 0x88, 0x8E,
	0x5A, 0x05, 0x76, 0x24, 0x6A, 0xCE, 0x74, 0x5B,
	0xB1, 0x63, 0x40, 0x8B, 0x5A, 0x9A, 0xE3, 0xC1,
	0x94, 0xA8, 0x5C, 0x21, 0x90, 0x7B, 0x37, 0xB1,
	0x62, 0x8E, 0xCD, 0x9A, 0x15, 0xEC, 0x20, 0x24,
	0x87, 0x30, 0x27, 0x34, 0x44, 0xB0, 0xA2, 0xF4,
	0x54, 0xF9, 0xB4, 0x73, 0x0F, 0x33, 0x91, 0x50,
	0x47, 0x6E, 0xE0, 0x70, 0x98, 0xF6, 0xBC, 0x1B,
	0x97, 0xCC, 0x1B, 0xD3, 0xB8, 0xC1, 0xA2, 0xEB,
	0x0E, 0x50, 0xA7, 0x82, 0xF2, 0x11, 0x5D, 0xF6,
	0x17, 0x49, 0x6F, 0x6C, 0x6F, 0x8C, 0x09, 0xB0,
	0x5F, 0x88, 0x8D, 0x9E, 0x93, 0x3D, 0x28, 0x77,
	0x46, 0xC6, 0x31, 0xB3, 0x10, 0x87, 0x26, 0xB0,
	0xC4, 0xA7, 0xC8, 0x8B, 0x09, 0xC8, 0x60, 0xAC,
	0xD7, 0x52, 0x35, 0x70, 0xC8, 0x02, 0xBD, 0x38,
	0x72, 0x43, 0x16, 0x2D, 0x12, 0x8C, 0xA2, 0x29,
	0x4C, 0x83, 0x43, 0x18, 0xCC, 0x21, 0xFB, 0x14,
	0xD2, 0xAB, 0x37, 0x3F, 0x22, 0x4E, 0x3F, 0xD4,
	0x98, 0x43, 0x85, 0x95, 0x09, 0xF4, 0xCA, 0x1A,
	0x1A, 0x56, 0x6C, 0x05, 0x67, 0x88, 0xA3, 0xAA,
	0x48, 0x4A, 0xAA, 0xBD, 0xF1, 0xA0, 0x8F, 0x1B,
	0x44, 0xC7, 0x56, 0xAB, 0x2A, 0x0C, 0x8B, 0xC5,
	0x85, 0x1E, 0xE2, 0xEB, 0x23, 0x03, 0x4C, 0x2E,
	0xAA, 0x5A, 0xC1, 0x5F, 0x89, 0x75, 0x1B, 0xD5,
	0xCA, 0xE1, 0xD5, 0x90, 0x89, 0xD5, 0x00, 0x2B,
	0xB1, 0xB3, 0xBF, 0xCA, 0x4A, 0xDE, 0x09, 0x82,
	0xAD, 0x67, 0x83, 0xE9, 0xD3, 0x7F, 0xB3, 0xE3,
	0x20, 0xAC, 0x98, 0x55, 0xDF, 0x66, 0x6B, 0xFD,
	0x7C, 0x6B, 0x87, 0xA9, 0xA4, 0x6E, 0x25, 0x97,
	0x12, 0x56, 0x11, 0x61, 0x50, 0x7E, 0x17, 0x35,
	0x98, 0xF9, 0x88, 0xAC, 0xB6, 0xF9, 0xAB, 0x3A,
	0x10, 0x92, 0x63, 0x24, 0x46, 0x88, 0xC7, 0x08,
	0x78, 0x75, 0x8F, 0xF4, 0xD4, 0x31, 0x3B, 0x76,
	0x64, 0xF0, 0xF5, 0x10, 0xE9, 0x13, 0xCA, 0x01,
	0xC5, 0x2B, 0x3A, 0x1B, 0x46, 0x53, 0x51, 0x44,
	0xE2, 0xDB, 0x0C, 0xAC, 0xE8, 0xA6, 0x46, 0x66,
	0x00, 0xCF, 0x2A, 0x87, 0x83, 0x50, 0x07, 0xE1,
	0x6A, 0xA5, 0x07, 0x80, 0x1D, 0x86, 0xB7, 0x38,
	0x5E, 0x66, 0x9B, 0xFA, 0xF6, 0xBA, 0x1D, 0xF5,
	0x68, 0x31, 0x63, 0xC2, 0xD5, 0x70, 0x53, 0x2F,
	0xD7, 0x43, 0x90, 0x54, 0x64, 0x0F, 0x24, 0x4C,
	0x52, 0x87, 0x13, 0x59, 0xD2, 0x84, 0x2F, 0xC3,
	0x37, 0xA0, 0x60, 0x03, 0x3A, 0xF0, 0x5E, 0xAA,
	0x00, 0x1C, 0x34, 0xFC, 0x7B, 0xD8, 0xF9, 0x10,
	0x29, 0xE4, 0x6C, 0x29, 0x43, 0x36, 0x27, 0x64,
	0x5D, 0x67, 0x86, 0x64, 0x21, 0xD6, 0x61, 0xAF,
	0x25, 0x74, 0x80, 0x53, 0x2B, 0x88, 0x78, 0x50,
	0xDC, 0x49, 0x9F, 0xFE, 0xD5, 0xB1, 0x40, 0x98,
	0xA0, 0x33, 0x72, 0x5E, 0x82, 0x0A, 0x5B, 0xE1,
	0x40, 0x0A, 0x0C, 0xB7, 0x03, 0x74, 0x1C, 0xA7,
	0x4B, 0x47, 0x86, 0x73, 0xAA, 0xCF, 0x85, 0x16,
	0x6E, 0xE8, 0xA1, 0x84, 0xDB, 0x2C, 0x58, 0x54,
	0x9C, 0x22, 0x40, 0xB8, 0x30, 0x8C, 0x27, 0xBE,
	0xBA, 0x40, 0xC3, 0xB9, 0xD7, 0x29, 0xED, 0xB1,
	0x8C, 0xC2, 0x06, 0x8E, 0xCD, 0xB2, 0x7D, 0xA2,
	0x2B, 0x2C, 0xD3, 0xE7, 0xA0, 0xBA, 0xA5, 0x30,
	0xC5, 0x19, 0x3C, 0xD3, 0xC8, 0x6B, 0xF8, 0x6A,
	0x44, 0xF0, 0x79, 0x9E, 0x51, 0x55, 0xB0, 0x9B,
	0xE2, 0x50, 0x98, 0x85, 0x23, 0xB3, 0xA7, 0x31,
	0xDD, 0x89, 0x1B, 0xC2, 0x00, 0x60, 0x11, 0xA0,
	0x65, 0xC0, 0xAB, 0x57, 0xF1, 0xA6, 0xC1, 0x78,
	0x89, 0x55, 0xE0, 0x13, 0x5F, 0xA5, 0xCA, 0x8F,
	0x3E, 0x52, 0xC7, 0x5D, 0x37, 0x16, 0x97, 0x3A,
	0x2F, 0xB2, 0x2C, 0x0E, 0xB1, 0x7C, 0x1B, 0x32,
	0x85, 0x29, 0xD9, 0xFA, 0x76, 0x56, 0xD7, 0x4D,
	0xEA, 0x74, 0x0D, 0x9F, 0x07, 0x97, 0x77, 0xC3,
	0x6C, 0x17, 0xA1, 0x9C, 0x19, 0x58, 0x9E, 0x84,
	0xD2, 0xB8, 0xE4, 0xD1, 0xBC, 0x31, 0x07, 0xCB,
	0xD2, 0x78, 0x14, 0x22, 0x48, 0x35, 0x48, 0x44,
	0x6C, 0x89, 0x93, 0x14, 0x77, 0x44, 0xAA, 0x9E,
	0xC1, 0xC5, 0x93, 0xEC, 0x2D, 0x5B, 0xAA, 0xC8,
	0x6A, 0x0A, 0xF6, 0x4A, 0x85, 0xE9, 0x09, 0xDF,
	0x8E, 0x28, 0x16, 0x60, 0x5D, 0x20, 0xB4, 0xE3,
	0x82, 0xB3, 0x0B, 0xBB, 0x61, 0xBF, 0x3A, 0x5F,
	0x82, 0x1A, 0x0B, 0x5D, 0xBA, 0x9A, 0xD3, 0xE7,
	0xE5, 0xBD, 0x1B, 0x37, 0xA7, 0x5E, 0x0F, 0x09,
	0x29, 0x74, 0xE8, 0x46, 0xE8, 0xC3, 0x7C, 0x45,
	0x48, 0x7D, 0x60, 0x73, 0x9F, 0x99, 0x35, 0x17,
	0x19, 0xA5, 0x39, 0x47, 0x23, 0x26, 0x2B, 0x3B,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

uint8_t mlkem_512_ek_noseed[] = {
	0xDF, 0x17, 0x84, 0x86, 0x77, 0x41, 0x6E, 0x95,
	0x4D, 0x66, 0xF9, 0xB0, 0x9E, 0x12, 0x81, 0x53,
	0x2A, 0x2E, 0x8F, 0x0C, 0x6A, 0xBE, 0x00, 0x37,
	0xE7, 0xE8, 0x11, 0x90, 0x97, 0xC9, 0xEC, 0x84,
	0x5A, 0xA0, 0x69, 0x85, 0xC0, 0x88, 0x55, 0x2C,
	0x41, 0xB6, 0x15, 0x17, 0x36, 0x42, 0xC1, 0x02,
	0x51, 0xC0, 0x6E, 0x91, 0xA2, 0x5C, 0x24, 0x3C,
	0x02, 0x63, 0xB6, 0x75, 0xC7, 0x20, 0x7D, 0x58,
	0x70, 0x1D, 0x13, 0xA5, 0x2E, 0xAB, 0x92, 0x56,
	0x5E, 0xF1, 0xA1, 0xDE, 0xFB, 0xAE, 0xFE, 0x4C,
	0x0B, 0x03, 0xF5, 0x04, 0x44, 0xA3, 0xBE, 0x20,
	0xB1, 0x71, 0x31, 0x0B, 0xA3, 0xF2, 0x08, 0x52,
	0xF3, 0xA2, 0xA1, 0x8E, 0x72, 0x29, 0x40, 0x70,
	0x15, 0x64, 0x60, 0x52, 0xD2, 0xE7, 0x3A, 0xBE,
	0xE3, 0x18, 0xD7, 0x55, 0x89, 0x9C, 0x78, 0x4A,
	0x6F, 0xB5, 0xAA, 0xB1, 0x7C, 0x90, 0x2C, 0xB6,
	0x8A, 0xFA, 0x36, 0x57, 0xB9, 0x01, 0x2E, 0x1B,
	0xB8, 0xB6, 0xAA, 0xC2, 0x68, 0x9A, 0xA4, 0x8C,
	0x2D, 0x42, 0x62, 0xBB, 0x29, 0x63, 0xB0, 0x6A,
	0x24, 0x22, 0x82, 0x54, 0x1C, 0xF6, 0x14, 0x19,
	0xD9, 0x0E, 0x2B, 0xD3, 0x66, 0xBE, 0xE7, 0xA4,
	0x36, 0x9B, 0x72, 0x41, 0xA6, 0x6B, 0xF2, 0x64,
	0x5D, 0x8E, 0x90, 0x4C, 0x3F, 0x7B, 0x30, 0x0C,
	0x04, 0x1D, 0x56, 0x87, 0x39, 0x3D, 0x2C, 0xA9,
	0x29, 0x75, 0xCB, 0xFD, 0xC0, 0x18, 0xDD, 0xC6,
	0x0A, 0xF5, 0x62, 0x00, 0x9E, 0xB0, 0x88, 0x8E,
	0x5A, 0x05, 0x76, 0x24, 0x6A, 0xCE, 0x74, 0x5B,
	0xB1, 0x63, 0x40, 0x8B, 0x5A, 0x9A, 0xE3, 0xC1,
	0x94, 0xA8, 0x5C, 0x21, 0x90, 0x7B, 0x37, 0xB1,
	0x62, 0x8E, 0xCD, 0x9A, 0x15, 0xEC, 0x20, 0x24,
	0x87, 0x30, 0x27, 0x34, 0x44, 0xB0, 0xA2, 0xF4,
	0x54, 0xF9, 0xB4, 0x73, 0x0F, 0x33, 0x91, 0x50,
	0x47, 0x6E, 0xE0, 0x70, 0x98, 0xF6, 0xBC, 0x1B,
	0x97, 0xCC, 0x1B, 0xD3, 0xB8, 0xC1, 0xA2, 0xEB,
	0x0E, 0x50, 0xA7, 0x82, 0xF2, 0x11, 0x5D, 0xF6,
	0x17, 0x49, 0x6F, 0x6C, 0x6F, 0x8C, 0x09, 0xB0,
	0x5F, 0x88, 0x8D, 0x9E, 0x93, 0x3D, 0x28, 0x77,
	0x46, 0xC6, 0x31, 0xB3, 0x10, 0x87, 0x26, 0xB0,
	0xC4, 0xA7, 0xC8, 0x8B, 0x09, 0xC8, 0x60, 0xAC,
	0xD7, 0x52, 0x35, 0x70, 0xC8, 0x02, 0xBD, 0x38,
	0x72, 0x43, 0x16, 0x2D, 0x12, 0x8C, 0xA2, 0x29,
	0x4C, 0x83, 0x43, 0x18, 0xCC, 0x21, 0xFB, 0x14,
	0xD2, 0xAB, 0x37, 0x3F, 0x22, 0x4E, 0x3F, 0xD4,
	0x98, 0x43, 0x85, 0x95, 0x09, 0xF4, 0xCA, 0x1A,
	0x1A, 0x56, 0x6C, 0x05, 0x67, 0x88, 0xA3, 0xAA,
	0x48, 0x4A, 0xAA, 0xBD, 0xF1, 0xA0, 0x8F, 0x1B,
	0x44, 0xC7, 0x56, 0xAB, 0x2A, 0x0C, 0x8B, 0xC5,
	0x85, 0x1E, 0xE2, 0xEB, 0x23, 0x03, 0x4C, 0x2E,
	0xAA, 0x5A, 0xC1, 0x5F, 0x89, 0x75, 0x1B, 0xD5,
	0xCA, 0xE1, 0xD5, 0x90, 0x89, 0xD5, 0x00, 0x2B,
	0xB1, 0xB3, 0xBF, 0xCA, 0x4A, 0xDE, 0x09, 0x82,
	0xAD, 0x67, 0x83, 0xE9, 0xD3, 0x7F, 0xB3, 0xE3,
	0x20, 0xAC, 0x98, 0x55, 0xDF, 0x66, 0x6B, 0xFD,
	0x7C, 0x6B, 0x87, 0xA9, 0xA4, 0x6E, 0x25, 0x97,
	0x12, 0x56, 0x11, 0x61, 0x50, 0x7E, 0x17, 0x35,
	0x98, 0xF9, 0x88, 0xAC, 0xB6, 0xF9, 0xAB, 0x3A,
	0x10, 0x92, 0x63, 0x24, 0x46, 0x88, 0xC7, 0x08,
	0x78, 0x75, 0x8F, 0xF4, 0xD4, 0x31, 0x3B, 0x76,
	0x64, 0xF0, 0xF5, 0x10, 0xE9, 0x13, 0xCA, 0x01,
	0xC5, 0x2B, 0x3A, 0x1B, 0x46, 0x53, 0x51, 0x44,
	0xE2, 0xDB, 0x0C, 0xAC, 0xE8, 0xA6, 0x46, 0x66,
	0x00, 0xCF, 0x2A, 0x87, 0x83, 0x50, 0x07, 0xE1,
	0x6A, 0xA5, 0x07, 0x80, 0x1D, 0x86, 0xB7, 0x38,
	0x5E, 0x66, 0x9B, 0xFA, 0xF6, 0xBA, 0x1D, 0xF5,
	0x68, 0x31, 0x63, 0xC2, 0xD5, 0x70, 0x53, 0x2F,
	0xD7, 0x43, 0x90, 0x54, 0x64, 0x0F, 0x24, 0x4C,
	0x52, 0x87, 0x13, 0x59, 0xD2, 0x84, 0x2F, 0xC3,
	0x37, 0xA0, 0x60, 0x03, 0x3A, 0xF0, 0x5E, 0xAA,
	0x00, 0x1C, 0x34, 0xFC, 0x7B, 0xD8, 0xF9, 0x10,
	0x29, 0xE4, 0x6C, 0x29, 0x43, 0x36, 0x27, 0x64,
	0x5D, 0x67, 0x86, 0x64, 0x21, 0xD6, 0x61, 0xAF,
	0x25, 0x74, 0x80, 0x53, 0x2B, 0x88, 0x78, 0x50,
	0xDC, 0x49, 0x9F, 0xFE, 0xD5, 0xB1, 0x40, 0x98,
	0xA0, 0x33, 0x72, 0x5E, 0x82, 0x0A, 0x5B, 0xE1,
	0x40, 0x0A, 0x0C, 0xB7, 0x03, 0x74, 0x1C, 0xA7,
	0x4B, 0x47, 0x86, 0x73, 0xAA, 0xCF, 0x85, 0x16,
	0x6E, 0xE8, 0xA1, 0x84, 0xDB, 0x2C, 0x58, 0x54,
	0x9C, 0x22, 0x40, 0xB8, 0x30, 0x8C, 0x27, 0xBE,
	0xBA, 0x40, 0xC3, 0xB9, 0xD7, 0x29, 0xED, 0xB1,
	0x8C, 0xC2, 0x06, 0x8E, 0xCD, 0xB2, 0x7D, 0xA2,
	0x2B, 0x2C, 0xD3, 0xE7, 0xA0, 0xBA, 0xA5, 0x30,
	0xC5, 0x19, 0x3C, 0xD3, 0xC8, 0x6B, 0xF8, 0x6A,
	0x44, 0xF0, 0x79, 0x9E, 0x51, 0x55, 0xB0, 0x9B,
	0xE2, 0x50, 0x98, 0x85, 0x23, 0xB3, 0xA7, 0x31,
	0xDD, 0x89, 0x1B, 0xC2, 0x00, 0x60, 0x11, 0xA0,
	0x65, 0xC0, 0xAB, 0x57, 0xF1, 0xA6, 0xC1, 0x78,
	0x89, 0x55, 0xE0, 0x13, 0x5F, 0xA5, 0xCA, 0x8F,
	0x3E, 0x52, 0xC7, 0x5D, 0x37, 0x16, 0x97, 0x3A,
	0x2F, 0xB2, 0x2C, 0x0E, 0xB1, 0x7C, 0x1B, 0x32,
	0x85, 0x29, 0xD9, 0xFA, 0x76, 0x56, 0xD7, 0x4D,
	0xEA, 0x74, 0x0D, 0x9F, 0x07, 0x97, 0x77, 0xC3,
	0x6C, 0x17, 0xA1, 0x9C, 0x19, 0x58, 0x9E, 0x84,
	0xD2, 0xB8, 0xE4, 0xD1, 0xBC, 0x31, 0x07, 0xCB,
	0xD2, 0x78, 0x14, 0x22, 0x48, 0x35, 0x48, 0x44,
	0x6C, 0x89, 0x93, 0x14, 0x77, 0x44, 0xAA, 0x9E,
	0xC1, 0xC5, 0x93, 0xEC, 0x2D, 0x5B, 0xAA, 0xC8,
	0x6A, 0x0A, 0xF6, 0x4A, 0x85, 0xE9, 0x09, 0xDF,
	0x8E, 0x28, 0x16, 0x60, 0x5D, 0x20, 0xB4, 0xE3,
	0x82, 0xB3, 0x0B, 0xBB, 0x61, 0xBF, 0x3A, 0x5F,
	0x82, 0x1A, 0x0B, 0x5D, 0xBA, 0x9A, 0xD3, 0xE7,
};

struct crypto_testsuite_mlkem_params mlkem_keygen_test_vectors[] = {
	{
		.name = "mlkem_512_keygen",
		.type = RTE_CRYPTO_ML_KEM_512,
		.d = {
			.data = mlkem_512_d,
			.length = sizeof(mlkem_512_d),
		},
		.z = {
			.data = mlkem_512_z,
			.length = sizeof(mlkem_512_z),
		},
	},
	{
		.name = "mlkem_512_keygen (deterministic)",
		.type = RTE_CRYPTO_ML_KEM_512,
		.d = {
			.data = (uint8_t [32]) {0},
			.length = 32,
		},
		.z = {
			.data = (uint8_t [32]) {0},
			.length = 32,
		},
		.dk = {
			.data = mlkem_512_dk_noseed,
			.length = sizeof(mlkem_512_dk_noseed),
		},
		.ek = {
			.data = mlkem_512_ek_noseed,
			.length = sizeof(mlkem_512_ek_noseed),
		}
	},
};

struct crypto_testsuite_mlkem_params mlkem_encap_test_vectors[] = {
	{
		.name = "mlkem_512_encap",
		.type = RTE_CRYPTO_ML_KEM_512,
		.dk = {
			.data = mlkem_512_dk,
			.length = sizeof(mlkem_512_dk),
		},
		.ek = {
			.data = mlkem_512_ek,
			.length = sizeof(mlkem_512_ek),
		},
		.message = {
			.data = NULL,
			.length = 0,
		},
	},
	{
		.name = "mlkem_512_encap (deterministic)",
		.type = RTE_CRYPTO_ML_KEM_512,
		.dk = {
			.data = mlkem_512_dk,
			.length = sizeof(mlkem_512_dk),
		},
		.ek = {
			.data = mlkem_512_ek,
			.length = sizeof(mlkem_512_ek),
		},
		.message = {
			.data = mlkem_512_message,
			.length = 32,
		},
		.cipher = {
			.data = mlkem_512_cipher,
			.length = sizeof(mlkem_512_cipher),
		},
		.sk = {
			.data = mlkem_512_sk,
			.length = sizeof(mlkem_512_sk),
		},
	},
};

struct crypto_testsuite_mlkem_params mlkem_decap_test_vectors[] = {
	{
		.name = "mlkem_512_decap",
		.type = RTE_CRYPTO_ML_KEM_512,
		.cipher = {
			.data = mlkem_512_cipher,
			.length = sizeof(mlkem_512_cipher),
		},
		.dk = {
			.data = mlkem_512_dk,
			.length = sizeof(mlkem_512_dk),
		},
		.sk = {
			.data = mlkem_512_sk,
			.length = sizeof(mlkem_512_sk),
		},
	}
};

#endif /* __TEST_CRYPTODEV_ML_KEM_TEST_VECTORS_H__ */
