/************************************ */
/* Rule Set Based Access Control      */
/*                                    */
/* Author and (c) 1999-2003:          */
/*   Amon Ott <ao@rsbac.org>          */
/*                                    */
/* Debug and logging functions for all parts */
/*                                    */
/* Last modified: 01/Jul/2003         */
/************************************ */
 
#include <asm/uaccess.h>
#include <rsbac/types.h>
#include <rsbac/aci.h>
#include <rsbac/aci_data_structures.h>
#include <rsbac/debug.h>
#include <rsbac/error.h>
#include <rsbac/proc_fs.h>
#include <rsbac/getname.h>
#include <rsbac/net_getname.h>
#include <rsbac/adf.h>
#include <rsbac/rkmem.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/console.h>
#include <linux/in.h>

#ifdef CONFIG_RSBAC_DEBUG
/* Boolean debug switch for data structures */
int  rsbac_debug_ds = 0;

/* Boolean debug switch for writing of data structures */
int  rsbac_debug_write = 0;

/* Boolean debug switch for AEF */
EXPORT_SYMBOL(rsbac_debug_aef);
int  rsbac_debug_aef = 0;

/* Boolean debug switch for NO_WRITE (global) */
int  rsbac_debug_no_write = 0;

/* Boolean debug switch for stack debugging */
int  rsbac_debug_stack = 0;

/* Boolean debug switch for generic lists */
int  rsbac_debug_lists = 0;

#ifdef CONFIG_RSBAC_NET
int rsbac_debug_ds_net = 0;
int rsbac_debug_adf_net = 0;
int rsbac_debug_aef_net = 0;
#endif

#if defined(CONFIG_RSBAC_MAC)
/* Boolean debug switch for MAC data structures */
int  rsbac_debug_ds_mac = 0;
/* Boolean debug switch for MAC syscalls / AEF */
int  rsbac_debug_aef_mac = 0;
/* Boolean debug switch for MAC decisions / ADF */
int  rsbac_debug_adf_mac = 0;
#endif

#if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
/* Boolean debug switch for PM data structures */
int  rsbac_debug_ds_pm = 0;
/* Boolean debug switch for PM syscalls / AEF */
int  rsbac_debug_aef_pm = 0;
/* Boolean debug switch for PM decisions / ADF */
int  rsbac_debug_adf_pm = 0;
#endif

#if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
/* Boolean debug switch for MS decisions / ADF */
int  rsbac_debug_adf_ms = 0;
#endif

#if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
/* Boolean debug switch for RC data structures */
int  rsbac_debug_ds_rc = 0;
/* Boolean debug switch for RC syscalls / AEF */
int  rsbac_debug_aef_rc = 0;
/* Boolean debug switch for RC decisions / ADF */
int  rsbac_debug_adf_rc = 0;
#endif

#if defined(CONFIG_RSBAC_AUTH) || defined(CONFIG_RSBAC_AUTH_MAINT)
/* Boolean debug switch for AUTH data structures */
int  rsbac_debug_ds_auth = 0;
/* Boolean debug switch for AUTH syscalls / AEF */
int  rsbac_debug_aef_auth = 0;
/* Boolean debug switch for AUTH decisions / ADF */
int  rsbac_debug_adf_auth = 0;
#endif

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
/* Boolean debug switch for REG */
int  rsbac_debug_reg = 0;
#endif

#if defined(CONFIG_RSBAC_ACL) || defined(CONFIG_RSBAC_ACL_MAINT)
/* Boolean debug switch for ACL data structures */
int  rsbac_debug_ds_acl = 0;
/* Boolean debug switch for ACL syscalls / AEF */
int  rsbac_debug_aef_acl = 0;
/* Boolean debug switch for ACL decisions / ADF */
int  rsbac_debug_adf_acl = 0;
#endif

#if defined(CONFIG_RSBAC_JAIL)
/* Boolean debug switch for JAIL syscalls / AEF */
int  rsbac_debug_aef_jail = 0;
/* Boolean debug switch for JAIL decisions / ADF */
int  rsbac_debug_adf_jail = 0;
#endif

#if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
  int  rsbac_debug_auto = 0;
#endif /* CONFIG_RSBAC_AUTO_WRITE > 0 */

#endif /* DEBUG */

#if defined(CONFIG_RSBAC_AUTH) || defined(CONFIG_RSBAC_AUTH_MAINT)
/* Boolean switch for AUTH init: set may_setuid for /bin/login */
int  rsbac_auth_enable_login = 0;
#endif

/* Suppress default list creation for complete restore */
int  rsbac_no_defaults = 0;

static rsbac_list_handle_t log_levels_handle = NULL;

#ifdef CONFIG_RSBAC_SOFTMODE
/* Boolean switch for RSBAC soft mode */
int  rsbac_softmode = 0;
#ifdef CONFIG_RSBAC_SOFTMODE_IND
int  rsbac_ind_softmode[SW_NONE] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0};
#endif

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_in_softmode);
#endif
int rsbac_in_softmode(void)
  {
    return rsbac_softmode;
  }
#endif

#if defined(CONFIG_RSBAC_CAP_PROC_HIDE)
int  rsbac_cap_process_hiding = 0;
#endif

#ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_FULL
/* Boolean switch for disabling Linux DAC */
int  rsbac_dac_disable = 0;

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_dac_is_disabled);
#endif
int rsbac_dac_is_disabled(void)
  {
    return rsbac_dac_disable;
  }
#endif

/* Boolean switch for no syslog option*/
#ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
int  rsbac_nosyslog = 0;
#endif

/* Boolean switch for delayed init option*/
#ifdef CONFIG_RSBAC_INIT_DELAY
int  rsbac_delay_init = 1;
kdev_t rsbac_delayed_root = MKDEV(0,0);
#endif

/* Array of Boolean debug switches for ADF */
int  rsbac_debug_adf_default = 1;
rsbac_log_entry_t  rsbac_log_levels[R_NONE+1];

boolean rsbac_debug_adf_dirty = FALSE;

/* variables for rsbac_logging */
#if defined(CONFIG_RSBAC_RMSG)
#include <linux/poll.h>
#include <linux/smp.h>
#define RLOG_BUF_LEN (1 << 14)
#define RLOG_BUF_MASK (RLOG_BUF_LEN-1)
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static struct wait_queue * rlog_wait = NULL;
static u_long rlog_size = 0;
#else
DECLARE_WAIT_QUEUE_HEAD(rlog_wait);
static unsigned long log_end = 0;		/* Index into log_buf: most-recently-written-char + 1 */
#endif
static char rlog_buf[RLOG_BUF_LEN];
#define RLOG_BUF(idx) (rlog_buf[(idx) & RLOG_BUF_MASK])
static unsigned long log_start = 0;
static unsigned long logged_chars = 0;

#if defined(CONFIG_RSBAC_LOG_REMOTE)
#define REMOTE_RLOG_BUF_LEN (16384)
#define REMOTE_RLOG_BUF_MASK (REMOTE_RLOG_BUF_LEN-1)
static DECLARE_WAIT_QUEUE_HEAD(rsbaclogd_wait);
#ifndef CONFIG_RSBAC_LOG_REMOTE_SYNC
static struct timer_list rsbac_log_remote_timer;
u_int rsbac_log_remote_interval = CONFIG_RSBAC_LOG_INTERVAL;
#endif
rsbac_pid_t rsbaclogd_pid=0;
#define REMOTE_SEND_BUF_LEN 1024
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static u_long remote_rlog_size = 0;
#else
static unsigned long remote_log_end;
#endif
static char remote_rlog_buf[REMOTE_RLOG_BUF_LEN];
static unsigned long remote_log_start = 0;
static unsigned long remote_logged_chars = 0;
#define REMOTE_RLOG_BUF(idx) (remote_rlog_buf[(idx) & REMOTE_RLOG_BUF_MASK])
#endif

#endif

void  rsbac_adf_log_switch(rsbac_adf_request_int_t request,
                           enum rsbac_target_t target,
                           rsbac_enum_t value)
  {
    if(   (request < R_NONE)
       && (target <= T_NONE)
       && (value <= LL_full)
      )
      {
        rsbac_log_levels[request][target] = value;
        if(log_levels_handle)
          rsbac_list_add(log_levels_handle, &request, rsbac_log_levels[request]);
      }
  };

int rsbac_get_adf_log(rsbac_adf_request_int_t request,
                      enum rsbac_target_t target,
                      u_int * value_p)
  {
    if(   (request < R_NONE)
       && (target <= T_NONE)
      )
      {
        *value_p = rsbac_log_levels[request][target];
        return 0;
      }
    else
      return -RSBAC_EINVALIDVALUE;
  }

inline boolean rsbac_parse_koptions(char * line)
  {
    /* RSBAC: suppress defaults? */
    if (!strcmp(line,"rsbac_no_defaults"))
      {
        rsbac_no_defaults = 1;
        return(TRUE);
      }
    #if defined(CONFIG_RSBAC_AUTH) || defined(CONFIG_RSBAC_AUTH_MAINT)
    /* RSBAC: AUTH - set auth_may_setuid for /bin/login? */
    if (!strcmp(line,"rsbac_auth_enable_login"))
      {
        rsbac_auth_enable_login = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_SOFTMODE)
    /* RSBAC: softmode on? */
    if (!strcmp(line,"rsbac_softmode"))
      {
        rsbac_softmode = 1;
        return(TRUE);
      }
    #if defined(CONFIG_RSBAC_SOFTMODE_IND)
    /* RSBAC: softmode on for a module? */
    if (!strcmp(line,"rsbac_softmode_mac"))
      {
        rsbac_ind_softmode[MAC] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_fc"))
      {
        rsbac_ind_softmode[FC] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_sim"))
      {
        rsbac_ind_softmode[SIM] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_pm"))
      {
        rsbac_ind_softmode[PM] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_ms"))
      {
        rsbac_ind_softmode[MS] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_ff"))
      {
        rsbac_ind_softmode[FF] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_rc"))
      {
        rsbac_ind_softmode[RC] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_auth"))
      {
        rsbac_ind_softmode[AUTH] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_reg"))
      {
        rsbac_ind_softmode[REG] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_acl"))
      {
        rsbac_ind_softmode[ACL] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_cap"))
      {
        rsbac_ind_softmode[CAP] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_jail"))
      {
        rsbac_ind_softmode[JAIL] = 1;
        return(TRUE);
      }
    if (!strcmp(line,"rsbac_softmode_res"))
      {
        rsbac_ind_softmode[RES] = 1;
        return(TRUE);
      }
    #endif
    #endif
    #if defined(CONFIG_RSBAC_CAP_PROC_HIDE)
    /* RSBAC: hide processes? */
    if (!strcmp(line,"rsbac_cap_process_hiding"))
      {
        rsbac_cap_process_hiding = 1;
        return(TRUE);
      }
    #endif
    #ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_FULL
    /* RSBAC: disable Linux DAC? */
    if (!strcmp(line,"rsbac_dac_disable"))
      {
        rsbac_dac_disable = 1;
        return(TRUE);
      }
    #endif
    #ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
    if (   !strcmp(line,"rsbac_nosyslog")
        || !strcmp(line,"rsbac_no_syslog")
       )
      {
        rsbac_nosyslog = 1;
        return(TRUE);
      }
    #endif
    #ifdef CONFIG_RSBAC_INIT_DELAY
    if (   !strcmp(line,"rsbac_delay_init")
       )
      {
        rsbac_delay_init = 1;
        return(TRUE);
      }
    if (   !strcmp(line,"rsbac_no_delay_init")
        || !strcmp(line,"rsbac_no_init_delay")
       )
      {
        rsbac_delay_init = 0;
        return(TRUE);
      }
    if (   !strncmp(line,"rsbac_delayed_root=",19)
       )
      {
        char * p = line+19;
        u_int major = 0;
        u_int minor = 0;

        if(*p)
          {
            major = simple_strtoul(p, NULL, 0);
            while((*p != ':') && (*p != '\0'))
              p++;
            if(*p)
              {
                p++;
                minor = simple_strtoul(p, NULL, 0);
              }
            rsbac_delayed_root = MKDEV(major,minor);
          }
        return(TRUE);
      }
    #endif

#ifdef CONFIG_RSBAC_DEBUG
    #ifdef CONFIG_RSBAC_NET
    /* RSBAC: debug for all of net? */
    if (!strcmp(line,"rsbac_debug_net"))
      {
        rsbac_debug_ds_net = 1;
        rsbac_debug_aef_net = 1;
        rsbac_debug_adf_net = 1;
        return(TRUE);
      }
    /* RSBAC: debug for net data structures? */
    if (!strcmp(line,"rsbac_debug_ds_net"))
      {
        rsbac_debug_ds_net = 1;
        return(TRUE);
      }
    /* RSBAC: debug for net syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_net"))
      {
        rsbac_debug_aef_net = 1;
        return(TRUE);
      }
    /* RSBAC: debug for net decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_net"))
      {
        rsbac_debug_adf_net = 1;
        return(TRUE);
      }
    #endif

    #if defined(CONFIG_RSBAC_MAC)
    /* RSBAC: debug for all of mac? */
    if (!strcmp(line,"rsbac_debug_mac"))
      {
        rsbac_debug_ds_mac = 1;
        rsbac_debug_aef_mac = 1;
        rsbac_debug_adf_mac = 1;
        return(TRUE);
      }
    /* RSBAC: debug for mac data structures? */
    if (!strcmp(line,"rsbac_debug_ds_mac"))
      {
        rsbac_debug_ds_mac = 1;
        return(TRUE);
      }
    /* RSBAC: debug for MAC-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_mac"))
      {
        rsbac_debug_aef_mac = 1;
        return(TRUE);
      }
    /* RSBAC: debug for MAC-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_mac"))
      {
        rsbac_debug_adf_mac = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
    /* RSBAC: debug for all of pm? */
    if (!strcmp(line,"rsbac_debug_pm"))
      {
        rsbac_debug_ds_pm = 1;
        rsbac_debug_aef_pm = 1;
        rsbac_debug_adf_pm = 1;
        return(TRUE);
      }
    /* RSBAC: debug for pm data structures? */
    if (!strcmp(line,"rsbac_debug_ds_pm"))
      {
        rsbac_debug_ds_pm = 1;
        return(TRUE);
      }
    /* RSBAC: debug for PM-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_pm"))
      {
        rsbac_debug_aef_pm = 1;
        return(TRUE);
      }
    /* RSBAC: debug for PM-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_pm"))
      {
        rsbac_debug_adf_pm = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
    /* RSBAC: debug for MS-decisions/ADF? */
    if (   !strcmp(line,"rsbac_debug_adf_ms")
        || !strcmp(line,"rsbac_debug_ms")
       )
      {
      	rsbac_debug_adf_ms = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
    /* RSBAC: debug for all of rc? */
    if (!strcmp(line,"rsbac_debug_rc"))
      {
        rsbac_debug_ds_rc = 1;
        rsbac_debug_aef_rc = 1;
        rsbac_debug_adf_rc = 1;
        return(TRUE);
      }
    /* RSBAC: debug for rc data structures? */
    if (!strcmp(line,"rsbac_debug_ds_rc"))
      {
        rsbac_debug_ds_rc = 1;
        return(TRUE);
      }
    /* RSBAC: debug for RC-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_rc"))
      {
        rsbac_debug_aef_rc = 1;
        return(TRUE);
      }
    /* RSBAC: debug for RC-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_rc"))
      {
        rsbac_debug_adf_rc = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_AUTH) || defined(CONFIG_RSBAC_AUTH_MAINT)
    /* RSBAC: debug for all of auth? */
    if (!strcmp(line,"rsbac_debug_auth"))
      {
        rsbac_debug_ds_auth = 1;
        rsbac_debug_aef_auth = 1;
        rsbac_debug_adf_auth = 1;
        return(TRUE);
      }
    /* RSBAC: debug for auth data structures? */
    if (!strcmp(line,"rsbac_debug_ds_auth"))
      {
        rsbac_debug_ds_auth = 1;
        return(TRUE);
      }
    /* RSBAC: debug for AUTH-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_auth"))
      {
        rsbac_debug_aef_auth = 1;
        return(TRUE);
      }
    /* RSBAC: debug for AUTH-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_auth"))
      {
        rsbac_debug_adf_auth = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
    /* RSBAC: debug for all of reg? */
    if (!strcmp(line,"rsbac_debug_reg"))
      {
        rsbac_debug_reg = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_ACL) || defined(CONFIG_RSBAC_ACL_MAINT)
    /* RSBAC: debug for all of acl? */
    if (!strcmp(line,"rsbac_debug_acl"))
      {
        rsbac_debug_ds_acl = 1;
        rsbac_debug_aef_acl = 1;
        rsbac_debug_adf_acl = 1;
        return(TRUE);
      }
    /* RSBAC: debug for ACL data structures? */
    if (!strcmp(line,"rsbac_debug_ds_acl"))
      {
        rsbac_debug_ds_acl = 1;
        return(TRUE);
      }
    /* RSBAC: debug for ACL-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_acl"))
      {
        rsbac_debug_aef_acl = 1;
        return(TRUE);
      }
    /* RSBAC: debug for ACL-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_acl"))
      {
        rsbac_debug_adf_acl = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_JAIL)
    /* RSBAC: debug for all of jail? */
    if (!strcmp(line,"rsbac_debug_jail"))
      {
        rsbac_debug_aef_jail = 1;
        rsbac_debug_adf_jail = 1;
        return(TRUE);
      }
    /* RSBAC: debug for JAIL-syscalls/AEF? */
    if (!strcmp(line,"rsbac_debug_aef_jail"))
      {
        rsbac_debug_aef_jail = 1;
        return(TRUE);
      }
    /* RSBAC: debug for JAIL-decisions/ADF? */
    if (!strcmp(line,"rsbac_debug_adf_jail"))
      {
        rsbac_debug_adf_jail = 1;
        return(TRUE);
      }
    #endif
    #if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
    /* RSBAC: debug for auto-write? */
    if (!strcmp(line,"rsbac_debug_auto"))
      {
        rsbac_debug_auto = 1;
        return(TRUE);
      }
    #endif
    /* RSBAC: debug for all? */
    if (!strcmp(line,"rsbac_debug_all"))
      {
	rsbac_debug_ds = 1;
	rsbac_debug_write = 1;
	rsbac_debug_aef = 1;
	rsbac_debug_adf_default = 2;
        #if defined(CONFIG_RSBAC_MAC)
	rsbac_debug_ds_mac = 1;
	rsbac_debug_aef_mac = 1;
	rsbac_debug_adf_mac = 1;
        #endif
        #if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
	rsbac_debug_ds_pm = 1;
	rsbac_debug_aef_pm = 1;
	rsbac_debug_adf_pm = 1;
        #endif
        #if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
	rsbac_debug_adf_ms = 1;
        #endif
        #if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
	rsbac_debug_ds_rc = 1;
	rsbac_debug_aef_rc = 1;
	rsbac_debug_adf_rc = 1;
        #endif
        #if defined(CONFIG_RSBAC_AUTH)
	rsbac_debug_ds_auth = 1;
	rsbac_debug_aef_auth = 1;
	rsbac_debug_adf_auth = 1;
        #endif
        #if defined(CONFIG_RSBAC_ACL)
	rsbac_debug_ds_acl = 1;
	rsbac_debug_aef_acl = 1;
	rsbac_debug_adf_acl = 1;
        #endif
        #if defined(CONFIG_RSBAC_JAIL)
	rsbac_debug_aef_jail = 1;
	rsbac_debug_adf_jail = 1;
        #endif
        #if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
        rsbac_debug_auto = 1;
        #endif

	return(TRUE);
      }
    /* RSBAC: debug_lists */
    if (!strcmp(line,"rsbac_debug_lists"))
      {
        rsbac_debug_lists = 1;
        return(TRUE);
      }
    /* RSBAC: debug_stack */
    if (!strcmp(line,"rsbac_debug_stack"))
      {
        rsbac_debug_stack = 1;
        return(TRUE);
      }
    /* RSBAC: debug for data structures? */
    if (!strcmp(line,"rsbac_debug_ds"))
      {
        rsbac_debug_ds = 1;
        return(TRUE);
      }
    /* RSBAC: debug for writing of data structures? */
    if (!strcmp(line,"rsbac_debug_write"))
      {
        rsbac_debug_write = 1;
        return(TRUE);
      }
    /* RSBAC: debug for AEF? */
    if (!strcmp(line,"rsbac_debug_aef"))
      {
        rsbac_debug_aef = 1;
        return(TRUE);
      }
    /* RSBAC: debug_no_write for ds */
    if (!strcmp(line,"rsbac_debug_no_write"))
      {
        rsbac_debug_no_write = 1;
        return(TRUE);
      }
    /* RSBAC: debug default 0 for ADF */
    if (!strcmp(line,"rsbac_debug_no_adf"))
      {
        rsbac_debug_adf_default = 0;
        return(TRUE);
      }
    /* RSBAC: debug default 1 for ADF, log denied requests? (default value) */
    if (!strcmp(line,"rsbac_debug_adf"))
      {
        rsbac_debug_adf_default = 1;
        return(TRUE);
      }
    /* RSBAC: debug: log all for ADF, even GRANTED and DO_NOT_CARE? */
    if (!strcmp(line,"rsbac_debug_adf_all"))
      {
        rsbac_debug_adf_default = 2;
        return(TRUE);
      }
#endif /* DEBUG */

    /* nothing matched */
    return(FALSE);
  }

#if defined(CONFIG_RSBAC_RMSG)

static spinlock_t rsbac_log_lock = SPIN_LOCK_UNLOCKED;

#if defined(CONFIG_RSBAC_LOG_REMOTE)
static spinlock_t rsbac_log_remote_lock = SPIN_LOCK_UNLOCKED;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
/*
 * Commands to rsbac_log:
 *
 * 	0 -- Close the log.  Currently a NOP.
 * 	1 -- Open the log. Currently a NOP.
 * 	2 -- Read from the log.
 * 	3 -- Read up to the last 4k of messages in the ring buffer.
 * 	4 -- Read and clear last 4k of messages in the ring buffer
 * 	5 -- Clear ring buffer.
 */
int rsbac_log(int type, char * buf, int len)
{
	int do_clear = 0;
	char c;
	int error = -EPERM;
	int i,j;
	u_long flags;
	u_long count;

        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;

        lock_kernel();

        /* RSBAC */
        rsbac_target_id.scd = ST_rsbaclog;
        rsbac_attribute_value.dummy = 0;
        if ((type == 4) || (type == 5))
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "rsbac_log(): function %u, calling ADF for MODIFY_SYSTEM_DATA\n", type);
#endif
            if (!rsbac_adf_request(R_MODIFY_SYSTEM_DATA,
                                   current->pid,
                                   T_SCD,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                error = -EPERM;
                goto out;
              }
          }
        else
        if(type >= 1)
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "rsbac_log(): function %u, calling ADF for GET_STATUS_DATA\n", type);
#endif
            if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                   current->pid,
                                   T_SCD,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                error = -EPERM;
                goto out;
              }
          }

	error = 0;
	switch (type) {
	case 0:		/* Close log */
		break;
	case 1:		/* Open log */
		break;
	case 2:		/* Read from log */
		error = -EINVAL;
		if (!buf || len < 0)
			goto out;
		error = 0;
		if (!len)
			goto out;
		error = verify_area(VERIFY_WRITE,buf,len);
		if (error)
			goto out;
		cli();
		error = -ERESTARTSYS;
		while (!rlog_size) {
			if (signal_pending(current)) {
				sti();
				goto out;
			}
			interruptible_sleep_on(&rlog_wait);
		}
		i = 0;
		while (rlog_size && i < len) {
			c = *((char *) rlog_buf+log_start);
			log_start++;
			rlog_size--;
			log_start &= RLOG_BUF_MASK;
			sti();
			__put_user(c,buf);
			buf++;
			i++;
			cli();
		}
		sti();
		error = i;
		break;
	case 4:		/* Read/clear last kernel messages */
		do_clear = 1; 
		/* FALL THRU */
	case 3:		/* Read last kernel messages */
		error = -EINVAL;
		if (!buf || len < 0)
			goto out;
		error = 0;
		if (!len)
			goto out;
		error = verify_area(VERIFY_WRITE,buf,len);
		if (error)
			goto out;
		/*
		 * The logged_chars, log_start, and rlog_size values may
		 * change from an interrupt, so we disable interrupts.
		 */
		__save_flags(flags);
		__cli();
		count = len;
		if (count > RLOG_BUF_LEN)
			count = RLOG_BUF_LEN;
		if (count > logged_chars)
			count = logged_chars;
		j = log_start + rlog_size - count;
		__restore_flags(flags);
		for (i = 0; i < count; i++) {
			c = *((char *) rlog_buf+(j++ & RLOG_BUF_MASK));
			__put_user(c, buf++);
		}
		if (do_clear)
			logged_chars = 0;
		error = i;
		break;
	case 5:		/* Clear ring buffer */
		logged_chars = 0;
		break;
	default:
		error = -EINVAL;
		break;
	}
out:
        unlock_kernel();
	return error;
}

#else /* Kernel Version >= 2.3.0 */

/*
 * Commands to do_syslog:
 *
 * 	0 -- Close the log.  Currently a NOP.
 * 	1 -- Open the log. Currently a NOP.
 * 	2 -- Read from the log.
 * 	3 -- Read all messages remaining in the ring buffer.
 * 	4 -- Read and clear all messages remaining in the ring buffer
 * 	5 -- Clear ring buffer.
 *	9 -- Return number of unread characters in the log buffer
 */
int rsbac_log(int type, char * buf, int len)
{
	unsigned long i, j, limit, count;
	int do_clear = 0;
	char c;
	int error = 0;

        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;

        /* RSBAC */
        rsbac_target_id.scd = ST_rsbaclog;
        rsbac_attribute_value.dummy = 0;
        if ((type == 4) || (type == 5))
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "rsbac_log(): function %u, calling ADF for MODIFY_SYSTEM_DATA\n", type);
#endif
            if (!rsbac_adf_request(R_MODIFY_SYSTEM_DATA,
                                   current->pid,
                                   T_SCD,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                error = -EPERM;
                goto out;
              }
          }
        else
        if(type >= 1)
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "rsbac_log(): function %u, calling ADF for GET_STATUS_DATA\n", type);
#endif
            if (!rsbac_adf_request(R_GET_STATUS_DATA,
                                   current->pid,
                                   T_SCD,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                error = -EPERM;
                goto out;
              }
          }

	switch (type) {
	case 0:		/* Close log */
		break;
	case 1:		/* Open log */
		break;
	case 2:		/* Read from log */
		error = -EINVAL;
		if (!buf || len < 0)
			goto out;
		error = 0;
		if (!len)
			goto out;
		error = verify_area(VERIFY_WRITE,buf,len);
		if (error)
			goto out;
		error = wait_event_interruptible(rlog_wait, (log_start - log_end));
		if (error)
			goto out;
		i = 0;
		spin_lock_irq(&rsbac_log_lock);
		while ((log_start != log_end) && i < len) {
			c = RLOG_BUF(log_start);
			log_start++;
			spin_unlock_irq(&rsbac_log_lock);
			__put_user(c,buf);
			buf++;
			i++;
			spin_lock_irq(&rsbac_log_lock);
		}
		spin_unlock_irq(&rsbac_log_lock);
		error = i;
		break;
	case 4:		/* Read/clear last kernel messages */
		do_clear = 1; 
		/* FALL THRU */
	case 3:		/* Read last kernel messages */
		error = -EINVAL;
		if (!buf || len < 0)
			goto out;
		error = 0;
		if (!len)
			goto out;
		error = verify_area(VERIFY_WRITE,buf,len);
		if (error)
			goto out;
		count = len;
		if (count > RLOG_BUF_LEN)
			count = RLOG_BUF_LEN;
		spin_lock_irq(&rsbac_log_lock);
		if (count > logged_chars)
			count = logged_chars;
		if (do_clear)
			logged_chars = 0;
		limit = log_end;
		/*
		 * __put_user() could sleep, and while we sleep
		 * printk() could overwrite the messages 
		 * we try to copy to user space. Therefore
		 * the messages are copied in reverse. <manfreds>
		 */
		for(i=0;i < count;i++) {
			j = limit-1-i;
			if (j+RLOG_BUF_LEN < log_end)
				break;
			c = RLOG_BUF(j);
			spin_unlock_irq(&rsbac_log_lock);
			__put_user(c,&buf[count-1-i]);
			spin_lock_irq(&rsbac_log_lock);
		}
		spin_unlock_irq(&rsbac_log_lock);
		error = i;
		if(i != count) {
			int offset = count-error;
			/* buffer overflow during copy, correct user buffer. */
			for(i=0;i<error;i++) {
				__get_user(c,&buf[i+offset]);
				__put_user(c,&buf[i]);
			}
		}

		break;
	case 5:		/* Clear ring buffer */
		spin_lock_irq(&rsbac_log_lock);
		logged_chars = 0;
		spin_unlock_irq(&rsbac_log_lock);
		break;
	case 9:		/* Number of chars in the log buffer */
		spin_lock_irq(&rsbac_log_lock);
		error = log_end - log_start;
		spin_unlock_irq(&rsbac_log_lock);
		break;
	default:
		error = -EINVAL;
		break;
	}
out:
	return error;
}
#endif /* Kernel Version */

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_printk);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
int rsbac_printk(const char *fmt, ...)
{
	va_list args;
	int i;
        static u_int log_seq = 0;
#if defined(CONFIG_RSBAC_LOG_REMOTE)
	int j;
        static char remote_buf[1024];
        static u_int remote_log_seq = 0;
#endif
	char *p, *buf_end;
        static char buf[1024];
	int line_feed;
	static signed char msg_level = -1;
	long flags;

	spin_lock_irqsave(&rsbac_log_lock, flags);
	va_start(args, fmt);
	i = vsprintf(buf + 14, fmt, args); /* hopefully i < sizeof(buf)-15 */
	buf_end = buf + i + 14;
	va_end(args);
	for (p = buf + 14; p < buf_end; p++) {
		if (msg_level < 0) {
			if (
				p[0] != '<' ||
				p[1] < '0' || 
				p[1] > '7' ||
				p[2] != '>'
			) {
				p -= 14;
				p[0] = '<';
				p[1] = RSBAC_DEF_MESS_LOGLEVEL + '0';
				p[2] = '>';
				sprintf(p + 3, "%010u", log_seq++);
				p[13] = '|';
			} else {
				p -= 11;
				p[0] = '<';
				p[1] = p[12];
				p[2] = '>';
				sprintf(p + 3, "%010u", log_seq++);
				p[13] = '|';
			}
			msg_level = p[1] - '0';
		}
		line_feed = 0;
		for (; p < buf_end; p++) {
			rlog_buf[(log_start+rlog_size) & (RLOG_BUF_LEN-1)] = *p;
			if (rlog_size < RLOG_BUF_LEN)
				rlog_size++;
			else {
				log_start++;
				log_start &= RLOG_BUF_LEN-1;
			}
			logged_chars++;
			if (*p == '\n') {
				line_feed = 1;
				break;
			}
		}
		if (line_feed)
			msg_level = -1;
	}
	spin_unlock_irqrestore(&rsbac_log_lock, flags);
	wake_up_interruptible(&rlog_wait);

#if defined(CONFIG_RSBAC_LOG_REMOTE)
	spin_lock_irqsave(&rsbac_log_remote_lock, flags);
	va_start(args, fmt);
	j = vsprintf(remote_buf + 14, fmt, args); /* hopefully j < sizeof(buf)-15 */
	buf_end = remote_buf + j + 14;
	va_end(args);
	for (p = remote_buf + 14; p < buf_end; p++) {
		if (msg_level < 0) {
			if (
				p[0] != '<' ||
				p[1] < '0' || 
				p[1] > '7' ||
				p[2] != '>'
			) {
				p -= 14;
				p[0] = '<';
				p[1] = RSBAC_DEF_MESS_LOGLEVEL + '0';
				p[2] = '>';
				sprintf(p + 3, "%010u", remote_log_seq++);
				p[13] = '|';
			} else {
				p -= 11;
				p[0] = '<';
				p[1] = p[12];
				p[2] = '>';
				sprintf(p + 3, "%010u", remote_log_seq++);
				p[13] = '|';
			}
			msg_level = p[1] - '0';
		}
		line_feed = 0;
		for (; p < buf_end; p++) {
			remote_rlog_buf[(remote_log_start+remote_rlog_size) & (REMOTE_RLOG_BUF_LEN-1)] = *p;
			if (remote_rlog_size < REMOTE_RLOG_BUF_LEN)
				remote_rlog_size++;
			else {
				remote_log_start++;
				remote_log_start &= REMOTE_RLOG_BUF_LEN-1;
			}
			remote_logged_chars++;
			if (*p == '\n') {
				line_feed = 1;
				break;
			}
		}
		if (line_feed)
			msg_level = -1;
	}
	spin_unlock_irqrestore(&rsbac_log_remote_lock, flags);
#ifdef CONFIG_RSBAC_LOG_REMOTE_SYNC
        wake_up_interruptible(&rsbaclogd_wait);
#endif
#endif

	return i;
}

#else /* Kernel Version >= 2.3.0 */

static void emit_log_char(char c)
{
	RLOG_BUF(log_end) = c;
	log_end++;
	if (log_end - log_start > RLOG_BUF_LEN)
		log_start = log_end - RLOG_BUF_LEN;
	if (logged_chars < RLOG_BUF_LEN)
		logged_chars++;
}

#if defined(CONFIG_RSBAC_LOG_REMOTE)
static void emit_remote_log_char(char c)
{
	REMOTE_RLOG_BUF(remote_log_end) = c;
	remote_log_end++;
	if (remote_log_end - remote_log_start > REMOTE_RLOG_BUF_LEN)
		remote_log_start = remote_log_end - REMOTE_RLOG_BUF_LEN;
	if (remote_logged_chars < REMOTE_RLOG_BUF_LEN)
		remote_logged_chars++;
}
#endif

int rsbac_printk(const char *fmt, ...)
{
	va_list args;
	unsigned long flags;
	int printed_len;
	char *p;
        static char buf[2048];
	static int log_level_unknown = 1;
        static u_int log_seq = 0;
#if defined(CONFIG_RSBAC_LOG_REMOTE)
        static u_int remote_log_seq = 0;
	static int remote_log_level_unknown = 1;
#endif

	if (oops_in_progress) {
		/* If a crash is occurring, make sure we can't deadlock */
		spin_lock_init(&rsbac_log_lock);
	}

	/* This stops the holder of console_sem just where we want him */
	spin_lock_irqsave(&rsbac_log_lock, flags);

	/* Emit the output into the temporary buffer */
	va_start(args, fmt);
	printed_len = vsnprintf(buf + 14, sizeof(buf) - 14, fmt, args);
	va_end(args);

	/*
	 * Copy the output into log_buf.  If the caller didn't provide
	 * appropriate log level tags, we insert them here
	 */
	for (p = buf + 14; *p; p++) {
		if (log_level_unknown) {
			if (p[0] != '<' || p[1] < '0' || p[1] > '7' || p[2] != '>') {
				p -= 14;
				p[0] = '<';
				p[1] = RSBAC_DEF_MESS_LOGLEVEL + '0';
				p[2] = '>';
				sprintf(p + 3, "%010u", log_seq++);
				p[13] = '|';
			} else {
				p -= 11;
				p[0] = '<';
				p[1] = p[12];
				p[2] = '>';
				sprintf(p + 3, "%010u", log_seq++);
				p[13] = '|';
			}
			log_level_unknown = 0;
		}
		emit_log_char(*p);
		if (*p == '\n')
			log_level_unknown = 1;
	}
        spin_unlock_irqrestore(&rsbac_log_lock, flags);
	wake_up_interruptible(&rlog_wait);

#if defined(CONFIG_RSBAC_LOG_REMOTE)
	if (oops_in_progress) {
		/* If a crash is occurring, make sure we can't deadlock */
		spin_lock_init(&rsbac_log_remote_lock);
	}

	/* This stops the holder of console_sem just where we want him */
	spin_lock_irqsave(&rsbac_log_remote_lock, flags);

	/* Emit the output into the temporary buffer */
	va_start(args, fmt);
	printed_len = vsnprintf(buf + 14, sizeof(buf) - 14, fmt, args);
	va_end(args);

	/*
	 * Copy the output into log_buf.  If the caller didn't provide
	 * appropriate log level tags, we insert them here
	 */
	for (p = buf + 14; *p; p++) {
		if (remote_log_level_unknown) {
			if (p[0] != '<' || p[1] < '0' || p[1] > '7' || p[2] != '>') {
				p -= 14;
				p[0] = '<';
				p[1] = RSBAC_DEF_MESS_LOGLEVEL + '0';
				p[2] = '>';
				sprintf(p + 3, "%010u", remote_log_seq++);
				p[13] = '|';
			} else {
				p -= 11;
				p[0] = '<';
				p[1] = p[12];
				p[2] = '>';
				sprintf(p + 3, "%010u", remote_log_seq++);
				p[13] = '|';
			}
			remote_log_level_unknown = 0;
		}
		emit_remote_log_char(*p);
		if (*p == '\n')
			remote_log_level_unknown = 1;
	}
	spin_unlock_irqrestore(&rsbac_log_remote_lock, flags);
#ifdef CONFIG_RSBAC_LOG_REMOTE_SYNC
        wake_up_interruptible(&rsbaclogd_wait);
#endif
#endif

	return printed_len;
}
#endif /* Kernel Version */

#if defined(CONFIG_RSBAC_PROC) && defined(CONFIG_PROC_FS)
static int rmsg_open(struct inode * inode, struct file * file)
{
	return rsbac_log(1,NULL,0);
}

static int rmsg_release(struct inode * inode, struct file * file)
{
	(void) rsbac_log(0,NULL,0);
	return 0;
}

static ssize_t rmsg_read(struct file * file, char * buf,
			 size_t count, loff_t *ppos)
{
	return rsbac_log(2,buf,count);
}

static unsigned int rmsg_poll(struct file *file, poll_table * wait)
{
	poll_wait(file, &rlog_wait, wait);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	if (rlog_size)
		return POLLIN | POLLRDNORM;
#else
	if (rsbac_log(9, 0, 0))
		return POLLIN | POLLRDNORM;
#endif
	return 0;
}

static struct file_operations proc_rmsg_operations = {
      #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
        NULL,           /* owner */
      #endif
	NULL,		/* rmsg_lseek */
	rmsg_read,
	NULL,		/* rmsg_write */
	NULL,		/* rmsg_readdir */
	rmsg_poll,	/* rmsg_poll */
	NULL,		/* rmsg_ioctl */
	NULL,		/* mmap */
	rmsg_open,
	NULL,		/* flush */
	rmsg_release,
	NULL		/* can't fsync */
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,48)
static struct inode_operations proc_rmsg_inode_operations = {
	&proc_rmsg_operations,	/* default base directory file-ops */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};
#endif /* VERSION < 2.3.48 */
#endif /* PROC */
#endif /* RMSG */

#if defined(CONFIG_RSBAC_PROC) && defined(CONFIG_PROC_FS)
#ifndef PROC_BLOCK_SIZE
#define PROC_BLOCK_SIZE	(3*1024)  /* 4K page size but our output routines use some slack for overruns */
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static int
log_levels_proc_info(char *buffer, char **start, off_t offset, int length, int dummy)
#else
static int
log_levels_proc_info(char *buffer, char **start, off_t offset, int length)
#endif
{
  int len = 0;
  int i,j;
  off_t pos   = 0;
  off_t begin = 0;
  char * name;
  char * name2;

  union rsbac_target_id_t       rsbac_target_id;
  union rsbac_attribute_value_t rsbac_attribute_value;

  if (!rsbac_is_initialized())
    return (-ENOSYS);

#ifdef CONFIG_RSBAC_DEBUG
  if (rsbac_debug_aef) printk(KERN_DEBUG "log_levels_proc_info(): calling ADF\n");
#endif
  rsbac_target_id.scd = ST_rsbac;
  rsbac_attribute_value.dummy = 0;
  if (!rsbac_adf_request(R_GET_STATUS_DATA,
                         current->pid,
                         T_SCD,
                         rsbac_target_id,
                         A_none,
                         rsbac_attribute_value))
    {
      return -EPERM;
    }

  name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
  if(!name)
    return -ENOMEM;
  name2 = rsbac_kmalloc(RSBAC_MAXNAMELEN);
  if(!name2)
    {
      rsbac_kfree(name);
      return -ENOMEM;
    }
    
  len += sprintf(buffer, "RSBAC Log Levels\n----------------\n");
  len += sprintf(buffer+len, "Name\t\t\tFILE\tDIR\tFIFO\tSYMLINK\tDEV\tIPC\tSCD\tUSER\tPROCESS\tNETDEV\tNETTEMP\tNETOBJ\tNETT_NT\tNONE\n");

  for (i = 0; i<R_NONE; i++)
    {
      len += sprintf(buffer + len, "%-23s",
                     get_request_name(name, i));
      for(j = 0; j<=T_NONE; j++)
        {
          if(j != T_FD)
            len += sprintf(buffer + len, "\t%u",
                           rsbac_log_levels[i][j]);
        }
      len += sprintf(buffer + len, "\n");
      pos = begin + len;
      if (pos < offset)
        {
          len = 0;
          begin = pos;
        }
      if (pos > offset+length)
        break;
    }

  *start = buffer + (offset - begin);
  len -= (offset - begin);
  
  if (len > length)
    len = length;
  rsbac_kfree(name);
  rsbac_kfree(name2);

  return len;
}

static ssize_t log_levels_proc_write(struct file * file, const char * buf,
                                     u_long count, void *ppos)
{
    ssize_t err = -EINVAL;
    char * k_buf;
    char * p;
    unsigned int log_level;
    char rname[RSBAC_MAXNAMELEN];
    int i,j;

    union rsbac_target_id_t       rsbac_target_id;
    union rsbac_attribute_value_t rsbac_attribute_value;

    if(count > PROC_BLOCK_SIZE) {
	return(-EOVERFLOW);
    }

    if (!(k_buf = (char *) __get_free_page(GFP_KERNEL)))
      return(-ENOMEM);
    copy_from_user(k_buf, buf, count);

  if(count < 15 || strncmp("log_levels", k_buf, 10))
    {
      goto out;
    }
  if (!rsbac_is_initialized())
    {
      err=-ENOSYS;
      goto out;
    }

    /*
     * Usage: echo "log_levels request #N" > /proc/rsbac_info/log_levels
     *   to set log level for request to given value
     */
    for(i=0; i<R_NONE; i++)
      {
        get_request_name(rname,i);
        if(!strncmp(rname, k_buf + 11, strlen(rname))) 
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "log_levels_proc_write(): calling ADF\n");
#endif
            rsbac_target_id.dummy = 0;
            rsbac_attribute_value.request = i;
            if (!rsbac_adf_request(R_SWITCH_LOG,
                                   current->pid,
                                   T_NONE,
                                   rsbac_target_id,
                                   A_request,
                                   rsbac_attribute_value))
              {
                err = -EPERM;
                goto out;
              }
	    p = k_buf + 11 + strlen(rname)+1;

            if( *p == '\0' )
              goto out;

            log_level = simple_strtoul(p, NULL, 0);
            /* only accept 0 or 1 */
            if(   (log_level == LL_none)
               || (log_level == LL_denied)
               || (log_level == LL_full)
              )
              {
                printk(KERN_INFO
                       "log_levels_proc_write(): setting %s log level for all target types to %u\n",
                       rname, log_level);
                for(j = 0; j<=T_NONE; j++)
                  {
                    rsbac_log_levels[i][j] = log_level;
                  }
                err = count;
                goto out;
              }
            else
              {
                printk(KERN_INFO
                       "log_levels_proc_write(): rejecting invalid log level (should be %u, %u or %u)\n",
                       LL_none, LL_denied, LL_full);
                goto out;
              }
          }
      }

out:
  free_page((ulong) k_buf);
  return(err);
  }


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static int
debug_proc_info(char *buffer, char **start, off_t offset, int length, int dummy)
#else
static int
debug_proc_info(char *buffer, char **start, off_t offset, int length)
#endif
{
  int len = 0;
  off_t pos   = 0;
  off_t begin = 0;

  union rsbac_target_id_t       rsbac_target_id;
  union rsbac_attribute_value_t rsbac_attribute_value;

  if (!rsbac_is_initialized())
    return (-ENOSYS);

#ifdef CONFIG_RSBAC_DEBUG
  if (rsbac_debug_aef) printk(KERN_DEBUG "debug_proc_info(): calling ADF\n");
#endif
  rsbac_target_id.scd = ST_rsbac;
  rsbac_attribute_value.dummy = 0;
  if (!rsbac_adf_request(R_GET_STATUS_DATA,
                         current->pid,
                         T_SCD,
                         rsbac_target_id,
                         A_none,
                         rsbac_attribute_value))
    {
      return -EPERM;
    }
  len += sprintf(buffer, "RSBAC Debug Settings\n--------------------\n");

#ifdef CONFIG_RSBAC_SOFTMODE
  len += sprintf(buffer + len, "rsbac_softmode is %i\n",
                 rsbac_softmode);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#ifdef CONFIG_RSBAC_SOFTMODE_IND
#ifdef CONFIG_RSBAC_MAC
  len += sprintf(buffer + len, "rsbac_ind_softmode[MAC] is %i\n",
                 rsbac_ind_softmode[MAC]);
#endif
#ifdef CONFIG_RSBAC_FC
  len += sprintf(buffer + len, "rsbac_ind_softmode[FC] is %i\n",
                 rsbac_ind_softmode[FC]);
#endif
#ifdef CONFIG_RSBAC_SIM
  len += sprintf(buffer + len, "rsbac_ind_softmode[SIM] is %i\n",
                 rsbac_ind_softmode[SIM]);
#endif
#ifdef CONFIG_RSBAC_PM
  len += sprintf(buffer + len, "rsbac_ind_softmode[PM] is %i\n",
                 rsbac_ind_softmode[PM]);
#endif
#ifdef CONFIG_RSBAC_MS
  len += sprintf(buffer + len, "rsbac_ind_softmode[MS] is %i\n",
                 rsbac_ind_softmode[MS]);
#endif
#ifdef CONFIG_RSBAC_FF
  len += sprintf(buffer + len, "rsbac_ind_softmode[FF] is %i\n",
                 rsbac_ind_softmode[FF]);
#endif
#ifdef CONFIG_RSBAC_RC
  len += sprintf(buffer + len, "rsbac_ind_softmode[RC] is %i\n",
                 rsbac_ind_softmode[RC]);
#endif
#ifdef CONFIG_RSBAC_AUTH
  len += sprintf(buffer + len, "rsbac_ind_softmode[AUTH] is %i\n",
                 rsbac_ind_softmode[AUTH]);
#endif
#ifdef CONFIG_RSBAC_REG
  len += sprintf(buffer + len, "rsbac_ind_softmode[REG] is %i\n",
                 rsbac_ind_softmode[REG]);
#endif
#ifdef CONFIG_RSBAC_ACL
  len += sprintf(buffer + len, "rsbac_ind_softmode[ACL] is %i\n",
                 rsbac_ind_softmode[ACL]);
#endif
#ifdef CONFIG_RSBAC_CAP
  len += sprintf(buffer + len, "rsbac_ind_softmode[CAP] is %i\n",
                 rsbac_ind_softmode[CAP]);
#endif
#ifdef CONFIG_RSBAC_JAIL
  len += sprintf(buffer + len, "rsbac_ind_softmode[JAIL] is %i\n",
                 rsbac_ind_softmode[JAIL]);
#endif
#ifdef CONFIG_RSBAC_RES
  len += sprintf(buffer + len, "rsbac_ind_softmode[RES] is %i\n",
                 rsbac_ind_softmode[RES]);
#endif
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

#endif
#endif
#ifdef CONFIG_RSBAC_CAP_PROC_HIDE
  len += sprintf(buffer + len, "rsbac_cap_process_hiding is %i\n",
                 rsbac_cap_process_hiding);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_FULL
  len += sprintf(buffer + len, "rsbac_dac_disable is %i\n",
                 rsbac_dac_disable);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
  len += sprintf(buffer + len, "rsbac_nosyslog is %i\n",
                 rsbac_nosyslog);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#ifdef CONFIG_RSBAC_INIT_DELAY
  len += sprintf(buffer + len, "rsbac_delay_init is %i\n",
                 rsbac_delay_init);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
  len += sprintf(buffer + len, "rsbac_delayed_root is %02u:%02u\n",
                 MAJOR(rsbac_delayed_root), MINOR(rsbac_delayed_root));
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_AUTH)
  len += sprintf(buffer + len, "rsbac_auth_enable_login is %i\n",
                 rsbac_auth_enable_login);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

  len += sprintf(buffer + len, "rsbac_no_defaults is %i\n",
                 rsbac_no_defaults);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

#ifdef CONFIG_RSBAC_DEBUG
  len += sprintf(buffer + len, "rsbac_debug_write is %i\n",
                 rsbac_debug_write);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "rsbac_debug_stack is %i\n",
                 rsbac_debug_stack);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "rsbac_debug_lists is %i\n",
                 rsbac_debug_lists);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "rsbac_debug_ds is %i\n",
                 rsbac_debug_ds);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "rsbac_debug_aef is %i\n",
                 rsbac_debug_aef);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "rsbac_debug_no_write is %i\n",
                 rsbac_debug_no_write);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
/* Boolean debug switch for REG */
  len += sprintf(buffer + len, "rsbac_debug_reg is %i\n",
                 rsbac_debug_reg);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_NET)
/* Boolean debug switch for NET data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_net is %i\n",
                 rsbac_debug_ds_net);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for NET syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_net is %i\n",
                 rsbac_debug_aef_net);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for NET decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_net is %i\n",
                 rsbac_debug_adf_net);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_MAC)
/* Boolean debug switch for MAC data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_mac is %i\n",
                 rsbac_debug_ds_mac);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for MAC syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_mac is %i\n",
                 rsbac_debug_aef_mac);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for MAC decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_mac is %i\n",
                 rsbac_debug_adf_mac);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
/* Boolean debug switch for PM data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_pm is %i\n",
                 rsbac_debug_ds_pm);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for PM syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_pm is %i\n",
                 rsbac_debug_aef_pm);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for PM decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_pm is %i\n",
                 rsbac_debug_adf_pm);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
/* Boolean debug switch for MS decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_ms is %i\n",
                 rsbac_debug_adf_ms);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
/* Boolean debug switch for RC data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_rc is %i\n",
                 rsbac_debug_ds_rc);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for RC syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_rc is %i\n",
                 rsbac_debug_aef_rc);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for RC decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_rc is %i\n",
                 rsbac_debug_adf_rc);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_AUTH)
/* Boolean debug switch for AUTH data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_auth is %i\n",
                 rsbac_debug_ds_auth);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for AUTH syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_auth is %i\n",
                 rsbac_debug_aef_auth);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for AUTH decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_auth is %i\n",
                 rsbac_debug_adf_auth);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_ACL)
/* Boolean debug switch for ACL data structures */
  len += sprintf(buffer + len, "rsbac_debug_ds_acl is %i\n",
                 rsbac_debug_ds_acl);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for ACL syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_acl is %i\n",
                 rsbac_debug_aef_acl);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for ACL decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_acl is %i\n",
                 rsbac_debug_adf_acl);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_JAIL)
/* Boolean debug switch for JAIL syscalls / AEF */
  len += sprintf(buffer + len, "rsbac_debug_aef_jail is %i\n",
                 rsbac_debug_aef_jail);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

/* Boolean debug switch for JAIL decisions / ADF */
  len += sprintf(buffer + len, "rsbac_debug_adf_jail is %i\n",
                 rsbac_debug_adf_jail);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif

#if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
  len += sprintf(buffer + len, "rsbac_debug_auto is %i\n",
                 rsbac_debug_auto);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;
#endif /* CONFIG_RSBAC_AUTO_WRITE > 0 */
#endif /* DEBUG */

out:
  *start = buffer + (offset - begin);
  len -= (offset - begin);
  
  if (len > length)
    len = length;
  return len;
}

static ssize_t debug_proc_write(struct file * file, const char * buf,
                                u_long count, void *ppos)
{
    ssize_t err = -EINVAL;
    char * k_buf;
    char * p;
    unsigned int debug_level;
#ifdef CONFIG_RSBAC_SOFTMODE_IND
    enum rsbac_switch_target_t sw_target;
#endif

    union rsbac_target_id_t       rsbac_target_id;
    union rsbac_attribute_value_t rsbac_attribute_value;

    if(count > PROC_BLOCK_SIZE) {
	return(-EOVERFLOW);
    }

    if (!(k_buf = (char *) __get_free_page(GFP_KERNEL)))
      return(-ENOMEM);
    copy_from_user(k_buf, buf, count);

  if(count < 10)
    goto out;
  if(!strncmp("debug", k_buf, 5))
    {
      p = k_buf + 6;
    }
  else
  if(!strncmp("rsbac_debug", k_buf, 11))
    {
      p = k_buf + 12;
    }
  else
    goto out;

  if (!rsbac_is_initialized())
    {
      err=-ENOSYS;
      goto out;
    }


#ifdef CONFIG_RSBAC_SOFTMODE
#ifdef CONFIG_RSBAC_SOFTMODE_IND
/* Boolean switch for RSBAC individual soft mode */
    /*
     * Usage: echo "debug ind_softmode modname #N" > /proc/rsbac_info/debug
     *   to set rsbac_ind_softmode[module] to given value
     */
    if(!strncmp("ind_softmode", k_buf + 6, 12)) 
      {
        char tmp[20];

	p += 13;

        if( *p == '\0' )
          goto out;

        sw_target = get_switch_target_nr(p);
        if(sw_target == SW_NONE)
          goto out;
        get_switch_target_name(tmp, sw_target);
        p += strlen(tmp)+1;
        if( *p == '\0' )
          goto out;
        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
#if defined(CONFIG_RSBAC_DEBUG)
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "debug_proc_write(): calling ADF for switching\n");
#endif
            rsbac_target_id.dummy = 0;
            rsbac_attribute_value.switch_target = sw_target;
            if (!rsbac_adf_request(R_SWITCH_MODULE,
                                   current->pid,
                                   T_NONE,
                                   rsbac_target_id,
                                   A_switch_target,
                                   rsbac_attribute_value))
              {
                err = -EPERM;
                goto out;
              }
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_ind_softmode[%s] to %u\n",
                   tmp,
                   debug_level);
            rsbac_ind_softmode[sw_target] = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid ind_softmode value (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

/* Boolean switch for RSBAC soft mode */
    /*
     * Usage: echo "debug softmode #N" > /proc/rsbac_info/debug
     *   to set rsbac_softmode to given value
     */
    if(!strncmp("softmode", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
#if defined(CONFIG_RSBAC_DEBUG)
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "debug_proc_write(): calling ADF for softmode\n");
#endif
            rsbac_target_id.dummy = 0;
            rsbac_attribute_value.switch_target = SOFTMODE;
            if (!rsbac_adf_request(R_SWITCH_MODULE,
                                   current->pid,
                                   T_NONE,
                                   rsbac_target_id,
                                   A_switch_target,
                                   rsbac_attribute_value))
              {
                err = -EPERM;
                goto out;
              }
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_softmode to %u\n",
                   debug_level);
            rsbac_softmode = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid softmode value (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_FULL
/* Boolean switch for disabling Linux DAC */
    /*
     * Usage: echo "debug dac_disable #N" > /proc/rsbac_info/debug
     *   to set rsbac_softmode to given value
     */
    if(!strncmp("dac_disable", k_buf + 6, 11)) 
      {
	p += 12;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
#if defined(CONFIG_RSBAC_DEBUG)
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "debug_proc_write(): calling ADF for dac_disable\n");
#endif
            rsbac_target_id.dummy = 0;
            rsbac_attribute_value.dummy = 0;
            if (!rsbac_adf_request(R_MODIFY_PERMISSIONS_DATA,
                                   current->pid,
                                   T_NONE,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                err = -EPERM;
                goto out;
              }
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_dac_disable to %u\n",
                   debug_level);
            rsbac_dac_disable = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid dac_disabled value (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
/* Boolean switch for disabling logging to syslog */
    /*
     * Usage: echo "debug nosyslog #N" > /proc/rsbac_info/debug
     *   to set rsbac_nosyslog to given value
     */
    if(!strncmp("nosyslog", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
#if defined(CONFIG_RSBAC_DEBUG)
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "debug_proc_write(): calling ADF for nosyslog\n");
#endif
            rsbac_target_id.dummy = 0;
            rsbac_attribute_value.dummy = 0;
            if (!rsbac_adf_request(R_SWITCH_LOG,
                                   current->pid,
                                   T_NONE,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                err = -EPERM;
                goto out;
              }
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_nosyslog to %u\n",
                   debug_level);
            rsbac_nosyslog = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid nosyslog value (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_DEBUG)
    if (rsbac_debug_aef) printk(KERN_DEBUG "debug_proc_write(): calling ADF\n");
    rsbac_target_id.scd = ST_rsbac;
    rsbac_attribute_value.dummy = 0;
    if (!rsbac_adf_request(R_MODIFY_SYSTEM_DATA,
                           current->pid,
                           T_SCD,
                           rsbac_target_id,
                           A_none,
                           rsbac_attribute_value))
      {
        err = -EPERM;
        goto out;
      }

#if defined(CONFIG_RSBAC_NET)
/* Boolean debug switch for NET data structures */
    /*
     * Usage: echo "debug ds_net #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_net to given value
     */
    if(!strncmp("ds_net", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_net to %u\n",
                   debug_level);
            rsbac_debug_ds_net = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for NET syscalls / AEF */
    /*
     * Usage: echo "debug aef_net #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_net to given value
     */
    if(!strncmp("aef_net", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_net to %u\n",
                   debug_level);
            rsbac_debug_aef_net = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for NET decisions / ADF */
    /*
     * Usage: echo "debug adf_net #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_net to given value
     */
    if(!strncmp("adf_net", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_net to %u\n",
                   debug_level);
            rsbac_debug_adf_net = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_MAC)
/* Boolean debug switch for MAC data structures */
    /*
     * Usage: echo "debug ds_mac #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_mac to given value
     */
    if(!strncmp("ds_mac", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_mac to %u\n",
                   debug_level);
            rsbac_debug_ds_mac = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for MAC syscalls / AEF */
    /*
     * Usage: echo "debug aef_mac #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_mac to given value
     */
    if(!strncmp("aef_mac", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_mac to %u\n",
                   debug_level);
            rsbac_debug_aef_mac = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for MAC decisions / ADF */
    /*
     * Usage: echo "debug adf_mac #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_mac to given value
     */
    if(!strncmp("adf_mac", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_mac to %u\n",
                   debug_level);
            rsbac_debug_adf_mac = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
/* Boolean debug switch for PM data structures */
    /*
     * Usage: echo "debug ds_pm #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_pm to given value
     */
    if(!strncmp("ds_pm", k_buf + 6, 5)) 
      {
	p += 6;

        if( *p == '\0' )
          goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_pm to %u\n",
                   debug_level);
            rsbac_debug_ds_pm = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for PM syscalls / AEF */
    /*
     * Usage: echo "debug aef_pm #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_pm to given value
     */
    if(!strncmp("aef_pm", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_pm to %u\n",
                   debug_level);
            rsbac_debug_aef_pm = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for PM decisions / ADF */
    /*
     * Usage: echo "debug adf_pm #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_pm to given value
     */
    if(!strncmp("adf_pm", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_pm to %u\n",
                   debug_level);
            rsbac_debug_adf_pm = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
/* Boolean debug switch for PM decisions / ADF */
    /*
     * Usage: echo "debug adf_ms #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_ms to given value
     */
    if(!strncmp("adf_ms", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_ms to %u\n",
                   debug_level);
            rsbac_debug_adf_ms = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
/* Boolean debug switch for RC data structures */
    /*
     * Usage: echo "debug ds_rc #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_rc to given value
     */
    if(!strncmp("ds_rc", k_buf + 6, 5)) 
      {
	p += 6;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_rc to %u\n",
                   debug_level);
            rsbac_debug_ds_rc = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for RC syscalls / AEF */
    /*
     * Usage: echo "debug aef_rc #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_rc to given value
     */
    if(!strncmp("aef_rc", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_rc to %u\n",
                   debug_level);
            rsbac_debug_aef_rc = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for RC decisions / ADF */
    /*
     * Usage: echo "debug adf_rc #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_rc to given value
     */
    if(!strncmp("adf_rc", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_rc to %u\n",
                   debug_level);
            rsbac_debug_adf_rc = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_AUTH)
/* Boolean debug switch for AUTH data structures */
    /*
     * Usage: echo "debug ds_auth #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_auth to given value
     */
    if(!strncmp("ds_auth", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_auth to %u\n",
                   debug_level);
            rsbac_debug_ds_auth = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for AUTH syscalls / AEF */
    /*
     * Usage: echo "debug aef_auth #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_auth to given value
     */
    if(!strncmp("aef_auth", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_auth to %u\n",
                   debug_level);
            rsbac_debug_aef_auth = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for AUTH decisions / ADF */
    /*
     * Usage: echo "debug adf_auth #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_auth to given value
     */
    if(!strncmp("adf_auth", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_auth to %u\n",
                   debug_level);
            rsbac_debug_adf_auth = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
/* Boolean debug switch for REG */
    /*
     * Usage: echo "debug reg #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_reg to given value
     */
    if(!strncmp("reg", k_buf + 6, 3)) 
      {
	p += 3;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_reg to %u\n",
                   debug_level);
            rsbac_debug_reg = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_ACL)
/* Boolean debug switch for ACL data structures */
    /*
     * Usage: echo "debug ds_acl #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds_acl to given value
     */
    if(!strncmp("ds_acl", k_buf + 6, 6)) 
      {
	p += 7;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds_acl to %u\n",
                   debug_level);
            rsbac_debug_ds_acl = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
/* Boolean debug switch for ACL syscalls / AEF */
    /*
     * Usage: echo "debug aef_acl #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_acl to given value
     */
    if(!strncmp("aef_acl", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_acl to %u\n",
                   debug_level);
            rsbac_debug_aef_acl = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for ACL decisions / ADF */
    /*
     * Usage: echo "debug adf_acl #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_acl to given value
     */
    if(!strncmp("adf_acl", k_buf + 6, 7)) 
      {
	p += 8;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_acl to %u\n",
                   debug_level);
            rsbac_debug_adf_acl = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

#if defined(CONFIG_RSBAC_JAIL)
/* Boolean debug switch for JAIL syscalls / AEF */
    /*
     * Usage: echo "debug aef_jail #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef_jail to given value
     */
    if(!strncmp("aef_jail", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef_jail to %u\n",
                   debug_level);
            rsbac_debug_aef_jail = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for JAIL decisions / ADF */
    /*
     * Usage: echo "debug adf_jail #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_adf_jail to given value
     */
    if(!strncmp("adf_jail", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_adf_jail to %u\n",
                   debug_level);
            rsbac_debug_adf_jail = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif

    /*
     * Usage: echo "debug ds #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_ds to given value
     */
    if(!strncmp("ds", k_buf + 6, 2)) 
      {
	p += 3;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_ds to %u\n",
                   debug_level);
            rsbac_debug_ds = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

    /*
     * Usage: echo "debug write #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_write to given value
     */
    if(!strncmp("write", k_buf + 6, 5)) 
      {
	p += 6;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_write to %u\n",
                   debug_level);
            rsbac_debug_write = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

    /*
     * Usage: echo "debug stack #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_stack to given value
     */
    if(!strncmp("stack", k_buf + 6, 5)) 
      {
	p += 6;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_stack to %u\n",
                   debug_level);
            rsbac_debug_stack = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

    /*
     * Usage: echo "debug lists #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_lists to given value
     */
    if(!strncmp("lists", k_buf + 6, 5)) 
      {
	p += 6;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_lists to %u\n",
                   debug_level);
            rsbac_debug_lists = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

    /* Boolean debug switch for AEF */
    /*
     * Usage: echo "debug aef #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_aef to given value
     */
    if(!strncmp("aef", k_buf + 6, 3)) 
      {
	p += 4;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_aef to %u\n",
                   debug_level);
            rsbac_debug_aef = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

/* Boolean debug switch for NO_WRITE */
    /*
     * Usage: echo "debug no_write #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_no_write to given value
     */
    if(!strncmp("no_write", k_buf + 6, 8)) 
      {
	p += 9;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_no_write to %u\n",
                   debug_level);
            rsbac_debug_no_write = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }

#if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
    /*
     * Usage: echo "debug auto #N" > /proc/rsbac_info/debug
     *   to set rsbac_debug_auto to given value
     */
    if(!strncmp("auto", k_buf + 6, 4)) 
      {
	p += 5;

        if( *p == '\0' )
            goto out;

        debug_level = simple_strtoul(p, NULL, 0);
        /* only accept 0 or 1 */
        if(!debug_level || (debug_level == 1))
          {
            printk(KERN_INFO
                   "debug_proc_write(): setting rsbac_debug_auto to %u\n",
                   debug_level);
            rsbac_debug_auto = debug_level;
            err = count;
            goto out;
          }
        else
          {
            printk(KERN_INFO
                   "debug_proc_write(): rejecting invalid debug level (should be 0 or 1)\n");
            goto out;
          }
      }
#endif /* CONFIG_RSBAC_AUTO_WRITE > 0 */
#endif /* DEBUG */

out:
  free_page((ulong) k_buf);
  return(err);
  }
#endif /* defined(CONFIG_RSBAC_PROC) && defined(CONFIG_PROC_FS) */

#if defined(CONFIG_RSBAC_LOG_REMOTE)
/* declare net functions */
long sys_socket(int family, int type, int protocol);
long sys_bind(int fd, struct sockaddr *umyaddr, int addrlen);
long sys_sendto(int fd, void * buff, size_t len, unsigned flags,
                struct sockaddr *addr, int addr_len);

#ifndef CONFIG_RSBAC_LOG_REMOTE_SYNC
/* rsbac kernel timer for auto-write */
static void wakeup_rsbaclogd(u_long dummy)
  {
    wake_up(&rsbaclogd_wait);
  }
#endif

/* rsbac kernel daemon for remote logging */
static int rsbaclogd(void * dummy)
  {
    struct task_struct *tsk = current;
    int err;
    int i;
    char c;
    int sock_fd;
    struct sockaddr_in addr;
    char * tmp = rsbac_kmalloc(RSBAC_MAXNAMELEN);
    static char remote_send_buf[REMOTE_SEND_BUF_LEN];
    mm_segment_t oldfs;

    printk(KERN_INFO "rsbaclogd(): Initializing.\n");
    tsk->session = 1;
    tsk->pgrp = 1;
    strcpy(tsk->comm, "rsbaclogd");
/*    lock_kernel(); */

#ifdef CONFIG_RSBAC_DEBUG
    printk(KERN_DEBUG "rsbaclogd(): Setting auto timer.\n");
#endif
#ifndef CONFIG_RSBAC_LOG_REMOTE_SYNC
    init_timer(&rsbac_log_remote_timer);
    rsbac_log_remote_timer.function = wakeup_rsbaclogd;
    rsbac_log_remote_timer.data = 0;
    rsbac_log_remote_timer.expires = jiffies + rsbac_log_remote_interval;
    add_timer(&rsbac_log_remote_timer);
#endif
    interruptible_sleep_on(&rsbaclogd_wait);

    /* create a socket */
    sock_fd = sys_socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sock_fd < 0)
      {
        printk(KERN_WARNING
               "rsbaclogd(): creating local log socket failed with error %s, exiting!\n",
               get_error_name(tmp, sock_fd));
        rsbaclogd_pid = 0;
        return -RSBAC_EWRITEFAILED;
      }
    /* bind local address */
    addr.sin_family = PF_INET;
    addr.sin_port = htons(CONFIG_RSBAC_LOG_LOCAL_PORT);
    err = rsbac_net_str_to_inet(CONFIG_RSBAC_LOG_LOCAL_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "rsbaclogd(): converting local socket address %s failed with error %s, exiting!\n",
               CONFIG_RSBAC_LOG_LOCAL_ADDR,
               get_error_name(tmp, err));
        sys_close(sock_fd);
        rsbaclogd_pid = 0;
        return -RSBAC_EINVALIDVALUE;
      }
    /* change data segment - sys_bind reads address from user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    err = sys_bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr));
    set_fs(oldfs);
    if(err < 0)
      {
        printk(KERN_WARNING
               "rsbaclogd(): binding local socket address %u.%u.%u.%u:%u failed with error %s, exiting!\n",
               NIPQUAD(addr.sin_addr.s_addr),
               CONFIG_RSBAC_LOG_LOCAL_PORT,
               get_error_name(tmp, err));
        sys_close(sock_fd);
        rsbaclogd_pid = 0;
        return -RSBAC_EWRITEFAILED;
      }

    /* convert remote address */
    addr.sin_family = PF_INET;
    addr.sin_port = htons(CONFIG_RSBAC_LOG_REMOTE_PORT);
    err = rsbac_net_str_to_inet(CONFIG_RSBAC_LOG_REMOTE_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "rsbaclogd(): converting remote socket address %s failed with error %s, exiting!\n",
               CONFIG_RSBAC_LOG_REMOTE_ADDR,
               get_error_name(tmp, err));
        sys_close(sock_fd);
        rsbaclogd_pid = 0;
        return -RSBAC_EINVALIDVALUE;
      }

#ifdef CONFIG_RSBAC_DEBUG
    if(rsbac_debug_stack)
      {
        unsigned long * n = (unsigned long *) (current+1);

        while (!*n)
          n++;
	printk(KERN_DEBUG "rsbaclogd: free stack: %lu\n",
	       (unsigned long) n - (unsigned long)(current+1));
      }
#endif
    for(;;)
      {
        /* wait */
        spin_lock_irq(&tsk->sigmask_lock);
        flush_signals(tsk);
        sigfillset(&tsk->blocked);
        recalc_sigpending(tsk);
        spin_unlock_irq(&tsk->sigmask_lock);
/*
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_auto)
          printk(KERN_DEBUG
                 "rsbacd(): calling rsbac_write()\n");
#endif
*/

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
	while(remote_rlog_size)
	  {
            cli();
            i = 0;
            c = 0;
            while (remote_rlog_size && i < REMOTE_SEND_BUF_LEN && c != '\n')
              {
                c = *((char *) remote_rlog_buf+remote_log_start);
		remote_log_start++;
		remote_rlog_size--;
		remote_log_start &= REMOTE_RLOG_BUF_MASK;
		sti();
		remote_send_buf[i] = c;
		i++;
		cli();
              }
            sti();
#else
	while(remote_log_start != remote_log_end)
	  {
            i = 0;
            c = 0;
            spin_lock_irq(&rsbac_log_remote_lock);
            while (   (remote_log_start != remote_log_end)
                   && (i < sizeof(remote_send_buf))
                   && (c != '\n')
                  )
              {
                c = REMOTE_RLOG_BUF(remote_log_start);
                remote_log_start++;
                remote_logged_chars--;
		remote_send_buf[i] = c;
//                spin_unlock_irq(&rsbac_log_remote_lock);
		i++;
//                spin_lock_irq(&rsbac_log_remote_lock);
              }
            spin_unlock_irq(&rsbac_log_remote_lock);
#endif

            /* change data segment - sys_sendto reads data and address from user space */
            oldfs = get_fs();
            set_fs(KERNEL_DS);
            err = sys_sendto(sock_fd,
                             remote_send_buf,
                             i,
                             MSG_DONTWAIT,
                             (struct sockaddr *)&addr,
                             sizeof(addr));
            set_fs(oldfs);
            if(   (err < 0)
//               && (err != -EPERM)
              )
              {
                printk(KERN_WARNING
                       "rsbaclogd(): sending to remote socket address %u.%u.%u.%u:%u failed with error %i!\n",
                       NIPQUAD(addr.sin_addr.s_addr),
                       CONFIG_RSBAC_LOG_REMOTE_PORT,
                       err);
                continue;
              }
          }
#ifndef CONFIG_RSBAC_LOG_REMOTE_SYNC
        /* set new timer (only, if not woken up by rsbac_printk()) */
        mod_timer(&rsbac_log_remote_timer, jiffies + rsbac_log_remote_interval);
#endif
        interruptible_sleep_on(&rsbaclogd_wait);
        /* sleep */
      }
  }
#endif


/********************************/
/*             Init             */
/********************************/

#ifdef CONFIG_RSBAC_INIT_DELAY
inline void rsbac_init_debug(void)
#else
inline void __init rsbac_init_debug(void)
#endif
  {
    int i;
    #if defined(CONFIG_RSBAC_PROC) && defined(CONFIG_PROC_FS)
    struct proc_dir_entry * tmp_entry_p;
    #endif
    
    if (!rsbac_is_initialized())
      {
        struct rsbac_list_info_t info;
        int tmperr;
        rsbac_log_entry_t def_data;

#if 0 && defined(CONFIG_RSBAC_RMSG)
        /* init rsbac_log_lock */
        spin_lock_init(&rsbac_log_lock);
#endif
        /* register log_levels list */
        for(i=0; i<=T_NONE; i++)
          def_data[i] = LL_denied;
        info.version = RSBAC_LOG_LEVEL_VERSION;
        info.key = RSBAC_LOG_LEVEL_KEY;
        info.desc_size = sizeof(rsbac_adf_request_int_t);
        info.data_size = sizeof(rsbac_log_entry_t);
        info.max_age = 0;
        tmperr = rsbac_list_register(RSBAC_LIST_VERSION,
                                     &log_levels_handle,
                                     info,
                                     RSBAC_LIST_PERSIST | RSBAC_LIST_DEF_DATA,
                                     NULL,
                                     NULL,
                                     &def_data,
                                     RSBAC_LOG_LEVEL_LIST_NAME,
                                     0);
        if(tmperr)
          {
            char * tmp;

            tmp = rsbac_kmalloc(RSBAC_MAXNAMELEN);
            if(tmp)
              {
                printk(KERN_WARNING
                       "rsbac_init_debug(): registering log levels list ll failed with error %s!\n",
                       get_error_name(tmp, tmperr));
                rsbac_kfree(tmp);
              }
            memset(rsbac_log_levels, LL_denied, sizeof(rsbac_log_levels));
          }
        else
          {
            rsbac_adf_request_int_t req;

            for(req = 0; req < R_NONE; req++)
              rsbac_list_get_data(log_levels_handle, &req, rsbac_log_levels[req]);
          }

        #if defined(CONFIG_RSBAC_PROC) && defined(CONFIG_PROC_FS)
        tmp_entry_p = create_proc_entry("log_levels",
                                        S_IFREG | S_IRUGO | S_IWUGO,
                                        proc_rsbac_root_p);
        if(tmp_entry_p)
          {
            tmp_entry_p->get_info = log_levels_proc_info;
            tmp_entry_p->write_proc = log_levels_proc_write;
          }

        tmp_entry_p = create_proc_entry("debug",
                                        S_IFREG | S_IRUGO | S_IWUGO,
                                        proc_rsbac_root_p);
        if(tmp_entry_p)
          {
            tmp_entry_p->get_info = debug_proc_info;
            tmp_entry_p->write_proc = debug_proc_write;
          }

        #if defined(CONFIG_RSBAC_RMSG)
        tmp_entry_p = create_proc_entry("rmsg",
                                        S_IFREG | S_IRUGO | S_IWUGO,
                                        proc_rsbac_root_p);
        if(tmp_entry_p)
          {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,48)
            tmp_entry_p->ops = &proc_rmsg_inode_operations;
#else
            tmp_entry_p->proc_fops = &proc_rmsg_operations;
#endif
          }
        #endif
        #endif

        #if defined(CONFIG_RSBAC_LOG_REMOTE)
        /* init rsbac_console lock */
//        spin_lock_init(&rsbac_log_remote_lock);
        /* init the rsbaclogd wait queue head */
        init_waitqueue_head(&rsbaclogd_wait);
        /* Start rsbac logging thread for auto write */
        rsbaclogd_pid = kernel_thread(rsbaclogd, NULL, 0);
        printk(KERN_INFO "rsbac_init_debug(): Started rsbaclogd thread with pid %u\n",
               rsbaclogd_pid);
        #endif
      }

    #ifdef CONFIG_RSBAC_SOFTMODE
    if(rsbac_softmode)
      printk(KERN_DEBUG "rsbac_softmode is set\n");
    #endif
    #ifdef CONFIG_RSBAC_CAP_PROC_HIDE
    if(rsbac_cap_process_hiding)
      printk(KERN_DEBUG "rsbac_cap_process_hiding is set\n");
    #endif
    #ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_FULL
    if(rsbac_dac_disable)
      printk(KERN_DEBUG "rsbac_dac_disable is set\n");
    #endif
    #ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
    if(rsbac_nosyslog)
      printk(KERN_DEBUG "rsbac_nosyslog is set\n");
    #endif
    #ifdef CONFIG_RSBAC_INIT_DELAY
    if(rsbac_delay_init)
      printk(KERN_DEBUG "rsbac_delay_init is set\n");
    if(MAJOR(rsbac_delayed_root) || MINOR(rsbac_delayed_root))
      printk(KERN_DEBUG "rsbac_delayed_root is %02u:%02u\n",
             MAJOR(rsbac_delayed_root), MINOR(rsbac_delayed_root));
    #endif
    if(rsbac_no_defaults)
      printk(KERN_DEBUG "rsbac_no_defaults is set\n");

#if defined(CONFIG_RSBAC_DEBUG)
    if(rsbac_debug_ds)
      printk(KERN_DEBUG "rsbac_debug_ds is set\n");
    if(rsbac_debug_write)
      printk(KERN_DEBUG "rsbac_debug_write is set\n");
    if(rsbac_debug_no_write)
      printk(KERN_DEBUG "rsbac_debug_no_write is set\n");
    if(rsbac_debug_stack)
      printk(KERN_DEBUG "rsbac_debug_stack is set\n");
    if(rsbac_debug_lists)
      printk(KERN_DEBUG "rsbac_debug_lists is set\n");
    if(rsbac_debug_aef)
      printk(KERN_DEBUG "rsbac_debug_aef is set\n");
    if(rsbac_debug_adf_default != 1)
      printk(KERN_DEBUG "rsbac_debug_adf_default is set to %i\n",
             rsbac_debug_adf_default);

    #if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
    if(rsbac_debug_reg)
      printk(KERN_DEBUG "rsbac_debug_reg is set\n");
    #endif

    #if defined(CONFIG_RSBAC_NET)
    if(rsbac_debug_ds_net)
      printk(KERN_DEBUG "rsbac_debug_ds_net is set\n");
    if(rsbac_debug_aef_net)
      printk(KERN_DEBUG "rsbac_debug_aef_net is set\n");
    if(rsbac_debug_adf_net)
      printk(KERN_DEBUG "rsbac_debug_adf_net is set\n");
    #endif

    #if defined(CONFIG_RSBAC_MAC)
    if(rsbac_debug_ds_mac)
      printk(KERN_DEBUG "rsbac_debug_ds_mac is set\n");
    if(rsbac_debug_aef_mac)
      printk(KERN_DEBUG "rsbac_debug_aef_mac is set\n");
    if(rsbac_debug_adf_mac)
      printk(KERN_DEBUG "rsbac_debug_adf_mac is set\n");
    #endif

    #if defined(CONFIG_RSBAC_PM) || defined(CONFIG_RSBAC_PM_MAINT)
    if(rsbac_debug_ds_pm)
      printk(KERN_DEBUG "rsbac_debug_ds_pm is set\n");
    if(rsbac_debug_aef_pm)
      printk(KERN_DEBUG "rsbac_debug_aef_pm is set\n");
    if(rsbac_debug_adf_pm)
      printk(KERN_DEBUG "rsbac_debug_adf_pm is set\n");
    #endif

    #if defined(CONFIG_RSBAC_MS) || defined(CONFIG_RSBAC_MS_MAINT)
    if(rsbac_debug_adf_ms)
      printk(KERN_DEBUG "rsbac_debug_adf_ms is set\n");
    #endif

    #if defined(CONFIG_RSBAC_RC) || defined(CONFIG_RSBAC_RC_MAINT)
    if(rsbac_debug_ds_rc)
      printk(KERN_DEBUG "rsbac_debug_ds_rc is set\n");
    if(rsbac_debug_aef_rc)
      printk(KERN_DEBUG "rsbac_debug_aef_rc is set\n");
    if(rsbac_debug_adf_rc)
      printk(KERN_DEBUG "rsbac_debug_adf_rc is set\n");
    #endif

    #if defined(CONFIG_RSBAC_AUTH) || defined(CONFIG_RSBAC_AUTH_MAINT)
    if(rsbac_debug_ds_auth)
      printk(KERN_DEBUG "rsbac_debug_ds_auth is set\n");
    if(rsbac_debug_aef_auth)
      printk(KERN_DEBUG "rsbac_debug_aef_auth is set\n");
    if(rsbac_debug_adf_auth)
      printk(KERN_DEBUG "rsbac_debug_adf_auth is set\n");
    #endif

    #if defined(CONFIG_RSBAC_ACL) || defined(CONFIG_RSBAC_ACL_MAINT)
    if(rsbac_debug_ds_acl)
      printk(KERN_DEBUG "rsbac_debug_ds_acl is set\n");
    if(rsbac_debug_aef_acl)
      printk(KERN_DEBUG "rsbac_debug_aef_acl is set\n");
    if(rsbac_debug_adf_acl)
      printk(KERN_DEBUG "rsbac_debug_adf_acl is set\n");
    #endif

    #if defined(CONFIG_RSBAC_JAIL)
    if(rsbac_debug_aef_jail)
      printk(KERN_DEBUG "rsbac_debug_aef_jail is set\n");
    if(rsbac_debug_adf_jail)
      printk(KERN_DEBUG "rsbac_debug_adf_jail is set\n");
    #endif

    #if defined(CONFIG_RSBAC_AUTO_WRITE) && (CONFIG_RSBAC_AUTO_WRITE > 0)
    if(rsbac_debug_auto)
      printk(KERN_DEBUG "rsbac_debug_auto is set\n");
    #endif
#endif /* DEBUG */
  };

