/* linux/net/inet/arp.c
 *
 * Version:	$Id: arp.c,v 1.99 2001/08/30 22:55:42 davem Exp $
 *
 * Copyright (C) 1994 by Florian  La Roche
 *
 * This module implements the Address Resolution Protocol ARP (RFC 826),
 * which is used to convert IP addresses (or in the future maybe other
 * high-level addresses) into a low-level hardware address (like an Ethernet
 * address).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * Fixes:
 *		Alan Cox	:	Removed the Ethernet assumptions in 
 *					Florian's code
 *		Alan Cox	:	Fixed some small errors in the ARP 
 *					logic
 *		Alan Cox	:	Allow >4K in /proc
 *		Alan Cox	:	Make ARP add its own protocol entry
 *		Ross Martin     :       Rewrote arp_rcv() and arp_get_info()
 *		Stephen Henson	:	Add AX25 support to arp_get_info()
 *		Alan Cox	:	Drop data when a device is downed.
 *		Alan Cox	:	Use init_timer().
 *		Alan Cox	:	Double lock fixes.
 *		Martin Seine	:	Move the arphdr structure
 *					to if_arp.h for compatibility.
 *					with BSD based programs.
 *		Andrew Tridgell :       Added ARP netmask code and
 *					re-arranged proxy handling.
 *		Alan Cox	:	Changed to use notifiers.
 *		Niibe Yutaka	:	Reply for this device or proxies only.
 *		Alan Cox	:	Don't proxy across hardware types!
 *		Jonathan Naylor :	Added support for NET/ROM.
 *		Mike Shaver     :       RFC1122 checks.
 *		Jonathan Naylor :	Only lookup the hardware address for
 *					the correct hardware type.
 *		Germano Caronni	:	Assorted subtle races.
 *		Craig Schlenter :	Don't modify permanent entry 
 *					during arp_rcv.
 *		Russ Nelson	:	Tidied up a few bits.
 *		Alexey Kuznetsov:	Major changes to caching and behaviour,
 *					eg intelligent arp probing and 
 *					generation
 *					of host down events.
 *		Alan Cox	:	Missing unlock in device events.
 *		Eckes		:	ARP ioctl control errors.
 *		Alexey Kuznetsov:	Arp free fix.
 *		Manuel Rodriguez:	Gratuitous ARP.
 *              Jonathan Layes  :       Added arpd support through kerneld 
 *                                      message queue (960314)
 *		Mike Shaver	:	/proc/sys/net/ipv4/arp_* support
 *		Mike McLagan    :	Routing by source
 *		Stuart Cheshire	:	Metricom and grat arp fixes
 *					*** FOR 2.1 clean this up ***
 *		Lawrence V. Stefani: (08/12/96) Added FDDI support.
 *		Alan Cox 	:	Took the AP1000 nasty FDDI hack and
 *					folded into the mainstream FDDI code.
 *					Ack spit, Linus how did you allow that
 *					one in...
 *		Jes Sorensen	:	Make FDDI work again in 2.1.x and
 *					clean up the APFDDI & gen. FDDI bits.
 *		Alexey Kuznetsov:	new arp state machine;
 *					now it is in net/core/neighbour.c.
 *		Krzysztof Halasa:	Added Frame Relay ARP support.
 *		Julian Anastasov:	"hidden" flag: hide the
 *					interface and don't reply for it
 *		Julian Anastasov:	ARP filtering via netlink
 */

#include <linux/types.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/config.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/errno.h>
#include <linux/in.h>
#include <linux/mm.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/fddidevice.h>
#include <linux/if_arp.h>
#include <linux/trdevice.h>
#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>
#include <linux/netlink.h>
#include <linux/init.h>
#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif

#include <net/ip.h>
#include <net/icmp.h>
#include <net/route.h>
#include <net/protocol.h>
#include <net/tcp.h>
#include <net/sock.h>
#include <net/arp.h>
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
#include <net/ax25.h>
#if defined(CONFIG_NETROM) || defined(CONFIG_NETROM_MODULE)
#include <net/netrom.h>
#endif
#endif
#ifdef CONFIG_ATM_CLIP
#include <net/atmclip.h>
#endif

#include <asm/system.h>
#include <asm/uaccess.h>

#include <linux/netfilter_arp.h>

/*
 *	Interface to generic neighbour cache.
 */
static u32 arp_hash(const void *pkey, const struct net_device *dev);
static int arp_constructor(struct neighbour *neigh);
static void arp_solicit(struct neighbour *neigh, struct sk_buff *skb);
static void arp_error_report(struct neighbour *neigh, struct sk_buff *skb);
static void parp_redo(struct sk_buff *skb);

static struct neigh_ops arp_generic_ops = {
	family:			AF_INET,
	solicit:		arp_solicit,
	error_report:		arp_error_report,
	output:			neigh_resolve_output,
	connected_output:	neigh_connected_output,
	hh_output:		dev_queue_xmit,
	queue_xmit:		dev_queue_xmit,
};

static struct neigh_ops arp_hh_ops = {
	family:			AF_INET,
	solicit:		arp_solicit,
	error_report:		arp_error_report,
	output:			neigh_resolve_output,
	connected_output:	neigh_resolve_output,
	hh_output:		dev_queue_xmit,
	queue_xmit:		dev_queue_xmit,
};

static struct neigh_ops arp_direct_ops = {
	family:			AF_INET,
	output:			dev_queue_xmit,
	connected_output:	dev_queue_xmit,
	hh_output:		dev_queue_xmit,
	queue_xmit:		dev_queue_xmit,
};

struct neigh_ops arp_broken_ops = {
	family:			AF_INET,
	solicit:		arp_solicit,
	error_report:		arp_error_report,
	output:			neigh_compat_output,
	connected_output:	neigh_compat_output,
	hh_output:		dev_queue_xmit,
	queue_xmit:		dev_queue_xmit,
};

struct neigh_table arp_tbl = {
	family:		AF_INET,
	entry_size:	sizeof(struct neighbour) + 4,
	key_len:	4,
	hash:		arp_hash,
	constructor:	arp_constructor,
	proxy_redo:	parp_redo,
	id:		"arp_cache",
	parms: {
		tbl:			&arp_tbl,
		base_reachable_time:	30 * HZ,
		retrans_time:		1 * HZ,
		gc_staletime:		60 * HZ,
		reachable_time:		30 * HZ,
		delay_probe_time:	5 * HZ,
		queue_len:		3,
		ucast_probes:		3,
		mcast_probes:		3,
		anycast_delay:		1 * HZ,
		proxy_delay:		(8 * HZ) / 10,
		proxy_qlen:		64,
		locktime:		1 * HZ,
	},
	gc_interval:	30 * HZ,
	gc_thresh1:	128,
	gc_thresh2:	512,
	gc_thresh3:	1024,
};

struct arpf_node {
	struct arpf_node *	at_next;
	u32			at_pref;
	u32			at_from;
	u32			at_from_mask;
	u32			at_to;
	u32			at_to_mask;
	u32			at_src;
	atomic_t		at_packets;
	atomic_t		at_refcnt;
	unsigned		at_flags;
	unsigned char		at_from_len;
	unsigned char		at_to_len;
	unsigned char		at_action;
	char			at_dead;
	unsigned char		at_llfrom_len;
	unsigned char		at_llto_len;
	unsigned char		at_llsrc_len;
	unsigned char		at_lldst_len;
	unsigned char		at_iif_len;
	unsigned char		at_oif_len;
	unsigned short		at__pad1;
	unsigned char		at_llfrom[MAX_ADDR_LEN];
	unsigned char		at_llto[MAX_ADDR_LEN];
	unsigned char		at_llsrc[MAX_ADDR_LEN];
	unsigned char		at_lldst[MAX_ADDR_LEN];
	char			at_iif[IFNAMSIZ];
	char			at_oif[IFNAMSIZ];
};

static struct arpf_node *arp_tabs[3];

static kmem_cache_t *arpf_cachep;

static rwlock_t arpf_lock = RW_LOCK_UNLOCKED;

static void
arpf_send(int table, struct sk_buff *skb, u32 sip, u32 tip,
	  unsigned char *from_hw, unsigned char *to_hw,
	  struct net_device *idev, struct net_device *odev);

int arp_mc_map(u32 addr, u8 *haddr, struct net_device *dev, int dir)
{
	switch (dev->type) {
	case ARPHRD_ETHER:
	case ARPHRD_FDDI:
	case ARPHRD_IEEE802:
		ip_eth_mc_map(addr, haddr);
		return 0; 
	case ARPHRD_IEEE802_TR:
		ip_tr_mc_map(addr, haddr);
		return 0;
	default:
		if (dir) {
			memcpy(haddr, dev->broadcast, dev->addr_len);
			return 0;
		}
	}
	return -EINVAL;
}

/* RSBAC */
#ifdef CONFIG_RSBAC_NET_DEV
#include <rsbac/adf.h>
#endif

static u32 arp_hash(const void *pkey, const struct net_device *dev)
{
	u32 hash_val;

	hash_val = *(u32*)pkey;
	hash_val ^= (hash_val>>16);
	hash_val ^= hash_val>>8;
	hash_val ^= hash_val>>3;
	hash_val = (hash_val^dev->ifindex)&NEIGH_HASHMASK;

	return hash_val;
}

static int arp_constructor(struct neighbour *neigh)
{
	u32 addr = *(u32*)neigh->primary_key;
	struct net_device *dev = neigh->dev;
	struct in_device *in_dev = in_dev_get(dev);

	if (in_dev == NULL)
		return -EINVAL;

	neigh->type = inet_addr_type(addr);
	if (in_dev->arp_parms)
		neigh->parms = in_dev->arp_parms;

	in_dev_put(in_dev);

	if (dev->hard_header == NULL) {
		neigh->nud_state = NUD_NOARP;
		neigh->ops = &arp_direct_ops;
		neigh->output = neigh->ops->queue_xmit;
	} else {
		/* Good devices (checked by reading texts, but only Ethernet is
		   tested)

		   ARPHRD_ETHER: (ethernet, apfddi)
		   ARPHRD_FDDI: (fddi)
		   ARPHRD_IEEE802: (tr)
		   ARPHRD_METRICOM: (strip)
		   ARPHRD_ARCNET:
		   etc. etc. etc.

		   ARPHRD_IPDDP will also work, if author repairs it.
		   I did not it, because this driver does not work even
		   in old paradigm.
		 */

#if 1
		/* So... these "amateur" devices are hopeless.
		   The only thing, that I can say now:
		   It is very sad that we need to keep ugly obsolete
		   code to make them happy.

		   They should be moved to more reasonable state, now
		   they use rebuild_header INSTEAD OF hard_start_xmit!!!
		   Besides that, they are sort of out of date
		   (a lot of redundant clones/copies, useless in 2.1),
		   I wonder why people believe that they work.
		 */
		switch (dev->type) {
		default:
			break;
		case ARPHRD_ROSE:	
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
		case ARPHRD_AX25:
#if defined(CONFIG_NETROM) || defined(CONFIG_NETROM_MODULE)
		case ARPHRD_NETROM:
#endif
			neigh->ops = &arp_broken_ops;
			neigh->output = neigh->ops->output;
			return 0;
#endif
		;}
#endif
		if (neigh->type == RTN_MULTICAST) {
			neigh->nud_state = NUD_NOARP;
			arp_mc_map(addr, neigh->ha, dev, 1);
		} else if (dev->flags&(IFF_NOARP|IFF_LOOPBACK)) {
			neigh->nud_state = NUD_NOARP;
			memcpy(neigh->ha, dev->dev_addr, dev->addr_len);
		} else if (neigh->type == RTN_BROADCAST || dev->flags&IFF_POINTOPOINT) {
			neigh->nud_state = NUD_NOARP;
			memcpy(neigh->ha, dev->broadcast, dev->addr_len);
		}
		if (dev->hard_header_cache)
			neigh->ops = &arp_hh_ops;
		else
			neigh->ops = &arp_generic_ops;
		if (neigh->nud_state&NUD_VALID)
			neigh->output = neigh->ops->connected_output;
		else
			neigh->output = neigh->ops->output;
	}
	return 0;
}

static void arp_error_report(struct neighbour *neigh, struct sk_buff *skb)
{
	dst_link_failure(skb);
	kfree_skb(skb);
}

static void arp_solicit(struct neighbour *neigh, struct sk_buff *skb)
{
	u32 saddr;
	int from_skb;
	struct in_device *in_dev2 = NULL;
	struct net_device *dev2 = NULL;
	u8  *dst_ha = NULL;
	struct net_device *dev = neigh->dev;
	u32 target = *(u32*)neigh->primary_key;
	int probes = atomic_read(&neigh->probes);
	unsigned char tha[MAX_ADDR_LEN];

	from_skb = (skb &&
		(dev2 = ip_dev_find(skb->nh.iph->saddr)) != NULL &&
		(in_dev2 = in_dev_get(dev2)) != NULL &&
		!IN_DEV_HIDDEN(in_dev2));
	if (dev2) {
		if (in_dev2) in_dev_put(in_dev2);
		dev_put(dev2);
	}
	if (from_skb)
		saddr = skb->nh.iph->saddr;
	else
		saddr = inet_select_addr(dev, target, RT_SCOPE_LINK);

	if (!saddr)
		return;
	if ((probes -= neigh->parms->ucast_probes) < 0) {
		if (!(neigh->nud_state&NUD_VALID))
			printk(KERN_DEBUG "trying to ucast probe in NUD_INVALID\n");
		dst_ha = tha;
		read_lock_bh(&neigh->lock);
		memcpy(dst_ha, neigh->ha, dev->addr_len);
		read_unlock_bh(&neigh->lock);
	} else if ((probes -= neigh->parms->app_probes) < 0) {
#ifdef CONFIG_ARPD
		neigh_app_ns(neigh);
#endif
		return;
	}

	arpf_send(ARPA_TABLE_OUTPUT,skb,saddr,target,NULL,dst_ha,NULL,dev);
}

static int arp_filter(__u32 sip, __u32 tip, struct net_device *dev)
{
	struct rtable *rt;
	int flag = 0; 
	/*unsigned long now; */

	if (ip_route_output(&rt, sip, tip, 0, 0) < 0) 
		return 1;
	if (rt->u.dst.dev != dev) { 
		NET_INC_STATS_BH(ArpFilter);
		flag = 1;
	} 
	ip_rt_put(rt); 
	return flag; 
} 

/* OBSOLETE FUNCTIONS */

/*
 *	Find an arp mapping in the cache. If not found, post a request.
 *
 *	It is very UGLY routine: it DOES NOT use skb->dst->neighbour,
 *	even if it exists. It is supposed that skb->dev was mangled
 *	by a virtual device (eql, shaper). Nobody but broken devices
 *	is allowed to use this function, it is scheduled to be removed. --ANK
 */

static int arp_set_predefined(int addr_hint, unsigned char * haddr, u32 paddr, struct net_device * dev)
{
	switch (addr_hint) {
	case RTN_LOCAL:
		printk(KERN_DEBUG "ARP: arp called for own IP address\n");
		memcpy(haddr, dev->dev_addr, dev->addr_len);
		return 1;
	case RTN_MULTICAST:
		arp_mc_map(paddr, haddr, dev, 1);
		return 1;
	case RTN_BROADCAST:
		memcpy(haddr, dev->broadcast, dev->addr_len);
		return 1;
	}
	return 0;
}


int arp_find(unsigned char *haddr, struct sk_buff *skb)
{
	struct net_device *dev = skb->dev;
	u32 paddr;
	struct neighbour *n;

	if (!skb->dst) {
		printk(KERN_DEBUG "arp_find is called with dst==NULL\n");
		kfree_skb(skb);
		return 1;
	}

	paddr = ((struct rtable*)skb->dst)->rt_gateway;

	if (arp_set_predefined(inet_addr_type(paddr), haddr, paddr, dev))
		return 0;

	n = __neigh_lookup(&arp_tbl, &paddr, dev, 1);

	if (n) {
		n->used = jiffies;
		if (n->nud_state&NUD_VALID || neigh_event_send(n, skb) == 0) {
			read_lock_bh(&n->lock);
 			memcpy(haddr, n->ha, dev->addr_len);
			read_unlock_bh(&n->lock);
			neigh_release(n);
			return 0;
		}
		neigh_release(n);
	} else
		kfree_skb(skb);
	return 1;
}

/* END OF OBSOLETE FUNCTIONS */

int arp_bind_neighbour(struct dst_entry *dst)
{
	struct net_device *dev = dst->dev;
	struct neighbour *n = dst->neighbour;

	if (dev == NULL)
		return -EINVAL;
	if (n == NULL) {
		u32 nexthop = ((struct rtable*)dst)->rt_gateway;
		if (dev->flags&(IFF_LOOPBACK|IFF_POINTOPOINT))
			nexthop = 0;
		n = __neigh_lookup_errno(
#ifdef CONFIG_ATM_CLIP
		    dev->type == ARPHRD_ATM ? &clip_tbl :
#endif
		    &arp_tbl, &nexthop, dev);
		if (IS_ERR(n))
			return PTR_ERR(n);
		dst->neighbour = n;
	}
	return 0;
}

/*
 * Check if we can use proxy ARP for this path
 */

static inline int arp_fwd_proxy(struct in_device *in_dev, struct rtable *rt)
{
	struct in_device *out_dev;
	int imi, omi = -1;

	if (!IN_DEV_PROXY_ARP(in_dev))
		return 0;

	if ((imi = IN_DEV_MEDIUM_ID(in_dev)) == 0)
		return 1;
	if (imi == -1)
		return 0;

	/* place to check for proxy_arp for routes */

	if ((out_dev = in_dev_get(rt->u.dst.dev)) != NULL) {
		omi = IN_DEV_MEDIUM_ID(out_dev);
		in_dev_put(out_dev);
	}
	return (omi != imi && omi != -1);
}

/*
 *	Interface to link layer: send routine and receive handler.
 */

/*
 *	Create an arp packet. If (dest_hw == NULL), we create a broadcast
 *	message.
 */
struct sk_buff *arp_create(int type, int ptype, u32 dest_ip,
			   struct net_device *dev, u32 src_ip,
			   unsigned char *dest_hw, unsigned char *src_hw,
			   unsigned char *target_hw)
{
	struct sk_buff *skb;
	struct arphdr *arp;
	unsigned char *arp_ptr;

	/*
	 *	Allocate a buffer
	 */
	
	skb = alloc_skb(sizeof(struct arphdr)+ 2*(dev->addr_len+4)
				+ dev->hard_header_len + 15, GFP_ATOMIC);
	if (skb == NULL)
		return NULL;

	skb_reserve(skb, (dev->hard_header_len+15)&~15);
	skb->nh.raw = skb->data;
	arp = (struct arphdr *) skb_put(skb,sizeof(struct arphdr) + 2*(dev->addr_len+4));
	skb->dev = dev;
	skb->protocol = htons (ETH_P_ARP);
	if (src_hw == NULL)
		src_hw = dev->dev_addr;
	if (dest_hw == NULL)
		dest_hw = dev->broadcast;

	/*
	 *	Fill the device header for the ARP frame
	 */
	if (dev->hard_header &&
	    dev->hard_header(skb,dev,ptype,dest_hw,src_hw,skb->len) < 0)
		goto out;

	/*
	 * Fill out the arp protocol part.
	 *
	 * The arp hardware type should match the device type, except for FDDI,
	 * which (according to RFC 1390) should always equal 1 (Ethernet).
	 */
	/*
	 *	Exceptions everywhere. AX.25 uses the AX.25 PID value not the
	 *	DIX code for the protocol. Make these device structure fields.
	 */
	switch (dev->type) {
	default:
		arp->ar_hrd = htons(dev->type);
		arp->ar_pro = htons(ETH_P_IP);
		break;

#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
	case ARPHRD_AX25:
		arp->ar_hrd = htons(ARPHRD_AX25);
		arp->ar_pro = htons(AX25_P_IP);
		break;

#if defined(CONFIG_NETROM) || defined(CONFIG_NETROM_MODULE)
	case ARPHRD_NETROM:
		arp->ar_hrd = htons(ARPHRD_NETROM);
		arp->ar_pro = htons(AX25_P_IP);
		break;
#endif
#endif

#ifdef CONFIG_FDDI
	case ARPHRD_FDDI:
		arp->ar_hrd = htons(ARPHRD_ETHER);
		arp->ar_pro = htons(ETH_P_IP);
		break;
#endif
#ifdef CONFIG_TR
	case ARPHRD_IEEE802_TR:
		arp->ar_hrd = htons(ARPHRD_IEEE802);
		arp->ar_pro = htons(ETH_P_IP);
		break;
#endif
	}

	arp->ar_hln = dev->addr_len;
	arp->ar_pln = 4;
	arp->ar_op = htons(type);

	arp_ptr=(unsigned char *)(arp+1);

	memcpy(arp_ptr, src_hw, dev->addr_len);
	arp_ptr+=dev->addr_len;
	memcpy(arp_ptr, &src_ip,4);
	arp_ptr+=4;
	if (target_hw != NULL)
		memcpy(arp_ptr, target_hw, dev->addr_len);
	else
		memset(arp_ptr, 0, dev->addr_len);
	arp_ptr+=dev->addr_len;
	memcpy(arp_ptr, &dest_ip, 4);

	return skb;

out:
	kfree_skb(skb);
	return NULL;
}

/*
 *	Send an arp packet.
 */
void arp_xmit(struct sk_buff *skb)
{
	/* Send it off, maybe filter it using firewalling first.  */
	NF_HOOK(NF_ARP, NF_ARP_OUT, skb, NULL, skb->dev, dev_queue_xmit);
}

/*
 *	Create and send an arp packet.
 */
void arp_send(int type, int ptype, u32 dest_ip, 
	      struct net_device *dev, u32 src_ip, 
	      unsigned char *dest_hw, unsigned char *src_hw,
	      unsigned char *target_hw)
{
	struct sk_buff *skb;

	/*
	 *	No arp on this interface.
	 */
	
	if (dev->flags&IFF_NOARP)
		return;

	skb = arp_create(type, ptype, dest_ip, dev, src_ip,
			 dest_hw, src_hw, target_hw);
	if (skb == NULL) {
		return;
	}

	arp_xmit(skb);
}

static void parp_redo(struct sk_buff *skb)
{
	arp_rcv(skb, skb->dev, NULL);
}

/*
 *	Process an arp request.
 */

int arp_process(struct sk_buff *skb)
{
	struct net_device *dev = skb->dev;
	struct in_device *in_dev = in_dev_get(dev);
	struct arphdr *arp;
	unsigned char *arp_ptr;
	struct rtable *rt;
	unsigned char *sha, *tha;
	u32 sip, tip;
	u16 dev_type = dev->type;
	int addr_type;
	struct neighbour *n;

	/* arp_rcv below verifies the ARP header, verifies the device
	 * is ARP'able, and linearizes the SKB (if needed).
	 */

	if (in_dev == NULL)
		goto out;

	arp = skb->nh.arph;
	arp_ptr= (unsigned char *)(arp+1);

	switch (dev_type) {
	default:	
		if (arp->ar_pro != htons(ETH_P_IP))
			goto out;
		if (htons(dev_type) != arp->ar_hrd)
			goto out;
		break;
#ifdef CONFIG_NET_ETHERNET
	case ARPHRD_ETHER:
		/*
		 * ETHERNET devices will accept ARP hardware types of either
		 * 1 (Ethernet) or 6 (IEEE 802.2).
		 */
		if (arp->ar_hrd != htons(ARPHRD_ETHER) &&
		    arp->ar_hrd != htons(ARPHRD_IEEE802))
			goto out;
		if (arp->ar_pro != htons(ETH_P_IP))
			goto out;
		break;
#endif
#ifdef CONFIG_TR
	case ARPHRD_IEEE802_TR:
		/*
		 * Token ring devices will accept ARP hardware types of either
		 * 1 (Ethernet) or 6 (IEEE 802.2).
		 */
		if (arp->ar_hrd != htons(ARPHRD_ETHER) &&
		    arp->ar_hrd != htons(ARPHRD_IEEE802))
			goto out;
		if (arp->ar_pro != htons(ETH_P_IP))
			goto out;
		break;
#endif
#ifdef CONFIG_FDDI
	case ARPHRD_FDDI:
		/*
		 * According to RFC 1390, FDDI devices should accept ARP hardware types
		 * of 1 (Ethernet).  However, to be more robust, we'll accept hardware
		 * types of either 1 (Ethernet) or 6 (IEEE 802.2).
		 */
		if (arp->ar_hrd != htons(ARPHRD_ETHER) &&
		    arp->ar_hrd != htons(ARPHRD_IEEE802))
			goto out;
		if (arp->ar_pro != htons(ETH_P_IP))
			goto out;
		break;
#endif
#ifdef CONFIG_NET_FC
	case ARPHRD_IEEE802:
		/*
		 * According to RFC 2625, Fibre Channel devices (which are IEEE
		 * 802 devices) should accept ARP hardware types of 6 (IEEE 802)
		 * and 1 (Ethernet).
		 */
		if (arp->ar_hrd != htons(ARPHRD_ETHER) &&
		    arp->ar_hrd != htons(ARPHRD_IEEE802))
			goto out;
		if (arp->ar_pro != htons(ETH_P_IP))
			goto out;
		break;
#endif
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
	case ARPHRD_AX25:
		if (arp->ar_pro != htons(AX25_P_IP))
			goto out;
		if (arp->ar_hrd != htons(ARPHRD_AX25))
			goto out;
		break;
#if defined(CONFIG_NETROM) || defined(CONFIG_NETROM_MODULE)
	case ARPHRD_NETROM:
		if (arp->ar_pro != htons(AX25_P_IP))
			goto out;
		if (arp->ar_hrd != htons(ARPHRD_NETROM))
			goto out;
		break;
#endif
#endif
	}

	/* Understand only these message types */

	if (arp->ar_op != htons(ARPOP_REPLY) &&
	    arp->ar_op != htons(ARPOP_REQUEST))
		goto out;

/*
 *	Extract fields
 */
	sha=arp_ptr;
	arp_ptr += dev->addr_len;
	memcpy(&sip, arp_ptr, 4);
	arp_ptr += 4;
	tha=arp_ptr;
	arp_ptr += dev->addr_len;
	memcpy(&tip, arp_ptr, 4);
/* 
 *	Check for bad requests for 127.x.x.x and requests for multicast
 *	addresses.  If this is one such, delete it.
 */
	if (LOOPBACK(tip) || MULTICAST(tip))
		goto out;

/*
 *     Special case: We must set Frame Relay source Q.922 address
 */
	if (dev_type == ARPHRD_DLCI)
		sha = dev->broadcast;

/*
 *  Process entry.  The idea here is we want to send a reply if it is a
 *  request for us or if it is a request for someone else that we hold
 *  a proxy for.  We want to add an entry to our cache if it is a reply
 *  to us or if it is a request for our address.  
 *  (The assumption for this last is that if someone is requesting our 
 *  address, they are probably intending to talk to us, so it saves time 
 *  if we cache their address.  Their address is also probably not in 
 *  our cache, since ours is not in their cache.)
 * 
 *  Putting this another way, we only care about replies if they are to
 *  us, in which case we add them to the cache.  For requests, we care
 *  about those for us and those for our proxies.  We reply to both,
 *  and in the case of requests for us we add the requester to the arp 
 *  cache.
 */

	/* Special case: IPv4 duplicate address detection packet (RFC2131) */
	if (sip == 0) {
		int reply;
		struct net_device *dev2 = NULL;
		struct in_device *in_dev2 = NULL;

		reply =
		    (arp->ar_op == htons(ARPOP_REQUEST) &&
		    (dev2 = ip_dev_find(tip)) != NULL &&
		    (dev2 == dev ||
		    ((in_dev2 = in_dev_get(dev2)) != NULL &&
		    !IN_DEV_HIDDEN(in_dev2))));
		if (dev2) {
		    if (in_dev2) in_dev_put(in_dev2);
		    dev_put(dev2);
		    if (reply)
			arpf_send(ARPA_TABLE_INPUT,skb,sip,tip,sha,tha,dev,NULL);
		}
		goto out;
	}

	if (arp->ar_op == htons(ARPOP_REQUEST) &&
	    ip_route_input(skb, tip, sip, 0, dev) == 0) {

		rt = (struct rtable*)skb->dst;
		addr_type = rt->rt_type;

		if (addr_type == RTN_LOCAL) {
			n = neigh_event_ns(&arp_tbl, sha, &sip, dev);
			if (n) {
				int dont_send = 0;
				if (ipv4_devconf.hidden &&
				    skb->pkt_type != PACKET_HOST) {
					struct net_device *dev2 = NULL;
					struct in_device *in_dev2 = NULL;

					dont_send |=
					  ((dev2 = ip_dev_find(tip)) != NULL &&
					  dev2 != dev &&
					  (in_dev2=in_dev_get(dev2)) != NULL &&
					  IN_DEV_HIDDEN(in_dev2));
					if (dev2) {
					    if (in_dev2) in_dev_put(in_dev2);
					    dev_put(dev2);
					}
				}
				if (IN_DEV_ARPFILTER(in_dev))
					dont_send |= arp_filter(sip,tip,dev); 
				if (!dont_send)
					arpf_send(ARPA_TABLE_INPUT,skb,
						  sip,tip,sha,tha,dev,NULL);

				neigh_release(n);
			}
			goto out;
		} else if (IN_DEV_FORWARD(in_dev)) {
			if ((rt->rt_flags&RTCF_DNAT) ||
			    (addr_type == RTN_UNICAST  && rt->u.dst.dev != dev &&
			     (arp_fwd_proxy(in_dev, rt) || pneigh_lookup(&arp_tbl, &tip, dev, 0)))) {
				n = neigh_event_ns(&arp_tbl, sha, &sip, dev);
				if (n)
					neigh_release(n);

				if (skb->stamp.tv_sec == 0 ||
				    skb->pkt_type == PACKET_HOST ||
				    in_dev->arp_parms->proxy_delay == 0) {
					arpf_send(ARPA_TABLE_FORWARD,skb,
						  sip,tip,sha,tha,dev,
						  rt->u.dst.dev);
				} else {
					pneigh_enqueue(&arp_tbl, in_dev->arp_parms, skb);
					in_dev_put(in_dev);
					return 0;
				}
				goto out;
			}
		}
	}

	/* Update our ARP tables */

	n = __neigh_lookup(&arp_tbl, &sip, dev, 0);

#ifdef CONFIG_IP_ACCEPT_UNSOLICITED_ARP
	/* Unsolicited ARP is not accepted by default.
	   It is possible, that this option should be enabled for some
	   devices (strip is candidate)
	 */
	if (n == NULL &&
	    arp->ar_op == htons(ARPOP_REPLY) &&
	    inet_addr_type(sip) == RTN_UNICAST)
		n = __neigh_lookup(&arp_tbl, &sip, dev, -1);
#endif

	if (n) {
		int state = NUD_REACHABLE;
		int override = 0;

		/* If several different ARP replies follows back-to-back,
		   use the FIRST one. It is possible, if several proxy
		   agents are active. Taking the first reply prevents
		   arp trashing and chooses the fastest router.
		 */
		if (jiffies - n->updated >= n->parms->locktime)
			override = 1;

		/* Broadcast replies and request packets
		   do not assert neighbour reachability.
		 */
		if (arp->ar_op != htons(ARPOP_REPLY) ||
		    skb->pkt_type != PACKET_HOST)
			state = NUD_STALE;
		neigh_update(n, sha, state, override, 1);
		neigh_release(n);
	}

out:
	if (in_dev)
		in_dev_put(in_dev);
	kfree_skb(skb);
	return 0;
}


/*
 *	Receive an arp request from the device layer.
 */

int arp_rcv(struct sk_buff *skb, struct net_device *dev, struct packet_type *pt)
{
	struct arphdr *arp = skb->nh.arph;

	if (arp->ar_hln != dev->addr_len ||
	    dev->flags & IFF_NOARP ||
	    skb->pkt_type == PACKET_OTHERHOST ||
	    skb->pkt_type == PACKET_LOOPBACK ||
	    arp->ar_pln != 4)
		goto freeskb;

	if ((skb = skb_share_check(skb, GFP_ATOMIC)) == NULL)
		goto out_of_mem;

	if (skb_is_nonlinear(skb)) {
		if (skb_linearize(skb, GFP_ATOMIC) != 0)
			goto freeskb;
	}

	return NF_HOOK(NF_ARP, NF_ARP_IN, skb, dev, NULL, arp_process);

freeskb:
	kfree_skb(skb);
out_of_mem:
	return 0;
}

/*
 *	User level interface (ioctl, /proc)
 */

/*
 *	Set (create) an ARP cache entry.
 */

int arp_req_set(struct arpreq *r, struct net_device * dev)
{
	u32 ip = ((struct sockaddr_in *) &r->arp_pa)->sin_addr.s_addr;
	struct neighbour *neigh;
	int err;

	if (r->arp_flags&ATF_PUBL) {
		u32 mask = ((struct sockaddr_in *) &r->arp_netmask)->sin_addr.s_addr;
		if (mask && mask != 0xFFFFFFFF)
			return -EINVAL;
		if (!dev && (r->arp_flags & ATF_COM)) {
			dev = dev_getbyhwaddr(r->arp_ha.sa_family, r->arp_ha.sa_data);
			if (!dev)
				return -ENODEV;
		}
		if (mask) {
			if (pneigh_lookup(&arp_tbl, &ip, dev, 1) == NULL)
				return -ENOBUFS;
			return 0;
		}
		if (dev == NULL) {
			ipv4_devconf.proxy_arp = 1;
			return 0;
		}
		if (__in_dev_get(dev)) {
			__in_dev_get(dev)->cnf.proxy_arp = 1;
			return 0;
		}
		return -ENXIO;
	}

	if (r->arp_flags & ATF_PERM)
		r->arp_flags |= ATF_COM;
	if (dev == NULL) {
		struct rtable * rt;
		if ((err = ip_route_output(&rt, ip, 0, RTO_ONLINK, 0)) != 0)
			return err;
		dev = rt->u.dst.dev;
		ip_rt_put(rt);
		if (!dev)
			return -EINVAL;
	}
	if (r->arp_ha.sa_family != dev->type)	
		return -EINVAL;

	neigh = __neigh_lookup_errno(&arp_tbl, &ip, dev);
	err = PTR_ERR(neigh);
	if (!IS_ERR(neigh)) {
		unsigned state = NUD_STALE;
		if (r->arp_flags & ATF_PERM)
			state = NUD_PERMANENT;
		err = neigh_update(neigh, (r->arp_flags&ATF_COM) ?
				   r->arp_ha.sa_data : NULL, state, 1, 0);
		neigh_release(neigh);
	}
	return err;
}

static unsigned arp_state_to_flags(struct neighbour *neigh)
{
	unsigned flags = 0;
	if (neigh->nud_state&NUD_PERMANENT)
		flags = ATF_PERM|ATF_COM;
	else if (neigh->nud_state&NUD_VALID)
		flags = ATF_COM;
	return flags;
}

/*
 *	Get an ARP cache entry.
 */

static int arp_req_get(struct arpreq *r, struct net_device *dev)
{
	u32 ip = ((struct sockaddr_in *) &r->arp_pa)->sin_addr.s_addr;
	struct neighbour *neigh;
	int err = -ENXIO;

	neigh = neigh_lookup(&arp_tbl, &ip, dev);
	if (neigh) {
		read_lock_bh(&neigh->lock);
		memcpy(r->arp_ha.sa_data, neigh->ha, dev->addr_len);
		r->arp_flags = arp_state_to_flags(neigh);
		read_unlock_bh(&neigh->lock);
		r->arp_ha.sa_family = dev->type;
		strncpy(r->arp_dev, dev->name, sizeof(r->arp_dev));
		neigh_release(neigh);
		err = 0;
	}
	return err;
}

int arp_req_delete(struct arpreq *r, struct net_device * dev)
{
	int err;
	u32 ip = ((struct sockaddr_in *)&r->arp_pa)->sin_addr.s_addr;
	struct neighbour *neigh;

	if (r->arp_flags & ATF_PUBL) {
		u32 mask = ((struct sockaddr_in *) &r->arp_netmask)->sin_addr.s_addr;
		if (mask == 0xFFFFFFFF)
			return pneigh_delete(&arp_tbl, &ip, dev);
		if (mask == 0) {
			if (dev == NULL) {
				ipv4_devconf.proxy_arp = 0;
				return 0;
			}
			if (__in_dev_get(dev)) {
				__in_dev_get(dev)->cnf.proxy_arp = 0;
				return 0;
			}
			return -ENXIO;
		}
		return -EINVAL;
	}

	if (dev == NULL) {
		struct rtable * rt;
		if ((err = ip_route_output(&rt, ip, 0, RTO_ONLINK, 0)) != 0)
			return err;
		dev = rt->u.dst.dev;
		ip_rt_put(rt);
		if (!dev)
			return -EINVAL;
	}
	err = -ENXIO;
	neigh = neigh_lookup(&arp_tbl, &ip, dev);
	if (neigh) {
		if (neigh->nud_state&~NUD_NOARP)
			err = neigh_update(neigh, NULL, NUD_FAILED, 1, 0);
		neigh_release(neigh);
	}
	return err;
}

/*
 *	Handle an ARP layer I/O control request.
 */

int arp_ioctl(unsigned int cmd, void *arg)
{
	int err;
	struct arpreq r;
	struct net_device * dev = NULL;

        /* RSBAC */
        #ifdef CONFIG_RSBAC_NET_DEV
        enum  rsbac_adf_request_t     rsbac_request = R_NONE;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	switch(cmd) {
		case SIOCDARP:
		case SIOCSARP:
			if (!capable(CAP_NET_ADMIN))
				return -EPERM;
			#ifdef CONFIG_RSBAC_NET_DEV
			rsbac_request = R_MODIFY_SYSTEM_DATA;
			#endif
		case SIOCGARP:
			err = copy_from_user(&r, arg, sizeof(struct arpreq));
			if (err)
				return -EFAULT;
			#ifdef CONFIG_RSBAC_NET_DEV
			if(rsbac_request == R_NONE)
				rsbac_request = R_GET_STATUS_DATA;
			#endif
			break;
		default:
			return -EINVAL;
	}

	if (r.arp_pa.sa_family != AF_INET)
		return -EPFNOSUPPORT;

	if (!(r.arp_flags & ATF_PUBL) &&
	    (r.arp_flags & (ATF_NETMASK|ATF_DONTPUB)))
		return -EINVAL;
	if (!(r.arp_flags & ATF_NETMASK))
		((struct sockaddr_in *)&r.arp_netmask)->sin_addr.s_addr=htonl(0xFFFFFFFFUL);

	rtnl_lock();
	if (r.arp_dev[0]) {
		err = -ENODEV;
		if ((dev = __dev_get_by_name(r.arp_dev)) == NULL)
			goto out;

		/* Mmmm... It is wrong... ARPHRD_NETROM==0 */
		if (!r.arp_ha.sa_family)
			r.arp_ha.sa_family = dev->type;
		err = -EINVAL;
		if ((r.arp_flags & ATF_COM) && r.arp_ha.sa_family != dev->type)
			goto out;

		/* RSBAC */
		#ifdef CONFIG_RSBAC_NET_DEV
#ifdef CONFIG_RSBAC_DEBUG
		if (rsbac_debug_aef)
		  printk(KERN_DEBUG "arp_ioctl(): calling ADF\n");
#endif
		strncpy(rsbac_target_id.netdev, r.arp_dev, RSBAC_IFNAMSIZ);
		rsbac_target_id.netdev[RSBAC_IFNAMSIZ] = 0;
		rsbac_attribute_value.dummy = 0;
		if (!rsbac_adf_request(rsbac_request,
				    current->pid,
				    T_NETDEV,
				    rsbac_target_id,
				    A_none,
				    rsbac_attribute_value))
		  {
		    err = -EPERM;
		    goto out;
		  }
		#endif
	} else if (cmd == SIOCGARP) {
		err = -ENODEV;
		goto out;
	}

	switch(cmd) {
	case SIOCDARP:
	        err = arp_req_delete(&r, dev);
		break;
	case SIOCSARP:
		err = arp_req_set(&r, dev);
		break;
	case SIOCGARP:
		err = arp_req_get(&r, dev);
		if (!err && copy_to_user(arg, &r, sizeof(r)))
			err = -EFAULT;
		break;
	}
out:
	rtnl_unlock();
	return err;
}

/*
 *	Write the contents of the ARP cache to a PROCfs file.
 */
#ifndef CONFIG_PROC_FS
static int arp_get_info(char *buffer, char **start, off_t offset, int length) { return 0; }
#else
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
static char *ax2asc2(ax25_address *a, char *buf);
#endif
#define HBUFFERLEN 30

static int arp_get_info(char *buffer, char **start, off_t offset, int length)
{
	int len=0;
	off_t pos=0;
	int size;
	char hbuffer[HBUFFERLEN];
	int i,j,k;
	const char hexbuf[] =  "0123456789ABCDEF";

	size = sprintf(buffer,"IP address       HW type     Flags       HW address            Mask     Device\n");

	pos+=size;
	len+=size;

	for(i=0; i<=NEIGH_HASHMASK; i++) {
		struct neighbour *n;
		read_lock_bh(&arp_tbl.lock);
		for (n=arp_tbl.hash_buckets[i]; n; n=n->next) {
			struct net_device *dev = n->dev;
			int hatype = dev->type;

			/* Do not confuse users "arp -a" with magic entries */
			if (!(n->nud_state&~NUD_NOARP))
				continue;

			read_lock(&n->lock);

/*
 *	Convert hardware address to XX:XX:XX:XX ... form.
 */
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
			if (hatype == ARPHRD_AX25 || hatype == ARPHRD_NETROM)
				ax2asc2((ax25_address *)n->ha, hbuffer);
			else {
#endif
			for (k=0,j=0;k<HBUFFERLEN-3 && j<dev->addr_len;j++) {
				hbuffer[k++]=hexbuf[(n->ha[j]>>4)&15 ];
				hbuffer[k++]=hexbuf[n->ha[j]&15     ];
				hbuffer[k++]=':';
			}
			hbuffer[--k]=0;

#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
		}
#endif

			{
				char tbuf[16];
				sprintf(tbuf, "%u.%u.%u.%u", NIPQUAD(*(u32*)n->primary_key));
				size = sprintf(buffer+len, "%-16s 0x%-10x0x%-10x%s"
							"     *        %s\n",
					tbuf,
					hatype,
					arp_state_to_flags(n), 
					hbuffer,
					dev->name);
			}

			read_unlock(&n->lock);

			len += size;
			pos += size;
		  
			if (pos <= offset)
				len=0;
			if (pos >= offset+length) {
				read_unlock_bh(&arp_tbl.lock);
 				goto done;
			}
		}
		read_unlock_bh(&arp_tbl.lock);
	}

	for (i=0; i<=PNEIGH_HASHMASK; i++) {
		struct pneigh_entry *n;
		for (n=arp_tbl.phash_buckets[i]; n; n=n->next) {
			struct net_device *dev = n->dev;
			int hatype = dev ? dev->type : 0;

			{
				char tbuf[16];
				sprintf(tbuf, "%u.%u.%u.%u", NIPQUAD(*(u32*)n->key));
				size = sprintf(buffer+len, "%-16s 0x%-10x0x%-10x%s"
							"     *        %s\n",
					tbuf,
					hatype,
 					ATF_PUBL|ATF_PERM,
					"00:00:00:00:00:00",
					dev ? dev->name : "*");
			}

			len += size;
			pos += size;
		  
			if (pos <= offset)
				len=0;
			if (pos >= offset+length)
				goto done;
		}
	}

done:
  
	*start = buffer+len-(pos-offset);	/* Start of wanted data */
	len = pos-offset;			/* Start slop */
	if (len>length)
		len = length;			/* Ending slop */
	if (len<0)
		len = 0;
	return len;
}
#endif

/* Note, that it is not on notifier chain.
   It is necessary, that this routine was called after route cache will be
   flushed.
 */
void arp_ifdown(struct net_device *dev)
{
	neigh_ifdown(&arp_tbl, dev);
}


static void arpf_destroy(struct arpf_node *afp)
{
	if (!afp->at_dead) {
		printk(KERN_ERR "Destroying alive arp table node %p from %08lx\n", afp,
		       *(((unsigned long*)&afp)-1));
		return;
	}
	kmem_cache_free(arpf_cachep, afp);
}

static inline void arpf_put(struct arpf_node *afp)
{
	if (atomic_dec_and_test(&afp->at_refcnt))
		arpf_destroy(afp);
}

static inline struct arpf_node *
arpf_lookup(int table, struct sk_buff *skb, u32 sip, u32 tip,
	    unsigned char *from_hw, unsigned char *to_hw,
	    struct net_device *idev, struct net_device *odev)
{
	int sz_iif = idev? strlen(idev->name) : 0;
	int sz_oif = odev? strlen(odev->name) : 0;
	int alen;
	struct arpf_node *afp;

	if (ARPA_TABLE_OUTPUT != table) {
		alen = idev->addr_len;
	} else {
		if (!from_hw) from_hw = odev->dev_addr;
		if (!to_hw) to_hw = odev->broadcast;
		alen = odev->addr_len;
	}

	read_lock(&arpf_lock);
	for (afp = arp_tabs[table]; afp; afp = afp->at_next) {
		if ((tip ^ afp->at_to) & afp->at_to_mask)
			continue;
		if ((sip ^ afp->at_from) & afp->at_from_mask)
			continue;
		if (afp->at_llfrom_len &&
		    (afp->at_llfrom_len > alen ||
		     memcmp(from_hw, afp->at_llfrom, afp->at_llfrom_len)))
			continue;
		if (afp->at_llto_len &&
		    (afp->at_llto_len > alen ||
		     memcmp(to_hw, afp->at_llto, afp->at_llto_len)))
			continue;
		if (afp->at_iif_len &&
		    (afp->at_iif_len > sz_iif ||
		     memcmp(afp->at_iif, idev->name, afp->at_iif_len) ||
		     (sz_iif != afp->at_iif_len &&
		      !(afp->at_flags & ARPM_F_WILDIIF))))
			continue;
		if (afp->at_oif_len &&
		    (afp->at_oif_len > sz_oif ||
		     memcmp(afp->at_oif, odev->name, afp->at_oif_len) ||
		     (sz_oif != afp->at_oif_len &&
		      !(afp->at_flags & ARPM_F_WILDOIF))))
			continue;
		if (afp->at_flags & ARPM_F_BROADCAST &&
		    skb->pkt_type == PACKET_HOST)
			continue;
		if (afp->at_flags & ARPM_F_UNICAST &&
		    skb->pkt_type != PACKET_HOST)
			continue;
		if (afp->at_llsrc_len && afp->at_llsrc_len != alen)
			continue;
		if (afp->at_lldst_len && afp->at_lldst_len != alen)
			continue;
		atomic_inc(&afp->at_packets);
		break;
	}
	read_unlock(&arpf_lock);
	return afp;
}

static void
arpf_send(int table, struct sk_buff *skb, u32 sip, u32 tip,
	  unsigned char *from_hw, unsigned char *to_hw,
	  struct net_device *idev, struct net_device *odev)
{
	struct arpf_node *afp = NULL;

	if (!arp_tabs[table] ||
	    !(afp = arpf_lookup(table, skb, sip, tip,
				from_hw, to_hw, idev, odev))) {
		switch (table) {
		case ARPA_TABLE_INPUT:
			if (!sip) {
				arp_send(ARPOP_REPLY, ETH_P_ARP, tip, idev, tip,
					 from_hw, idev->dev_addr,
					 idev->dev_addr);
				break;
			}
			/* continue */
		case ARPA_TABLE_FORWARD:
			arp_send(ARPOP_REPLY, ETH_P_ARP, sip, idev, tip,
				 from_hw, idev->dev_addr, from_hw);
			break;
		case ARPA_TABLE_OUTPUT:
			arp_send(ARPOP_REQUEST, ETH_P_ARP, tip, odev, sip,
				 to_hw, odev->dev_addr, NULL);
			break;
		}
		return;
	}

	/* deny? */
	if (!afp->at_action) goto out;

	switch (table) {
	case ARPA_TABLE_INPUT:
		if (!sip) {
			arp_send(ARPOP_REPLY, ETH_P_ARP, tip, idev, tip,
				from_hw,
				afp->at_llsrc_len?afp->at_llsrc:idev->dev_addr,
				afp->at_llsrc_len?afp->at_llsrc:idev->dev_addr);
			break;
		}
		/* continue */
	case ARPA_TABLE_FORWARD:
		arp_send(ARPOP_REPLY, ETH_P_ARP, sip, idev, tip,
			 afp->at_lldst_len?afp->at_lldst:from_hw,
			 afp->at_llsrc_len?afp->at_llsrc:idev->dev_addr,
			 afp->at_lldst_len?afp->at_lldst:from_hw);
		break;
	case ARPA_TABLE_OUTPUT:
		if (afp->at_flags & ARPM_F_PREFSRC && afp->at_src == 0) {
			struct rtable *rt;

			if (ip_route_output(&rt, tip, 0, 0, odev->ifindex) < 0)
				break;
			sip = rt->rt_src;
			ip_rt_put(rt);
			if (!sip)
				break;
		}
		arp_send(ARPOP_REQUEST, ETH_P_ARP, tip, odev, afp->at_src?:sip,
			 afp->at_lldst_len?afp->at_lldst:to_hw,
			 afp->at_llsrc_len?afp->at_llsrc:odev->dev_addr,
			 NULL);
		break;
	}

out:
	arpf_put(afp);
}

static int
arpf_fill_node(struct sk_buff *skb, u32 pid, u32 seq, unsigned flags,
	       int event, int table, struct arpf_node *afp)
{
	struct arpmsg	*am;
	struct nlmsghdr	*nlh;
	unsigned char	*b = skb->tail;
	u32 packets = atomic_read(&afp->at_packets);

	nlh = NLMSG_PUT(skb, pid, seq, event, sizeof(*am));
	nlh->nlmsg_flags = flags;
	am = NLMSG_DATA(nlh);
	am->arpm_family = AF_UNSPEC;
	am->arpm_table = table;
	am->arpm_action = afp->at_action;
	am->arpm_from_len = afp->at_from_len;
	am->arpm_to_len = afp->at_to_len;
	am->arpm_pref = afp->at_pref;
	am->arpm_flags = afp->at_flags;
	if (afp->at_from_len)
		RTA_PUT(skb, ARPA_FROM, 4, &afp->at_from);
	if (afp->at_to_len)
		RTA_PUT(skb, ARPA_TO, 4, &afp->at_to);
	if (afp->at_src || afp->at_flags & ARPM_F_PREFSRC)
		RTA_PUT(skb, ARPA_SRC, 4, &afp->at_src);
	if (afp->at_iif[0])
		RTA_PUT(skb, ARPA_IIF, sizeof(afp->at_iif), afp->at_iif);
	if (afp->at_oif[0])
		RTA_PUT(skb, ARPA_OIF, sizeof(afp->at_oif), afp->at_oif);
	if (afp->at_llfrom_len)
		RTA_PUT(skb, ARPA_LLFROM, afp->at_llfrom_len, afp->at_llfrom);
	if (afp->at_llto_len)
		RTA_PUT(skb, ARPA_LLTO, afp->at_llto_len, afp->at_llto);
	if (afp->at_llsrc_len)
		RTA_PUT(skb, ARPA_LLSRC, afp->at_llsrc_len, afp->at_llsrc);
	if (afp->at_lldst_len)
		RTA_PUT(skb, ARPA_LLDST, afp->at_lldst_len, afp->at_lldst);
	RTA_PUT(skb, ARPA_PACKETS, 4, &packets);
	nlh->nlmsg_len = skb->tail - b;
	return skb->len;

nlmsg_failure:
rtattr_failure:
	skb_trim(skb, b - skb->data);
	return -1;
}

static int
arpmsg_notify(struct sk_buff *oskb, struct nlmsghdr *n, int table,
	      struct arpf_node *afp, int event)
{
	struct sk_buff *skb;
	u32 pid = oskb ? NETLINK_CB(oskb).pid : 0;
	int size = NLMSG_SPACE(sizeof(struct arpmsg)+256);

	skb = alloc_skb(size, GFP_KERNEL);
	if (!skb)
		return -ENOBUFS;

	if (arpf_fill_node(skb, pid, n->nlmsg_seq, 0, event, table, afp) <= 0) {
		kfree_skb(skb);
		return -EINVAL;
	}

	return rtnetlink_send(skb, pid, RTMGRP_ARP, n->nlmsg_flags&NLM_F_ECHO);
}

static inline int
arpf_str_size(int a, struct rtattr **rta, int maxlen)
{
	int size = 0;

	if (rta[a-1] && (size = RTA_PAYLOAD(rta[a-1]))) {
		if (size > maxlen)
			size = maxlen;
	}
	return size;
}

static inline int
arpf_get_str(int a, struct rtattr **rta, unsigned char *p,
	     int maxlen, unsigned char *l)
{
	int size = arpf_str_size(a, rta, maxlen);

	if (size) {
		memcpy(p, RTA_DATA(rta[a-1]), size);
		*l = size;
	}
	return size;
}

#define ARPF_MATCH_U32(ind, field)	(			\
	(!rta[ind-1] && r->at_ ## field == 0) ||		\
	(rta[ind-1] &&						\
	 *(u32*) RTA_DATA(rta[ind-1]) == r->at_ ## field))

#define ARPF_MATCH_STR(ind, field)	(			\
	(!rta[ind-1] && r->at_ ## field ## _len == 0) ||	\
	(rta[ind-1] && r->at_ ## field ## _len &&		\
	 r->at_ ## field ## _len < RTA_PAYLOAD(rta[ind-1]) &&	\
	 strcmp(RTA_DATA(rta[ind-1]), r->at_ ## field) == 0))

#define ARPF_MATCH_DATA(ind, field)	(			\
	(!rta[ind-1] && r->at_ ## field ## _len == 0) ||	\
	(rta[ind-1] && r->at_ ## field ## _len &&		\
	 r->at_ ## field ## _len == RTA_PAYLOAD(rta[ind-1]) &&	\
	 memcmp(RTA_DATA(rta[ind-1]), &r->at_ ## field,		\
		r->at_ ## field ## _len) == 0))

/* RTM_NEWARPRULE/RTM_DELARPRULE/RTM_GETARPRULE */

int arpf_rule_ctl(struct sk_buff *skb, struct nlmsghdr* n, void *arg)
{
	struct rtattr **rta = arg;
	struct arpmsg *am = NLMSG_DATA(n);
	struct arpf_node *r, **rp, **prevp = 0, **delp = 0, *newp = 0;
	unsigned pref = 1;
	int size, ret = -EINVAL;

	if (am->arpm_table >= sizeof(arp_tabs)/sizeof(arp_tabs[0]))
		goto out;
	if (!((~am->arpm_flags) & (ARPM_F_BROADCAST|ARPM_F_UNICAST)))
		goto out;
	if (am->arpm_action > 1)
		goto out;
	if (am->arpm_to_len > 32 || am->arpm_from_len > 32)
		goto out;
	if (am->arpm_flags & ARPM_F_WILDIIF &&
	    (!rta[ARPA_IIF-1] || !RTA_PAYLOAD(rta[ARPA_IIF-1]) ||
	    !*(char*)RTA_DATA(rta[ARPA_IIF-1])))
		am->arpm_flags &= ~ARPM_F_WILDIIF;
	if (am->arpm_flags & ARPM_F_WILDOIF &&
	    (!rta[ARPA_OIF-1] || !RTA_PAYLOAD(rta[ARPA_OIF-1]) ||
	    !*(char*)RTA_DATA(rta[ARPA_OIF-1])))
		am->arpm_flags &= ~ARPM_F_WILDOIF;
	switch (am->arpm_table) {
	case ARPA_TABLE_INPUT:
		if (rta[ARPA_SRC-1] || rta[ARPA_OIF-1])
			goto out;
		break;
	case ARPA_TABLE_OUTPUT:
		if (rta[ARPA_IIF-1])
			goto out;
		if (am->arpm_flags & (ARPM_F_BROADCAST|ARPM_F_UNICAST))
			goto out;
		break;
	case ARPA_TABLE_FORWARD:
		if (rta[ARPA_SRC-1])
			goto out;
		break;
	}
	if (rta[ARPA_SRC-1] && !*(u32*) RTA_DATA(rta[ARPA_SRC-1]))
		am->arpm_flags |= ARPM_F_PREFSRC;
	else
		am->arpm_flags &= ~ARPM_F_PREFSRC;

	for (rp = &arp_tabs[am->arpm_table]; (r=*rp) != NULL; rp=&r->at_next) {
		if (pref < r->at_pref)
			prevp = rp;
		if (am->arpm_pref == r->at_pref ||
		    (!am->arpm_pref &&
		     am->arpm_to_len == r->at_to_len &&
		     am->arpm_from_len == r->at_from_len &&
		     !((am->arpm_flags ^ r->at_flags) &
		       (ARPM_F_BROADCAST | ARPM_F_UNICAST |
		        ARPM_F_WILDIIF | ARPM_F_WILDOIF)) &&
		     ARPF_MATCH_U32(ARPA_TO, to) &&
		     ARPF_MATCH_U32(ARPA_FROM, from) &&
		     ARPF_MATCH_DATA(ARPA_LLFROM, llfrom) &&
		     ARPF_MATCH_DATA(ARPA_LLTO, llto) &&
		     ARPF_MATCH_STR(ARPA_IIF, iif) &&
		     ARPF_MATCH_STR(ARPA_OIF, oif) &&
		     (n->nlmsg_type != RTM_DELARPRULE ||
		      /* DEL matches more keys */
		      (am->arpm_flags == r->at_flags &&
		       am->arpm_action == r->at_action &&
		       ARPF_MATCH_U32(ARPA_SRC, src) &&
		       ARPF_MATCH_DATA(ARPA_LLSRC, llsrc) &&
		       ARPF_MATCH_DATA(ARPA_LLDST, lldst)
		      )
		     )
		    )
		   )
			break;
		if (am->arpm_pref && r->at_pref > am->arpm_pref) {
			r = NULL;
			break;
		}
		pref = r->at_pref+1;
	}

	/*
	 * r=NULL:	*rp != NULL (stopped before next pref), pref: not valid
	 *		*rp == NULL (not found), pref: ready to use
	 * r!=NULL:	found, pref: not valid
	 *
	 * prevp=NULL:	no free slot
	 * prevp!=NULL:	free slot for rule
	 */

	if (n->nlmsg_type == RTM_DELARPRULE) {
		if (!r)
			return -ESRCH;
		delp = rp;
		goto dequeue;
	}

	if (r) {
		/* Existing rule */
		ret = -EEXIST;
		if (n->nlmsg_flags&NLM_F_EXCL)
			goto out;

		if (n->nlmsg_flags&NLM_F_REPLACE) {
			pref = r->at_pref;
			prevp = delp = rp;
			goto replace;
		}
	}

	if (n->nlmsg_flags&NLM_F_APPEND) {
		if (r) {
			pref = r->at_pref+1;
			for (rp=&r->at_next; (r=*rp) != NULL; rp=&r->at_next) {
				if (pref != r->at_pref)
					break;
				pref ++;
			}
			ret = -EBUSY;
			if (!pref)
				goto out;
		} else if (am->arpm_pref)
			pref = am->arpm_pref;
		prevp = rp;
	}

	if (!(n->nlmsg_flags&NLM_F_CREATE)) {
		ret = -ENOENT;
		if (n->nlmsg_flags&NLM_F_EXCL || r)
			ret = 0;
		goto out;
	}

	if (!(n->nlmsg_flags&NLM_F_APPEND)) {
		if (!prevp) {
			ret = -EBUSY;
			if (r || *rp ||
			    (!am->arpm_pref && arp_tabs[am->arpm_table]))
				goto out;
			prevp = rp;
			pref = am->arpm_pref? : 99;
		} else {
			if (r || !am->arpm_pref) {
				pref = (*prevp)->at_pref - 1;
				if (am->arpm_pref && am->arpm_pref < pref)
					pref = am->arpm_pref;
			} else {
				prevp = rp;
				pref = am->arpm_pref;
			}
		}
	}

replace:

	ret = -ENOMEM;
	r = kmem_cache_alloc(arpf_cachep, SLAB_KERNEL);
	if (!r)
		return ret;
	memset(r, 0, sizeof(*r));

	arpf_get_str(ARPA_LLFROM, rta, r->at_llfrom, MAX_ADDR_LEN,
		     &r->at_llfrom_len);
	arpf_get_str(ARPA_LLTO, rta, r->at_llto, MAX_ADDR_LEN,
		     &r->at_llto_len);
	arpf_get_str(ARPA_LLSRC, rta, r->at_llsrc, MAX_ADDR_LEN,
		     &r->at_llsrc_len);
	arpf_get_str(ARPA_LLDST, rta, r->at_lldst, MAX_ADDR_LEN,
		     &r->at_lldst_len);

	if (delp)
		r->at_next = (*delp)->at_next;
	else if (*prevp)
		r->at_next = *prevp;

	r->at_pref = pref;
	r->at_from_len = am->arpm_from_len;
	r->at_from_mask = inet_make_mask(r->at_from_len);
	if (rta[ARPA_FROM-1])
		r->at_from = *(u32*) RTA_DATA(rta[ARPA_FROM-1]);
	r->at_from &= r->at_from_mask;
	r->at_to_len = am->arpm_to_len;
	r->at_to_mask = inet_make_mask(r->at_to_len);
	if (rta[ARPA_TO-1])
		r->at_to = *(u32*) RTA_DATA(rta[ARPA_TO-1]);
	r->at_to &= r->at_to_mask;
	if (rta[ARPA_SRC-1])
		r->at_src = *(u32*) RTA_DATA(rta[ARPA_SRC-1]);
	if (rta[ARPA_PACKETS-1]) {
		u32 packets = *(u32*) RTA_DATA(rta[ARPA_PACKETS-1]);
		atomic_set(&r->at_packets, packets);
	}
	atomic_set(&r->at_refcnt, 1);
	r->at_flags = am->arpm_flags;
	r->at_action = am->arpm_action;

	if (rta[ARPA_IIF-1] && (size = RTA_PAYLOAD(rta[ARPA_IIF-1]))) {
		if (size >= sizeof(r->at_iif))
			size = sizeof(r->at_iif)-1;
		memcpy(r->at_iif, RTA_DATA(rta[ARPA_IIF-1]), size);
		r->at_iif_len = strlen(r->at_iif);
	}
	if (rta[ARPA_OIF-1] && (size = RTA_PAYLOAD(rta[ARPA_OIF-1]))) {
		if (size >= sizeof(r->at_oif))
			size = sizeof(r->at_oif)-1;
		memcpy(r->at_oif, RTA_DATA(rta[ARPA_OIF-1]), size);
		r->at_oif_len = strlen(r->at_oif);
	}

	newp = r;

dequeue:

	if (delp) {
		r = *delp;
		write_lock_bh(&arpf_lock);
		if (newp) {
			if (!rta[ARPA_PACKETS-1])
				atomic_set(&newp->at_packets,
					atomic_read(&r->at_packets));
			*delp = newp;
		} else {
			*delp = r->at_next;
		}
		r->at_dead = 1;
		write_unlock_bh(&arpf_lock);
		arpmsg_notify(skb, n, am->arpm_table, r, RTM_DELARPRULE);
		arpf_put(r);
		prevp = 0;
	}

	if (newp) {
		if (prevp) {
			write_lock_bh(&arpf_lock);
			*prevp = newp;
			write_unlock_bh(&arpf_lock);
		}
		arpmsg_notify(skb, n, am->arpm_table, newp, RTM_NEWARPRULE);
	}

	ret = 0;

out:
	return ret;
}

int arpf_dump_table(int t, struct sk_buff *skb, struct netlink_callback *cb)
{
	int idx, ret = -1;
	struct arpf_node *afp;
	int s_idx = cb->args[1];

	for (idx=0, afp = arp_tabs[t]; afp; afp = afp->at_next, idx++) {
		if (idx < s_idx)
			continue;
		if (arpf_fill_node(skb, NETLINK_CB(cb->skb).pid,
		    cb->nlh->nlmsg_seq, NLM_F_MULTI, RTM_NEWARPRULE, t, afp) < 0)
			goto out;
	}

	ret = skb->len;

out:
	cb->args[1] = idx;

	return ret;
}

int arpf_dump_rules(struct sk_buff *skb, struct netlink_callback *cb)
{
	int idx;
	int s_idx = cb->args[0];

	read_lock_bh(&arpf_lock);
	for (idx = 0; idx < sizeof(arp_tabs)/sizeof(arp_tabs[0]); idx++) {
		if (idx < s_idx)
			continue;
		if (idx > s_idx)
			memset(&cb->args[1], 0, sizeof(cb->args)-1*sizeof(cb->args[0]));
		if (arpf_dump_table(idx, skb, cb) < 0)
			break;
	}
	read_unlock_bh(&arpf_lock);
	cb->args[0] = idx;

	return skb->len;
}

/*
 *	Called once on startup.
 */

static struct packet_type arp_packet_type = {
	type:	__constant_htons(ETH_P_ARP),
	func:	arp_rcv,
	data:	(void*) 1, /* understand shared skbs */
};

void __init arp_init (void)
{
	struct rtnetlink_link *link_p = rtnetlink_links[PF_UNSPEC];

	arpf_cachep = kmem_cache_create("ip_arpf_cache",
					sizeof(struct arpf_node), 0,
					SLAB_HWCACHE_ALIGN, NULL, NULL);
	if (!arpf_cachep)
		panic("IP: failed to allocate ip_arpf_cache\n");

	if (link_p) {
		link_p[RTM_NEWARPRULE-RTM_BASE].doit = arpf_rule_ctl;
		link_p[RTM_DELARPRULE-RTM_BASE].doit = arpf_rule_ctl;
		link_p[RTM_GETARPRULE-RTM_BASE].dumpit = arpf_dump_rules;
	}

	neigh_table_init(&arp_tbl);

	dev_add_pack(&arp_packet_type);

	proc_net_create ("arp", 0, arp_get_info);

#ifdef CONFIG_SYSCTL
	neigh_sysctl_register(NULL, &arp_tbl.parms, NET_IPV4, NET_IPV4_NEIGH, "ipv4");
#endif
}


#ifdef CONFIG_PROC_FS
#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)

/*
 *	ax25 -> ASCII conversion
 */
char *ax2asc2(ax25_address *a, char *buf)
{
	char c, *s;
	int n;

	for (n = 0, s = buf; n < 6; n++) {
		c = (a->ax25_call[n] >> 1) & 0x7F;

		if (c != ' ') *s++ = c;
	}
	
	*s++ = '-';

	if ((n = ((a->ax25_call[6] >> 1) & 0x0F)) > 9) {
		*s++ = '1';
		n -= 10;
	}
	
	*s++ = n + '0';
	*s++ = '\0';

	if (*buf == '\0' || *buf == '-')
	   return "*";

	return buf;

}

#endif
#endif
