/*
 * inotify.c - frontend for really simple inotify example
 *
 * Robert Love	<rml@novell.com>
 */

#include <config.h>

#include <glib.h>

#include "inotify-lib.h"

/*
 * my_cb - our callback.  Events are relative to the watch, which is given by
 * "wd".  This simple example ignores it, for the most part, but realworld uses
 * probably want to map the watch descriptor back to the object they are
 * watching.  "event" describes the event (see inotify.h) and "cookie" is used
 * to synchronize two separate IN_MOVED_{TO,FROM} events (not done here).
 */
static gboolean
my_cb (const char *name, int wd, unsigned int event, unsigned int cookie)
{
	const char *type = "file";

	if (event & IN_ISDIR)
		type = "dir";

	if (event & IN_ACCESS)
		g_print ("wd=%d: %s (%s) was read\n", wd, name, type);
	if (event & IN_MODIFY)
		g_print ("wd=%d: %s (%s) was written\n", wd, name, type);
	if (event & IN_ATTRIB)
		g_print ("wd=%d: %s's (%s) metadata changed\n", wd, name, type);
	if (event & IN_CLOSE_WRITE)
		g_print ("wd=%d: %s (%s) was closed (was writable)\n",
			 wd, name, type);
	if (event & IN_CLOSE_NOWRITE)
		g_print ("wd=%d: %s (%s) was closed (was not writable)\n",
			 wd, name, type);
	if (event & IN_OPEN)
		g_print ("wd=%d: %s (%s) was opened\n", wd, name, type);
	if (event & IN_MOVED_FROM)
		g_print ("wd=%d: %s (%s) was moved away\n", wd, name, type);
	if (event & IN_MOVED_TO)
		g_print ("wd=%d: %s (%s) was moved here\n", wd, name, type);
	if (event & IN_DELETE)
		g_print ("wd=%d: %s (%s) was deleted\n", wd, name, type);
	if (event & IN_CREATE)
		g_print ("wd=%d: %s (%s) was created\n", wd, name, type);
	if (event & IN_DELETE_SELF)
		g_print ("wd=%d: The watch was deleted!", wd);
	if (event & IN_UNMOUNT)
		g_print ("wd=%d: %s was unmounted\n", wd, name);
	if (event & IN_Q_OVERFLOW)
		g_print ("The queue overflowed!");
	if (event & IN_IGNORED)
		g_print ("wd=%d: %s is no longer watched\n", wd, name);

	return TRUE;
}

int
main (int argc, char *argv[])
{
	GMainLoop *mainloop;
	GIOChannel *gio;
	int i;

	if (argc < 2) {
		g_warning ("Usage: %s [directories or files ...]\n", argv[0]);
		return -1;
	}

	gio = inotify_open ();
	if (!gio)
		return -1;

	for (i = 1; i < argc; i++) {
		int wd;

		g_print ("Adding watch on %s\n", argv[i]);
		wd = inotify_add_watch (gio, argv[i]);
	}

	inotify_callback (gio, my_cb);

	mainloop = g_main_new (FALSE);
	g_main_run (mainloop);

	inotify_close (gio);

	return 0;
}
