/* Copyright (C) 2002, 2003 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#define _GNU_SOURCE

#include <pwd.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <shadow.h>
#include <unistd.h>
#include <getopt.h>
#include <syslog.h>
#include <security/pam_appl.h>
#include <security/pam_misc.h>
#include <locale.h>
#include <libintl.h>

#ifndef _
#define _(String) gettext (String)
#endif

#include "main.h"
#include "getdef.h"
#include "public.h"
#include "error_codes.h"

/* How often should we try to lock the passwd database ?  */
#define MAX_LOCK_RETRIES 3

static const struct pam_conv conv = {
  misc_conv,
  NULL
};

static long
conv2long (const char *param)
{
  long val;
  char *cp;

  val = strtol (param, &cp, 10);
  if (*cp)
    return -2;
  return val;
}

static const char *
pw_status (const char *pass)
{
  if (*pass == '*' || *pass == '!' || *pass == 'x')
    return "LK";
  if (*pass == '\0')
    return "NP";
  return "PS";
}

static char *
date_to_str (time_t t)
{
  static char buf[80];
  struct tm *tm;

  tm = gmtime (&t);
  strftime (buf, sizeof buf, "%m/%d/%Y", tm);
  return buf;
}

static void
display_pw (const struct passwd *pw)
{
  struct spwd *sp;

#define DAY (24L*3600L)
#define SCALE DAY

  sp = getspnam (pw->pw_name);
  if (sp)
    {
      printf ("%s %s %s %ld %ld %ld %ld\n",
              sp->sp_namp,
              pw_status (sp->sp_pwdp),
              date_to_str (sp->sp_lstchg * SCALE),
              (sp->sp_min * SCALE) / DAY,
              (sp->sp_max * SCALE) / DAY,
              (sp->sp_warn * SCALE) / DAY,
              (sp->sp_inact * SCALE) / DAY);
    }
  else
    printf ("%s %s\n", pw->pw_name, pw_status (pw->pw_passwd));
}

/*
 * passwd - change a user's password file information
 *
 *      This command controls the password file and commands which are
 *      used to modify it.
 *
 *      The valid options are
 *
 *      -l      lock the named account (*)
 *      -u      unlock the named account (*)
 *      -d      delete the password for the named account (*)
 *      -e      expire the password for the named account (*)
 *      -x #    set sp_max to # days (*)
 *      -n #    set sp_min to # days (*)
 *      -w #    set sp_warn to # days (*)
 *      -i #    set sp_inact to # days (*)
 *      -h      change home directory (*)
 *      -S      show password status of named account
 *      -a      with -S show password status for all accounts
 *      -g      execute gpasswd command to interpret flags
 *      -f      execute chfn command to interpret flags
 *      -s      execute chsh command to interpret flags
 *      -k      change password only if expired
 *      -D binddn Use binddn for LDAP binding
 *
 *      (*) requires administrator permission to execute.
 *
 *      All of the time fields are entered in days and converted to the
 *      appropriate internal format.  For finer resolute the chage
 *      command must be used.
 */

static void
print_usage (FILE *stream, const char *program)
{
  fprintf (stream, _("Usage: %s [-f|-g|-s|-k[-q]] [name]\n"), program);
  fprintf (stream, _("       %s [-D binddn] [-n min] [-x max] [-w warn] [-i inact] user\n"), program);
  fprintf (stream, _("       %s {-l|-u|-d|-S[-a]|-e|-h} user\n"), program);
}

static void
print_help (const char *program)
{
  print_usage (stdout, program);
  fprintf (stdout, _("%s - change password information\n\n"), program);

  fputs (_("  -f             Change the finger (gecos) information\n"),
         stdout);
  fputs (_("  -h             Change the home directory\n"), stdout);
  fputs (_("  -s             Change the login shell\n"), stdout);
  fputs (_("  -g             Change the group password\n"), stdout);
  fputs (_("  -k             Change the password only if expired\n"), stdout);
  fputs (_("  -q             Try to be quiet\n"), stdout);
  fputs (_("  -S             Show the password attributes\n"), stdout);
  fputs (_("  -a             Only with -S, show for all accounts\n"), stdout);
  fputs (_("  -d             Delete the password for the named account\n"), stdout);
  fputs (_("  -l             Locks the password entry for \"user\"\n"), stdout);
  fputs (_("  -u             Try to unlock the password entry for \"user\"\n"), stdout);
  fputs (_("  -e             Force the user to change password at next login\n"), stdout);
  fputs (_("  -n min         Set minimum field for \"user\"\n"), stdout);
  fputs (_("  -x max         Set maximum field for \"user\"\n"), stdout);
  fputs (_("  -w warn        Set warn field for \"user\"\n"), stdout);
  fputs (_("  -r service     Use nameservice 'service'\n"), stdout);
  fputs (_("  -D binddn      Use dn \"binddn\" to bind to the LDAP directory\n"),
	 stdout);
  fputs (_("  --help         Give this help list\n"), stdout);
  fputs (_("  --usage        Give a short usage message\n"), stdout);
  fputs (_("  --version      Print program version\n"), stdout);
  fputs (_("Valid services for -r are: files, nis, nisplus, ldap\n"), stdout);
}

int
passwd_main (const char *program, int argc, char **argv)
{
  int admin_only = 0;
  int silent = 0;
  uid_t uid = getuid ();
  user_t *pw_data = NULL;
  char *caller_name = NULL;
  char *use_service = NULL;
  char *binddn = NULL;
  int k_flag = 0, a_flag = 0, d_flag = 0, e_flag = 0, h_flag = 0,
    i_flag = 0, l_flag = 0, n_flag = 0, u_flag = 0, x_flag = 0,
    S_flag = 0, w_flag = 0;
  long inact = 0, age_min = 0, age_max = 0, warn = 0;

  /* Parse program arguments */
  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
	{
	  {"binddn", required_argument, NULL, 'D'},
	  {"service", required_argument, NULL, 'r'},
	  {"version", no_argument, NULL, '\255'},
	  {"usage", no_argument, NULL, '\254'},
	  {"help", no_argument, NULL, '\253'},
	  {NULL, 0, NULL, '\0'}
	};

      c = getopt_long (argc, argv, "adD:hlqr:uSekn:x:i:w:", long_options,
                       &option_index);
      if (c == EOF)
        break;
      switch (c)
	{
	case 'a':
	  a_flag = 1;
	  break;
        case 'd':
          d_flag = 1;
          admin_only = 1;
          break;
	case 'D':
	  binddn = optarg;
	  break;
        case 'e':
          e_flag = 1;
          admin_only = 1;
          break;
        case 'h':
          h_flag = 1;
          admin_only = 1;
          break;
        case 'i':
	  i_flag = 1;
          inact = conv2long (optarg);
          if (inact <= -2)
	    {
	      print_error (program);
	      return E_BAD_ARG;
	    }
	  admin_only = 1;
          break;
        case 'k':
          k_flag = 1;  /* ok for users */
          break;
        case 'l':
          l_flag = 1;
          admin_only = 1;
          break;
        case 'n':
	  n_flag = 1;
          age_min = conv2long (optarg);
	  if (age_min <= -2)
	    {
	      print_error (program);
	      return E_BAD_ARG;
	    }
          admin_only = 1;
          break;
        case 'q':
          silent = 1;  /* ok for users */
          break;
        case 'r':
          if (use_service != NULL)
            {
              print_error (program);
              return E_BAD_ARG;
            }
          if (strcasecmp (optarg, "yp") == 0 ||
              strcasecmp (optarg, "nis") == 0)
            use_service = "nis";
          else if (strcasecmp (optarg, "nis+") == 0 ||
                   strcasecmp (optarg, "nisplus") == 0)
            use_service = "nisplus";
          else if (strcasecmp (optarg, "files") == 0)
            use_service = "files";
	  else if (strcasecmp (optarg, "ldap") == 0)
	    use_service = "ldap";
          else
            {
              fprintf (stderr, _("Service \"%s\" not supported\n"), optarg);
              print_usage (stderr, program);
              return E_BAD_ARG;
            }
          break;
        case 'x':
	  x_flag = 1;
          age_max = conv2long (optarg);
	  if (age_max <= -2)
	    {
	      print_error (program);
	      return E_BAD_ARG;
	    }
	  admin_only = 1;
          break;
        case 'S':
          S_flag = 1;  /* ok for users */
          break;
        case 'u':
          u_flag = 1;
          admin_only = 1;
          break;
        case 'w':
	  w_flag = 1;
          warn = conv2long (optarg);
          if (warn <= -2)
	    {
	      print_error (program);
	      return E_BAD_ARG;
	    }
          admin_only = 1;
          break;
	case '\253':
          print_help (program);
          return 0;
        case '\255':
          print_version (program);
          return 0;
        case '\254':
          print_usage (stdout, program);
          return E_USAGE;
        default:
          print_error (program);
          return E_BAD_ARG;
        }
    }

  argc -= optind;
  argv += optind;

  /* We have more than one username or we have -S -a with a
     username */
  if (argc > 1|| (a_flag && S_flag && argc != 0))
    {
      fprintf (stderr, _("%s: Too many arguments\n"), program);
      print_error (program);
      return E_USAGE;
    }

  /* For admin only commands we need a user name */
  if (argc == 0 && admin_only)
    {
      fprintf (stderr, _("%s: user argument missing\n"), program);
      print_error (program);
      return E_USAGE;
    }

  /* Print a list of all users with status informations.
     The -a flag requires -S, no other flags, no username, and
     you must be root.  */
  if (a_flag)
    {
      struct passwd *pw;

      if (admin_only || !S_flag || (argc != 0))
	{
	  print_error (program);
	  return E_USAGE;
	}
      if (uid != 0)
        {
          fprintf (stderr, _("%s: Permission denied.\n"), program);
          return E_NOPERM;
        }

      setpwent ();
      while ((pw = getpwent ()))
        display_pw (pw);
      endpwent ();
      return E_SUCCESS;
    }

  if (S_flag && (admin_only || k_flag))
    {
      print_error (program);
      return E_USAGE;
    }
  else
    {
      int buflen = 256;
      char *buffer = alloca (buflen);
      struct passwd resultbuf;
      struct passwd *pw;

      /* Determine our own user name for PAM authentication.  */
      while (getpwuid_r (uid, &resultbuf, buffer, buflen, &pw) != 0
	     && errno == ERANGE)
	{
	  errno = 0;
	  buflen += 256;
	  buffer = alloca (buflen);
	}
      if (!pw)
	{
	  fprintf (stderr, _("%s: Cannot determine your user name.\n"),
		   program);
	  return E_NOPERM;
	}
      caller_name = strdupa (pw->pw_name);

      /* We change the passwd information for another user, get that
         data, too.  */
      if (argc == 1)
        {
          while (getpwnam_r (argv[0], &resultbuf, buffer, buflen, &pw) != 0
                 && errno == ERANGE)
            {
              errno = 0;
              buflen += 256;
              buffer = alloca (buflen);
            }
          if (!pw)
            {
              fprintf (stderr, _("%s: Unknown user %s\n"), program, argv[0]);
              return E_NOPERM;
            }
        }
      pw_data = do_getpwnam (pw->pw_name, use_service);
      if (pw_data == NULL || pw_data->service == S_NONE)
        {
          if (use_service)
            fprintf (stderr, _("%s: User %s is not known to service %s\n"),
                     program, pw->pw_name, use_service);
          else
            fprintf (stderr, _("%s: Unknown user %s\n"), program,
                     pw->pw_name);
          return E_NOPERM;
        }
    }

  /* Check if normal users are allowed to change the data. For NIS+ and
     LDAP, we let the service decide if the user is allowed.  */
  if (uid != 0 && pw_data->service != S_NISPLUS && pw_data->service != S_LDAP)
    {
      if (admin_only)
	{
          fprintf(stderr, _("%s: Permission denied\n"), program);
	  free_user_t (pw_data);
	  return E_NOPERM;
	}
      if (pw_data->pw.pw_uid != uid)
	{
	  fprintf (stderr, _("You cannot change the shadow data for \"%s\".\n"),
		   pw_data->pw.pw_name);
	  syslog (LOG_WARNING, "%d cannot change shadow data for \"%s\"",
		  uid, pw_data->pw.pw_name);
	  free_user_t (pw_data);
	  return E_NOPERM;
	}
    }

  if (S_flag)
    {
      display_pw (&pw_data->pw);
      free_user_t (pw_data);
      return E_SUCCESS;
    }

  /* We only change the password, let PAM do it.  */
  if (!admin_only)
    {
      pam_handle_t *pamh = NULL;
      int flags = 0, ret;

      if (!silent)
	printf (_("Changing password for %s.\n"), pw_data->pw.pw_name);

      if (silent)
	flags |= PAM_SILENT;
      if (k_flag)
	flags |= PAM_CHANGE_EXPIRED_AUTHTOK;

      ret = pam_start ("passwd", pw_data->pw.pw_name, &conv, &pamh);
      if (ret != PAM_SUCCESS)
	{
          fprintf (stderr, _("%s: PAM Failure, aborting: %s\n"),
                   program, pam_strerror (pamh, ret));
          syslog (LOG_ERR, "Couldn't initialize PAM: %s",
                  pam_strerror (pamh, ret));
          free_user_t (pw_data);
	  return E_PAM_ERROR;
	}

      ret = pam_chauthtok (pamh, flags);
      if (ret != PAM_SUCCESS)
	{
          syslog (LOG_ERR, "User %s: %s", caller_name,
                  pam_strerror (pamh, ret));
          sleep (getdef_num ("FAIL_DELAY", 1));
	  fprintf (stderr, "%s: %s\n", program, pam_strerror (pamh, ret));
	  free_user_t (pw_data);
	  return E_PAM_ERROR;
	}

      pam_end (pamh, PAM_SUCCESS);
      return E_SUCCESS;
    }

  if (binddn)
    {
      char prompt[130+strlen (binddn)], *cp;

      pw_data->binddn = strdup (binddn);
      snprintf (prompt, sizeof (prompt), _("Enter LDAP Password:"));
      cp = getpass (prompt);
      pw_data->oldclearpwd = strdup (cp);
    }

  if (d_flag)
    pw_data->newpassword = strdup ("");

  if (u_flag)
    {
      if (pw_data->use_shadow)
	{
	  if (pw_data->sp.sp_pwdp[0] == '!')
	    pw_data->newpassword = strdup (&pw_data->sp.sp_pwdp[1]);
	  else
	    {
	      fprintf (stderr, _("Cannot unlock the password for \"%s\"!\n"),
		       pw_data->pw.pw_name);
	      return E_FAILURE;
	    }
	}
      else
	{
	  if (pw_data->pw.pw_passwd[0] == '!')
	    pw_data->newpassword = strdup (&pw_data->pw.pw_passwd[1]);
	  else
	    {
	      fprintf (stderr, _("Cannot unlock the password for \"%s\"!\n"),
		       pw_data->pw.pw_name);
	      return E_FAILURE;
	    }
	}
    }

  if (l_flag)
    {
      if (pw_data->use_shadow)
	{
	  if (pw_data->sp.sp_pwdp[0] != '!')
	    {
	      pw_data->newpassword = malloc (strlen (pw_data->sp.sp_pwdp) + 2);
	      if (pw_data->newpassword == NULL)
		return E_FAILURE;
	      strcpy (&pw_data->newpassword[1], pw_data->sp.sp_pwdp);
	      pw_data->newpassword[0] = '!';
	    }
	  else
	    {
	      fprintf (stderr, _("Password for \"%s\" is already locked!\n"),
		       pw_data->pw.pw_name);
	      return E_FAILURE;
	    }
	}
      else
	{
	  if (pw_data->pw.pw_passwd[0] != '!')
	    {
	      pw_data->newpassword =
		malloc (strlen (pw_data->pw.pw_passwd) + 2);
	      if (pw_data->newpassword == NULL)
		return E_FAILURE;
	      strcpy (&pw_data->newpassword[1], pw_data->pw.pw_passwd);
	      pw_data->newpassword[0] = '!';
	    }
	  else
	    {
	      fprintf (stderr, _("Password for \"%s\" is already locked!\n"),
		       pw_data->pw.pw_name);
	      return E_FAILURE;
	    }
	}
    }

  if (x_flag)
    pw_data->spn.sp_max = (age_max * DAY) / SCALE;
  if (n_flag)
    pw_data->spn.sp_min = (age_min * DAY) / SCALE;
  if (w_flag)
    pw_data->spn.sp_warn = (warn * DAY) / SCALE;
  if (i_flag)
    pw_data->spn.sp_inact = (inact * DAY) / SCALE;
  if (e_flag)
    pw_data->spn.sp_lstchg = 0;
  if (x_flag || n_flag || w_flag || i_flag || e_flag || e_flag)
    pw_data->sp_changed = TRUE;

  if (do_setpwnam (pw_data) != 0)
    {
      if (pw_data->sp_changed)
	fprintf (stderr,
		 _("Error while changing password expiry information.\n"));
      else
	fprintf (stderr, _("Error while changing password.\n"));
      free_user_t (pw_data);
      return E_FAILURE;
    }
  else
    {
      nscd_flush_cache ("passwd");
      if (!silent)
	{
	  if (pw_data->sp_changed)
	    printf (_("Password expiry information changed.\n"));
	  else
	    printf (_("Password changed.\n"));
	}
    }

  free_user_t (pw_data);

  return E_SUCCESS;
}
