""" 
   Copyright (C) 2001 PimenTech SARL (http://www.pimentech.net)

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
"""

from dbhandler import *
from sax import *
from map import *
from table import *
from set import *
from cStringIO import StringIO

import sys

class QmlDb(DbHandler):
	
	class QmlHandler(CommonHandler):
		def __init__(self, name, qmldb):
			CommonHandler.__init__(self, name)
			self.qmldb = qmldb
			self.tablesToSelect = Set("TablesToSelect")
			self.tablesToInsert = Set("TablesToInsert")
			self.tablesToUpdate = Set("TablesToUpdate")
			self.tablesToCheckBeforeUpdate = Set("TablesToCheckBeforeUpdate")
			self.vparentsChildMap = Map("vParentsChildMap")
			self.tablesInTransaction = Set("TablesInTransaction")			
			self.currentTable = None
			self.envMap = Map('EnvMap') # map environnement
			
		def startElement(self, name, attrs):
			CommonHandler.startElement(self, name, attrs)
			
			self.envMap.clear() # on vide !!!!
			for attr in attrs:
				self.envMap[attr] = attrs[attr]

			if name == 'select' or name == 'insert' or name == 'update' or name == 'update-or-insert':
				self.currentTable = Table(self.envMap['table'])
				self.tablesInTransaction.insert(self.currentTable)
				
			if name == 'select' or name == 'insert':
				vcurrentTable=self.qmldb.pgmlgraph[self.currentTable]
				if vcurrentTable:
					for parent in vcurrentTable.object.get_table_and_parents():
						self.vparentsChildMap[parent] = vcurrentTable
						self.qmldb.stderr.write('self.vparentsChildMap[%s] = %s\n' % (parent,vcurrentTable))
				
			if name == 'transaction':
				self.qmldb.query("begin;")

			elif name == 'select':
				self.tablesToSelect.insert(self.currentTable)
				
			elif name == 'insert':
				self.tablesToInsert.insert(self.currentTable)
				
			elif name == 'update' or name == 'update-or-insert':
				self.tablesToUpdate.insert(self.currentTable)
				if name == 'update-or-insert':
					self.tablesToCheckBeforeUpdate.insert(self.currentTable)

		def characters(self,data,start,length):
			self.envMap[self.stack.top()] = data[start:start+length]
				
		def endElement(self, name):
			CommonHandler.endElement(self, name)

			if name == 'transaction':

				if self.processTransaction():
					self.qmldb.query("commit;")
				else:
					self.qmldb.query("rollback;")
					
				self.tablesToSelect.clear()
				self.tablesToInsert.clear()
				self.tablesToUpdate.clear()
				self.tablesToCheckBeforeUpdate.clear()
				self.vparentsChildMap.clear()
				self.tablesInTransaction.clear()
				
			elif name == 'select' or name == 'insert' or name == 'update' or name == 'update-or-insert':
				self.currentTable = None
				
			elif name == 'attribute':

				if self.parentTag == 'select' or self.parentTag == 'update' or self.parentTag == 'insert' or self.parentTag == 'update-or-insert':
					
					if self.envMap['value'] != None:
						self.currentTable.insert_attribute(self.envMap['name'],'value',self.envMap['value'])
					else:
						self.currentTable.insert_attribute(self.envMap['name'],'value',self.envMap[name])
						
				if self.parentTag == 'select':
					
					if self.envMap['op']:
						self.currentTable.insert_attribute(self.envMap['name'],'op',self.envMap['op'])
					else:
						self.currentTable.insert_attribute(self.envMap['name'],'op','equal') # par defaut
							
				elif self.parentTag == 'update' or self.parentTag == 'update-or-insert':

					if self.envMap['op']:
						self.currentTable.insert_attribute(self.envMap['name'],'op',self.envMap['op'])
						
				elif self.parentTag == 'insert':

					pass

		def _get_ref_between(self, table1, relation, table2):
			if table1['ref_%s' % table2]:
				return 'ref_%s' % table2
			else:
				return 'ref_%s_%s' % (relation,table2)

		def _compute_insert(self, table, refList, refUidListMap):

			uidList = []
			if refList:
				refName = refList[0]
				for uid in refUidListMap[refList[0]]:
					table.insert_attribute(refName,'value',uid)
					uidList = uidList + self._compute_insert(table, refList[1:], refUidListMap)
			else:
				src = table.get_insert()
				uid = self.qmldb.uid_insert(src)
				if uid:
					uidList = [ uid ]
				
			return uidList

		def _compute_select(self, table, refList, refUidListMap):

			uidList = []
			if refList:
				refName = refList[0]
				for uid in refUidListMap[refList[0]]:

					if not table.has_key(refName):
						table.insert_attribute(refName,'op','equal')
					table.insert_attribute(refName,'value',uid)
					
					uidList = uidList + self._compute_select(table, refList[1:], refUidListMap)
			else:
				src = table.get_uid_select()
				
				query = self.qmldb.query(src)
				if query: # PAS TRES BEAU TOUT CA
					self.qmldb.lastSelect = query.getresult() # on sauvegarde le dernier select
					self.qmldb.lastSelectFields = table.get_select_fields() # avec le nom des champs mais sans l'uid du debut !!!
					self.qmldb.lastSelectTableName = str(table) # et le nom de la table
					uidList = map(lambda l:l[0], self.qmldb.lastSelect)

			return uidList
		
		def _compute_update(self, table, refList, refUidListMap):
			
			uidList = []
			
			if refList:
				refName = refList[0]
				for uid in refUidListMap[refList[0]]:
					
					if not table.has_key(refName):
						table.insert_attribute(refName,'op','equal')
					table.insert_attribute(refName,'value',uid)
					
					uidList = uidList + self._compute_update(table, refList[1:], refUidListMap)
					if not uidList:
						return uidList # breaking on error
			else:
				query = 0
				if self.tablesToCheckBeforeUpdate[table]: # si on doit checke l'existence par rapport a une cle
					src = table.get_uid_select()          # update-or-insert
					query = self.qmldb.query(src)
					if not query or not query.ntuples():
						src = table.get_insert() # ben on insere alors
						uid = self.qmldb.uid_insert(src)
						if uid:
							uidList = [ uid ]
				else:
					query = 1 # je fais un update comme prevu (cas update tout court)
					
				if query:
					src = table.get_update()
					query = self.qmldb.query(src)
					if query:
						uidList = [ None ] # ne retourne pas d'uids mais y faut que le parcours s'arrete donc que le tag soit non nul
				
			return uidList # vide forcement en update
		
		def _processTransaction(self, vtable):

			if vtable.tag:
				self.qmldb.stderr.write('already processed %s\n' % vtable)
				return vtable.tag
			
			self.qmldb.stderr.write('processing %s\n' % vtable)
			
			refUidListMap = Map("refUidListMap")
			
			table = vtable.object
			
			for table_or_parent in table.get_table_and_parents():
				
				vtable_or_vparent = self.qmldb.pgmlgraph[table_or_parent]

				self.qmldb.stderr.write('processing parent %s of %s\n' % (vtable_or_vparent,vtable))
				
				for (label, svrelation) in vtable_or_vparent.items():

					self.qmldb.stderr.write('checking label %s\n' % label)
					
					if self.qmldb.pgmlgraph._label_dir(label) == '1': # on est en presence d'une relation x,x,1
						
						for vrelation in svrelation.values():
							
							self.qmldb.stderr.write('checking relation %s\n' % vrelation)
							
							for (label, svtable) in vrelation.items():
								# si on est en presence d'une relation x,x,0 (toujours car il est simplifie ?) :
								if self.qmldb.pgmlgraph._label_dir(label) == '0': 
									
									for vtable_adj in svtable.values(): # vtable - x,x,1 - relation - x,x,0 - vtable_adj

										table_adj = vtable_adj.object
										if table_or_parent.isa == table_adj: # si table herites de table_adj on oublie car
											# on parcours le graphe pas en dfs mais differemment
											self.qmldb.stderr.write('not checked %s -- %s -- %s\n' % (vtable_or_vparent,vrelation,vtable_adj))
											continue

										self.qmldb.stderr.write('checking %s -- %s -- %s\n' % (vtable_or_vparent,vrelation,vtable_adj))
										
										self.qmldb.stderr.write('self.vparentsChildMap[%s] = %s\n' % (table_adj, self.vparentsChildMap[table_adj]))
										
										if (self.tablesToInsert[self.vparentsChildMap[table_adj]] or
											self.tablesToSelect[self.vparentsChildMap[table_adj]]):
											# retournent des uids
												
											refName = self._get_ref_between(table_or_parent,vrelation,table_adj)
											self.qmldb.stderr.write('found ref %s\n' % refName)
											
											uidList = self._processTransaction(self.vparentsChildMap[table_adj])
											refUidListMap[refName] = uidList

			qtable = self.tablesInTransaction[table]
				
			if self.tablesToInsert[qtable]:
					
				self.qmldb.stderr.write('processing insert on %s\n' % qtable)
					
				# on tagge et on sauvegarde les calculs en meme temps :
				vtable.tag = self._compute_insert(qtable,refUidListMap.keys(),refUidListMap)
				return vtable.tag
				 
			if self.tablesToUpdate[qtable]:
				
				self.qmldb.stderr.write('processing update on %s\n' % qtable)
				
				# on tagge et on sauvegarde les calculs en meme temps :
				vtable.tag = self._compute_update(qtable,refUidListMap.keys(),refUidListMap)
				return vtable.tag
			
			if self.tablesToSelect[qtable]:
					
				self.qmldb.stderr.write('processing select on %s\n' % qtable)

				# on tagge et on sauvegarde les calculs en meme temps :					
				vtable.tag = self._compute_select(qtable,refUidListMap.keys(),refUidListMap)
				return vtable.tag
			
			return [] # failed
		
		def processTransaction(self):
			
			self.qmldb.lastTransaction = []
			self.qmldb.lastSelect = None
			self.qmldb.lastSelectFields = None
			self.qmldb.lastSelectTableName = None			
			
			self.qmldb.pgmlgraph.clear_tags()
			
			for qtable in self.tablesInTransaction.values(): # pour chaque table issue du qml (transaction)

				self.qmldb.stderr.write('top processing request on %s\n' % qtable)
				vtable = self.qmldb.pgmlgraph[qtable] # on recupere le sommet ds le graphe issue du pgml
				
				if not vtable:
					self.qmldb.log.warning("no such table %s in pgml schema" % qtable)
					self.qmldb.lastTransaction = [] # rollback
					self.qmldb.lastSelect = None # rollback
					break
				
				if not vtable.tag:
					self.qmldb.lastTransaction = self._processTransaction(vtable)

				if not self.qmldb.lastTransaction:
					self.qmldb.log.warning("transaction failed for table %s" % qtable)
					break
					
			return self.qmldb.lastTransaction
			
	def __init__(self, pgmlgraph, DBNAME, DBUSER, DBPWD, DBHOST = 'localhost', DBPORT = '5432', log = LOG('QmlDb.log'), name = 'QmlDb', debug = 1):
		DbHandler.__init__(self, DBNAME, DBUSER, DBPWD, DBHOST, DBPORT, log, name, debug)
		self.pgmlgraph = pgmlgraph
		self.pgmlgraph.simplify() # avant tout on simplifie le graphe (les refs et plus de relation de degre > 2
		self.lastTransaction = []
		self.lastSelect = None
		self.lastSelectFields = None
		self.lastSelectTableName = None
		self.parser = self._get_parser()

	def write_last_select(self, output):

		if not self.lastSelectTableName: return
		output.write("<table name='%s'>\n" % self.lastSelectTableName)

		if self.lastSelectFields and self.lastSelect:
			
			for line in self.lastSelect:
				
				output.write("<row>\n")
				i = 1
				for attr in self.lastSelectFields:
					output.write("<attribute name='%s'>" % attr)
					output.write("<![CDATA[%s]]>" % line[i])
					output.write("</attribute>\n")
					i = i + 1
				output.write("</row>\n")
				
		output.write("</table>\n")
		
	def _get_parser(self):

		pf=saxexts.ParserFactory()
		p=pf.make_parser('xml.sax.drivers.drv_xmlproc')
		p.setDocumentHandler(self.QmlHandler('doc_handler', self))
		p.setDTDHandler(self.QmlHandler('dtd_handler', self))
		p.setErrorHandler(self.QmlHandler('err_handler', self))
		p.setEntityResolver(self.QmlHandler('ent_handler', self))
		
		return p

	def parse(self, filename):
		try:
			self.parser.parse(filename)
			return self.lastTransaction
		except:
			return []
		
	def parseFile(self, fileobj):
		try:
			self.parser.parseFile(fileobj)
			return self.lastTransaction
		except:
			return []
	
	def read(self, filename):
		
		return self.parse(filename)
			
	def readFile(self, fileobj):

		return self.parseFile(fileobj)

	def xquery(self, strquery):

		stringIO = StringIO()
		stringIO.write(strquery)
		stringIO.seek(0)
		return self.readFile(stringIO)
