------------------------------------------------
-- Produces an interval value from an I-schedule
-- broken into components.
------------------------------------------------
create or replace function pgj_schedule_to_interval
(
	smallint,	-- $1 day
	smallint,	-- $2 hour
	smallint	-- $3 minute
)
returns interval
as '
	declare
		schDay		alias for $1;
		schHour		alias for $2;
		schMinute	alias for $3;

		tiStr		text;

	begin
		tiStr := '''';

		if schDay is not null then
			tiStr := tiStr || schDay || '' day '';
		end if;

		if schHour is not null then
			tiStr := tiStr || schHour || '' hour '';
		end if;

		tiStr := tiStr || schMinute || '' minute '';

		return cast( tiStr as interval );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the number of minutes in an interval.
------------------------------------------------
create or replace function pgj_interval_to_minutes
(
	interval	-- $1 ti
)
returns int
as '
	declare
		ti		alias for $1;

	begin
		return	extract( ''day''	from ti ) * 24 * 60 +
				extract( ''hour''	from ti ) * 60 +
				extract( ''minute''	from ti );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of an I-schedule.
------------------------------------------------
create or replace function pgj_schedule_trigger_time_I
(
	timestamp,	-- $1 startupTime
	timestamp,	-- $2 currentTime
	smallint,	-- $3 schDay
	smallint,	-- $4 schHour
	smallint	-- $5 schMinute
)
returns timestamp
as '
	declare
		startupTime		alias for $1;
		currentTime		alias for $2;
		schDay			alias for $3;
		schHour			alias for $4;
		schMinute		alias for $5;

		schInterval		interval;
		upInterval		interval;
		schMinutes		int;
		upMinutes		int;
		nextIteration	int;

	begin
		schInterval	:= pgj_schedule_to_interval( schDay, schHour, schMinute );
		upInterval	:= currentTime - startupTime;

		schMinutes	:= pgj_interval_to_minutes( schInterval );
		upMinutes	:= pgj_interval_to_minutes( upInterval );
		
		nextIteration := ceil( cast( upMinutes as real ) / cast( schMinutes as real ) );
		
		return startupTime + ( schInterval * nextIteration );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of a D- or
-- W-schedule.
------------------------------------------------
create or replace function pgj_schedule_trigger_time_DW
(
	timestamp,	-- $1 currentTime
	char,		-- $2 schType
	smallint,	-- $3 schYear
	smallint,	-- $4 schMonth
	smallint,	-- $5 schDay
	smallint,	-- $6 schHour
	smallint	-- $7 schMinute
)
returns timestamp
as '
	declare
		currentTime		alias for $1;
		schType			alias for $2;
		schYear			alias for $3;
		schMonth		alias for $4;
		schDay			alias for $5;
		schHour			alias for $6;
		schMinute		alias for $7;

		triggerTime		timestamp;
		retryInterval	interval;
		tsStr			text;
		yyyy			smallint;
		mm				smallint;
		dd				smallint;
		hh				smallint;
		nn				smallint;
		dow				smallint;

	begin
		retryInterval := interval ''00:00'';

		if schYear is null then
			retryInterval := interval ''1 year'';
			yyyy := extract( ''year'' from currentTime );
		else
			yyyy := schYear;
		end if;

		if schMonth is null then
			retryInterval := interval ''1 month'';
			mm := extract( ''month'' from currentTime );
		else
			mm := schMonth;
		end if;

		if schDay is null then
			retryInterval := interval ''1 day'';
			dd := extract( ''day'' from currentTime );
		else
			if schType = ''D'' then
				dd := schDay;
			else
				dow := extract( ''dow'' from currentTime );
				dd  := extract( ''day'' from currentTime ) + ( schDay + 7 - dow ) % 7;
				retryInterval := ''1 week'';
			end if;
		end if;

		if schHour is null then
			retryInterval := interval ''1 hour'';
			hh := extract( ''hour'' from currentTime );
		else
			hh := schHour;
		end if;

		nn := schMinute;

		tsStr := yyyy || ''-'' || mm || ''-'' || dd || '' '' || hh || '':'' || nn;
		triggerTime := cast( tsStr as timestamp );
		if triggerTime < currentTime then
			triggerTime := triggerTime + retryInterval;
		end if;

		if triggerTime < currentTime then
			triggerTime := null;
		end if;

		return triggerTime;
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of a schedule.
-- Redirects the call to the function specific
-- to the type.
------------------------------------------------
create or replace function pgj_schedule_trigger_time
(
	timestamp,	-- $1 startupTime
	timestamp,	-- $2 currentTime
	char,		-- $3 schType
	smallint,	-- $4 schYear
	smallint,	-- $5 schMonth
	smallint,	-- $6 schDay
	smallint,	-- $7 schHour
	smallint	-- $8 schMinute
)
returns timestamp
as '
	declare
		startupTime		alias for $1;
		currentTime		alias for $2;
		schType			alias for $3;
		schYear			alias for $4;
		schMonth		alias for $5;
		schDay			alias for $6;
		schHour			alias for $7;
		schMinute		alias for $8;

		triggerTime		timestamp;
	begin
		if schType in ( ''D'', ''W'' ) then
			triggerTime := pgj_schedule_trigger_time_DW( currentTime, schType,
									schYear, schMonth, schDay, schHour, schMinute );
		elsif schType = ''I'' then
			triggerTime := pgj_schedule_trigger_time_I( startupTime, currentTime,
													schDay, schHour, schMinute );
		else
			triggerTime := null;
		end if;
		
		return triggerTime;
	end;
' language 'plpgsql';
