/*  XNECVIEW - a program for previewing NEC2 structure files
 *
 *  Copyright (C) 1998, P.T. de Boer -- pa3fwm@amsat.org / ptdeboer@cs.utwente.nl
 *
 *  Distributed on the conditions of the GPL: see the files README and
 *  COPYING, which accompany this source file.
 */

/* ------------------------------------------------------------------------ */

/* some defines: */

#define ini_phi 30         /* initial rotation around Z-axis */
#define ini_theta 70       /* initial tilting of Z-axis */
#define ini_zoom 1.0       /* initial zoom-factor */
#define ini_trx 0          /* initial horizontal displacement of origin w.r.t. center of window */
#define ini_try 0          /* initial vertical displacement of origin w.r.t. center of window */

#define ini_winsize 500    /* initial size of window (pixels) */

#define MAXWIRES 1000      /* maximum number of wires */
#define MAXSURFACES 10     /* maximum number of surfaces */

#define Axislen 1.1        /* length of axes, compared to largest dimension of the antenna */

#define DRAG_NEG 1         /* put 0 in order not to draw the negative axes while dragging; this may be useful on slow systems because the negative axes are dotted lines */

/* ------------------------------------------------------------------------ */

#include  <X11/StringDefs.h> 
#include  <X11/Intrinsic.h> 
#include  <X11/Core.h> 
#include  <X11/Xaw/Box.h> 
#include  <X11/Xlib.h> 
#include  <X11/keysym.h> 
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

Display *display;
Drawable window;
Widget drawing;
GC gc;
int redraw=1;        /* global flag which signifies need for redrawing */
int dragging=0;      /* global flag to indicate that user is dragging the structure, so drawing should be quick */


/* ------------------------- interpreting the NEC data -------------------- */

typedef struct {
   double x,y,z;
} Point;

typedef struct {
   Point p0,p1;
   double thick;
} Wire;

int numwires;
Wire wires[MAXWIRES];

typedef struct {
   Point p0,p1,p2;
} Surface;

int numsurfaces;
Surface surfaces[MAXSURFACES];


double xmax,ymax,zmax;
double xmin,ymin,zmin;
double extr;

void updateextremes(Point *p)
{
   if (p->x > xmax) { xmax=p->x; if (xmax>extr) extr=xmax; }
   if (p->y > ymax) { ymax=p->y; if (ymax>extr) extr=ymax; }
   if (p->z > zmax) { zmax=p->z; if (zmax>extr) extr=zmax; }
   if (p->x < xmin) { xmin=p->x; if (-xmin>extr) extr=-xmin; }
   if (p->y < ymin) { ymin=p->y; if (-ymin>extr) extr=-ymin; }
   if (p->z < zmin) { zmin=p->z; if (-zmin>extr) extr=-zmin; }
}


void reflect(int plane)           /* perform a reflection of all antenna elements */
{
   Wire *w,*w0;
   int i;

   for ( i=numwires, w0=wires, w=wires+numwires ; i>0 ; i--, w0++, w++, numwires++ ) {
      if (numwires==MAXWIRES) { puts("Too many wires."); exit(1); }
      *w=*w0;
      switch (plane) {
         case 0: w->p0.x=-w->p0.x; w->p1.x=-w->p1.x; break;
         case 1: w->p0.y=-w->p0.y; w->p1.y=-w->p1.y; break;
         case 2: w->p0.z=-w->p0.z; w->p1.z=-w->p1.z; break;
      }
   }
}


void removecommas(char *s)        /* change commas in string into spaces */
{
   char *p;
   p=s; while (p=strchr(p,',')) *p++=' ';
}


void read_nec(void)               /* read the input file, and update the arrays of wires and surfaces accordingly */
{
   char s[100];
   Wire *w;
   Surface *su;
   int i;

   xmax=ymax=zmax=xmin=ymin=zmin=extr=0;

   gets(s);
   while (!feof(stdin) && (s[0]!='E' || s[1]!='N')) {
      if (s[0]=='C' && s[1]=='M') {
         puts(s);
      } else if (s[0]=='G') switch (s[1]) {
         case 'W':
            if (numwires==MAXWIRES) { puts("Too many wires."); exit(1); }
            w=wires+numwires++;
            removecommas(s);
            sscanf(s,"GW%*i%*i%lg%lg%lg%lg%lg%lg%lg",&w->p0.x,&w->p0.y,&w->p0.z,&w->p1.x,&w->p1.y,&w->p1.z,&w->thick);
            updateextremes(&w->p0);
            updateextremes(&w->p1);
            break;
         case 'X':
            removecommas(s);
            sscanf(s,"GX%*i%i",&i);
            if (i%10==1) reflect(2);
            if ((i/10)%10==1) reflect(1);
            if ((i/100)==1) reflect(0);
            break;
         case 'A':
         case 'F':
         case 'H':
         case 'M':
         case 'R':
            printf("Warning: unhandled cardtype in %s\n",s);
            break;
      } else if (s[0]=='S') switch (s[1]) {
         case 'M':
            if (numsurfaces==MAXSURFACES) { puts("Too many surfaces."); exit(1); }
            su=surfaces+numsurfaces;
            removecommas(s);
            sscanf(s,"SM%*i%*i%lg%lg%lg%lg%lg%lg",&su->p0.x,&su->p0.y,&su->p0.z,&su->p1.x,&su->p1.y,&su->p1.z);
            updateextremes(&su->p0);
            updateextremes(&su->p1);
            break;
         case 'C':
            su=surfaces+numsurfaces++;
            removecommas(s);
            sscanf(s,"SC%*i%*i%lg%lg%lg",&su->p2.x,&su->p2.y,&su->p2.z);
            updateextremes(&su->p2);
            break;
         case 'P':
            printf("Warning: unhandled cardtype in %s\n",s);
            break;
      }
      gets(s);
   }
}


/* ------------------------- coordinate conversion, drawing routine -------------------- */

                               /* these define the projection: */
double phi=ini_phi,theta=ini_theta;    /* angles defining direction of eye */
double zoom=ini_zoom;                  /* zoom factor */
double trx=ini_trx,try=ini_try;        /* 2D-translation, as a fraction of winsize */
int winsizex,winsizey;                 /* size of window in pixels */

                               /* and these are calculated from them: */
double Xx,Xy,Xz,Yx,Yy,Yz;         /* projection matrix */
double Xtr,Ytr;                   /* 2D-translation */
int winsize;                      /* min(winsizex,winsizey) */



void calcproj(void)     /* calculate the projection matrix */
{
   double ph,th;
   double scale;

   if (winsizey<winsizex) winsize=winsizey;
   else winsize=winsizex;
   scale=winsize*zoom/(3.0*extr);
   ph=phi*(M_PI/180.);
   th=theta*(M_PI/180.);
   Xz = 0;
   Yz = -scale*sin(th);
   Xx = -scale*sin(ph);
   Xy = scale*cos(ph);
   Yx = scale*cos(ph)*cos(th);
   Yy = scale*sin(ph)*cos(th);
   Xtr = 0.5 + trx*zoom*winsize + 0.5*winsizex;
   Ytr = 0.5 + try*zoom*winsize + 0.5*winsizey;
}


void proj(Point *p,int *c)      /* perform the projection: *p defines a point in 3D-space, *c defines a point in the window */
{
   c[0] = Xtr + Xx*p->x + Xy*p->y + Xz*p->z;
   c[1] = Ytr + Yx*p->x + Yy*p->y + Yz*p->z;
}



void draw_antenna()              /* draw the entire antenna, including axes; interrupt drawing if any event (e.g. mouseclick) happens */
{
   Wire *wi;
   Surface *su;
   int i;
   int c0[2],c1[2],c2[2],c3[2];
   Point pp;

   if (XtPending()) return;
   XClearWindow(XtDisplay(drawing),XtWindow(drawing));

   if (XtPending()) return;
   XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
   pp.x=0; pp.y=0; pp.z=0; proj(&pp,c0);
   pp.x=Axislen*extr; pp.y=0; pp.z=0; proj(&pp,c1);
   XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
   pp.x=0; pp.y=Axislen*extr; pp.z=0; proj(&pp,c1);
   XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
   pp.x=0; pp.y=0; pp.z=Axislen*extr; proj(&pp,c1);
   XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);

   if (XtPending()) return;
   XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);
   for (i=0,wi=wires;i<numwires;i++,wi++) {
      proj(&wi->p0,c0);
      proj(&wi->p1,c1);
      XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
   }

   XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);

   for (i=0,su=surfaces;i<numsurfaces;i++,su++) {
      if (XtPending()) return;
      proj(&su->p0,c0);
      proj(&su->p1,c1);
      proj(&su->p2,c2);
      pp.x=su->p0.x+su->p2.x-su->p1.x;
      pp.y=su->p0.y+su->p2.y-su->p1.y;
      pp.z=su->p0.z+su->p2.z-su->p1.z;
      proj(&pp,c3);
      XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
      XDrawLine(display, window, gc, c1[0], c1[1], c2[0], c2[1]);
      XDrawLine(display, window, gc, c2[0], c2[1], c3[0], c3[1]);
      XDrawLine(display, window, gc, c3[0], c3[1], c0[0], c0[1]);
      XDrawLine(display, window, gc, c0[0], c0[1], c2[0], c2[1]);
      XDrawLine(display, window, gc, c1[0], c1[1], c3[0], c3[1]);
   }

   if (DRAG_NEG || !dragging)
   {
      if (XtPending()) return;
      XSetLineAttributes(display, gc, 1, LineOnOffDash, CapRound, JoinRound);
      pp.x=0; pp.y=0; pp.z=0; proj(&pp,c0);
      pp.x=-Axislen*extr; pp.y=0; pp.z=0; proj(&pp,c1);
      XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
      if (XtPending()) return;
      pp.x=0; pp.y=-Axislen*extr; pp.z=0; proj(&pp,c1);
      XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
      if (XtPending()) return;
      pp.x=0; pp.y=0; pp.z=-Axislen*extr; proj(&pp,c1);
      XDrawLine(display, window, gc, c0[0], c0[1], c1[0], c1[1]);
   }

   redraw=0;
}


/* -------------------- X windows stuff (except for drawing) ---------------------- */

void redisplay_event(Widget w,XtPointer client,XExposeEvent *ev)
{
   if(ev->count != 0) return;
   redraw=1;
}


void resize_event(Widget w,XtPointer client,XConfigureEvent *ev)
{
   if(ev->type != ConfigureNotify) return;
   winsizex=ev->width;
   winsizey=ev->height;
   calcproj();
   redraw=1;
}


int lastx=0;
int lasty=0;

void buttonpress_event(Widget w,XtPointer client,XButtonEvent *ev)
{
   lastx=ev->x;
   lasty=ev->y;
}


void buttonrelease_event(Widget w,XtPointer client,XButtonEvent *ev)
{
   if (dragging) {
      dragging=0;
      if (!DRAG_NEG) redraw=1;
      return;
   }

   if (ev->button==Button1) {
      zoom*=1.4142;
      trx-=(double)(ev->x-winsizex/2)/zoom/winsize;
      try-=(double)(ev->y-winsizey/2)/zoom/winsize;
      calcproj();
      redraw=1;
   }
   if (ev->button==Button2) { 
      zoom=ini_zoom;
      phi=ini_phi;
      theta=ini_theta;
      trx=ini_trx;
      try=ini_try;
      calcproj();
      redraw=1;
   }
   if (ev->button==Button3) {
      zoom/=1.4142;
      calcproj();
      redraw=1;
   }
}


void motion_event(Widget w,XtPointer client,XMotionEvent *ev)
{
   int dx,dy;

   dx=ev->x-lastx;
   dy=ev->y-lasty;
   if (ev->state&Button1Mask) {
      theta-=180.*(double)dy/(double)winsizey;
      phi-=180.*(double)dx/(double)winsizex;
   } else if (ev->state&Button3Mask) {
      trx+=(double)(dx)/zoom/winsize;
      try+=(double)(dy)/zoom/winsize;
   } else return;

   calcproj();
   redraw=1;

   dragging=1;

   lastx=ev->x;
   lasty=ev->y;
}


void keypress_event(Widget w,XtPointer client,XKeyPressedEvent *ev)
{
   KeySym key;

   key=XLookupKeysym(ev,0);
   if (key==XK_Q || key==XK_q) {
      XFreeGC(display, gc);
      exit(0);
   }
}



void initX(int argc,char **argv)
{
   Widget toplevel;
   int n;
   Arg wargs[10];

   winsize=winsizex=winsizey=ini_winsize;

   toplevel = XtInitialize(argv[0],"xnecview", NULL, 0, &argc, argv);
   drawing = XtCreateManagedWidget("xnecview",coreWidgetClass, toplevel, NULL, 0);

   n = 0;
   XtSetArg(wargs[n], XtNheight, winsizey); n++;
   XtSetArg(wargs[n], XtNwidth, winsizex); n++;
   XtSetValues(drawing, wargs, n);

   XtAddEventHandler(drawing, ExposureMask, FALSE, (XtEventHandler)redisplay_event, NULL);
   XtAddEventHandler(drawing, StructureNotifyMask, FALSE, (XtEventHandler)resize_event, NULL);
   XtAddEventHandler(drawing, ButtonPressMask, FALSE, (XtEventHandler)buttonpress_event, NULL);
   XtAddEventHandler(drawing, ButtonReleaseMask, FALSE, (XtEventHandler)buttonrelease_event, NULL);
   XtAddEventHandler(drawing, PointerMotionMask, FALSE, (XtEventHandler)motion_event, NULL);
   XtAddEventHandler(drawing, KeyPressMask, FALSE, (XtEventHandler)keypress_event, NULL);

   XtRealizeWidget(toplevel);

   display = XtDisplay(drawing);
   window = XtWindow(drawing);
   gc = XCreateGC(display, window, 0, NULL);
   XSetForeground(display, gc, BlackPixel(display,0));
   XSetBackground(display, gc, WhitePixel(display,0));
}





/* -------------------- main program ------------------------------------------- */

main(int argc,char **argv)
{
   puts("XNECVIEW 0.1 -- preview NEC-2 structure files");
   puts("Copyright (C) 1998 P.T. de Boer -- pa3fwm@amsat.org\n");
   puts("Xnecview comes with ABSOLUTELY NO WARRANTY. This is free software, and you are\n"
        "welcome to redistribute it under certain conditions. For more information: see\n"
        "the files README and COPYING which should accompany this software, or ask the\n"
        "author.\n");

   initX(argc,argv);
   read_nec();
   calcproj();

   for (;;) {
      XEvent xe;
      XtNextEvent(&xe);
      XtDispatchEvent(&xe);
      if (redraw && !XtPending()) draw_antenna();
   }
}


