
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netdb.h>

#include "hmalloc.h"
#include "af_inet.h"
#include "csock.h"
#include "log.h"

/*
 *	Convert address & port to string
 */

char *aptoa(struct in_addr sin_addr, int sin_port)
{
	static char s[15+1+7];
	
	if (sin_addr.s_addr == INADDR_ANY)
		sprintf(s, "*:%d", sin_port);
	else
		sprintf(s, "%s:%d", inet_ntoa(sin_addr), sin_port);
	
	return s;
}

/*
 *	Convert return values of gethostbyname() to a string
 */

char *h_strerror(int i)
{
	static char s[80];
	switch (i) {
		case HOST_NOT_FOUND:
			sprintf(s, "Host not found");
			break;
		case NO_ADDRESS:
			sprintf(s, "No IP address found for name");
			break;
		case NO_RECOVERY:
			sprintf(s, "A non-recovable name server error occurred");
			break;
		case TRY_AGAIN:
			sprintf(s, "A temporary error on an authoritative name server");
			break;
		default:
			sprintf(s, "%d", i);
	}
	return s;
}

/*
 *	Open up a TCP listening socket
 */

void tcp_listen(struct listenq_t *lq)
{
	int on = 1;
	struct sockaddr_in sin;
	
	log(L_SINFO, "Binding TCP interface %s%s", 
		aptoa(lq->sockaddr.inet.sin_addr, lq->sockaddr.inet.sin_port),
		lq->compress ? " (compressed)" : "");
	
	if ((lq->fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		log(L_CRIT, "tcp_listen(): socket(): %s", strerror(errno));
		exit(1);
	}
	
	if (lq->fd > maxfd)
		maxfd = lq->fd;
	
	/* Set socket options.  We try to make the port reusable and have it
	   close as fast as possible without waiting in unnecessary wait states
	   on close. */
	setsockopt(lq->fd, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on));
	
	/* Initialize the socket address. */
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr = lq->sockaddr.inet.sin_addr;
	sin.sin_port = htons(lq->sockaddr.inet.sin_port);
	
	/* Bind the socket to the desired port. */
	if (bind(lq->fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		log(L_CRIT, "Could not bind() TCP interface %s: %s", 
			aptoa(lq->sockaddr.inet.sin_addr, lq->sockaddr.inet.sin_port), strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	if (listen(lq->fd, 5) < 0) {
		log(L_CRIT, "Could not listen() on TCP interface %s: %s",
			aptoa(lq->sockaddr.inet.sin_addr, lq->sockaddr.inet.sin_port), strerror(errno));
		exit(1);
	}
	
	FD_SET(lq->fd, &readfds);
}

/*
 *	Open up all TCP listener sockets
 */
 
void tcp_listens(void)
{
	struct listenq_t *lq;
	
	for (lq = listenq; (lq); lq = lq->next)
		if (lq->af_type == AF_INET)
			tcp_listen(lq);
}

/*
 *	Accept a TCP connection
 */

void tcp_accept(struct listenq_t *lq)
{
	int addrlen, new_fd, i;
	struct sockaddr_in sin;
	struct csock_t *s;
	
	addrlen = sizeof(sin);
	new_fd = accept(lq->fd, (struct sockaddr *)&sin, &addrlen);
	
	if (new_fd < 0) {
		if (errno == EINTR)
			return;
			
		log(L_ERR, "TCP accept() error: %s", strerror(errno));
		return;
	}
	
	log(L_SINFO, "New TCP connection on fd %d from %s",
		new_fd, aptoa(sin.sin_addr, sin.sin_port));
	
	i = 1;
	ioctl(new_fd, FIONBIO, &i);	/* Make the socket non-blocking */
	
	s = sock_alloc(new_fd, AF_INET, DEF_BUFLEN_IN, 80, lq->compress);
	memcpy(&s->sockaddr, &sin, sizeof(sin));
	s->addrlen = addrlen;
	s->eoltype = inet_eol;
	s->eol = INET_EOL;
	
	s->node = hstrdup(inet_ntoa(sin.sin_addr));
	s->port = hstrdup(lq->port);
	
	sock_login(s);
}

