
/*
 *	net_pc.c
 *
 *	PacketCluster (TM) linking "protocol"
 */
 
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#include "net_pc.h"
#include "network.h"
#include "net_user.h"
#include "net_ping.h"
#include "log.h"
#include "cfgfile.h"
#include "hmalloc.h"
#include "timer.h"
#include "ctime.h"

int pc_10(struct link_t *l, int argc, char **argv);
int pc_11(struct link_t *l, int argc, char **argv);
int pc_12(struct link_t *l, int argc, char **argv);
int pc_16(struct link_t *l, int argc, char **argv);
int pc_17(struct link_t *l, int argc, char **argv);
int pc_18(struct link_t *l, int argc, char **argv);
int pc_19(struct link_t *l, int argc, char **argv);
int pc_20(struct link_t *l, int argc, char **argv);
int pc_21(struct link_t *l, int argc, char **argv);
int pc_22(struct link_t *l, int argc, char **argv);
int pc_23(struct link_t *l, int argc, char **argv);
int pc_38(struct link_t *l, int argc, char **argv);
int pc_50(struct link_t *l, int argc, char **argv);
int pc_51(struct link_t *l, int argc, char **argv);

struct pcmd pcmds[] = {
	{ 10,	pc_10	},
	{ 11,	pc_11	},
	{ 12,	pc_12	},
	{ 16,	pc_16	},
	{ 17,	pc_17	},
	{ 18,	pc_18	},
	{ 19,	pc_19	},
	{ 20,	pc_20	},
	{ 21,	pc_21	},
	{ 22,	pc_22	},
	{ 23,	pc_23	},
	{ 38,	pc_38	},
	{ 50,	pc_50	},
	{ 51,	pc_51	},
	{ 0,	NULL	}
};

int pc_msgstats[51];

/*
 *	Convert PC hop counters
 */

char *hops2pcstr(int hops)
{
	static char s[6];
	
	snprintf(s, 4, "H%d", 99 - hops);
	s[4] = '\0';
	
	return s;
}

int pchops2int(char *s)
{
	int i;
	
	if (*s != 'H')
		return 99;
	s++;
	if (!*s)
		return 99;
	i = atoi(s);
	if ((i > 99) || (i < 1))
		return 99;
	
	return 100 - i;
}

/*
 *	Send message to a PC link
 */

int pc_send(struct link_t *via, const char *fmt, ...)
{
	char s[PCMAXLEN+1];
	va_list args;
	
	va_start(args, fmt);
	vsnprintf(s, PCMAXLEN, fmt, args);
	va_end(args);
	
	return csputs(via->sock, s);
}

/*
 *	Send message to all PC links, except for the one it came from
 */

int pc_sendall(struct link_t *via, const char *fmt, ...)
{
	struct link_t *l;
	int i = 0;
	char s[PCMAXLEN+1];
	va_list args;
	
	va_start(args, fmt);
	vsnprintf(s, PCMAXLEN, fmt, args);
	va_end(args);
	
	for (l = links; (l); l = l->next)
		if (l != via && l->proto == lp_pc && l->state == ls_linked) {
			csputs(l->sock, s);
			i++;
		}
		
	return i;
}

/*
 *	Beacon
 */

void pc_beacon(struct link_t *l)
{
	pc_send(l, "PC50^%s^%d^%s^\n", l->mycall, luser_count, hops2pcstr(0));
}

/*
 *	Send a ping
 */

void pc_ping(struct link_t *l, call_t *to, call_t *from, int flag)
{
	pc_send(l, "PC51^%s^%s^%d^\n", to, from, flag);
	csflush(l->sock);
}

/*
 *	Request link closing
 */

void pc_reqclose(struct link_t *l, char *reason)
{
	pc_send(l, "PC39^%s^%s^\n", l->mycall, reason);
	csflush(l->sock);
}

/*
 *	Send full node list in sane order
 */

char *add_node_str(char *s, struct node_t *n)
{
	n->locked = 1;
	sprintf(s, "^%c^%s^0^%d", (n->here) ? '1' : '0', n->call, n->version);
	return s;
}

void pc_sendnodes(struct link_t *l, struct node_t *n)
{
	struct node_t *p;
	char s[PCMAXLEN];
	int i;
	int chosen;
	
	clear_nlocks();
	pc_send(l, "PC19^0^%s^0^%d^H99^\n", l->mycall, pc_version);
	while (1) {
		strcpy(s, "PC19");
		i = 0;
		chosen = 0;
		p = n;
		while (p) {
			if ((chosen && p->via != l && p->hops_ok && p->hops == chosen && !p->locked)
			  || (!chosen && p->via != l && p->hops_ok && !p->locked && p->hops < 99 && p->hops > 0)) {
				if (!chosen)
					chosen = p->hops;
				add_node_str(s + strlen(s), p);
				i++;
			}
			if (strlen(s) > PCMAXLEN - 30 || (!p->next && (i))) {
				sprintf(s + strlen(s), "^%s\n", hops2pcstr(chosen));
				pc_send(l, "%s", s);
				i = 0;
				strcpy(s, "PC19");
			}
			p = p->next;
		}
		if (!chosen)
			return;
	}
	
}

/*
 *	Node list
 */

void pc_nodeadd(struct node_t *n)
{
	struct link_t *l;
	
	for (l = links; (l); l = l->next)
		if (l != n->via && l->proto == lp_pc && l->state == ls_linked)
			pc_sendnodes(l, n);
}

void pc_nodedel(struct node_t *n, char *reason)
{
	if (n->hops <= 98)
		pc_send(n->via, "PC21^%s^%s^%s^\n", n->call, reason, hops2pcstr(n->hops));
}

/*
 *	User list
 */


/*
 *	Send a list of users
 */

void pc_sendusers(struct link_t *l, struct nuser_t *nu)
{
	struct nuser_t *p;
	struct node_t *n;
	char s[PCMAXLEN];
	int i;
	
	for (n = nodes; (n); n = n->next) {
		sprintf(s, "PC16^%s", n->call);
		i = 0;
		
		p = nu;
		while (p) {
			if (p->node == n) {
				sprintf(s + strlen(s), "^%s %c %c", p->call, '-', (p->here) ? '1' : '0');
				i++;
			}
			if (strlen(s) > PCMAXLEN - 30 || (!p->next && (i))) {
				sprintf(s + strlen(s), "^%s^\n", hops2pcstr(n->hops));
				pc_send(l, "%s", s);
				i = 0;
				sprintf(s, "PC16^%s", n->call);
			}
			p = p->next;
		}
	}
}

void pc_useradd(struct nuser_t *nu)
{
	struct link_t *l;
	
	for (l = links; (l); l = l->next)
		if (l != nu->node->via && l->proto == lp_pc && l->state == ls_linked)
			pc_sendusers(l, nu);
}

void pc_userdel(struct nuser_t *nu)
{
	if (nu->hops <= 98)
		pc_sendall(nu->node->via, "PC17^%s^%s^%s^\n", nu->call, nu->node->call, hops2pcstr(nu->hops));
}

void pc_useraway(struct nuser_t *nu)
{
	if (nu->here && nu->hops <= 98) {
		/* Only send PC24 if the user did go away for the first
		   time. Changing the away string doesn"t affect the PC. */
		
		pc_sendall(nu->node->via, "PC24^%s^0^%s^\n", nu->call, hops2pcstr(nu->hops));
	}
}

void pc_userhere(struct nuser_t *nu)
{
	if (nu->hops <= 98)
		pc_sendall(nu->node->via, "PC24^%s^1^%s^\n", nu->call, hops2pcstr(nu->hops));
	
	return;
}

/*
 *	Cluster events
 */

void pc_dx(struct dx_t *dx)
{
	if (dx->hops <= 98)
		pc_sendall(dx->via, "PC11^%s^%s^%s^%s^%s^%s^%s^%s^~\n",
			freq2str(dx->freq), dx->call, pcdatestr(dx->time),
			pctimestr(dx->time), dx->info, dx->fromcall, dx->fromnodec, hops2pcstr(dx->hops));
}

void pc_announce(struct ann_t *ann)
{
	int c = '\0';
	if (strlen(ann->message) > 190) {
		c = ann->message[189];
		ann->message[189] = '\0';
	}
	
	if (ann->hops <= 98)
		pc_sendall(ann->via, "PC12^%s^%s^%s^%s^%s^%s^%s^~\n",
			ann->fromcall, ann->tonodec,
			(*ann->message) ? ann->message : " ",
			(ann->sysop) ? "*" : " ", ann->fromnodec,
			(ann->wx) ? "1" : "0", hops2pcstr(ann->hops));
			
	if (c)
		ann->message[189] = c;
}

void pc_wwv(struct wwv_t *wwv)
{
	if (wwv->hops <= 98)
		pc_sendall(wwv->via, "PC23^%s^%d^%d^%d^%d^%s^%s^%s^%s^~\n",
			pcdatestr(wwv->time), wwv->hour, wwv->sfi, wwv->a,
			wwv->k,
			(*wwv->forecast) ? wwv->forecast : " ",
			wwv->fromcall, wwv->fromnodec,
			hops2pcstr(wwv->hops));
}

void pc_talk(struct talk_t *talk)
{
	struct nuser_t *nu;
	struct node_t *n;
	
	nu = get_nuser(&talk->tocall);
	n = get_node(&talk->tocall);
	
	pc_send(talk->tonode->via, "PC10^%s^%s^%s^%c^%s^%s^~\n",
		talk->fromcall,
		(nu) ? talk->tocall : talk->tonode->call,
		(*talk->message) ? talk->message : " ",
		'*',
		((nu) || (n)) ? " " : talk->tocall,
		talk->fromnodec);
}

/*
 *	PC ^ line parsing
 */

int parse_pc(char *argv[], char *cmd)
{
	int ct = 0;
	
	while (ct < 255)
	{
		if (*cmd == 0)
			break;
		argv[ct++] = cmd;
		while (*cmd && *cmd != '^')
			cmd++;
		if (*cmd)
			*cmd++ = 0;
	}
	argv[ct] = NULL;
	return ct;
}

/*
 *	Bad amount of arguments received
 */
 
int invalid_argc(struct link_t *l, int wanted, int argc, char **argv)
{
	log(L_ERR, "PC: %s: Got %d args, expected %d: %s",
		l->name, argc, wanted, argstr(0, argc, argv));
	return 0;
}

/*
 *	Handshaking
 */

void pc_sendloopcheck(struct link_t *l)
{
	struct node_t *n;
	char s[PCMAXLEN+1];
	int i = 1;
	
	sprintf(s, "PC38^%s", localnode->call);
	for (n = nodes; (n); n = n->next) {
		if (n != localnode) {
			i++;
			if (i > 1)
				strcat(s, ",");
			strcat(s, n->call);
		}
		if (strlen(s) > PCMAXLEN-20 || n->next == NULL) {
			strcat(s, "^~\n");
			pc_send(l, "%s", s);
			sprintf(s, "PC38^");
			i = 0;
		}
	}
}

void pc_sendcluid(struct link_t *l)
{
	pc_send(l, "PC18^%s^%d^~\n", SOFT_STRING, pc_version);
}

void pc_conn_handler(struct link_t *l)
{
	log(L_LINK, "PC: %s: Incoming connection, requesting initialisation",
		l->name);
	pc_sendloopcheck(l);
	pc_sendcluid(l);
}

/*
 *	Disconnection handler
 */

void pc_disc_handler(struct csock_t *s)
{
	log(L_LINK, "PC: %s disconnected", s->link->call);
	link_logout(s);
	
	return;
}

/*
 *	Input handler for the PC protocol
 */
 
int pc_handler(struct csock_t *s, void *data, int len)
{
	char *p = (char *)data;
	int argc;
	char *argv[256];
	int cmdid = 50;
	struct pcmd *cmdp;
	
	if ((argc = parse_pc(argv, p)) == 0 || *argv[0] == '#')
		return 0;
	
	if (strstr(argv[0], "PC") != argv[0] || strlen(argv[0]) != 4)
		return 0;
	
	p = argv[0] + 2;
	cmdid = atoi(p);
	if (cmdid < 10 || cmdid > 51)
		return 0;
	
	for (cmdp = pcmds; cmdp->function != NULL; cmdp++)
		if (cmdp->id == cmdid)
			break;
	if (cmdp->function == NULL)
		return 0;	/* unsupported */
	return (*cmdp->function)(s->link, argc, argv);
}

/*
 *	PC command handlers
 */
 
 /* Talk */
int pc_10(struct link_t *l, int argc, char **argv)
{
	struct talk_t t;
	struct nuser_t *nu;
	
	if (argc != 8)
		return invalid_argc(l, 8, argc, argv);
	
	strncpy(t.fromcall, argv[1], sizeof(call_t));
	strncpy(t.tocall, argv[2], sizeof(call_t));
	strncpy(t.message, argv[3], sizeof(t.message));
	strncpy(t.fromnodec, argv[6], sizeof(call_t));
	t.via = l;
	t.time = now;
	
	if ((nu = get_nuser(&t.tocall)))
		t.tonode = nu->node;
	else {
		t.tonode = get_node(&t.tocall);
		if (*argv[5] != ' ')
			strncpy(t.tocall, argv[5], sizeof(call_t));
	}
	
	if (!t.tonode)
		t.tonode = localnode; /* huh. */
		
	net_talk(&t);
	return 0;
}

 /* DX */
int pc_11(struct link_t *l, int argc, char **argv)
{
	struct dx_t *d;
	
	if (argc != 9)
		return invalid_argc(l, 9, argc, argv);
	
	d = hmalloc(sizeof(struct dx_t));
	
	d->freq = str2freq(argv[1]);
	strncpy(d->call, argv[2], sizeof(d->call));
	d->time = pcstr2t(argv[3], argv[4]);
	strncpy(d->info, argv[5], sizeof(d->info));
	strncpy(d->fromcall, argv[6], sizeof(call_t));
	strncpy(d->fromnodec, argv[7], sizeof(call_t));
	d->hops = pchops2int(argv[8]);
	d->via = l;
	
	net_dx(d);
	return 0;
}

 /* Announcement */
int pc_12(struct link_t *l, int argc, char **argv)
{
	struct ann_t *a;
	
	if (argc != 9)
		return invalid_argc(l, 9, argc, argv);
		
	a = hmalloc(sizeof(struct ann_t));
	
	strncpy(a->fromcall, argv[1], sizeof(call_t));
	strncpy(a->tonodec, argv[2], sizeof(call_t));
	strncpy(a->message, argv[3], sizeof(a->message));
	a->sysop = (*argv[4] == '*');
	strncpy(a->fromnodec, argv[5], sizeof(call_t));
	a->wx = (*argv[6] == '1');
	a->hops = pchops2int(argv[7]);
	a->time = now;
	a->via = l;
	
	net_announce(a);
	return 0;
}

 /* Add user */
int pc_16(struct link_t *l, int argc, char **argv)
{
	struct node_t *n;
	struct nuser_t *nu, *nul, **prevp;
	char *cm, *hr;
	int i;
	int hops;
	
	prevp = &nul;
	if (!(n = get_node((call_t *)argv[1])))
		return 0;
	hops = pchops2int(argv[argc-1]);
	
	for (i = 2; i < argc-1; i++) {
		if (!(cm = strchr(argv[i], ' ')))
			continue;
		*cm = '\0';
		cm++;
		if (!(hr = strchr(cm, ' ')))
			continue;
		*hr = '\0';
		hr++;
		nu = hmalloc(sizeof(struct nuser_t));
		*prevp = nu;
		nu->prevp = prevp;
		prevp = &nu->next;
		nu->next = NULL;
		strncpy(nu->call, argv[i], sizeof(nu->call));
		nu->here = (*hr == '1');
		nu->since = now;
		nu->name[0] = '\0';
		nu->node = n;
		nu->hops = hops;
		nu->away_str = NULL;
		nu->away_time = now;
		nu->sysop = 0;
		nu->priviledged = 0;
	}
	
	net_useradd(nul);
	
	return 0;
}

 /* Delete user */
int pc_17(struct link_t *l, int argc, char **argv)
{
	struct nuser_t *nu;
	struct node_t *n;
	
	if (argc != 4)
		return invalid_argc(l, 4, argc, argv);
		
	if ( ((n = get_node((call_t *)argv[2])))
	   && ((nu = get_nuserh((call_t *)argv[1], n))) )
		net_userdel(nu);
	
	return 0;
}

 /* Request link init */
int pc_18(struct link_t *l, int argc, char **argv)
{
	if (argc != 4)
		return invalid_argc(l, 4, argc, argv);
	log(L_LINK, "PC: %s: requests link initialisation, sending tables",
		l->name);
	l->version = atoi(argv[2]);
	
	pc_sendnodes(l, nodes);
	pc_sendusers(l, nusers);
	pc_send(l, "PC20^\n");
	
	return 0;
}

 /* Add node */
int pc_19(struct link_t *l, int argc, char **argv)
{
	struct node_t *n, *nl, **prevp;
	int i = 1;
	int hops;
	
	prevp = &nl;
	hops = pchops2int(argv[argc-1]);
	
	while (i < argc-2) {
		if (i > argc - 5)
			break;
		n = hmalloc(sizeof(struct node_t));
		*prevp = n;
		n->prevp = prevp;
		prevp = &n->next;
		n->next = NULL;
		strncpy(n->call, argv[i+1], sizeof(n->call));
		n->here = (*argv[i] == '1');
		n->version = atoi(argv[i+3]);
		n->via = l;
		n->since = now;
		n->hops = hops;
		n->hops_ok = 0;
		n->users_ok = 1;
		n->users_known = 0;
		n->users = 0;
		n->rtt = 0;
		n->rtt_ok = 0;
		n->pinging = 0;
		n->locked = 0;
		i += 4;
	}
	
	net_nodeadd(nl);
	
	return 0;
}

 /* Init done */
int pc_20(struct link_t *l, int argc, char **argv)
{
	log(L_LINK, "PC: %s: Finished receiving tables, sending mine - link complete",
		l->name);
		
	pc_sendnodes(l, nodes);
	pc_sendusers(l, nusers);
	pc_send(l, "PC22^\n");
	
	link_finished(l);
	
	return 0;
}

 /* Delete node */
int pc_21(struct link_t *l, int argc, char **argv)
{
	struct node_t *n;
	
	if (argc != 4)
		return invalid_argc(l, 4, argc, argv);
	
	if ((n = get_node((call_t *)argv[1])))
		net_nodedel(n, argv[2]);
	
	return 0;
}

 /* PCDone */
int pc_22(struct link_t *l, int argc, char **argv)
{
	log(L_LINK, "PC: %s: Finished receiving tables - link complete",
		l->name);
	
	link_finished(l);	
	
	return 0;
}

 /* WWV */
int pc_23(struct link_t *l, int argc, char **argv)
{
	struct wwv_t *w;
	struct tm *tm;
	
	if (argc != 11)
		return invalid_argc(l, 11, argc, argv);
	
	w = hmalloc(sizeof(struct wwv_t));
	
	w->time = pcstr2t(argv[1], argv[2]);
	tm = gmtime(&w->time);
	w->hour = tm->tm_hour;
	w->sfi = atoi(argv[3]);
	w->a = atoi(argv[4]);
	w->k = atoi(argv[5]);
	strncpy(w->forecast, argv[6], sizeof(w->forecast));
	strncpy(w->fromcall, argv[7], sizeof(call_t));
	strncpy(w->fromnodec, argv[8], sizeof(call_t));
	w->via = l;
	w->hops = pchops2int(argv[9]);
	
	net_wwv(w);
	
	return 0;
}

 /* Loop check */
int pc_38(struct link_t *l, int argc, char **argv)
{
	char *p, *p2;
	
	if (argc != 3)
		return invalid_argc(l, 3, argc, argv);
	
	p2 = p = argv[1];
	while (p) {
		p2 = strchr(p, ',');
		if (p2) {
			*p2 = '\0';
			p2++;
		}
		
		if (get_node((call_t *)p)) {
			/* LOOP */
		}
		
		p = p2;
	}
	
	return 0;
}

 /* User & hop count beacon */
int pc_50(struct link_t *l, int argc, char **argv)
{
	return 0;
}

 /* Ping */
int pc_51(struct link_t *l, int argc, char **argv)
{
	if (argc != 4)
		return invalid_argc(l, 4, argc, argv);
	
	net_ping(l, (call_t *)argv[1], (call_t *)argv[2], atoi(argv[3]));
	
	return 0;
}

