
/*
 *	net_user.c
 *
 *	network user list handling
 */
 
#include <string.h>
#include <stdio.h>

#include "net_user.h"
#include "network.h"
#include "net_link.h"
#include "hmalloc.h"
#include "log.h"
#include "cluster.h"
#include "f_nuser.h"
#include "timer.h"

struct nuser_t *nusers;	/* Users */

int nuser_count;		/* How many users on the network */
int kuser_count;		/* How many KNOWN users on the network */
int nuser_max;			/* Users high water mark */
int kuser_max;			/* Known users high water */

int away_strings;		/* Away strings allocated */

/* ====================================================================== */
/*  U S E R   T A B L E   H A N D L I N G                                 */
/* ====================================================================== */

/*
 *	Delete an user from the user list
 */

void free_nuser(struct nuser_t *nu)
{
	*nu->prevp = nu->next;
	if (nu->next)
		nu->next->prevp = nu->prevp;
	if (nu->away_str) {
		hfree(nu->away_str);
		away_strings--;
	}
	hfree(nu);
	nuser_count--;
	kuser_count--;
}

/*
 *	Delete all users on a node
 */

void free_nusersn(struct node_t *n)
{
	struct nuser_t *nu, *next;

	nu = nusers;
	while (nu) {
		next = nu->next;
		if (nu->node == n)
			free_nuser(nu);
		nu = next;
	}
}

/*
 * 	Add an user to the list 
 */

struct nuser_t *insert_nuser(struct nuser_t *nu)
{
	struct nuser_t **prevp, *p;
	
	prevp = &nusers;
	p = nusers;
	while ((p) && (strcmp(p->call, nu->call) < 0)) {
		prevp = &p->next;
		p = p->next;
	}
	
	*prevp = nu;
	nu->prevp = prevp;
	nu->next = p;
	if (nu->next)
		nu->next->prevp = &nu->next;
	nuser_count++;
	kuser_count++;
	
	if (nuser_count > nuser_max)	/* High water mark */
		nuser_max = nuser_count;
	if (kuser_count > kuser_max)	/* High water mark */
		kuser_max = kuser_count;
	
	return nu;
}

/*
 *	Find a record from the user list
 */

struct nuser_t *get_nuser(call_t *call)
{
	struct nuser_t *p;

	p = nusers;
	while ((p) && (strcmp(p->call, (char *)call) != 0))
		p = p->next;
	return p;
}

struct nuser_t *get_nuserh(call_t *call, struct node_t *node)
{
	struct nuser_t *p;

	p = nusers;
	while ((p) && (strcmp(p->call, (char *)call) || p->node != node))
		p = p->next;
	return p;
}

struct nuser_t *guess_nuser(call_t *call)
{
	struct nuser_t *user, *u, *p;
	int l, m;

	p = nusers;
	m = 0;
	l = strlen((char *)call);
	if (l != 0) {
		while ((p)) {
			/* Match substring: */
			if (strstr(p->call, (char *)call)) {
				if (!strcmp((char *)call, p->call))
					/* Whoa, exact match! Accept this, even if there are others */
					return p;
				m++;
				u = p;
			}
			p = p->next;
		}
	}
	if (m != 1)		/* Exactly one match */
		return NULL;
	user = u;
	strcpy((char *)call, u->call);
	return user;
}

/*
 *	Calculate the total users count nuser_count, based on the node list
 */

int count_users(void)
{
	struct node_t *n = nodes;
	
	nuser_count = 0;
	while (n) {
		nuser_count += n->users;
		n = n->next;
	}
	if (nuser_count > nuser_max)	/* High water mark */
		nuser_max = nuser_count;

	return nuser_count;
}

/*
 *	User came back
 */

void net_here(struct link_t *l, call_t *call)
{
	struct nuser_t *nu;

	if (!(nu = get_nuser(call)) || ((l) && (nu->node == localnode)) || 
	   (!nu->here))
		return;

	nu->here = 1;
	nu->away_time = now;
	if (nu->away_str) {
		hfree(nu->away_str);
		nu->away_str = NULL;
		away_strings--;
	}
	log(L_NUSER, "User: %s came back", call);
	user_userhere(nu);
	link_userhere(nu);
}


/*
 *	User went away
 */
 
void net_away(struct link_t *l, call_t *call, char *reason)
{
	struct nuser_t *u;

	if (!(u = get_nuser(call)) || ((l) && (u->node == localnode)) )
		return;

	if (reason) {
		if (u->away_str)	/* The user might just be changing the string! */
			hfree(u->away_str);
		else
			away_strings++;
		u->away_str = hstrdup(reason);
		log(L_NUSER, "User: %s went away: %s", call, reason);
	} else
		log(L_NUSER, "User: %s went away.", call);
	
	link_useraway(u);
	u->here = 0;
	u->away_time = now;
	user_useraway(u);
}

/*
 *	Add a list of users
 */

void net_useradd(struct nuser_t *nu)
{
	struct nuser_t *ul, *p, *next;
	
	ul = p = nu;
	p->prevp = &ul;
	
	/* Go through the users, remove invalid/duplicate ones, inform local users and files */
	while (p) {
		next = p->next;
		
		if (get_nuserh(&p->call, p->node) || (!valid_call(&p->call))) {
			if (p->next)
				p->next->prevp = p->prevp;
			*p->prevp = p->next;
			hfree(p);
		} else {
			if (p->node != localnode)
				log(L_NUSER, "User: %s@%s logged in", p->call, p->node->call);
			user_useradd(p);
			set_homenode(p, &p->call, &p->node->call, 5);
		}
		p = next;
	}
	
	link_useradd(ul);	/* feed links */
	
	/* Insert in user list */
	p = ul;
	while (p) {
		next = p->next;
		p->node->users++;
		p->node->users_known = 1;
		insert_nuser(p);
		p = next;
	}
}

/*
 *	Local User login
 */

struct nuser_t *lnuser_login(struct luser_t *lu)
{
	struct nuser_t *u;
	
	u = hmalloc(sizeof(struct nuser_t));
	
	strcpy(u->call, (char *)lu->f->call);
	u->node = localnode;
	*u->name = '\0';
	u->here = 1;
	u->hops = 0;
	u->away_str = NULL;
	u->away_time = now;
	u->sysop = 0;
	u->priviledged = 0;
	u->next = NULL;
	u->prevp = NULL;
	u->since = now;
	net_useradd(u);
	
	return u;
}

/*
 *	Remove a cluster user
 */

void net_userdel(struct nuser_t *nu)
{
	nu->since = now;
	user_userdel(nu);
	
	if (nu->node != localnode)
		log(L_NUSER, "User: %s@%s logged out", nu->call, nu->node->call);
		
	link_userdel(nu);
	
	nu->node->users--;
	free_nuser(nu);
}

