
/*
 *	csock.c
 *
 *	clusse socket abstraction layer
 *	buffering, non-blocking network I/O sockets
 *	supporting telnet, AX.25, NET/ROM
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdarg.h>
#include <zlib.h>
#include <netdb.h>
#include <fcntl.h>

#include "csock.h"
#include "af_inet.h"
#include "af_ax25.h"
#include "af_netrom.h"
#include "log.h"
#include "timer.h"
#include "luser.h"
#include "hmalloc.h"
#include "telnet.h"
#include "login.h"
#include "cstring.h"

#define CS_FLUSH 1
#define CS_NOFLUSH 0

struct csock_t *sockets = NULL;
struct listenq_t *listenq = NULL;

fd_set readfds, writefds;
int maxfd = -1;

int cs_errno = 0;

char *cs_errorstrs[] = {
	"No error?",
	"NULL"
};

/*
 *	Convert af_type to str
 */
 
char *afstr(int af_type)
{
	static char af_inet_s[] = "TCP";
	static char af_ax25_s[] = "AX.25";
	static char af_nr_s[] = "NET/ROM";
	static char af_rs_s[] = "ROSE";
	static char s[10];
	
	switch (af_type) {
	case AF_INET:
		return af_inet_s;
	case AF_AX25:
		return af_ax25_s;
	case AF_NETROM:
		return af_nr_s;
	case AF_ROSE:
		return af_rs_s;
	default:
		sprintf(s, "AF_%d", af_type);
		return s;
	}
	
	return NULL;
}

/************************************************************************
 ****  Socket list management  ******************************************
 ************************************************************************/

/*
 *	Allocate a socket
 */

struct csock_t *sock_alloc(int fd, int af_type, int def_buflen, 
				int obuf_tresh, int compressed)
{
	struct csock_t *s = sockets, **ps = &sockets;
	struct cscomp_t *c;
	
	while (s) {
		ps = &s->next;
		s = s->next;
	}
	
	s = hmalloc(sizeof(struct csock_t));
	*ps = s;
	
	s->prevp = ps;
	s->next = NULL;
	s->fd = fd;
	s->cs_errno = 0;
	
	s->type = cst_unknown;
	s->state = css_unknown;
	
	s->af_type = af_type;
	
	s->call = NULL;
	s->node = NULL;
	s->port = NULL;
	
	s->ibuf.len = def_buflen;
	s->ibuf.tresh = def_buflen;
	s->ibuf.pos = 0;
	s->ibuf.buf = hmalloc(s->ibuf.len);
	
	s->obuf.len = DEF_OBUFLEN;
	s->obuf.tresh = obuf_tresh;
	s->obuf.pos = 0;
	s->obuf.buf = hmalloc(DEF_OBUFLEN);
	
	s->in_count = 0;
	s->out_count = 0;
	
	s->eolmode = eol_text;
	s->eol = NULL;
	
	s->in_handler = NULL;
	s->conn_handler = NULL;
	s->disc_handler = NULL;
	
	if (compressed) {
		c = s->comp = hmalloc(sizeof(struct cscomp_t));
		if ((inflateInit(&c->zin) != Z_OK) || (deflateInit(&c->zout, 9) != Z_OK)) {
			log(L_CRIT, "fd %d: sock_alloc(): zlib init failed! Abandon ship!", s->fd);
			goto failed;
		}
		
		c->z_error = 0;
		
		s->ribuf = hmalloc(sizeof(struct sbuf_t));
		s->ribuf->len = def_buflen;
		s->ribuf->tresh = def_buflen;
		s->ribuf->pos = 0;
		s->ribuf->buf = hmalloc(s->ribuf->len);
		
		s->robuf = hmalloc(sizeof(struct sbuf_t));
		s->robuf->len = DEF_OBUFLEN;
		s->robuf->tresh = obuf_tresh;
		s->robuf->pos = 0;
		s->robuf->buf = hmalloc(DEF_OBUFLEN);
	} else {
		s->comp = NULL;
		s->ribuf = &s->ibuf;
		s->robuf = &s->obuf;
	}
	
	s->lu = NULL;
	
	FD_SET(fd, &readfds);
	if (fd > maxfd)
		maxfd = fd;
	
	log(L_SDEBUG, "fd %d: sock_alloc(): Allocated new socket", s->fd);
	
	return s;
	
failed:
	hfree(s->ibuf.buf);
	hfree(s->obuf.buf);
	hfree(s);
	return NULL;
}

/*
 *	Close and deallocate a socket
 */

void sock_close(struct csock_t *s)
{
	struct cscomp_t *c = s->comp;
	int i;
	
	log(L_SINFO, "Socket: Disconnected %s: %s%s%s:%s (%d) (%s)",
		afstr(s->af_type), (s->call) ? s->call : "", ((s->node) && (s->call)) ? ":" : "",
		(s->node) ? s->node : "", s->port, s->fd, strerror(s->cs_errno));
	FD_CLR(s->fd, &readfds);
	FD_CLR(s->fd, &writefds);
	close(s->fd);
	
	if (s->disc_handler) {
		(*s->disc_handler)(s);
		s->disc_handler = NULL;
	} else
		log(L_ERR, "fd %d: sock_close(): Ouch, no disc_handler defined!");
	
	*s->prevp = s->next;
	if (c) {
		log(L_COMPR, "fd %d: Compression statistics:", s->fd);
		log(L_COMPR, "fd %d:     input: raw %lu, compressed %lu, factor %.2f",
			s->fd, c->zin.total_out, c->zin.total_in,
			c->zin.total_out == 0 ? 0.0 : (double)c->zin.total_in / c->zin.total_out);
		log(L_COMPR, "fd %d:     output: raw %lu, compressed %lu, factor %.2f",
			s->fd, c->zout.total_in, c->zout.total_out,
			c->zout.total_in == 0 ? 0.0 : (double)c->zout.total_out / c->zout.total_in);
		log(L_COMPR, "fd %d:     total: raw %lu, compressed %lu, factor %.2f",
			s->fd, c->zin.total_out + c->zout.total_in, 
			c->zin.total_in + c->zout.total_out,
			(c->zin.total_out + c->zout.total_in == 0) ? 0.0 : 
				( (double)c->zin.total_in + (double)c->zout.total_out )
					/ (c->zin.total_out + c->zout.total_in));
		if ((i = deflateEnd(&c->zout)) != Z_OK && (c->zout.msg))
			log(L_ERR, "fd %d: deflateEnd returned %d: %s",
				s->fd, i, c->zout.msg);
		if ((i = inflateEnd(&c->zout)) != Z_OK && (c->zout.msg))
			log(L_ERR, "fd %d: inflateEnd returned %d: %s",
				s->fd, i, c->zout.msg);
		hfree(s->ribuf->buf);
		hfree(s->ribuf);
		hfree(s->robuf->buf);
		hfree(s->robuf);
		hfree(c);
	}
	hfree(s->ibuf.buf);
	hfree(s->obuf.buf);
	if (s->call)
		hfree(s->call);
	if (s->node)
		hfree(s->node);
	if (s->port)
		hfree(s->port);
	hfree(s);
}

/*
 *	Create an outgoing connection
 */

struct csock_t *sock_connect(int af_type, int compressed, int argc, char **argv)
{
	struct csock_t *s = NULL;
	
#ifdef HAVE_INET
	struct hostent *hp;
	struct servent *sp;
	struct in_addr inaddr;
#endif
	
	int i = 0;
	int fd = -1;
	
	switch (af_type) {

#ifdef HAVE_INET
	case AF_INET:
		hp = NULL;
		if (inet_aton(argv[0], &inaddr) != 0)
			hp = gethostbyaddr((char *)&inaddr, sizeof(struct in_addr), AF_INET);
		if (hp == NULL)
			hp = gethostbyname(argv[0]);
		if (hp == NULL) {
			log(L_ERR, "sock_connect(): Unknown host %s", argv[0]);
			goto trouble;
		}
		
		if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			log(L_ERR, "sock_connect(): socket(): Could not get an AF_INET socket: %s", strerror(errno));
			goto trouble;
		}
		
		if (!(s = sock_alloc(fd, af_type, DEF_BUFLEN_IN, 80, compressed)))
			goto trouble;
		
		memset(&s->sockaddr, 0, sizeof(s->sockaddr));
		s->addrlen = sizeof(struct in_addr);
		s->sockaddr.inet.sin_family = AF_INET;
		memcpy(&s->sockaddr.inet.sin_addr, hp->h_addr_list[0], hp->h_length);
		sp = NULL;
		if (argv[1] == NULL) {
			if ((sp = getservbyname("telnet", "tcp")) != NULL)
				s->sockaddr.inet.sin_port = sp->s_port;
			else
				s->sockaddr.inet.sin_port = htons(23);
		} else {
			s->sockaddr.inet.sin_port = htons(atoi(argv[1]));
			if ((sp = getservbyname(argv[1], "tcp")) != NULL ||
			    (sp = getservbyport(s->sockaddr.inet.sin_port, "tcp")) != NULL)
				s->sockaddr.inet.sin_port = sp->s_port;
		}
		if (s->sockaddr.inet.sin_port == 0) {
			log(L_ERR, "Unknown service %s", argv[1]);
			goto trouble;
		}
		s->node = hstrdup(hp->h_name);
		if (sp != NULL)
			s->port = hstrdup(sp->s_name);
		else {
			s->port = hmalloc(6);
			sprintf(s->port, "%d", ntohs(s->sockaddr.inet.sin_port));
		}
		s->eoltype = inet_eol;
		s->eol = INET_EOL;
		break;
#endif

#ifdef HAVE_AX25
	case AF_AX25:
		if ((fd = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			log(L_ERR, "sock_connect(): socket(): Could not get an AF_AX25 socket: %s", strerror(errno));
			goto trouble;
		}
		
		if (!(s = sock_alloc(fd, af_type, DEF_BUFLEN_AX, 80, compressed)))
			goto trouble;
		
		for (i = 1; i < argc; i++)
			hstrupr(argv[i]);
		
		if (ax25_prepare_outgoing(s, argc, argv))
			goto trouble;
		
		break;
#endif

#ifdef HAVE_NETROM
	case AF_NETROM:
		if ((fd = socket(AF_NETROM, SOCK_SEQPACKET, 0)) < 0) {
			log(L_ERR, "sock_connect(): socket(): Could not get an AF_NETROM socket: %s", strerror(errno));
			goto trouble;
		}
		
		if (!(s = sock_alloc(fd, af_type, DEF_BUFLEN_NR, 80, compressed)))
			goto trouble;
		
		for (i = 1; i < argc; i++)
			hstrupr(argv[i]);
		
		if (netrom_prepare_outgoing(s, argc, argv))
			goto trouble;
		
		break;
#endif

	default:
		log(L_ERR, "Socket: sock_connect() called for unknown af_type");
		goto trouble;
	}
	
	log(L_SINFO, "Socket: Connecting %s: %s%s%s:%s (%d)",
		afstr(s->af_type), (s->call) ? s->call : "", ((s->node) && (s->call)) ? ":" : "",
		(s->node) ? s->node : "", s->port, s->fd);
	
	/*
	 * Ok. Now set up a non-blocking connect...
	 */
	 
	if (fcntl(s->fd, F_SETFL, O_NONBLOCK) == -1) {
		log(L_ERR, "sock_connect(): fcntl O_NONBLOCK failed: %s", strerror(errno));
		goto trouble;
	}
	
	if (connect(s->fd, (struct sockaddr *)&s->sockaddr, sizeof(s->sockaddr)) == -1 && errno != EINPROGRESS) {
		log(L_ERR, "sock_connect(): connect: %s", strerror(errno));
		goto trouble;
	}
	
	s->type = cst_outgoing;
	s->state = css_connecting;
	
	return s;
	
trouble:
	if (fd >= 0)
		close(fd);
	if (s)
		sock_close(s);
		
	return NULL;
}

/*
 *	Notify lower levels of a successful connection
 */
 
void sock_connected(struct csock_t *s)
{
	s->state = css_connected;
	log(L_SINFO, "Socket: Connected %s: %s%s%s:%s (%d)",
		afstr(s->af_type), (s->call) ? s->call : "", ((s->node) && (s->call)) ? ":" : "",
		(s->node) ? s->node : "", s->port, s->fd);
	
	if (s->conn_handler) {
		(*s->conn_handler)(s);
		s->conn_handler = NULL;
	} else
		log(L_ERR, "fd %d: sock_connected(): Ouch, no conn_handler defined!");
	
	return;
}

/*
 *	Grow a buffer, if it's too small
 */

int grow_buf(struct sbuf_t *sb, int min_free)
{
	int space;
	
	if ((space = sb->len - sb->pos) < min_free) {
		sb->len += sb->tresh;
//		log(L_SDEBUG, "grow_buf(): Increasing to %d bytes", sb->len);
		sb->buf = hrealloc(sb->buf, sb->len);
		space = sb->len - sb->pos;
	}
	
	return space;
}

/************************************************************************
 ****  Compression  *****************************************************
 ************************************************************************/

/*
 *	Inflate data from ribuf to ibuf
 */

int sock_inflate(struct csock_t *s)
{
	int len, ret, av_in, av_out;
	z_stream *zp = &s->comp->zin;
	
	if (s->comp->z_error) {	/* Broken. Return immediately. */
		log(L_ERR, "fd %d: sock_inflate: Double error", s->fd);
		errno = s->comp->z_error;
		return -1;
	}
	
	/* Initialize zp from the socket buffers */
	zp->next_in = s->ribuf->buf;
	zp->avail_in = av_in = s->ribuf->pos;
	zp->next_out = s->ibuf.buf + s->ibuf.pos;
	zp->avail_out = av_out = s->ibuf.len - s->ibuf.pos;
	
	log(L_SDEBUG, "fd %d: sock_inflate avail_in=%d avail_out=%d", 
		s->fd, zp->avail_in, zp->avail_out);
		
	ret = inflate(zp, Z_PARTIAL_FLUSH);
	
	log(L_SDEBUG, "fd %d:      inflate ret=%d in=%d avail_in=%d out=%d avail_out=%d", 
		s->fd, ret, av_in - zp->avail_in, zp->avail_in, 
		av_out - zp->avail_out, zp->avail_out);
	
	/* Update socket buffers from zp */
	memmove(s->ribuf->buf, s->ribuf->buf + s->ribuf->pos - zp->avail_in, 
		zp->avail_in);
	s->ribuf->pos = zp->avail_in;
	len = s->ibuf.len - zp->avail_out;
	s->ibuf.pos += len;
	
	/* Handle errors */
	switch (ret) {
	case Z_OK:
		/* Hey. Great. */
		return len;
	case Z_BUF_ERROR:
		/*
		 *	Not enough new data to proceed,
		 *	let's hope we'll get some more below.
		 */
		 break;
	case Z_STREAM_END:
		/* Ouch. End of compressed stream. Something like EOF. */
		log(L_ERR, "fd %d: sock_inflate(): Z_STREAM_END (%s)", s->fd, zp->msg);
		errno = s->comp->z_error = ZERR_STREAM_END;
		return -1;
	case Z_STREAM_ERROR:
		/* Something weird */
		log(L_ERR, "fd %d: sock_inflate(): Z_STREAM_ERROR (%s)", s->fd, zp->msg);
		errno = s->comp->z_error = ZERR_STREAM_ERROR;
		return -1;
	case Z_DATA_ERROR:
		/* Compression protocol error */
		log(L_ERR, "fd %d: sock_inflate(): Z_DATA_ERROR (%s)", s->fd, zp->msg);
		errno = s->comp->z_error = ZERR_DATA_ERROR;
		return -1;
	case  Z_MEM_ERROR:
		/* Not enough memory */
		log(L_ERR, "fd %d: sock_inflate(): Z_MEM_ERROR (%s)", s->fd, zp->msg);
		errno = s->comp->z_error = ZERR_MEM_ERROR;
		return -1;
	default:
		log(L_ERR, "fd %d: sock_inflate(): inflate returned %d (%s)", s->fd, ret, zp->msg);
	}
	
	if (zp->avail_in != 0) {
		log(L_ERR, "fd %d: sock_inflate(): inflate didn't consume all input! (avail_in=%d, ret=%d)", zp->avail_in, ret);
		errno = s->comp->z_error = ZERR_UNKNOWN;
		return -1;
	}
	
	/*
	 * Our only hope is that inflate has consumed all input, has not 
	 * been able to decompress it yet, and we can get some more.
	 */
	
	return 0;
}

/*
 *	Deflate all data from obuf to robuf, flushing if necessary
 */

int sock_deflate(struct csock_t *s, int flush)
{
	z_stream *zp = &s->comp->zout;
	int ret, av_in, av_out;
	
	if (s->comp->z_error) {	/* Broken. Return immediately. */
		log(L_ERR, "fd %d: sock_deflate: Double error", s->fd);
		errno = s->comp->z_error;
		return -1;
	}
	
	/* Initialize zp from the socket buffers */
	zp->next_in = s->obuf.buf;
	zp->avail_in = s->obuf.pos;
	
	do {
		zp->next_out = s->robuf->buf + s->robuf->pos;
		av_out = zp->avail_out = s->robuf->len - s->robuf->pos;
		av_in = zp->avail_in;
		
		log(L_SDEBUG, "fd %d: sock_deflate avail_in=%d avail_out=%d %s", 
			s->fd, zp->avail_in, zp->avail_out,
			flush ? "CS_FLUSH" : "CS_NOFLUSH");
			
		ret = deflate(zp, flush ? Z_PARTIAL_FLUSH : Z_NO_FLUSH );
		
		log(L_SDEBUG, "fd %d:      deflate ret=%d in=%d avail_in=%d out=%d avail_out=%d", 
			s->fd, ret, av_in - zp->avail_in, zp->avail_in, 
			av_out - zp->avail_out, zp->avail_out);
		
		/* Update socket buffers from zp */
		s->robuf->pos = s->robuf->len - zp->avail_out;
		memmove(s->obuf.buf, s->obuf.buf + s->obuf.pos - zp->avail_in,
			zp->avail_in);
		s->obuf.pos = zp->avail_in;
		
		/* Handle errors */
		switch (ret) {
 		case Z_OK:
			/* Hey. Great. */
			return 0;
		case Z_BUF_ERROR:
			/*
			 *	Progress not possible, probably the output
			 *	buffer is full. We'll increase it below.
			 */
			 break;
		case Z_STREAM_END:
			/* Ouch. Can not happen. */
			log(L_ERR, "fd %d: sock_deflate(): Z_STREAM_END (%s)", s->fd, zp->msg);
			errno = s->comp->z_error = ZERR_STREAM_END;
			return -1;
		case Z_STREAM_ERROR:
			/* Something weird */
			log(L_ERR, "fd %d: sock_deflate(): Z_STREAM_ERROR (%s)", s->fd, zp->msg);
			errno = s->comp->z_error = ZERR_STREAM_ERROR;
			return -1;
		case  Z_MEM_ERROR:
			/* Not enough memory */
			log(L_ERR, "fd %d: sock_deflate(): Z_MEM_ERROR (%s)", s->fd, zp->msg);
			errno = s->comp->z_error = ZERR_MEM_ERROR;
			return -1;
		default:
			log(L_ERR, "fd %d: sock_deflate(): deflate returned %d (%s)", s->fd, ret, zp->msg);
			
		}
		if (zp->avail_out == 0)
			grow_buf(s->robuf, s->robuf->tresh);
	} while (zp->avail_out == 0);
	
	s->obuf.pos = 0;
	
	return 0;
}
 
/************************************************************************
 **** Socket input ******************************************************
 ************************************************************************/
 
/*
 *	Read data from socket
 */

int sock_read(struct csock_t *s)
{
	int len, space;
	
	space = grow_buf(s->ribuf, s->ribuf->tresh);
	len = read(s->fd, s->ribuf->buf + s->ribuf->pos, space);
	
	if (len == 0) {
		log(L_SDEBUG, "fd %d: sock_read(): read() returned EOF", s->fd);
		return -1;
	}
	
	if (len == -1) {
		s->cs_errno = errno;
		log(L_SDEBUG, "fd %d: sock_read(): read() error: %s", 
			s->fd, strerror(s->cs_errno));
		if (s->cs_errno == EINTR)
			return 0;
		return -1;
	}
	
	s->ribuf->pos += len;
	log(L_SDEBUG, "fd %d: sock_read(): Read %d bytes (ribuf pos %d len %d)",
		s->fd, len, s->ribuf->pos, s->ribuf->len);
	
	if (s->comp) /* If compressed, inflate from ribuf to ibuf */
		if ((len = sock_inflate(s)) == -1)
			return -1; /* Ouch. **/
	
	s->in_count += len;
	
	return len;
}

/*
 *	Handle the contents of the input buffer
 */

int sock_handle(struct csock_t *s)
{
	char *p, *cp, *cpp;
	int i, n;
	
	if (s->af_type == AF_INET)
		if (telnet_handle(s) == -1)
			return 0;	/* Needs more */
	
	if (s->eolmode == eol_text) {
		p = s->ibuf.buf;
		i = 0;
		
		/* Find EOL */
		while ((i < s->ibuf.pos) && (*p != s->eol[0]) 
		  && (*p != '\0')) {
			p++;
			i++;
		}
		
		if (i == s->ibuf.pos)
			return 0;		/* No EOL found */
		
		if (*p == s->eol[0]) {	
			/* Hm, EOL.. make sure we have all of it */
			cp = &s->eol[0];
			cpp = p;
			n = i;
			while ((n < s->ibuf.pos) && (*cp)) {
				if (*cp != *cpp)
					return 0;	/* Damn... missed */
				n++;
				cp++;
				cpp++;
			}
			if (*cp)
				return 0;		/* Ran out of data */
		}
		
		/* Feed line to handler */
		*p = '\0';
		if ((*s->in_handler)(s, s->ibuf.buf, i) == -2)
			return -2;
		
		/* Eat the rest of the EOL */
		while ((i < s->ibuf.pos) && ((*p == '\n') || (*p == '\r')
		   || (*p == '\0'))) {
		  	p++;
		  	i++;
		}
		
		if (i == s->ibuf.pos) {
			s->ibuf.pos = 0;	/* Out of data */
		} else {
			memmove(s->ibuf.buf, p, s->ibuf.pos - i);
			s->ibuf.pos -= i;
			return sock_handle(s);	/* Recursively call myself...
						   there might be another line
						   in the buffer */
		}
	} else {
		i = (*s->in_handler)(s, s->ibuf.buf, s->ibuf.pos);
		if (i == 0)
			s->ibuf.pos = 0;
		else
			return i;
	}
	
	return 0;
}

/************************************************************************
 ****  Socket output  ***************************************************
 ************************************************************************/
 
/*
 *	Flush the raw output of a socket
 */

int csrflush(struct csock_t *s)
{
	int len;
	
	if (s->robuf->pos == 0)
		return 0;
	
	len = write(s->fd, s->robuf->buf, s->robuf->pos);
	
	if (len == -1) {
		if (errno == EAGAIN) {
			log(L_SDEBUG, "fd %d: csrflush(): Would block, selecting on write", s->fd);
			FD_SET(s->fd, &writefds);
		} else {
			s->cs_errno = errno;
			log(L_SDEBUG, "fd %d: csrflush(): write() error: %s", 
				s->fd, strerror(s->cs_errno));
			if (errno == EINTR)
				return 0;
		}
		return -1;
	}
	
	log(L_SDEBUG, "fd %d: csrflush(): Wrote %d bytes out of %d, %d left", 
		s->fd, len, s->robuf->pos, s->robuf->pos - len);
	
	if (len == s->robuf->pos) {
		s->robuf->pos = 0;
		FD_CLR(s->fd, &writefds);
		return 0;
	}
	
	memmove(s->robuf->buf, s->robuf->buf + len, s->robuf->pos - len);
	s->robuf->pos -= len;
	
	return 0;
}

/*
 *	Flush the output of a socket
 */

int csflush(struct csock_t *s)
{
	if ((s->comp) && (s->obuf.pos))
		if (sock_deflate(s, CS_FLUSH) == -1)
			return -1;
		
	return csrflush(s);
}

/*
 *      Flush all sockets
 */

int csflush_all(void)
{
	int i;
	struct csock_t *s;
	
	for (s = sockets; (s); s = s->next) {
		csflush(s);
		i++;
	}
	
	return i;
}

/*
 *	Write a *raw* character in the output buffer, without escaping,
 *	handle buffer sizing and flushing
 */

int csrwritec(struct csock_t *s, char c)
{
	grow_buf(&s->obuf, 1);
	
	s->obuf.buf[s->obuf.pos] = c;
	s->obuf.pos++;
	s->out_count++;
	
	if ((s->comp) && (s->obuf.pos >= s->obuf.tresh))
		sock_deflate(s, CS_NOFLUSH);
	
	if (s->robuf->pos >= s->robuf->tresh)
		csrflush(s);
	
	return 1;
}

/*
 *	Write a (raw) character to a socket, do telnet escaping et al
 */

int cswritec(struct csock_t *s, char c)
{
	if ((s->af_type == AF_INET) && (c == (char)IAC))
		csrwritec(s, IAC);
		
	csrwritec(s, c);
	
	return 1;
}

/*
 *	Write a block of (raw) data to a socket, with telnet escaping
 */

int cswrite(struct csock_t *s, void *buf, int n)
{
	char *p = buf;
	int i;
	
	for (i = 0; i < n; i++) {
		cswritec(s, *p);
		p++;
	}
	
	return n;
}

/*
 *	Write a character to a socket, with EOL conversion
 */

int csputc(struct csock_t *s, char c)
{
	char *p;
	int i = 1;
	
	if ((s->eolmode == eol_text) && (c == '\n'))
		for (p = s->eol; (*p); p++) {
			cswritec(s, *p);
			i++;
		}
	else
		cswritec(s, c);
	
	return i;
}

/*
 *	Write a string to a socket, with EOL conversion
 */

int csputs(struct csock_t *s, char *str)
{
	char *p;
	int i = 0;
	
	for (p = str; (*p); p++)
		i += csputc(s, *p);
	
	return i;
}

/*
 *	Write formatted text to a socket
 */

int csprintf(struct csock_t *s, const char *fmt, ...)
{
	va_list args;
	char str[MAX_SENDLEN+1];
	
	va_start(args, fmt);
	vsnprintf(str, MAX_SENDLEN, fmt, args);
	va_end(args);
	
	str[MAX_SENDLEN] = '\0';
	
	return csputs(s, str);
}

/************************************************************************
 ****  Misc  ************************************************************
 ************************************************************************/

int strtoaf(char *s)
{
	if (!strcasecmp(s, "ax25"))
		return AF_AX25;
	
	if (!strcasecmp(s, "netrom"))
		return AF_NETROM;
	
	if (!strcasecmp(s, "telnet"))
		return AF_INET;
	
	return -1;
}

/************************************************************************
 ****  Connection, disconnection  ***************************************
 ************************************************************************/

/*
 *	Dispatch an incoming connection to a higher layer
 */

int sock_login(struct csock_t *s)
{
	s->type = cst_incoming;
	s->state = css_connected;
	
	log(L_SINFO, "Socket: Incoming %s: %s%s%s:%s (%d)",
		afstr(s->af_type), (s->call) ? s->call : "", ((s->node) && (s->call)) ? ":" : "",
		(s->node) ? s->node : "", s->port, s->fd);
	
	if (s->af_type == AF_INET)
		return auth_login(s);
	else
		return luser_login(s);
}

/*
 *	Disconnect a socket
 */

int sock_disconnect(struct csock_t *s)
{
	s->state = css_disconnecting;
	sock_close(s);
	return -2;
}

/************************************************************************
 ************************************************************************/

/*
 *	Open up all listening sockets
 */

void csock_init(void)
{
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	
	set_timer(0, 1, (void *)&csflush_all, NULL);
#ifdef HAVE_INET
	tcp_listens();
#endif
#ifdef HAVE_AX25
	ax25_listens();
#endif
#ifdef HAVE_NETROM
	netrom_listens();
#endif
}

/*
 *	The main select() loop
 */
 
void select_loop(void)
{
	int i, r;
	struct csock_t *s, *nexts;
	struct listenq_t *lq;
	fd_set readchk, writechk;
	
	while (1) {
		readchk = readfds;
		writechk = writefds;
		
		if ((i = select(maxfd + 1, &readchk, &writechk, NULL, NULL)) <= 0) {
			if (i < 0) {
				if ((errno == EINTR) && (timer_pending)) {
					timer_event();
					continue;
				} else
					log(L_ERR, "select() returned %d: %s",  i, strerror(errno));
			} else
				log(L_ERR, "select() timed out.");
			
			continue;
		}
		
		log(L_SDEBUG, "select() returned %d", i);
		
		for (lq = listenq; (lq); lq = lq->next) {
			if (FD_ISSET(lq->fd, &readchk)) {
				(*lq->accept)(lq);
				if (i == 1)
					continue;
			}
		}
		
		for (s = sockets; (s); s = nexts) {
			nexts = s->next;
			if (s == nexts) {
				log(L_CRIT, "OUCH. select_loop(): s->next == s (%s:%s/%s) - deadlock !!!", s->call, s->node, s->port);
				exit(1);
			}
			if (FD_ISSET(s->fd, &readchk)) {
				if ((r = sock_read(s)) < 0) {
					sock_close(s);
					continue;
				} else {
					if (s->state == css_connecting)
						sock_connected(s);
					if (sock_handle(s) == -2)
						continue;
				}
			}
			
			if (FD_ISSET(s->fd, &writechk))
				csflush(s);
		}
		
		timer_event();
	}
}

