/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.cql3;

import org.junit.Test;

import org.apache.cassandra.utils.Pair;
import org.assertj.core.api.Assertions;

public class CQLTesterTest extends CQLTester
{
    @Test
    public void testGetCreateIndexName()
    {
        schemaChange("CREATE KEYSPACE k WITH replication = {'class': 'SimpleStrategy', 'replication_factor': '1'}");
        schemaChange("CREATE KEYSPACE k2 WITH replication = {'class': 'SimpleStrategy', 'replication_factor': '1'}");
        String table = createTable("CREATE TABLE %s (k int PRIMARY KEY, c int)");

        // standard index with explicit name
        testGetCreateIndexName("k", "idx", "CREATE INDEX idx ON k.t (c)");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE INDEX idx ON t (c)");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE INDEX idx ON %s (c)");

        // custom index with explicit name
        testGetCreateIndexName("k", "idx", "CREATE CUSTOM INDEX idx ON k.t (c) USING 'class'");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE CUSTOM INDEX idx ON t (c) USING 'class'");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE CUSTOM INDEX idx ON %s (c) USING 'class'");

        // with an unqouted case-sensitive index name
        testGetCreateIndexName("k", "idx", "CREATE INDEX iDx ON k.t (c)");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE INDEX iDx ON t (c)");
        testGetCreateIndexName(KEYSPACE, "idx", "CREATE INDEX iDx ON %s (c)");

        // with a quoted case-sensitive index name
        testGetCreateIndexName("k", "iDx", "CREATE INDEX \"iDx\" ON k.t (c)");
        testGetCreateIndexName(KEYSPACE, "iDx", "CREATE INDEX \"iDx\" ON t (c)");
        testGetCreateIndexName(KEYSPACE, "iDx", "CREATE INDEX \"iDx\" ON %s (c)");

        // standard index with autogenerated name
        testGetCreateIndexName("k", "t_c_idx", "CREATE INDEX ON k.t (c)");
        testGetCreateIndexName(KEYSPACE, "t_c_idx", "CREATE INDEX ON t (c)");
        testGetCreateIndexName(KEYSPACE, table + "_c_idx", "CREATE INDEX ON %s (c)");

        // custom index with autogenerated name
        testGetCreateIndexName("k", "t_c_idx", "CREATE CUSTOM INDEX ON k.t (c) USING 'class'");
        testGetCreateIndexName(KEYSPACE, "t_c_idx", "CREATE CUSTOM INDEX ON t (c) USING 'class'");
        testGetCreateIndexName(KEYSPACE, table + "_c_idx", "CREATE CUSTOM INDEX ON %s (c) USING 'class'");

        // with a not-default working keyspace
        testGetCreateIndexName("k", "t_c_idx", "k2", "CREATE INDEX ON k.t (c)");
        testGetCreateIndexName("k2", "t_c_idx", "k2", "CREATE INDEX ON t (c)");
        testGetCreateIndexName("k2", table + "_c_idx", "k2", "CREATE INDEX ON %s (c)");
    }

    private void testGetCreateIndexName(String expectedKeyspaceName, String expectedIndexName, String query)
    {
        testGetCreateIndexName(expectedKeyspaceName, expectedIndexName, KEYSPACE, query);
    }

    private void testGetCreateIndexName(String expectedKeyspaceName, String expectedIndexName, String keyspace, String query)
    {
        Pair<String, String> name = getCreateIndexName(keyspace, formatQuery(keyspace, query));
        Assertions.assertThat(name.left).isEqualTo(expectedKeyspaceName);
        Assertions.assertThat(name.right).isEqualTo(expectedIndexName);
    }
}
