/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.webadmin.service;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAmount;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import javax.inject.Inject;
import org.apache.james.core.Username;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageManager;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.MailboxNotFoundException;
import org.apache.james.mailbox.model.MailboxPath;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.task.Task;
import org.apache.james.user.api.UsersRepository;
import org.apache.james.user.api.UsersRepositoryException;
import org.apache.james.util.DurationParser;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class ExpireMailboxService {
    private static final Logger LOGGER = LoggerFactory.getLogger(ExpireMailboxService.class);
    private final UsersRepository usersRepository;
    private final MailboxManager mailboxManager;

    @Inject
    public ExpireMailboxService(UsersRepository usersRepository, MailboxManager mailboxManager) {
        this.usersRepository = usersRepository;
        this.mailboxManager = mailboxManager;
    }

    public Mono<Task.Result> expireMailboxes(Context context, RunningOptions runningOptions, Date now) {
        SearchQuery expiration = SearchQuery.of((SearchQuery.Criterion[])new SearchQuery.Criterion[]{runningOptions.maxAgeDuration.map(maxAge -> {
            Date limit = Date.from(now.toInstant().minus((TemporalAmount)maxAge));
            return SearchQuery.internalDateBefore((Date)limit, (SearchQuery.DateResolution)SearchQuery.DateResolution.Second);
        }).orElse(SearchQuery.headerDateBefore((String)"Expires", (Date)now, (SearchQuery.DateResolution)SearchQuery.DateResolution.Second))});
        return Flux.from((Publisher)this.usersRepository.listReactive()).transform(ReactorUtils.throttle().elements(runningOptions.getUsersPerSecond()).per(Duration.ofSeconds(1L)).forOperation(username -> this.expireUserMailbox(context, (Username)username, runningOptions.getMailbox(), expiration))).reduce((Object)Task.Result.COMPLETED, Task::combine).onErrorResume(UsersRepositoryException.class, e -> {
            LOGGER.error("Error while accessing users from repository", (Throwable)e);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    private Mono<Task.Result> expireUserMailbox(Context context, Username username, String mailbox, SearchQuery expiration) {
        MailboxSession session = this.mailboxManager.createSystemSession(username);
        MailboxPath mailboxPath = MailboxPath.forUser((Username)username, (String)mailbox);
        return Mono.from((Publisher)this.mailboxManager.getMailboxReactive(mailboxPath, session)).onErrorResume(MailboxNotFoundException.class, ignore -> Mono.empty()).flatMap(mgr -> this.searchMessagesReactive((MessageManager)mgr, session, expiration).flatMap(list -> this.deleteMessagesReactive((MessageManager)mgr, session, (List<MessageUid>)list))).doOnNext(expired -> {
            if (expired > 0) {
                context.incrementExpiredCount();
                context.incrementMessagesDeleted(expired.intValue());
            }
            context.incrementProcessedCount();
        }).then(Mono.just((Object)Task.Result.COMPLETED)).onErrorResume(e -> {
            LOGGER.warn("Failed to expire user mailbox {}", (Object)username, e);
            context.incrementFailedCount();
            context.incrementProcessedCount();
            return Mono.just((Object)Task.Result.PARTIAL);
        }).doFinally(any -> this.mailboxManager.endProcessingRequest(session));
    }

    private Mono<List<MessageUid>> searchMessagesReactive(MessageManager mgr, MailboxSession session, SearchQuery expiration) {
        try {
            return Flux.from((Publisher)mgr.search(expiration, session)).collectList();
        }
        catch (MailboxException e) {
            return Mono.error((Throwable)e);
        }
    }

    private Mono<Integer> deleteMessagesReactive(MessageManager mgr, MailboxSession session, List<MessageUid> uids) {
        if (uids.isEmpty()) {
            return Mono.just((Object)0);
        }
        return Mono.from((Publisher)mgr.deleteReactive(uids, session)).thenReturn((Object)uids.size());
    }

    public static class Context {
        private final AtomicLong inboxesExpired = new AtomicLong(0L);
        private final AtomicLong inboxesFailed = new AtomicLong(0L);
        private final AtomicLong inboxesProcessed = new AtomicLong(0L);
        private final AtomicLong messagesDeleted = new AtomicLong(0L);

        public long getInboxesExpired() {
            return this.inboxesExpired.get();
        }

        public long getInboxesFailed() {
            return this.inboxesFailed.get();
        }

        public long getInboxesProcessed() {
            return this.inboxesProcessed.get();
        }

        public long getMessagesDeleted() {
            return this.messagesDeleted.get();
        }

        public void incrementExpiredCount() {
            this.inboxesExpired.incrementAndGet();
        }

        public void incrementFailedCount() {
            this.inboxesFailed.incrementAndGet();
        }

        public void incrementProcessedCount() {
            this.inboxesProcessed.incrementAndGet();
        }

        public void incrementMessagesDeleted(long count) {
            this.messagesDeleted.addAndGet(count);
        }
    }

    public static class RunningOptions {
        public static final RunningOptions DEFAULT = new RunningOptions(1, "INBOX", true, Optional.empty());
        private final int usersPerSecond;
        private final String mailbox;
        private final boolean byExpiresHeader;
        private final Optional<String> olderThan;
        @JsonIgnore
        private final Optional<Duration> maxAgeDuration;

        public static RunningOptions fromParams(Optional<String> byExpiresHeader, Optional<String> olderThan, Optional<String> usersPerSecond, Optional<String> mailbox) {
            try {
                if (byExpiresHeader.isPresent() == olderThan.isPresent()) {
                    throw new IllegalArgumentException("Must specify either 'olderThan' or 'byExpiresHeader' parameter");
                }
                return new RunningOptions(usersPerSecond.map(Integer::parseInt).orElse(DEFAULT.getUsersPerSecond()), mailbox.orElse(DEFAULT.getMailbox()), byExpiresHeader.isPresent(), olderThan);
            }
            catch (NumberFormatException ex) {
                throw new IllegalArgumentException("'usersPerSecond' must be numeric");
            }
        }

        @JsonCreator
        public RunningOptions(@JsonProperty(value="usersPerSecond") int usersPerSecond, @JsonProperty(value="mailbox") String mailbox, @JsonProperty(value="byExpiresHeader") boolean byExpiresHeader, @JsonProperty(value="olderThan") Optional<String> olderThan) {
            Preconditions.checkArgument((usersPerSecond > 0 ? 1 : 0) != 0, (Object)"'usersPerSecond' must be strictly positive");
            this.usersPerSecond = usersPerSecond;
            this.mailbox = mailbox;
            this.byExpiresHeader = byExpiresHeader;
            this.olderThan = olderThan;
            this.maxAgeDuration = olderThan.map(v -> DurationParser.parse((String)((String)olderThan.get()), (ChronoUnit)ChronoUnit.DAYS));
        }

        public int getUsersPerSecond() {
            return this.usersPerSecond;
        }

        public String getMailbox() {
            return this.mailbox;
        }

        public boolean getByExpiresHeader() {
            return this.byExpiresHeader;
        }

        public Optional<String> getOlderThan() {
            return this.olderThan;
        }
    }
}

