#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Conversion functions
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
char -> card
=============================================
|#

(deff char2card (char)
 (char-code char))

#|
=============================================
card -> char
=============================================
|#

(deff card2char (char)
 (code-char char))

#|
=============================================
int -> card
=============================================
|#

(deff int2card (int)
 (:when (<= int 0) (* int -2))
 (+ int int -1))

(etst (int2card  0) 0)
(etst (int2card  1) 1)
(etst (int2card -1) 2)
(etst (int2card  2) 3)
(etst (int2card -2) 4)
(etst (int2card  3) 5)
(etst (int2card -3) 6)

#|
=============================================
card -> int
=============================================
|#

(deff card2int (card)
 (:when (oddp card) (ceiling card 2))
 (- (floor card 2)))

(etst (card2int 0)  0)
(etst (card2int 1)  1)
(etst (card2int 2) -1)
(etst (card2int 3)  2)
(etst (card2int 4) -2)
(etst (card2int 5)  3)
(etst (card2int 6) -3)

#|
=============================================
string -> card*
=============================================
|#

(deff string2card* (string)
 (map 'list 'char2card (coerce string 'list)))

(etst (string2card* "abc") '(97 98 99))

#|
=============================================
card* -> string
=============================================
|#

(deff card*2string (char*)
 (map 'string 'card2char char*))

(etst (card*2string '(97 98 99)) "abc")

#|
=============================================
card* -> card
=============================================
|#

(deff card*2card (radix card*)
 (:when (null card*) 0)
 (+ (car card*) (* radix (card*2card radix (cdr card*)))))

(etst (card*2card 10 '(1 2 3 4 5)) 54321)

#|
=============================================
card -> card*
=============================================
|#

(deff card2card* (radix card)
 (:when (<debug card radix) (list card))
 (cons (mod card radix) (card2card* radix (floor card radix))))

(etst (card2card* 10 54321) '(1 2 3 4 5))

#|
=============================================
Vectors
=============================================
|#

(deff null-vector (length)
 (make-array
  (list length)
  :element-type '(unsigned-byte 8)
  :initial-element 0))

(deff make-vector (length)
 (make-array
  (list length)
  :element-type '(unsigned-byte 8)))

(etst (null-vector 3) #(0 0 0))
(etst (length (make-vector 3)) 3)

#|
=============================================
Convert card* to vector
=============================================
|#

(deff card*2vector (card*)
 (make-array
  (list (length card*))
  :element-type '(unsigned-byte 8)
  :initial-contents card*))

#|
=============================================
Convert vector to card*
=============================================
|#

(deff vector2card* (vector)
 (coerce vector 'list))

(etst (vector2card* (card*2vector '(0 1 254 255))) '(0 1 254 255))

#|
=============================================
Convert vector to vector*
=============================================
|#

(defc vector-array (make-array 256))
(dotimes (n 256) (setf (aref vector-array n) (card*2vector (list n))))

(deff vector-fct (n)
 (aref vector-array n))

(deff vector2vector* (vector)
 (map 'list 'vector-fct vector))

(etst (vector2vector* (card*2vector '(97 98 99))) '(#(97) #(98) #(99)))

#|
=============================================
Convert card* to vector*
=============================================
|#

(deff card*2vector* (card*)
 (map 'list 'vector-fct card*))

(etst (card*2vector* '(97 98 99)) '(#(97) #(98) #(99)))

#|
=============================================
Convert string to vector
=============================================
|#

(deff string2vector (string)
 (:let length (length string))
 (:let vector (make-vector length))
 (dotimes (i length) (setf (aref vector i) (char-code (aref string i))))
 vector)

(etst (string2vector "abc") #(97 98 99))

#|
=============================================
Convert vector to string
=============================================
|#

(deff vector2string (vector)
 (:let length (length vector))
 (:let string (make-string length))
 (dotimes (i length) (setf (aref string i) (code-char (aref vector i))))
 string)

(etst (vector2string #(97 98 99)) "abc")

#|
=============================================
Subvectors
=============================================
|#

(deff subvector (vector start &optional (end (length vector)))
 (:unless (<= 0 start end (length vector))
  (error "subvector indices out of range"))
 (make-array
  (list (- end start))
  :element-type '(unsigned-byte 8)
  :displaced-to vector
  :displaced-index-offset start))

(deff safe-subvector (vector start &optional (end))
 (:let end (default end (length vector)))
 (subvector vector (max start 0) (min end (length vector))))

(defc test-vector1 (card*2vector '(0 1 2 3 4 5 6 7 8)))
(etst (vector2card* (subvector test-vector1 2 7)) '(2 3 4 5 6))
(etst (vector2card* (subvector test-vector1 0 9)) '(0 1 2 3 4 5 6 7 8))
(etst (vector2card* (subvector test-vector1 0 0)) nil)
(etst (vector2card* (subvector test-vector1 5 5)) nil)
(etst (vector2card* (subvector test-vector1 9 9)) nil)

(defc test-vector2 (subvector test-vector1 2 8))
(etst (vector2card* test-vector2) '(2 3 4 5 6 7))
(etst (vector2card* (subvector test-vector2 2 4)) '(4 5))
(etst (vector2card* (subvector test-vector2 0 6)) '(2 3 4 5 6 7))
(etst (vector2card* (subvector test-vector2 0 0)) nil)
(etst (vector2card* (subvector test-vector2 2 2)) nil)
(etst (vector2card* (subvector test-vector2 6 6)) nil)

(setf (aref (subvector test-vector2 2 4) 1) 10)
(etst (vector2card* test-vector1) '(0 1 2 3 4 10 6 7 8))

#|
=============================================
Convert string to card
=============================================
On several occasions, cardinals are used to represent a sequence of bytes. In those cases, the byte sequence is extended with a one-byte and converted to a cardinal using little-endian base 256. In the following, bytes are called 'oktets' to avoid name conflicts with Lisp standard functions.
|#

(deff oktet (i)
 (byte 8 (* 8 i)))

(deff oktet-length (card)
 (:let length (1- (integer-length card)))
 (max 0 (floor length 8)))

(deff make-card (i)
 (ash 1 (* i 8)))

(deff string2card (string)
 (:when (integerp string) string)
 (:let length (length string))
 (:let card (make-card length))
 (dotimes (i length) (setf (ldb (oktet i) card) (char-code (aref string i))))
 card)

(etst (string2card "abc") #x01636261)

#|
=============================================
Convert vector to card
=============================================
|#

(deff vector2card (vector)
 (:when (integerp vector) vector)
 (:let length (length vector))
 (:let card (make-card length))
 (dotimes (i length) (setf (ldb (oktet i) card) (aref vector i)))
 card)

(etst (vector2card #(#x61 #x62 #x63)) #x01636261)

#|
=============================================
Convert card to string
=============================================
|#

(deff card2string (card)
 (:let length (oktet-length card))
 (:let string (make-string length))
 (dotimes (i length) (setf (aref string i) (code-char (ldb (oktet i) card))))
 string)

(etst (card2string #x01636261) "abc")

#|
=============================================
Convert card to vector
=============================================
|#

(deff card2vector (card)
 (:when (arrayp card) card)
 (:let length (oktet-length card))
 (:let vector (make-vector length))
 (dotimes (i length) (setf (aref vector i) (ldb (oktet i) card)))
 vector)

(etst (card2vector #x01636261) #(#x61 #x62 #x63))

#|
=============================================
Convert ct to card*
=============================================
|#

(deff ct2card* (ct)
 (ct2card*1 ct nil))

(deff ct2card*1 (ct result)
 (:when (null ct) result)
 (:when (consp ct) (ct2card*1 (car ct) (ct2card*1 (cdr ct) result)))
 (:when (integerp ct) (cons ct result))
 (:when (characterp ct) (cons (char2card ct) result))
 (:when (stringp ct) (append (string2card* ct) result))
 (:when (arrayp ct) (append (vector2card* ct) result))
 (error "Malformed ct: ~s" ct))

(etst (ct2card* '(("ab" 99) (#\d))) '(97 98 99 100))

#|
=============================================
Convert ct to length
=============================================
|#

(deff ct2length (ct)
 (ct2length1 ct 0))

(deff ct2length1 (ct result)
 (:when (null ct) result)
 (:when (consp ct) (ct2length1 (cdr ct) (ct2length1 (car ct) result)))
 (:when (consp ct) (+ (ct2length (car ct)) (ct2length (cdr ct))))
 (:when (integerp ct) (+ 1 result))
 (:when (characterp ct) (+ 1 result))
 (:when (stringp ct) (+ (length ct) result))
 (:when (arrayp ct) (+ (length ct) result))
 (error "Malformed ct: ~s" ct))

(etst (ct2length '(("ab" 99) (#\d))) 4)

#|
=============================================
Convert ct to vector
=============================================
(vecp x) is true if x is a vector (the name 'vectorp' is reserved by the Common Lisp standard). vecp is never used. Instead, x is considered to be a vector if it is an array but not a string.

(ct2vector ct) converts the given ct to a vector. It is 'cpu-time-safe' in the sense that if the ct is already a vector, it is returned unchanged. Furthermore, if the ct is a string then the ct is converted using string2vector.
|#

(deff vecp (x)
 (typep x '(simple-array (unsigned-byte 8))))

(deff ct2vector (ct)
 (:when (stringp ct) (string2vector ct))
 (:when (arrayp ct) ct)
 (:let length (ct2length ct))
 (:let vector (make-vector length))
 (:unless (equalp length (ct2vector1 vector ct 0)) (error "Internal error"))
 vector)

(deff ct2vector1 (vec ct i)
 (:when (null ct) i)
 (:when (consp ct) (ct2vector1 vec (cdr ct) (ct2vector1 vec (car ct) i)))
 (:when (integerp ct) (setf (aref vec i) ct) (+ i 1))
 (:when (characterp ct) (setf (aref vec i) (char-code ct)) (+ i 1))
 (:when (stringp ct)
  (:let length (length ct))
  (dotimes (j length) (setf (aref vec (+ i j)) (char-code (aref ct j))))
  (+ i length))
 (:when (arrayp ct)
  (:let length (length ct))
  (dotimes (j length) (setf (aref vec (+ i j)) (aref ct j)))
  (+ i length))
 (error "Malformed ct: ~s" ct))

(etst (ct2vector '(("ab" 99) (#\d) #(101 102))) #(97 98 99 100 101 102))

(deff vec (&rest ct)
 (ct2vector ct))

(etst (vec "a" "bc") #(97 98 99))

#|
=============================================
Convert ct to string
=============================================
|#

(deff ct2string (ct)
 (:let length (ct2length ct))
 (:let string (make-string length))
 (:unless (equalp length (ct2string1 string ct 0)) (error "Internal error"))
 string)

(deff ct2string1 (str ct i)
 (:when (null ct) i)
 (:when (consp ct) (ct2string1 str (cdr ct) (ct2string1 str (car ct) i)))
 (:when (integerp ct) (setf (aref str i) (code-char ct)) (+ i 1))
 (:when (characterp ct) (setf (aref str i) ct) (+ i 1))
 (:when (stringp ct)
  (:let length (length ct))
  (dotimes (j length) (setf (aref str (+ i j)) (aref ct j)))
  (+ i length))
 (:when (arrayp ct)
  (:let length (length ct))
  (dotimes (j length) (setf (aref str (+ i j)) (code-char (aref ct j))))
  (+ i length))
 (error "Malformed ct: ~s" ct))

(etst (ct2string '(("ab" 99) (#\d) #(101 102))) "abcdef")

#|
(deff ct2string (ct)
 (apply 'concatenate 'string (ct2string1 ct nil)))

(deff ct2string1 (ct result)
 (:when (null ct) result)
 (:when (consp ct) (ct2string1 (car ct) (ct2string1 (cdr ct) result)))
 (:when (integerp ct) (cons (card*2string (list ct)) result))
 (:when (characterp ct) (cons (coerce (list ct) 'string) result))
 (:when (stringp ct) (cons ct result))
 (:when (arrayp ct) (cons (vector2string ct) result))
 (error "Malformed ct: ~s" ct))
|#

#|
=============================================
Convert ct to ct
=============================================
|#

(deff ct2ct (ct)
 (:when (null ct) ct)
 (:when (consp ct) (cons (ct2ct (car ct)) (ct2ct (cdr ct))))
 (:when (characterp ct) (char2card ct))
 (:when (stringp ct) (string2card* ct))
 ct)

(etst (ct2ct '(("ab" 99) (#\d) :no-item)) '(((97 98) 99) (100) :no-item))

#|
=============================================
Print ct
=============================================
|#

(deff print-ct (ct)
 (print-ct1 ct 0))

(deff print-ct1 (ct column)
 (:when (> column 60) (terpri) (print-ct1 ct 0))
 (:when (null ct) column)
 (:when (consp ct) (print-ct1 (cdr ct) (print-ct1 (car ct) column)))
 (:when (characterp ct) (format t "~a" ct) (+ column 1))
 (:when (stringp ct) (format t "~a" ct) (+ column (length ct)))
 (:when (integerp ct) (format t "~a" (card2char ct)) (+ column 1))
 ct)

#|
=============================================
Convert reference to cardinal
=============================================
|#

(deff ref2card (ref)
 (:when (null ref) 1)
 (:let (card . ref) ref)
 (+ card (ash (ref2card ref) 8)))

(etst (ref2card '(2 3)) #x010302)

#|
=============================================
Convert cardinal to reference
=============================================
|#

(deff card2ref (card)
 (:when (<debug card 256) nil)
 (cons (logand card 255) (card2ref (ash card -8))))

(etst (card2ref #x010302) '(2 3))

#|
=============================================
Convert cardinal to base32 and 64
=============================================
|#

(defc *base32*
 "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567")
(defc *base64*
 "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_")

(etst (length *base32*) 32)
(etst (length *base64*) 64)

(defc *char-2-base32* (make-array 256))
(defc *base32-2-char* (make-array  32))
(defc *char-2-base64* (make-array 256))
(defc *base64-2-char* (make-array  64))

(deff init-base32 (card)
 (:let char (char2card (aref *base32* card)))
 (setf (aref *char-2-base32* char) card)
 (setf (aref *base32-2-char* card) char))

(deff init-base64 (card)
 (:let char (char2card (aref *base64* card)))
 (setf (aref *char-2-base64* char) card)
 (setf (aref *base64-2-char* card) char))

(dotimes (i 32) (init-base32 i))
(dotimes (i 64) (init-base64 i))

(deff card-2-base32 (card)
 (:when (<= card 0) nil)
 (cons (aref *base32-2-char* (mod card 32)) (card-2-base32 (floor card 32))))

(deff base32-2-card (char*)
 (:when (atom char*) 0)
 (:let (char . char*) char*)
 (:let card (aref *char-2-base32* char))
 (:when (null card) (raise))
 (+ card (* 32 (base32-2-card char*))))

(deff card-2-base64 (card)
 (:when (<= card 0) nil)
 (cons (aref *base64-2-char* (mod card 64)) (card-2-base64 (floor card 64))))

(deff base64-2-card (char*)
 (:when (atom char*) 0)
 (:let (char . char*) char*)
 (:let card (aref *char-2-base64* char))
 (:when (null card) (raise))
 (+ card (* 64 (base64-2-card char*))))

(etst (card-2-base32 32) (ct2ct "AB"))
(etst (base32-2-card (ct2ct "AB")) 32)
(etst (card-2-base64 64) (ct2ct "AB"))
(etst (base64-2-card (ct2ct "AB")) 64)

#|
=============================================
Convert card* to safe*
=============================================
(cons-safe card card*) converts a card in [0;63] to a 'safe' character (base64) and conses it on card*.

(card*2safe* card*) converts the given card* to a sequence of 'safe' characters. The output is a card* of length about 4/3 of the length of the input.

(card*2safe*-n where n is 0, 2, 4, or 6 all do a card*2safe* but have 0, 2, 4, or 6 bits left over from the previous card.
|#

(deff cons-safe (card card*)
 (cons (aref *base64-2-char* card) card*))

(deff cons-safe-three (card012 card*)
 (:let card (card*2card 256 card012))
 (:mlet (card card0) (floor card 64))
 (:mlet (card card1) (floor card 64))
 (:mlet (card3 card2) (floor card 64))
 (cons-safe card3 (cons-safe card2 (cons-safe card1 (cons-safe card0 card*)))))

(etst (cons-safe 1 2) '(66 . 2))

#|
(deff card*2safe* (card*)
 (card*2safe*-0 card* nil))
|#

(deff card*2safe* (card*)
 (card*2safe*-optimized card* nil))

(deff card*2safe*-optimized (card* result)
 (:when (null (cddr card*)) (card*2safe*-0 card* result))
 (:let result (cons-safe-three (subseq card* 0 3) result))
 (card*2safe*-optimized (cdddr card*) result))

(deff card*2safe*-0 (card* result)
 (:when (atom card*) (reverse result))
 (:let (card . card*) card*)
 (:mlet (bits safe) (floor card 64))
 (card*2safe*-2 card* bits (cons-safe safe result)))

(deff card*2safe*-2 (card* bits result)
 (:when (atom card*) (reverse (cons-safe bits result)))
 (:let (card . card*) card*)
 (:mlet (bits safe) (floor (+ bits (* card 4)) 64))
 (card*2safe*-4 card* bits (cons-safe safe result)))

(deff card*2safe*-4 (card* bits result)
 (:when (atom card*) (reverse (cons-safe bits result)))
 (:let (card . card*) card*)
 (:mlet (bits safe) (floor (+ bits (* card 16)) 64))
 (card*2safe*-6 card* bits (cons-safe safe result)))

(deff card*2safe*-6 (card* bits result)
 (card*2safe*-0 card* (cons-safe bits result)))

(ntst (card*2safe* nil))
(etst (card*2safe* '(  1))	 (ct2ct "BA"))
(etst (card*2safe* '(  2))	 (ct2ct "CA"))
(etst (card*2safe* '(  4))	 (ct2ct "EA"))
(etst (card*2safe* '(  8))	 (ct2ct "IA"))
(etst (card*2safe* '( 16))	 (ct2ct "QA"))
(etst (card*2safe* '( 32))	 (ct2ct "gA"))
(etst (card*2safe* '( 64))	 (ct2ct "AB"))
(etst (card*2safe* '(128))	 (ct2ct "AC"))
(etst (card*2safe* '(0   1))	 (ct2ct "AEA"))
(etst (card*2safe* '(0   2))	 (ct2ct "AIA"))
(etst (card*2safe* '(0   4))	 (ct2ct "AQA"))
(etst (card*2safe* '(0   8))	 (ct2ct "AgA"))
(etst (card*2safe* '(0  16))	 (ct2ct "AAB"))
(etst (card*2safe* '(0  32))	 (ct2ct "AAC"))
(etst (card*2safe* '(0  64))	 (ct2ct "AAE"))
(etst (card*2safe* '(0 128))	 (ct2ct "AAI"))
(etst (card*2safe* '(0 0   1))   (ct2ct "AAQA"))
(etst (card*2safe* '(0 0   2))   (ct2ct "AAgA"))
(etst (card*2safe* '(0 0   4))   (ct2ct "AAAB"))
(etst (card*2safe* '(0 0   8))   (ct2ct "AAAC"))
(etst (card*2safe* '(0 0  16))   (ct2ct "AAAE"))
(etst (card*2safe* '(0 0  32))   (ct2ct "AAAI"))
(etst (card*2safe* '(0 0  64))   (ct2ct "AAAQ"))
(etst (card*2safe* '(0 0 128))   (ct2ct "AAAg"))
(etst (card*2safe* '(0 0 0   1)) (ct2ct "AAAABA"))
(etst (card*2safe* '(0 0 0   2)) (ct2ct "AAAACA"))
(etst (card*2safe* '(0 0 0   4)) (ct2ct "AAAAEA"))
(etst (card*2safe* '(0 0 0   8)) (ct2ct "AAAAIA"))
(etst (card*2safe* '(0 0 0  16)) (ct2ct "AAAAQA"))
(etst (card*2safe* '(0 0 0  32)) (ct2ct "AAAAgA"))
(etst (card*2safe* '(0 0 0  64)) (ct2ct "AAAAAB"))
(etst (card*2safe* '(0 0 0 128)) (ct2ct "AAAAAC"))

#|
=============================================
Convert safe* to card*
=============================================
(cons-base64 card card*) converts a card in [0;63] to base64 and conses it on card*.

(safe*2card* safe*) converts the given card* in base64 encoding to a list of bytes. The output is a card* of length about 3/4 of the length of the input.

(safe*2card*-n where n is 0, 2, 4, or 6 all do a safe*2card* but have 0, 2, 4, or 6 bits left over from the previous card.

(revappend-safe* safe* result) does (revappend (safe*2card* safe*) result) except that revappend-safe* saves two list reversals.
|#

(deff uncons-safe (safe*)
 (:when (atom safe*) nil)
 (:let (safe . safe*) safe*)
 (:let card (aref *char-2-base64* safe))
 (:when (null card) (uncons-safe safe*))
 (cons card safe*))

#|
(deff safe*2card* (safe*)
 (reverse (safe*2card*-0 safe* nil)))
|#

(deff safe-invalid (safe)
 (null (aref *char-2-base64* safe)))

(deff safe*2card* (safe*)
 (:let safe* (remove-if 'safe-invalid safe*))
 (reverse (safe*2card*-optimized safe* nil)))

(deff safe*2card*-optimized (safe* result)
 (:when (null (cdddr safe*)) (safe*2card*-0 safe* result))
 (:let (card0 . safe*) (uncons-safe safe*))
 (:let (card1 . safe*) (uncons-safe safe*))
 (:let (card2 . safe*) (uncons-safe safe*))
 (:let (card3 . safe*) (uncons-safe safe*))
 (:let card (card*2card 64 (list card0 card1 card2 card3)))
 (:mlet (card card0) (floor card 256))
 (:mlet (card2 card1) (floor card 256))
 (safe*2card*-optimized safe* (list* card2 card1 card0 result)))

(deff revappend-safe* (safe* result)
 (safe*2card*-0 safe* result))

(deff safe*2card*-0 (safe* result)
 (:let safe* (uncons-safe safe*))
 (:when (atom safe*) result)
 (:let (safe . safe*) safe*)
 (safe*2card*-6 safe* safe result))

(deff safe*2card*-6 (safe* bits result)
 (:let safe* (uncons-safe safe*))
 (:when (atom safe*) result)
 (:let (safe . safe*) safe*)
 (:mlet (bits card) (floor (+ bits (* safe 64)) 256))
 (safe*2card*-4 safe* bits (cons card result)))

(deff safe*2card*-4 (safe* bits result)
 (:let safe* (uncons-safe safe*))
 (:when (atom safe*) result)
 (:let (safe . safe*) safe*)
 (:mlet (bits card) (floor (+ bits (* safe 16)) 256))
 (safe*2card*-2 safe* bits (cons card result)))

(deff safe*2card*-2 (safe* bits result)
 (:let safe* (uncons-safe safe*))
 (:when (atom safe*) result)
 (:let (safe . safe*) safe*)
 (:let card (+ bits (* safe 4)))
 (safe*2card*-0 safe* (cons card result)))

(ntst (safe*2card* nil))
(etst (safe*2card* (ct2ct " BA"))      '(  1))
(etst (safe*2card* (ct2ct "C A"))      '(  2))
(etst (safe*2card* (ct2ct "EA "))      '(  4))
(etst (safe*2card* (ct2ct " IA"))      '(  8))
(etst (safe*2card* (ct2ct "Q A"))      '( 16))
(etst (safe*2card* (ct2ct "gA "))      '( 32))
(etst (safe*2card* (ct2ct " AB"))      '( 64))
(etst (safe*2card* (ct2ct "A C"))      '(128))
(etst (safe*2card* (ct2ct "AE A"))     '(0   1))
(etst (safe*2card* (ct2ct "AIA "))     '(0   2))
(etst (safe*2card* (ct2ct " AQA"))     '(0   4))
(etst (safe*2card* (ct2ct "A gA"))     '(0   8))
(etst (safe*2card* (ct2ct "AA B"))     '(0  16))
(etst (safe*2card* (ct2ct "AAC "))     '(0  32))
(etst (safe*2card* (ct2ct " AAE"))     '(0  64))
(etst (safe*2card* (ct2ct "A AI"))     '(0 128))
(etst (safe*2card* (ct2ct "AA QA"))    '(0 0   1))
(etst (safe*2card* (ct2ct "AAg A"))    '(0 0   2))
(etst (safe*2card* (ct2ct "AAAB "))    '(0 0   4))
(etst (safe*2card* (ct2ct " AAAC"))    '(0 0   8))
(etst (safe*2card* (ct2ct "A AAE"))    '(0 0  16))
(etst (safe*2card* (ct2ct "AA AI"))    '(0 0  32))
(etst (safe*2card* (ct2ct "AAA Q"))    '(0 0  64))
(etst (safe*2card* (ct2ct "AAAg "))    '(0 0 128))
(etst (safe*2card* (ct2ct "AAAABA "))  '(0 0 0	1))
(etst (safe*2card* (ct2ct " AAAACA"))  '(0 0 0	2))
(etst (safe*2card* (ct2ct "A AAAEA"))  '(0 0 0	4))
(etst (safe*2card* (ct2ct "AA AAIA"))  '(0 0 0	8))
(etst (safe*2card* (ct2ct "AAA AQA"))  '(0 0 0  16))
(etst (safe*2card* (ct2ct "AAAA gA"))  '(0 0 0  32))
(etst (safe*2card* (ct2ct "AAAAA B"))  '(0 0 0  64))
(etst (safe*2card* (ct2ct "AAAAAC "))  '(0 0 0 128))

(etst (revappend-safe* (ct2ct "AAAg ")   '(1 2 3))  '(128 0 0 1 2 3))
(etst (revappend-safe* (ct2ct "AAAABA ") '(1 2 3))  '(1 0 0 0 1 2 3))

#|
=============================================
Linebreak safe*
=============================================
(safe*2safe* width linebreak safe*) inserts 'linebreak' in safe* every 'width' columns. 'linebreak' can be any ct such as #\Newline, 10, and (13 10).
|#

(deff safe*2safe* (width linebreak safe*)
 (safe*2safe*1 width (reverse (ct2card* linebreak)) width safe* nil))

(deff safe*2safe*1 (width linebreak column safe* result)
 (:when (atom safe*) (reverse result))
 (:when (= column 0)
  (safe*2safe*1 width linebreak width safe* (append linebreak result)))
 (:let (safe . safe*) safe*)
 (safe*2safe*1 width linebreak (- column 1) safe* (cons safe result)))

(etst (safe*2safe* 3 "->" (ct2ct "")) (ct2ct ""))
(etst (safe*2safe* 3 "->" (ct2ct "abcdefghi")) (ct2ct "abc->def->ghi"))
(etst (safe*2safe* 3 "->" (ct2ct "abcdefghij")) (ct2ct "abc->def->ghi->j"))

#|
=============================================
card2utf8*
=============================================
Convert a single unicode character to sequence of bytes using utf8

The utf8-encoding is thus:
#x00000000-#x0000007F: 0xxxxxxx
#x00000080-#x000007FF: 110xxxxx 10xxxxxx
#x00000800-#x0000FFFF: 1110xxxx 10xxxxxx 10xxxxxx
#x00010000-#x001FFFFF: 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
#x00200000-#x03FFFFFF: 111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
#x04000000-#x7FFFFFFF: 1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx

Or, stated as a function of the integer-length:
00-07: 0xxxxxxx
08-11: 110xxxxx 10xxxxxx
12-16: 1110xxxx 10xxxxxx 10xxxxxx
17-21: 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
22-26: 111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
27-31: 1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx

Now define utf8-length(x) = (floor (+ (integer-length x) 3) 5).

Numbers x between #x80 and #x7FFFFFFF are represented by utf8-length(x) bytes. The first byte consists of utf8-length(x) one-bits followed by one zero-bit followed by bits from x. The other bytes start with '10'.
|#

(deff card2utf8* (card)
 (:unless (numberp card) :wrong)
 (:unless (<= 0 card #x7FFFFFFF) :wrong)
 (:when (<debug card 128) (list card))
 (card2utf8*1 card 1 nil))

(deff card2utf8*1 (card head result)
 (:let length1 (integer-length card))
 (:let length2 (integer-length head))
 (:when (<debug (+ length1 length2) 8)
  (:let head (ash head (- 8 length2)))
  (cons (+ head card) result))
 (:let byte (+ #x80 (logand #x3F card)))
 (:let result (cons byte result))
 (:let head (+ 1 (ash head 1)))
 (:let card (ash card -6))
 (card2utf8*1 card head result))

(etst (card2utf8* #x00000000) '(#x00))
(etst (card2utf8* #x00000080) '(#xC2 #x80))
(etst (card2utf8* #x00000800) '(#xE0 #xA0 #x80))
(etst (card2utf8* #x00010000) '(#xF0 #x90 #x80 #x80))
(etst (card2utf8* #x00200000) '(#xF8 #x88 #x80 #x80 #x80))
(etst (card2utf8* #x04000000) '(#xFC #x84 #x80 #x80 #x80 #x80))

(etst (card2utf8* #x0000007F) '(#x7F))
(etst (card2utf8* #x000007FF) '(#xDF #xBF))
(etst (card2utf8* #x0000FFFF) '(#xEF #xBF #xBF))
(etst (card2utf8* #x001FFFFF) '(#xF7 #xBF #xBF #xBF))
(etst (card2utf8* #x03FFFFFF) '(#xFB #xBF #xBF #xBF #xBF))
(etst (card2utf8* #x7FFFFFFF) '(#xFD #xBF #xBF #xBF #xBF #xBF))

; Copyright sign (c.f. http://www.cl.cam.ac.uk/~mgk25/unicode.html#utf-8)
(etst (card2utf8* #x00A9) '(#xC2 #xA9))
; Unequal
(etst (card2utf8* #x2260) '(#xE2 #x89 #xA0))
; ae
(etst (card2utf8* #xE6) '(#xC3 #xA6))
; o-slash
(etst (card2utf8* #xF8) '(#xC3 #xB8))
; aa
(etst (card2utf8* #xE5) '(#xC3 #xA5))

#|
=============================================
utf2unicode
=============================================
(utf2unicode ct) converts the given utf-8 input to a list of unicode cardinals.
(utf2string ct) converts the given utf-8 input to a string.
|#

(deff utfhead-fct (byte)
 (:when (<debug byte #x80) (cons 0 (logandc1 #x80 byte)))
 (:when (<debug byte #xC0) (cons 0 (char-code #\?)))
 (:when (<debug byte #xE0) (cons 1 (logandc1 #xE0 byte)))
 (:when (<debug byte #xF0) (cons 2 (logandc1 #xF0 byte)))
 (:when (<debug byte #xF8) (cons 3 (logandc1 #xF8 byte)))
 (:when (<debug byte #xFC) (cons 4 (logandc1 #xFC byte)))
 (:when (<debug byte #xFE) (cons 5 (logandc1 #xFE byte)))
 (cons 0 (char-code #\?)))

(deff make-utfhead-table ()
 (:let table (make-array (list 256)))
 (dotimes (n 256) (setf (aref table n) (utfhead-fct n)))
 table)

(defc utfhead-table (make-utfhead-table))

(deff utf-wrong (result) 
 (cons (char-code #\?) result))

#|
CLISP does not seem to optimize mutual tail recursion, so the following does not work for large ct's.

(deff utf2unicode (ct)
 (utf2unicode1 (ct2vector ct) 0 nil))

(deff utf2unicode1 (vector index result)
 (when (= (mod index 1000) 0) (format t "~6d/~6d~%" index (length vector)))
 (:when (>= index (length vector)) (reverse result))
 (:let head (aref vector index))
 (:let index (+ index 1))
 (:let (length . value) (aref utfhead-table head))
 (utf2unicode2 vector index length value result))

(deff utf2unicode2 (vector index length char result)
 (:when (= length 0) (utf2unicode1 vector index (cons char result)))
 (:when (>= index (length vector)) (reverse (utf-wrong result)))
 (:let card (aref vector index))
 (:let index (+ index 1))
 (:unless (<= #x80 card #xBF) (utf2unicode1 vector index (utf-wrong result)))
 (:let char (+ (ash char 6) (logandc1 #xC0 card)))
 (utf2unicode2 vector index (- length 1) char result))
|#

(deff utf2unicode (ct)
 (cdr (utf2unicode2 (ct2vector ct) 0 0 0 nil)))

(deff utf2unicode2 (vector index length char result)
 (:when (= length 0)
  (:let result (cons char result))
  (:when (>= index (length vector)) (reverse result))
  (:let head (aref vector index))
  (:let index (+ index 1))
  (:let (length . value) (aref utfhead-table head))
  (utf2unicode2 vector index length value result))
 (:when (>= index (length vector)) (reverse (utf-wrong result)))
 (:let card (aref vector index))
 (:let index (+ index 1))
 (:unless (<= #x80 card #xBF)
; (utf2unicode1 vector index (utf-wrong result))
  (utf2unicode2 vector index 0 (char-code #\?) result))
 (:let char (+ (ash char 6) (logandc1 #xC0 card)))
 (utf2unicode2 vector index (- length 1) char result))

(deff utf2string (ct) 
 (card*2string (utf2unicode ct)))

(etst
 (utf2unicode
  (card*2vector
   '(#x00
     #xC2 #x80
     #xE0 #xA0 #x80
     #xF0 #x90 #x80 #x80
     #xF8 #x88 #x80 #x80 #x80
     #xFC #x84 #x80 #x80 #x80 #x80
     #x7F
     #xDF #xBF
     #xEF #xBF #xBF
     #xF7 #xBF #xBF #xBF
     #xFB #xBF #xBF #xBF #xBF
     #xFD #xBF #xBF #xBF #xBF #xBF
     #xC2 #xA9
     #xE2 #x89 #xA0
     #xC3 #xA6
     #xC3 #xB8
     #xC3 #xA5)))
 '(#x00000000
   #x00000080
   #x00000800
   #x00010000
   #x00200000
   #x04000000
   #x0000007F
   #x000007FF
   #x0000FFFF
   #x001FFFFF
   #x03FFFFFF
   #x7FFFFFFF
   #x00A9
   #x2260
   #xE6
   #xF8
   #xE5))

(rtst (utf2unicode (repeat 100000 0)))

#|
=============================================
Parse part of url
=============================================
|#

(deff code2card (code radix)
 (:let card (digit-char-p (code-char code) radix))
 (:when (null card) (raise))
 card)

(deff edge2ref (type edge)
 (:when (equalp type :go) (edge2hex edge))
 (:let edge (ct2ct edge))
 (:when (equalp type :base32) (card2ref (base32-2-card edge)))
 (:when (equalp type :base64) (card2ref (base64-2-card edge)))
 (raise))

(deff edge2hex (edge)
 (:when (null edge) nil)
 (:let (c1 c2 . edge) edge)
 (:when (null c2) (raise))
 (:let h1 (code2card c1 16))
 (:let h2 (code2card c2 16))
 (:let byte (+ h2 (* 16 h1)))
 (cons byte (edge2hex edge)))

(etst (edge2hex (ct2card* "90afAF")) '(#x90 #xAF #xAF))
(etst (edge2ref :go (ct2card* "90afAF")) '(#x90 #xAF #xAF))









