#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Time measurement
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Logiweb time schemes
=============================================

Logiweb uses the following time schemes

TAI: International atomic time
UTC: Universal coordinated time
MJD: Modified Julian day
GRD: Gregorian date
LGT: Logiweb time

---------------------------------------------
TAI
---------------------------------------------

TAI (International atomic time) is a 'paper' clock in the sense that it is a computed average of lots of real, atomic clocks located all over the world.

TAI counts seconds, minutes, and hours as regularly as possible. Each TAI day has 24 TAI hours, each TAI hour has 60 TAI minutes, and each TAI minute has 60 TAI seconds. Each TAI second is, as closely as possible, one SI second. An SI second is the duration of 9 192 631 770 periods of the radiation corresponding to the transition between the two hyperfine levels of the ground state of the caesium 133 atom.

TAI time is independent of the rotation of planet Earth.

In Logiweb, TAI hour hh, minute mm, and second ss is written TAI:hh:mm:ss. We have 00 <= hh <= 23, 00 <= mm < 59, and 00 <= ss <= 59. Occasionally, we shall use TAI:24:00:00 to denote TAI:00:00:00 on the following day. Decimal fractions of a second are written after a dot as in TAI:12:23:34.456 which denotes 0.456 seconds past TAI:12:23:34. The notation is compatible with ISO 8601 except that we prepend "TAI:" to emphasize the use of International Atomic Time.

---------------------------------------------
UTC
---------------------------------------------

UTC is a combination of TAI and yet another time scale named UT1.

UT1 is a measure of the rotation angle of planet Earth relative to the direction from the Earth to the Sun. Each UT1 day has 24 UT1 hours, each UT1 hour has 60 UT1 minutes, and each UT1 minute has 60 UT1 seconds. It is noon in UT1 when Greenwich is under the Sun. At the time of writing, UT1 is around 32 seconds behind TAI. In 1972, UT1 was 10 seconds behind TAI.

As mentioned, UTC is a combination of TAI and UT1. UTC equals TAI plus a politically decided offset. This UTC offset indicates how much UTC lacks behind TAI. At the time of writing, the UTC offset is 32 seconds indicating that UTC is 32 seconds behind TAI.

At any time, the UTC offset is an integral number of seconds, but the UTC offset may be incremented or decremented by decree from the International Earth Rotation Service (IERS). Hence, UTC depends on TAI and IERS, but IERS has the intension to keep the difference between UTC and UT1 below 0.9 seconds, so UTC indirectly depends on UT1.

UTC makes a leap whenever IERS increments or decrements the UTC offset. Such leaps are implemented by irregular UTC minutes.

A regular UTC minute has 60 UTC seconds. An irregular one either has 59 or 61. Apart from that UTC counts like TAI and UT1: days have 24 hours and hours have 60 minutes.

Whenever IERS increments (decrements) the UTC offset, the last minute of the last hour of a particular day has 61 (59) seconds. IERS intends to place irregular seconds at the end of June 30 and December 31 when necessary and intends to announce the leaps in advance.

In Logiweb, UTC hour hh, minute mm, and second ss is written UTC:hh:mm:ss. We have 00 <= hh <= 23, 00 <= mm < 59, and 00 <= ss <= 60. Occasionally, we shall use UTC:24:00:00 to denote UTC:00:00:00 on the following day. Decimal fractions of a second are written after a dot as in UTC:12:23:34.456 which denotes 0.456 seconds past UTC:12:23:34. The notation is compatible with ISO 8601 except that we prepend "UTC:" to emphasize the use of International Atomic Time.

At the time of writing, IERS has never decremented the UTC offset, but has incremented the UTC offset at the end of the following days:

GRD-1972-06-30
GRD-1972-12-31
GRD-1973-12-31
GRD-1974-12-31
GRD-1975-12-31
GRD-1976-12-31
GRD-1977-12-31
GRD-1978-12-31
GRD-1979-12-31
GRD-1981-06-30
GRD-1982-06-30
GRD-1983-06-30
GRD-1985-06-30
GRD-1987-12-31
GRD-1989-12-31
GRD-1990-12-31
GRD-1992-06-30
GRD-1993-06-30
GRD-1994-06-30
GRD-1995-12-31
GRD-1997-06-30
GRD-1998-12-31

Before GRD-1972-06-03, the UTC offset was 10 seconds.

---------------------------------------------
MJD
---------------------------------------------

MJD (Modified Julian Day) is a scheme for counting days in a completely regular fasion. Each day is simply expressed by the number of days since a particular day.

MJD is a regular and reliable day count used by astronomers. Furthermore, it is politically correct in the sence that, even though Julius Caesar was quite controversial in his own time, few people today are offended by a time scale named after him.

MJD is based on yet another time scale named JD (Julian Day). JD expresses the number of days since noon, January 1, year -4712 (year 4713 BC), in the Julian calender.

In ancient times, a day was measured from noon to noon, so people actually counted nights instead of days (as a reminiscence, a period of 14 days is still called a fortnight in the English tongue). Today, we prefer to step our day counters when the sun is on the other side of the planet, which is of course difficult to observe, but which possesses little problem for modern technology.

To get a day count based on JD which steps at midnight, the Modified Julian Day (MJD) is offset from JD by 2400000.5 days. In consequence, MJD counts the number of days since GRD-1858-11-17.

When we combine MJD with UTC, then MJD steps at UTC:00:00:00. When we combine MJD with TAI, then MJD steps at TAI:00:00:00. Hence, MJD/UTC and MJD/TAI are two different day counts, but at the time of writing they merely differ by 32 seconds.

In Logiweb, MJD day d is written MJD-d. As an example, GRD-1858-11-17 equals MJD-0 and MJD-51544 equals GRD-2000-01-01. The day before MJD-0 is named MJD--1 (i.e. Modified Julian Day hyphen minus one). The notation follows ISO 8601 in using a hyphen in connection with day counting, but is otherwise completely unrelated.

Combinations of day and second counting schemes are glued together with a dot. As an example, 0.456 seconds past TAI:12:23:34 on MJD-51544 is written MJD-51544.TAI:12:23:34.456. This follows ISO-8601 in putting the day before the second but does not follow the suggestion of ISO-8601 to separate day and second by a capital "T".

---------------------------------------------
GRD
---------------------------------------------

GRD (Gregorian Date) is a scheme for counting days in a fasion so complicated that it has taken millennia to screw it up. Furthermore, GRD counts days, not after "Jesus", but after "Our Lord, Jesus" (Anno Domini), which is not completely neutral.

But GRD is widespread, and hence we use it in the Human-Computer-Interfaces of Logiweb. In Logiweb itself, GRD has no place.

In GRD, day 0 of a year is named "January 1", and day 100 is named April 11 (except if the year is divisible by 4, in which case it is named April 10 (except if the year is divisible by 100, in which case it is named April 11 (except if the year is divisible by 400, in which case it is named April 10))).

In Logiweb, Gregorian year Y, month MM, and day DD is written GRD-Y-MM-DD. We have 01 <= MM <= 12 and 01 <= DD <= 31. The notation is compatible with ISO 8601 except for the following: (1) We prepend "GRD-" to emphasize that we label days like the Gregorian calender does (GRD for GRegorian Date). (2) We allow the year to have more than four digits after year 9999 and to have less then four digits before year 1000. (3) We allow the year to be zero and negative. As examples, GRD-0-01-01 and GRD--5-01-01 are January 1 on year 1 BC and 6 BC, respectively.

When we combine GRD with UTC, then GRD steps at UTC:00:00:00. When we combine GRD with TAI, then GRD steps at TAI:00:00:00. Hence, GRD/UTC and GRD/TAI are two different day counts, but at the time of writing they merely differ by 32 seconds.

Combinations of day and second counting schemes are glued together with a dot. As an example, 0.456 seconds past UTC:12:23:34 on GRD-2000-01-01 is written GRD-2000-01-01.UTC:12:23:34.456. This follows ISO-8601 in putting the day before the second but does not follow the suggestion of ISO-8601 to separate day and second by a capital "T".

---------------------------------------------
LGT
---------------------------------------------

LGT (Logiweb time) is the number of seconds since MJD-0.TAI:00:00:00.

Logiweb time is expressed on the form M*10^(-E) where M is an integer and E is a cardinal (i.e. a non-negative integer). In Logiweb, M is always non-negative, so one could as well say that M is a cardinal.

Logiweb time M*10^(-E) is written LGW-Me-E. As an example, LGW-1083564821686603e-6 equals GRD.2004-05-03.UTC:06:13:41.686603. The e-E may be replaced by the following decadic suffixes:

e-0 U (unit)
e-3 m (milli)
e-6 u (micro, a Greek mu may be used instead)
e-9 n (nano)
e-12 p (pico)
e-15 f (femto)
e-18 a (atto)
e-21 z (zepto)
e-24 y (yocto)

In a Logiweb time like LGW-1083564821686603e-6 one should not replace the small e by a capital one as that may cause confusion with the decadic suffix E (Exa) which stands for 10^+15.

By the way note the following: Logiweb is a computational system intended for mathematics. In physics, one uses decadic prefixes that glue in front of physical units. In computing systems it is better to use decadic suffixes that glue behind numbers. When needed, Logiweb uses the SI units meter, kilogram, second, etc., and derived units. As an example, font sizes are measured in meters. Period. A font size of twelve typographic points is 4218u, and a printer with a resolution of 600 dots per 0.0254 meters has a distance between pixels of 42.333u. An area of 1m by 1m (one milli meter by one milli meter) is 1m^2 (one square milli) or 1u (one micro) or 1e-6 measured in the derived SI unit of square meters. A weight of 1m is one gram (one milli kilogram). A weight of 1um is one microgram (one micro milli kilogram). This is almost but not completely different from the use of decadic prefixes in the SI system.

---------------------------------------------
GUTC
---------------------------------------------

When presenting Logiweb time to a user, we use GRD/UTC which we shall refer to as GUTC. This section describes GUTC in more detail than the individual sections on GRD and UTC.

GUTC is irregular compared to Logiweb time in that it occasionally includes leap seconds and, furthermore, it counts days in a rather complicated (Gregorian) manner, which includes leap days.

GUTC is built up from the following cycles:

GUTC second. The length of a GUTC second is one SI second (which equals one TAI and one UTC second). Each GUTC second starts at the 'tick' of the TAI 'paper' clock. TAI is a 'paper' clock in the sense that it is a computed average of lots of real, atomic clocks located all over the world. As an example, the duration from TAI:00:00:00 to TAI:00:00:01 on Gregor-2000-03-01 (March 1, year 2000) is a GUTC second.

GUTC minute. Regular GUTC minutes consist of 60 GUTC seconds. Irregular GUTC minutes consist of 61 GUTC seconds. In theory, a GUTC minute could consist of 59 GUTC seconds, but that has never happened. As an example, the duration from TAI:00:00:00 to TAI:00.01.00 on Gregor-2000-03-01 is a GUTC minute.

GUTC hour. GUTC hours consist of 60 GUTC minutes. As an example, the duration from TAI:00:00:00 to TAI:01.00.00 on Gregor- 2000-03-01 is a GUTC hour.

GUTC day. GUTC days consist of 24 GUTC hours. As an example, the duration from Gregor-2000-03-01.TAI:00:00:00 to Gregor-2000-03-02.TAI:00.00.00 is a GUTC day.

Long GUTC month. A long GUTC month consists of 31 GUTC days. As an example, the duration from Gregor-2000-03-01.TAI:00:00:00 to Gregor-2000-04-01.TAI:00.00.00 (i.e. March) is a long GUTC month.

Short GUTC month. A short GUTC month consists of 30 GUTC days. As an example, the duration from Gregor-2000-04-01.TAI:00:00:00 to Gregor-2000-05-01.TAI:00.00.00 (i.e. April) is a short GUTC month.

GUTC dimester. A GUTC dimester (dimester = duo menses = two months, compare trimester = tres menses = three months and semester = sex menses = six months) consists of a long GUTC month followed by a short one. As an example, the duration from March to April (inclusive) is a GUTC dimester.

GUTC quimester. A GUTC quimester (quimester = quinque menses = five months) consists of a long, a short, a long, a short, and a long GUTC month. In other words, a quimester consists of two regular dimesters followed by an irregular one that ends abruptly at the end of the quimester. As an example, the duration from March to July (inclusive) is a GUTC dimester. The duration from August to December is another quimester.

GUTC Roman year. A GUTC Roman year is the duration from March 1, inclusive, to the following March 1, exclusive. A regular Roman year has 365 GUTC days; an irregular one has one more. As an example, the period from Gregor-2000-03-01.TAI:00:00:00 to Gregor-2001-02-28.TAI:24:00:00 is GUTC Roman year 2000, which is regular. In contrast, the Gregorian year 2000 is a leap year and, hence, irregular. The difference arises because the Gregorian and Roman years have newyear before and after the leap day, respectively.

A GUTC Roman year consists of three quimesters, the third of which ends abruptly at the end of the year. As a consequence of the conventions mentioned until now, the last month of a regular GUTC Roman year (February) gets 28 GUTC days, and all the other months gets 30 and 31 GUTC days in the pattern prescribed by the Gregorian calender.

GUTC olympiad. A regular GUTC olympiad consists of three regular GUTC years followed by an irregular one. An irregular GUTC olympiad consists of four regular GUTC years. As an example, the period from March 1, 2000 to February 29, 2003 is a regular olympiad.

GUTC century. A regular GUTC century consists of 24 regular GUTC olympiads followed by an irregular one. An irregular GUTC olympiad consists of 25 regular GUTC olympiads. As an example, March 1, 2000 to February 28, 2100 is a regular GUTC century.

GUTC Gregorian cycle. A GUTC Gregorian cycle consists of three regular GUTC centuries followed by an irregular one.

---------------------------------------------
Lisp representations
---------------------------------------------

Logiweb does not need negative time in any of the time counting schemes. For that reason, all representations use cardinals (i.e. non-negative integers).

LGT is represented as a list (m e) of cardinals and which represents m*10^(-e) seconds past the Logiweb epoch which is at MJD-0.TAI:00:00:00.

MJD/TAI is represented as a list (f e s m h d) of cardinals which representes MJD-d.TAI:h:m:s.f where f must have e digits. As an example, (1 6 2 3 4 5) represents MJD-5.TAI:04:03:02.000001

GRD/UTC is represented as a list (f e s m h D M Y) of cardinals which represents GRD-Y-M-D.UTC:h:m:s.f where f must have e digits.

MJD is represented as a cardinal.

GRD is represented as a list (D M Y) of cardinals which represents GRD-Y-M-D.

---------------------------------------------
Lisp functions
---------------------------------------------

Conversion functions

(grd2mjd grd)
(mjd2grd mjd)
(grd-utc-2-lgt grd-utc)
(mjd-tai-2-lgt mjd-tai)
(lgt-2-mjd-tai lgt)
(lgt-2-grd-utc lgt)

Time reading function

(lgt)

Output functions

(lgt-2-string lgt)
(lgt-2-mjd-tai-string lgt)
(lgt-2-grd-utc-string lgt)
(lgt-2-filename lgt)



|#

#|
=============================================
Conversion between MJD and GRD
=============================================
|#

#|
---------------------------------------------
Computed conversion from MJD to GRD
---------------------------------------------
|#

(defc *mjd-of-grd-2000-01-01* 51544)
(defc *mjd-of-grd-2000-03-01* (+ 31 29 *mjd-of-grd-2000-01-01*))

(defc *seconds-per-minute* 60)
(defc *minutes-per-hour* 60)
(defc *hours-per-day* 24)
(defc *days-per-long-month* 31)
(defc *days-per-short-month* 30)
(defc *months-per-dimester* 2)
(defc *months-per-quimester* 5)
(defc *months-per-year* 12)
(defc *days-per-year* 365)
(defc *years-per-olympiad* 4)
(defc *olympiads-per-century* 25)
(defc *centuries-per-gregorian-cycle* 4)

(defc *seconds-per-hour* (* *seconds-per-minute* *minutes-per-hour*))
(defc *seconds-per-day* (* *seconds-per-hour* *hours-per-day*))

(defc *days-per-dimester*
 (+ *days-per-long-month* *days-per-short-month*))
(defc *days-per-quimester*
 (+ (* 3 *days-per-long-month*) (* 2 *days-per-short-month*)))

(defc *days-per-olympiad*
 (+ 1 (* *days-per-year* *years-per-olympiad*)))
(defc *days-per-century*
 (+ -1 (* *days-per-olympiad* *olympiads-per-century*)))
(defc *days-per-gregorian-cycle*
 (+ 1 (* *days-per-century* *centuries-per-gregorian-cycle*)))

(defc *years-per-century*
 (* *years-per-olympiad* *olympiads-per-century*))
(defc *years-per-gregorian-cycle*
 (* *years-per-century* *centuries-per-gregorian-cycle*))

(deff lfloor (dividend divisor limit)
 (:mlet (quotient remainder) (floor dividend divisor))
 (:when (<debug quotient limit) (values quotient remainder))
 (:let quotient (- limit 1))
 (:let remainder (- dividend (* quotient divisor)))
 (values quotient remainder))

(deff mjd2grd-by-computation (mjd)
 ; convert to GRD-2000-03-01 origin
 (:let day (- mjd *mjd-of-grd-2000-03-01*))
 ; compute year, month, and day relative to GRD-2000-03-01
 (:mlet (cycle day) (floor day *days-per-gregorian-cycle*))
 (:mlet (century day)
  (lfloor day *days-per-century* *centuries-per-gregorian-cycle*))
 (:mlet (olympiad day) (floor day *days-per-olympiad*))
 (:mlet (year day) (lfloor day *days-per-year* *years-per-olympiad*))
 (:mlet (quimester day) (floor day *days-per-quimester*))
 (:mlet (dimester day) (floor day *days-per-dimester*))
 (:mlet (month day) (floor day *days-per-long-month*))
 (:let dimester-month (* dimester *months-per-dimester*))
 (:let quimester-month (* quimester *months-per-quimester*))
 (:let month (+ month dimester-month quimester-month))
 (:let olympiad-year (* olympiad *years-per-olympiad*))
 (:let century-year (* century *years-per-century*))
 (:let cycle-year (* cycle *years-per-gregorian-cycle*))
 (:let year (+ year olympiad-year century-year cycle-year))
 ; convert year, month, and day relative to GRD-2000-03-01 to GRD
 (:mlet (carry month) (floor (+ month 2) *months-per-year*))
 (:let year (+ year 2000 carry))
 (:let month (+ month 1))
 (:let day (+ day 1))
 (list day month year))

(etst (mjd2grd-by-computation 0) '(17 11 1858))

#|
---------------------------------------------
Computed conversion from GRD to MJD
---------------------------------------------
|#

(defc *month-length* '(31 30 31 30 31   31 30 31 30 31   31))

(deff grd-sum (sum list)
 (:when (null list) (list sum))
 (cons sum (grd-sum (+ sum (car list)) (cdr list))))

(defc *month-sum* (grd-sum 0 *month-length*))

(etst (length *month-sum*) *months-per-year*)

(etst (+ 28 (car (last *month-sum*))) 365)

(deff grd2mjd-by-computation (grd)
 (:let (day month year) grd)
 ; convert to year, month, and day relative to GRD-2000-03-01
 (:let day (- day 1))
 (:mlet (carry month) (floor (- month 3) *months-per-year*))
 (:let year (- (+ carry year) 2000))
 ; Compute cycles
 (:let cycle (floor year *years-per-gregorian-cycle*))
 (:let century (floor year *years-per-century*))
 (:let olympiad (floor year *years-per-olympiad*))
 (+
  *mjd-of-grd-2000-03-01*
  day
  (nth month *month-sum*)
  (* year *days-per-year*)
  olympiad
  (- century)
  cycle))

(etst (grd2mjd-by-computation '(17 11 1858)) 0)

#|
---------------------------------------------
Table based conversion from MJD to GRD
---------------------------------------------
|#

(deff add-month (day month-year list)
 (:when (= day 0) list)
 (add-month (- day 1) month-year (acons day month-year list)))

(deff year-2-length-of-february (year)
 (:when (= (mod year 400) 0) 29)
 (:when (= (mod year 100) 0) 28)
 (:when (= (mod year 4) 0) 29)
 28)

(deff add-year (year list)
 (:let nn (year-2-length-of-february year))
 (:let year (list year))
 (:let list (add-month 31 (cons 12 year) list))
 (:let list (add-month 30 (cons 11 year) list))
 (:let list (add-month 31 (cons 10 year) list))
 (:let list (add-month 30 (cons  9 year) list))
 (:let list (add-month 31 (cons  8 year) list))
 (:let list (add-month 31 (cons  7 year) list))
 (:let list (add-month 30 (cons  6 year) list))
 (:let list (add-month 31 (cons  5 year) list))
 (:let list (add-month 30 (cons  4 year) list))
 (:let list (add-month 31 (cons  3 year) list))
 (:let list (add-month nn (cons  2 year) list))
 (:let list (add-month 31 (cons  1 year) list))
 list)

(deff add-years (year list)
 (:when (<debug year 0) list)
 (add-years (- year 1) (add-year year list)))

(defc grd2day-array (make-array '(32 13 400)))
(defc day2grd-array (make-array *days-per-gregorian-cycle*))

(deff init-grd-arrays ()
 (init-grd-arrays1 (add-years 399 nil) 0))

(deff init-grd-arrays1 (grd* day)
 (:when (null grd*) nil)
 (:let (grd . grd*1) grd*)
 (:let (day1 month year) grd)
 (setf (aref grd2day-array day1 month year) day)
 (setf (aref day2grd-array day) grd)
 (init-grd-arrays1 grd*1 (+ day 1)))

(init-grd-arrays)

(deff mjd2grd-by-table (mjd)
 (:let day (- mjd *mjd-of-grd-2000-01-01*))
 (:mlet (cycle day) (floor day *days-per-gregorian-cycle*))
 (:let (day month year) (aref day2grd-array day))
 (list day month (+ year (* cycle *years-per-gregorian-cycle*) 2000)))

(etst (mjd2grd-by-table 51544) '(1 1 2000))

(deff grd2mjd-by-table (grd)
 (:let (day month year) grd)
 (:let year (- year 2000))
 (:mlet (cycle year) (floor year *years-per-gregorian-cycle*))
 (:unless (<= 1 day 31) nil)
 (:unless (<= 1 month 12) nil)
 (:let day (aref grd2day-array day month year))
 (:when (null day) nil)
 (+ *mjd-of-grd-2000-01-01* day (* cycle *days-per-gregorian-cycle*)))

(etst (grd2mjd-by-table '( 1  1 2000)) 51544)
(ntst (grd2mjd-by-table '( 1  0 2000)))
(etst (grd2mjd-by-table '( 1 12 2000)) 51879)
(ntst (grd2mjd-by-table '( 1 13 2000)))
(ntst (grd2mjd-by-table '( 0  1 2000)))
(etst (grd2mjd-by-table '(31  1 2000)) 51574)
(ntst (grd2mjd-by-table '(32  1 2000)))

(etst (grd2mjd-by-table '(28  2 2000)) 51602)
(etst (grd2mjd-by-table '(29  2 2000)) 51603)
(ntst (grd2mjd-by-table '(30  2 2000)))

(etst (grd2mjd-by-table '(28  2 2001)) 51968)
(ntst (grd2mjd-by-table '(29  2 2001)))
(ntst (grd2mjd-by-table '(30  2 2001)))

(etst (grd2mjd-by-table '(28  2 2004)) 53063)
(etst (grd2mjd-by-table '(29  2 2004)) 53064)
(ntst (grd2mjd-by-table '(30  2 2004)))

(etst (grd2mjd-by-table '(28  2 2100)) 88127)
(ntst (grd2mjd-by-table '(29  2 2100)))
(ntst (grd2mjd-by-table '(30  2 2100)))

#|
---------------------------------------------
Test of conversion functions
---------------------------------------------
|#

(deff test-mjd (mjd end)
 (:when (> mjd end) nil)
 (:let grd1 (mjd2grd-by-computation mjd))
 (:let grd2 (mjd2grd-by-table mjd))
 (:when (unequal grd1 grd2)
  (error "MJD-~s converted to GRD-~s and GRD-~s" mjd grd1 grd2))
 (:let mjd1 (grd2mjd-by-computation grd1))
 (:when (unequal mjd mjd1)
  (error "MJD-~s converted to GRD-~s and back to MJD-~s" mjd grd1 mjd1))
 (:let mjd2 (grd2mjd-by-table grd1))
 (:when (unequal mjd mjd2)
  (error "MJD-~s converted to GRD-~s and back to MJD-~s" mjd grd1 mjd2))
;(format t "mjd=~s mjd1=~s mjd2=~s grd1=~s grd2=~s~%" mjd mjd1 mjd2 grd1 grd2)
 (test-mjd (+ mjd 1) end))

;(ntst (test-mjd 0 1000000))
(ntst (test-mjd -1000 1000))

#|
---------------------------------------------
Chosen conversions between MJD and GRD
---------------------------------------------
|#

(deff grd2mjd (grd) (grd2mjd-by-table grd))
(deff mjd2grd (mjd) (mjd2grd-by-table mjd))

#|
=============================================
Conversion between TAI and UTC
=============================================
|#

#|
---------------------------------------------
Input function
---------------------------------------------
|#

(deff parse-int (string)
 (:mlet (card index) (parse-integer string :junk-allowed t))
 (:unless (integerp card) nil)
 (cons card (subseq string index)))

(etst (parse-int "123") (cons 123 ""))
(etst (parse-int "123abc") (cons 123 "abc"))
(etst (parse-int "123 abc") (cons 123 " abc"))
(etst (parse-int "-123") (cons -123 ""))
(etst (parse-int "-123abc") (cons -123 "abc"))
(etst (parse-int "-123 abc") (cons -123 " abc"))
(etst (parse-int "0abc") (cons 0 "abc"))
(ntst (parse-int "abc0"))

(deff string2mjd (string)
 (:when (string-prefix "MJD-" string)
  (:let (mjd . string) (parse-int (subseq string 4)))
  (:unless (equalp string "") nil)
  mjd)
 (:unless (string-prefix "GRD-" string) nil)
 (:let (year . string) (parse-int (subseq string 4)))
 (:unless (string-prefix "-" string) nil)
 (:let (month . string) (parse-int (subseq string 1)))
 (:unless (string-prefix "-" string) nil)
 (:let (day . string) (parse-int (subseq string 1)))
 (:unless (equalp "" string) nil)
 (grd2mjd (list day month year)))

(ntst (string2mjd "ABC-123"))

(etst (string2mjd "MJD-123") 123)
(etst (string2mjd "MJD-0") 0)
(etst (string2mjd "MJD--123") -123)
(ntst (string2mjd "MJD-a"))
(ntst (string2mjd "MJD-1a"))

(etst (string2mjd "GRD-2000- 1- 1") 51544)
(ntst (string2mjd "GRD-2000- 0- 1"))
(etst (string2mjd "GRD-2000-12- 1") 51879)
(ntst (string2mjd "GRD-2000-13- 1"))
(ntst (string2mjd "GRD-2000- 1- 0"))
(etst (string2mjd "GRD-2000- 1-31") 51574)
(ntst (string2mjd "GRD-2000- 1-32"))

(etst (string2mjd "GRD-2000- 2-28") 51602)
(etst (string2mjd "GRD-2000- 2-29") 51603)
(ntst (string2mjd "GRD-2000- 2-30"))

(etst (string2mjd "GRD-2001- 2-28") 51968)
(ntst (string2mjd "GRD-2001- 2-29"))
(ntst (string2mjd "GRD-2001- 2-30"))

(etst (string2mjd "GRD-2004- 2-28") 53063)
(etst (string2mjd "GRD-2004- 2-29") 53064)
(ntst (string2mjd "GRD-2004- 2-30"))

(etst (string2mjd "GRD-2100- 2-28") 88127)
(ntst (string2mjd "GRD-2100- 2-29"))
(ntst (string2mjd "GRD-2100- 2-30"))

(ntst (string2mjd "GRD-1-1--1"))
(ntst (string2mjd "GRD-1--1-1"))
(etst (string2mjd "GRD--1-1-1") -679306)

#|
---------------------------------------------
Translate leap second definition strings
---------------------------------------------
|#

(defc *spaces* '(#\Space #\Tab))

(deff repeated-leap (leap*)
 (:when (atom leap*) nil)
 (:let ((:delta . mjd) . leap*) leap*)
 (repeated-leap1 mjd leap*))

(deff repeated-leap1 (mjd1 leap*)
 (:when (atom leap*) nil)
 (:let ((:delta . mjd2) . leap*) leap*)
 (:when (equalp mjd1 mjd2) mjd1)
 (repeated-leap1 mjd2 leap*))

(etst 5 (repeated-leap '((+1 . 5) (-1 . 5))))
(etst 5 (repeated-leap '((-1 . 6) (+1 . 5) (-1 . 5))))
(ntst (repeated-leap '((-1 . 6) (+1 . 5) (-1 . 6))))
(ntst (repeated-leap '((-1 . 6) (+1 . 5))))
(ntst (repeated-leap '((-1 . 6))))
(ntst (repeated-leap nil))

(deff string2leap (string)
 (:let (int . string) (parse-int string))
 (:unless (integerp int) nil)
 (:let string (string-left-trim *spaces* string))
 (:let date (string2mjd string))
 (:when (null date) nil)
 (cons int date))

(etst (string2leap "+1 GRD-2005-12-31") '( 1 . 53735))
(etst (string2leap "-1 GRD-2005-12-31") '(-1 . 53735))
(etst (string2leap "1 MJD-53735") '(1 . 53735))
(ntst (string2leap "a GRD-2005-2-29"))

(deff string*2leap*1 (string* result)
 (:when (atom string*) result)
 (:let (string . string*) string*)
 (:let leap (string2leap string))
 (:when (null leap) (complain "Invalid leap: ~s" string))
 (:let result (cons leap result))
 (string*2leap*1 string* result))

(deff string*2leap* (string*)
 (:let leap* (string*2leap*1 string* nil))
 (:let leap* (sort leap* '< :key 'cdr))
 (:let mjd (repeated-leap leap*))
 (:when mjd (complain "Multiple leaps at MJD-~a~%" mjd))
 leap*)

#|
---------------------------------------------
Translate string to list of strings
---------------------------------------------
|#

(deff string2list (string)
 (:when (equalp string "") nil)
 (:let separator (aref string 0))
 (string2list1 string separator 1))

(deff string2list1 (string separator start)
 (:let position (position separator string :start start))
 (:when (null position) (list (subseq string start)))
 (:let element (subseq string start position))
 (:let list (string2list1 string separator (+ position 1)))
 (cons element list))

(etst (string2list1 "ab,cd,ef" #\, 0) '("ab" "cd" "ef"))
(etst (string2list1 "ab,cd,ef," #\, 0) '("ab" "cd" "ef" ""))

(etst (string2list ",ab,cd,ef") '("ab" "cd" "ef"))
(etst (string2list ",ab,cd,ef,") '("ab" "cd" "ef" ""))

#|
---------------------------------------------
List of leap seconds according to the NIST Time Scale Data Archive at http://www.boulder.nist.gov/timefreq/pubs/bulletin/leapsecond.htm

The list has form ((delta . grd) ...) where delta is the amount of seconds added to the given Gregorian date (grd) in the UTC system.

The list is based on GRD rather than MJD because the MJD on http://www.boulder.nist.gov/timefreq/pubs/bulletin/leapsecond.htm seems to be broke (at least at the time the comment was wrote - I have been in contact with NIST and they have corrected it, thanks).
---------------------------------------------
|#

(defc *leap* "
+1 GRD-2005-12-31
+1 GRD-1998-12-31
+1 GRD-1997-06-30
+1 GRD-1995-12-31
+1 GRD-1994-06-30
+1 GRD-1993-06-30
+1 GRD-1992-06-30
+1 GRD-1990-12-31
+1 GRD-1989-12-31
+1 GRD-1987-12-31
+1 GRD-1985-06-30
+1 GRD-1983-06-30
+1 GRD-1982-06-30
+1 GRD-1981-06-30
+1 GRD-1979-12-31
+1 GRD-1978-12-31
+1 GRD-1977-12-31
+1 GRD-1976-12-31
+1 GRD-1975-12-31
+1 GRD-1974-12-31
+1 GRD-1973-12-31
+1 GRD-1972-12-31
+1 GRD-1972-06-30")

(defc *default-leap* (string*2leap* (string2list *leap*)))

; TAI was 10 seconds ahead of UTC at GRD-1972-01-01, i.e. before the first leap second.

(defc *init-offset* 10)

#|
---------------------------------------------
*mjd-leap-seconds* below is only used for test purposes now. Other conversion is based on the 'leap' option. *init-offset* below is still in use.
---------------------------------------------
|#

(deff year-month-day-2-mjd (year month day)
 (grd2mjd (list day month year)))

(defc *mjd-leap-seconds*
`(
  (+1 . ,(year-month-day-2-mjd 2005 12 31))
  (+1 . ,(year-month-day-2-mjd 1998 12 31))
  (+1 . ,(year-month-day-2-mjd 1997 06 30))
  (+1 . ,(year-month-day-2-mjd 1995 12 31))
  (+1 . ,(year-month-day-2-mjd 1994 06 30))
  (+1 . ,(year-month-day-2-mjd 1993 06 30))
  (+1 . ,(year-month-day-2-mjd 1992 06 30))
  (+1 . ,(year-month-day-2-mjd 1990 12 31))
  (+1 . ,(year-month-day-2-mjd 1989 12 31))
  (+1 . ,(year-month-day-2-mjd 1987 12 31))
  (+1 . ,(year-month-day-2-mjd 1985 06 30))
  (+1 . ,(year-month-day-2-mjd 1983 06 30))
  (+1 . ,(year-month-day-2-mjd 1982 06 30))
  (+1 . ,(year-month-day-2-mjd 1981 06 30))
  (+1 . ,(year-month-day-2-mjd 1979 12 31))
  (+1 . ,(year-month-day-2-mjd 1978 12 31))
  (+1 . ,(year-month-day-2-mjd 1977 12 31))
  (+1 . ,(year-month-day-2-mjd 1976 12 31))
  (+1 . ,(year-month-day-2-mjd 1975 12 31))
  (+1 . ,(year-month-day-2-mjd 1974 12 31))
  (+1 . ,(year-month-day-2-mjd 1973 12 31))
  (+1 . ,(year-month-day-2-mjd 1972 12 31))
  (+1 . ,(year-month-day-2-mjd 1972 06 30))))

(etst *default-leap* (reverse *mjd-leap-seconds*))

#|
---------------------------------------------
Conversion from GRD/UTC to LGT
---------------------------------------------
|#

(deff accumulate-leap-seconds (leap* sum result)
 (:when (null leap*) result)
 (:let ((delta . mjd) . leap*) leap*)
 (:let sum (+ sum delta))
 (accumulate-leap-seconds leap* sum (acons sum mjd result)))

(deff mjd2leaps (mjd list)
 (:when (null list) *init-offset*)
 (:let ((sum . mjd1) . list) list)
 (:when (> mjd mjd1) sum)
 (mjd2leaps mjd list))

(deff grd-utc-2-lgt (grd-utc &optional (pool *pool*))
 (:let (fraction exponent second minute hour . grd) grd-utc)
 (:let minute-seconds (* *seconds-per-minute* minute))
 (:let hour-seconds (* *seconds-per-hour* hour))
 (:let mjd (grd2mjd grd))
 (:let mjd-seconds (* *seconds-per-day* mjd))
 (:let total (pool-get* pool :leap :total))
 (:let leaps (mjd2leaps mjd total))
 (:let second (+ second minute-seconds hour-seconds mjd-seconds leaps))
 (:let resolution (expt 10 exponent))
 (:let mantissa (+ fraction (* resolution second)))
 (list mantissa exponent))

#|
---------------------------------------------
Conversion from MJD/TAI to LGT
---------------------------------------------
|#

(deff mjd-tai-2-lgt (mjd-tai)
 (:let (fraction exponent second minute hour mjd) mjd-tai)
 (:let minute-seconds (* *seconds-per-minute* minute))
 (:let hour-seconds (* *seconds-per-hour* hour))
 (:let mjd-seconds (* *seconds-per-day* mjd))
 (:let second (+ second minute-seconds hour-seconds mjd-seconds))
 (:let resolution (expt 10 exponent))
 (:let mantissa (+ fraction (* resolution second)))
 (list mantissa exponent))

#|
---------------------------------------------
Conversion from LGT to MJD/TAI
---------------------------------------------
|#

(deff lgt-2-mjd-tai (lgt)
 (:let (mantissa exponent) lgt)
 (:let resolution (expt 10 exponent))
 (:mlet (second fraction) (floor mantissa resolution))
 (:mlet (minute second) (floor second *seconds-per-minute*))
 (:mlet (hour minute) (floor minute *minutes-per-hour*))
 (:mlet (mjd hour) (floor hour *hours-per-day*))
 (list fraction exponent second minute hour mjd))

#|
---------------------------------------------
Conversion from LGT to GRD/UTC
---------------------------------------------
|#

(deff compute-correction (leap* sum result)
 (:when (null leap*) result)
 (:let ((delta . mjd) . leap*) leap*) ; mjd measured in utc
 (:let sum (+ sum delta))
 (:let end-of-jump (+ sum (* (+ 1 mjd) *seconds-per-day*))) ; in tai
 (:let entry (list sum delta end-of-jump))
 (compute-correction leap* sum (cons entry result)))

(deff second2correction (second lgt2leap)
 (:when (null lgt2leap) (list (- second *init-offset*) 0))
 (:let ((sum delta end-of-jump) . lgt2leap) lgt2leap)
 (:when (>= second end-of-jump) (list (- second sum) 0))
 (:when (<debug second (- end-of-jump delta)) (second2correction second lgt2leap))
 (list (- second sum) delta))

(deff lgt-2-grd-utc (lgt &optional (pool *pool*))
 (:let (mantissa exponent) lgt)
 (:let resolution (expt 10 exponent))
 (:mlet (second fraction) (floor mantissa resolution))
 (:let lgt2leap (pool-get* pool :leap :lgt2leap))
 (:let (second delta) (second2correction second lgt2leap))
 (:mlet (minute second) (floor second *seconds-per-minute*))
 (:mlet (hour minute) (floor minute *minutes-per-hour*))
 (:mlet (mjd hour) (floor hour *hours-per-day*))
 (:let grd (mjd2grd mjd))
 (list* fraction exponent (+ second delta) minute hour grd))

#|
---------------------------------------------
Add leap* to pool
---------------------------------------------
|#

(deff pool-put-leap (pool leap*)
 (:let total (accumulate-leap-seconds leap* *init-offset* nil))
 (:let pool (pool-put* pool total :leap :total))
 (:let lgt2leap (compute-correction leap* *init-offset* nil))
 (:let pool (pool-put* pool lgt2leap :leap :lgt2leap))
 pool)

(defc *pool0* (pool-put-leap nil *default-leap*))

#|
---------------------------------------------
Test of conversion functions
---------------------------------------------

Test cases from D. J. Bernstein: http://cr.yp.to/proto/utctai.html:

1997-06-30 23:59:59 UTC = 1997-07-01 00:00:29 TAI
1997-06-30 23:59:60 UTC = 1997-07-01 00:00:30 TAI
1997-07-01 00:00:00 UTC = 1997-07-01 00:00:31 TAI

Hence, we should have

GRD-1997-06-30.UTC:23:59:59.123 = MJD-50630.TAI:00:00:29.123
GRD-1997-06-30.UTC:23:59:60.123 = MJD-50630.TAI:00:00:30.123
GRD-1997-07-01.UTC:00:00:00.123 = MJD-50630.TAI:00:00:31.123
|#

(etst (lgt-2-mjd-tai (grd-utc-2-lgt (list 123 3 59 59 23 30 06 1997) *pool0*))
 (list 123 3 29 00 00 50630))
(etst (lgt-2-mjd-tai (grd-utc-2-lgt (list 123 3 60 59 23 30 06 1997) *pool0*))
 (list 123 3 30 00 00 50630))
(etst (lgt-2-mjd-tai (grd-utc-2-lgt (list 123 3 00 00 00 01 07 1997) *pool0*))
 (list 123 3 31 00 00 50630))

(etst (lgt-2-grd-utc (mjd-tai-2-lgt (list 123 3 29 00 00 50630)) *pool0*)
 (list 123 3 59 59 23 30 06 1997))
(etst (lgt-2-grd-utc (mjd-tai-2-lgt (list 123 3 30 00 00 50630)) *pool0*)
 (list 123 3 60 59 23 30 06 1997))
(etst (lgt-2-grd-utc (mjd-tai-2-lgt (list 123 3 31 00 00 50630)) *pool0*)
 (list 123 3 00 00 00 01 07 1997))

#|
---------------------------------------------
Test of conversion functions
---------------------------------------------
|#

(deff test-lgt (m e)
 (:let lgt (list m e))
 (and
  (equalp lgt (mjd-tai-2-lgt (lgt-2-mjd-tai lgt)))
  (equalp lgt (grd-utc-2-lgt (lgt-2-grd-utc lgt *pool0*) *pool0*))))

(ttst (test-lgt 0 0))
(ttst (test-lgt 0 1))
(ttst (test-lgt 0 6))
(ttst (test-lgt 1 0))
(ttst (test-lgt 1 1))
(ttst (test-lgt 1 6))
(ttst (test-lgt 1234567890 0))
(ttst (test-lgt 1234567890 1))
(ttst (test-lgt 1234567890 6))
(ttst (test-lgt -1234567890 0))
(ttst (test-lgt -1234567890 1))
(ttst (test-lgt -1234567890 6))

#|
=============================================
Get current Logiweb time
=============================================

The algorithm below uses unix gettimeofday.

The gettimeofday function is said to "ignore" leap seconds. I have found that terminology confusing since I would say that TAI ignores leap seconds and UTC takes them into account.

But gettimeofday keeps in sync with UTC and seems to "ignore" leap seconds in the three monkeys way (see no evil, hear no evil, speak no evil). It seems that gettimeofday is completely aware of leap seconds and "ignores" them actively: up to a leap second, gettimeofday closes its eyes so it doesn't see the clock move, it closes its ears so it doesn't hear the tick, and it shuts its mouth so it doesn't speak about the leap second. After the leap second has elapsed, gettimeofday pretends the the leap second never occurred. Hence, the count of seconds returned by gettimeofday is the number of non-leap seconds since the epoch, and gettimeofday has to be completely aware of leap seconds in order to avoid counting them.

I would very much like to avoid using gettimeofday. Rather, I would prefer to use a timer that counted seconds since e.g. the boot time of the computer, and then I would compute the offset between that timer and TAI once and for all when starting the Logiweb server. I have heard rumours about such a timer, but haven't been able to find it. Please tell me if you know where it is!

Until then, (lgt) below uses gettimeofday. And it converts the output from gettimeofday into LGT in a somewhat cumbersome way: Since gettimeofday is in sync with UTC, and since I already have routines for converting MJD to GRD and GRD/UTC to LGT, I first convert gettimeofday to MJD/UTC, then convert the MJD part to GRD so I have GRD/UTC, and then convert GRD/UTC to LGT. So I take the trouble to compute leap years twice: when going from MJD to GRD and when going back from GRD to LGT. But of course it is the computer that has the trouble. I just reuse some routines I already have.

|#

(defc *unix-epoch* (grd2mjd '(1 1 1970)))

(deff lgt ()
 (:let (mantissa exponent) (unix-time))
 (:mlet (second fraction) (floor mantissa (expt 10 exponent)))
 (:mlet (minute second) (floor second 60))
 (:mlet (hour minute) (floor minute 60))
 (:mlet (day hour) (floor hour 24))
 (:let mjd (+ day *unix-epoch*))
 (:let grd (mjd2grd mjd))
 (grd-utc-2-lgt (list* fraction exponent second minute hour grd)))

#|
=============================================
Time measurement for debugging/profiling
=============================================
|#

(defc *time* 0)

(deff set-time ()
 (setq *time* (car (unix-time))))

(deff get-time ()
 (:let (time exponent) (unix-time))
 (:let delta (- time *time*))
 (setq *time* time)
 (list delta exponent))

(deff print-time (msg)
 (:let (mantissa exponent) (get-time))
 (format t "After ~11:d e-~d seconds: ~a~%" mantissa exponent msg))

#|
=============================================
Output functions
=============================================
|#

(deff lgt-2-string (lgt)
 (:let (m e) lgt)
 (format nil "LGT-~de-~d" m e))

(deff lgt-2-raw-string (lgt)
 (:let (m e) lgt)
 (format nil "~de-~d" m e))

(deff lgt-2-mjd-tai-string (lgt)
 (:let (fraction exponent second minute hour day) (lgt-2-mjd-tai lgt))
 (if (= exponent 0)
  (format nil "MJD-~d.TAI:~2,'0d:~2,'0d:~2,'0dU"
   day hour minute second)
  (format nil "MJD-~d.TAI:~2,'0d:~2,'0d:~2,'0d.~v,'0d"
   day hour minute second exponent fraction)))

(deff lgt-2-grd-utc-string (lgt)
 (:let (fraction exponent second minute hour day month year)
  (lgt-2-grd-utc lgt))
 (if (= exponent 0)
  (format nil "GRD-~d-~2,'0d-~2,'0d.UTC:~2,'0d:~2,'0d:~2,'0dU"
   year month day hour minute second)
  (format nil "GRD-~d-~2,'0d-~2,'0d.UTC:~2,'0d:~2,'0d:~2,'0d.~v,'0d"
   year month day hour minute second exponent fraction)))

(deff lgt-2-filename (lgt)
 (:let (fraction exponent second minute hour day month year)
  (lgt-2-grd-utc lgt))
 (if (= exponent 0)
  (format nil "GRD-~d-~2,'0d-~2,'0d-UTC-~2,'0d-~2,'0d-~2,'0dU"
   year month day hour minute second)
  (format nil "GRD-~d-~2,'0d-~2,'0d-UTC-~2,'0d-~2,'0d-~2,'0d-~v,'0d"
   year month day hour minute second exponent fraction)))

(etst (lgt-2-string (list 0 0)) "LGT-0e-0")
(etst (lgt-2-string (list 1103 2)) "LGT-1103e-2")
(etst (lgt-2-mjd-tai-string (list 0 0)) "MJD-0.TAI:00:00:00U")
(etst (lgt-2-mjd-tai-string (list 1103 2)) "MJD-0.TAI:00:00:11.03")
(etst (lgt-2-grd-utc-string (list 0 0)) "GRD-1858-11-16.UTC:23:59:50U")
(etst (lgt-2-grd-utc-string (list 1103 2)) "GRD-1858-11-17.UTC:00:00:01.03")
(etst (lgt-2-filename (list 0 0)) "GRD-1858-11-16-UTC-23-59-50U")
(etst (lgt-2-filename (list 1103 2)) "GRD-1858-11-17-UTC-00-00-01-03")

#|
The following function gives TAI, UTC, GRD, MJD, and LGT. The values for TAI, UTC, GRD, and MJD may be compared manually against e.g. http://www.leapsecond.com/java/gpsclock.htm.
|#

(deff print-times ()
 (:let lgt (lgt))
 (format t "~a~%" (lgt-2-string lgt))
 (format t "~a~%" (lgt-2-grd-utc-string lgt))
 (format t "~a~%" (lgt-2-mjd-tai-string lgt))
 (values))

#|
The following test compares the lgt system with GRD-UTC from get-universal-time and decode-universal-time of Common Lisp.
|#

(deff offset-check ()
 (:let (mantissa exponent) (lgt))
 (:mlet (second minute hour day month year)
  (decode-universal-time (get-universal-time) 0))
 (:let (mantissa1)
  (grd-utc-2-lgt (list 0 exponent second minute hour day month year)))
 (:let resolution (expt 10 exponent))
 (format t "~a~%" (lgt-2-grd-utc-string (list mantissa exponent)))
 (format t "~a~%" (lgt-2-grd-utc-string (list mantissa1 exponent)))
 (/ (- mantissa mantissa1 0.0) resolution))

(ttst (<= (abs (offset-check)) 1))

#|
=============================================
Time manipulation functions
=============================================
(time< t1 t2) is true if duration t1 is smallar than duration t2.
time<=, time>, time>=, time=, and time/= are similar.
(time+ t1 t2) equals the sum of the two durations.
(time- t1 t2) equals the difference of the two durations.
(time* t1 t2) the product of t1 and t2. Useful e.g. when t2 has dimension baud.
(time-min t1 t2) the minimum of the two given times.
(time-max t1 t2) the maximum of the two given times.

When a duration is used to represent an absolute time, the duration is relative to the Logiweb epoch.
|#

(deff time-mantissa (time exponent)
 (:let (m e) time)
 (:when (= e exponent) m)
 (:when (<debug e exponent) (* m (expt 10 (- exponent e))))
 (floor m (expt 10 (- e exponent))))

(etst (time-mantissa '(117 6) 4) 1)
(etst (time-mantissa '(117 6) 5) 11)
(etst (time-mantissa '(117 6) 6) 117)
(etst (time-mantissa '(117 6) 7) 1170)
(etst (time-mantissa '(117 6) 8) 11700)

(deff max-exponent (t1 t2)
 (:let (:m1 e1) t1)
 (:let (:m2 e2) t2)
 (max e1 e2))

(etst (max-exponent '(117 6) '(118 5)) 6)
(etst (max-exponent '(117 6) '(118 6)) 6)
(etst (max-exponent '(117 6) '(118 7)) 7)

(deff time< (t1 t2)
 (:let max (max-exponent t1 t2))
 (<debug (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time<= (t1 t2)
 (:let max (max-exponent t1 t2))
 (<= (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time> (t1 t2)
 (:let max (max-exponent t1 t2))
 (> (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time>= (t1 t2)
 (:let max (max-exponent t1 t2))
 (>= (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time= (t1 t2)
 (:let max (max-exponent t1 t2))
 (= (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time/= (t1 t2)
 (:let max (max-exponent t1 t2))
 (/= (time-mantissa t1 max) (time-mantissa t2 max)))

(deff time+ (t1 t2)
 (:let max (max-exponent t1 t2))
 (list (+ (time-mantissa t1 max) (time-mantissa t2 max)) max))

(deff time- (t1 t2)
 (:let max (max-exponent t1 t2))
 (list (- (time-mantissa t1 max) (time-mantissa t2 max)) max))

(deff time* (t1 t2)
 (:let t1 (if (atom t1) (list t1 0) t1))
 (:let t2 (if (atom t2) (list t2 0) t2))
 (:let (m1 e1) t1)
 (:let (m2 e2) t2)
 (list (* m1 m2) (+ e1 e2)))

(deff timec* (c t2)
 (:let (m2 e2) t2)
 (list (* c m2) e2))

(deff time-min (t1 t2)
 (if (time< t1 t2) t1 t2))

(deff time-max (t1 t2)
 (if (time> t1 t2) t1 t2))

(defc test-t1 '(11 5))
(defc test-t2 '(117 6))
(defc test-t3 '(1170 7))
(defc test-t4 '(11701 8))

(ntst (time< test-t1 test-t1))
(ttst (time< test-t1 test-t2))
(ttst (time< test-t1 test-t3))
(ttst (time< test-t1 test-t4))
(ntst (time< test-t2 test-t1))
(ntst (time< test-t2 test-t2))
(ntst (time< test-t2 test-t3))
(ttst (time< test-t2 test-t4))
(ntst (time< test-t3 test-t1))
(ntst (time< test-t3 test-t2))
(ntst (time< test-t3 test-t3))
(ttst (time< test-t3 test-t4))
(ntst (time< test-t4 test-t1))
(ntst (time< test-t4 test-t2))
(ntst (time< test-t4 test-t3))
(ntst (time< test-t4 test-t4))

(ttst (time<= test-t1 test-t1))
(ttst (time<= test-t1 test-t2))
(ttst (time<= test-t1 test-t3))
(ttst (time<= test-t1 test-t4))
(ntst (time<= test-t2 test-t1))
(ttst (time<= test-t2 test-t2))
(ttst (time<= test-t2 test-t3))
(ttst (time<= test-t2 test-t4))
(ntst (time<= test-t3 test-t1))
(ttst (time<= test-t3 test-t2))
(ttst (time<= test-t3 test-t3))
(ttst (time<= test-t3 test-t4))
(ntst (time<= test-t4 test-t1))
(ntst (time<= test-t4 test-t2))
(ntst (time<= test-t4 test-t3))
(ttst (time<= test-t4 test-t4))

(ntst (time> test-t1 test-t1))
(ntst (time> test-t1 test-t2))
(ntst (time> test-t1 test-t3))
(ntst (time> test-t1 test-t4))
(ttst (time> test-t2 test-t1))
(ntst (time> test-t2 test-t2))
(ntst (time> test-t2 test-t3))
(ntst (time> test-t2 test-t4))
(ttst (time> test-t3 test-t1))
(ntst (time> test-t3 test-t2))
(ntst (time> test-t3 test-t3))
(ntst (time> test-t3 test-t4))
(ttst (time> test-t4 test-t1))
(ttst (time> test-t4 test-t2))
(ttst (time> test-t4 test-t3))
(ntst (time> test-t4 test-t4))

(ttst (time>= test-t1 test-t1))
(ntst (time>= test-t1 test-t2))
(ntst (time>= test-t1 test-t3))
(ntst (time>= test-t1 test-t4))
(ttst (time>= test-t2 test-t1))
(ttst (time>= test-t2 test-t2))
(ttst (time>= test-t2 test-t3))
(ntst (time>= test-t2 test-t4))
(ttst (time>= test-t3 test-t1))
(ttst (time>= test-t3 test-t2))
(ttst (time>= test-t3 test-t3))
(ntst (time>= test-t3 test-t4))
(ttst (time>= test-t4 test-t1))
(ttst (time>= test-t4 test-t2))
(ttst (time>= test-t4 test-t3))
(ttst (time>= test-t4 test-t4))

(ttst (time= test-t1 test-t1))
(ntst (time= test-t1 test-t2))
(ntst (time= test-t1 test-t3))
(ntst (time= test-t1 test-t4))
(ntst (time= test-t2 test-t1))
(ttst (time= test-t2 test-t2))
(ttst (time= test-t2 test-t3))
(ntst (time= test-t2 test-t4))
(ntst (time= test-t3 test-t1))
(ttst (time= test-t3 test-t2))
(ttst (time= test-t3 test-t3))
(ntst (time= test-t3 test-t4))
(ntst (time= test-t4 test-t1))
(ntst (time= test-t4 test-t2))
(ntst (time= test-t4 test-t3))
(ttst (time= test-t4 test-t4))

(ntst (time/= test-t1 test-t1))
(ttst (time/= test-t1 test-t2))
(ttst (time/= test-t1 test-t3))
(ttst (time/= test-t1 test-t4))
(ttst (time/= test-t2 test-t1))
(ntst (time/= test-t2 test-t2))
(ntst (time/= test-t2 test-t3))
(ttst (time/= test-t2 test-t4))
(ttst (time/= test-t3 test-t1))
(ntst (time/= test-t3 test-t2))
(ntst (time/= test-t3 test-t3))
(ttst (time/= test-t3 test-t4))
(ttst (time/= test-t4 test-t1))
(ttst (time/= test-t4 test-t2))
(ttst (time/= test-t4 test-t3))
(ntst (time/= test-t4 test-t4))

(defc test-t1 '(11 5))
(defc test-t2 '(117 6))
(defc test-t3 '(1170 7))
(defc test-t4 '(11701 8))

(etst (time+ test-t1 test-t1) '(   22 5))
(etst (time+ test-t1 test-t2) '(  227 6))
(etst (time+ test-t1 test-t3) '( 2270 7))
(etst (time+ test-t1 test-t4) '(22701 8))
(etst (time+ test-t2 test-t1) '(  227 6))
(etst (time+ test-t2 test-t2) '(  234 6))
(etst (time+ test-t2 test-t3) '( 2340 7))
(etst (time+ test-t2 test-t4) '(23401 8))
(etst (time+ test-t3 test-t1) '( 2270 7))
(etst (time+ test-t3 test-t2) '( 2340 7))
(etst (time+ test-t3 test-t3) '( 2340 7))
(etst (time+ test-t3 test-t4) '(23401 8))
(etst (time+ test-t4 test-t1) '(22701 8))
(etst (time+ test-t4 test-t2) '(23401 8))
(etst (time+ test-t4 test-t3) '(23401 8))
(etst (time+ test-t4 test-t4) '(23402 8))

(etst (time- test-t1 test-t1) '(   0 5))
(etst (time- test-t1 test-t2) '(  -7 6))
(etst (time- test-t1 test-t3) '( -70 7))
(etst (time- test-t1 test-t4) '(-701 8))
(etst (time- test-t2 test-t1) '(   7 6))
(etst (time- test-t2 test-t2) '(   0 6))
(etst (time- test-t2 test-t3) '(   0 7))
(etst (time- test-t2 test-t4) '(  -1 8))
(etst (time- test-t3 test-t1) '(  70 7))
(etst (time- test-t3 test-t2) '(   0 7))
(etst (time- test-t3 test-t3) '(   0 7))
(etst (time- test-t3 test-t4) '(  -1 8))
(etst (time- test-t4 test-t1) '( 701 8))
(etst (time- test-t4 test-t2) '(   1 8))
(etst (time- test-t4 test-t3) '(   1 8))
(etst (time- test-t4 test-t4) '(   0 8))

(etst (time* '(3 10) '(4 30)) '(12 40))
(etst (time* '2 '(4 30)) '(8 30))

(etst (time-min '(3 4) '(4 3)) '(3 4))
(etst (time-min '(4 3) '(3 4)) '(3 4))

(etst (time-max '(3 4) '(4 3)) '(4 3))
(etst (time-max '(4 3) '(3 4)) '(4 3))

#|
=============================================
Make unguessable, unique prefix
=============================================
|#

(deff make-prefix ()
 (:let ransize (ash 1 32))
 (:let (m) (lgt))
 (:let prefix (+ (random ransize) (* ransize m)))
 prefix)


