#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Server
=============================================
Constructs for defining constants and
functions
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
BUGHUNT
=============================================
|#

(defun less (x y u v)
 (when (or (null x) (null y))
  (format t "~s~%" u)
  (format t "~s~%" v))
 (< x y))

;(defmacro <debug (x y) `(less ,x ,y ',x ',y))
(defmacro <debug (x y) `(< ,x ,y))

#|
=============================================
Custom defconstant which
1) allows redefinition when reloading
2) always recomputes when reloading
=============================================
|#

(defvar const* nil)

(defmacro defc (x y)
 `(progn
   (push ',x const*)
   (defvar ,x)
   (when *testp* (format t "Evaluating DEFC ~s~%" ',x))
   (setq ,x ,y)))

(defun size (x)
 (cond
  ((null x) 0)
  ((symbolp x) 1)
  ((numberp x) 1)
  ((consp x) (+ (size (car x)) (size (cdr x))))
  ((hash-table-p x) (hash-table-size x))
  ((functionp x) 1)
  ((stringp x) (length x))
  ((characterp x) 1)
  ((arrayp x) (array-total-size x))
  ((pathnamep x) 1)
  ((equalp 'alien (and (consp (type-of x)) (car (type-of x)))) 1)
  (t 1)))

(defun size0 (x)
 (cond
  ((null x) 0)
  ((symbolp x) (format t "~s~%" x) 1)
  ((numberp x) 1)
  ((consp x) (+ (size (car x)) (size (cdr x))))
  ((hash-table-p x) (hash-table-size x))
  ((functionp x) 1)
  ((stringp x) (length x))
  ((characterp x) 1)
  ((arrayp x) (array-total-size x))
  ((pathnamep x) 1)
  ((equalp 'alien (and (consp (type-of x)) (car (type-of x)))) 1)
  (t 1)))

(defun sizes ()
 (setq const* (remove-duplicates const*))
 (sizes1 const* 0))

(defun sizes1 (list sum)
 (if (atom list)
  (format t "~10d ~a~%" sum "SUM")
  (let* ((size (size (symbol-value (car list)))))
   (format t "~10d ~a~%" size (car list))
   (sizes1 (cdr list) (+ sum size)))))

(defun all-sizes ()
 (do-all-symbols (x)
  (when (boundp x)
   (format t "~10d ~a~%" (size (symbol-value x)) (symbol-name x)))))

#|
=============================================
System for tracing execution
=============================================
(deff-trace-on) turns on tracing of subsequently macro-expanded deff constructs.
(deff-trace-off) turns tracing off.

(deff-clear) clears trace counts.
(deff-count key) increments the count associated to the given key.
(deff-report) reports the counts and clears the trace table.
|#

(defc *deff-trace-on* nil)

(defun deff-trace-on ()
 (setq *deff-trace-on* t))

(defun deff-trace-off ()
 (setq *deff-trace-on* nil))

(defc *deff-trace-table* (make-hash-table :test 'equalp))

(defun deff-clear ()
 (clrhash *deff-trace-table*))

(defun deff-count (key)
 (incf (gethash key *deff-trace-table* 0)))

(defun deff-report ()
 (deff-report-noclear)
 (deff-clear)
 (values))

(defun hash2assoc (hash)
 (let* ((assoc nil))
  (maphash
   #'(lambda (key val) (setq assoc (acons (reverse key) val assoc)))
   hash)
  assoc))

(ntst (hash2assoc *deff-trace-table*))
(deff-count '(a b c))
(deff-count '(a b c))
(etst (hash2assoc *deff-trace-table*) '(((c b a) . 2)))
(deff-clear)
(ntst (hash2assoc *deff-trace-table*))

(defun symbol< (x y)
 (string< (symbol-name x) (symbol-name y)))

(defun symbol*< (x y)
 (cond
  ((atom y) nil)
  ((atom x) t)
  ((symbol< (car x) (car y)) t)
  ((symbol< (car y) (car x)) nil)
  (t (symbol*< (cdr x) (cdr y)))))

(defun deff-report-noclear ()
 (let* (
   (assoc (hash2assoc *deff-trace-table*))
   (assoc1 (sort assoc 'symbol*< :key 'car)))
  (dolist (pair assoc1) (format t "~6d ~s~%" (cdr pair) (car pair)))))

#|
=============================================
Safe car and cdr
=============================================
|#

(defun head (x)
 (if (atom x) x (car x)))

(defun tail (x)
 (if (atom x) x (cdr x)))

#|
=============================================
Custom defun with a number of added features
=============================================
We shall refer to Lisp statements like (cons 2 3) and (declare (ignore x)) as 'units' and we shall refer to lists of units as 'bodies'.

A deff can replace a defun. Example:

(deff f (x y)
 (declare (ignore x))
 (:let z (cons y y))
 (:when (null y) z)
 (+ 2 3))

The function f above ignores x, lets z equal to (cons y y), returns z in case y if null, and returns 5 otherwise.

The deff macro does not recognise 'declare' and '+' above and, hence, it does not touch them. It recognises the following 'deff units':



(:when condition &rest body)

If the condition is non-nil, execute the units in the body and return the last value. The body may contain deff units so the body of a :when can contain other :when, :let, and so on. If the condition is nil, continue to the next unit in the enclosing body.



(:unless condition &rest body)

Same as (:when (not condition) &rest body).



(:let pattern value &rest ignored)

Bind the pattern to the value and continue to the next unit of the enclosing body.

Examples:

(:let (x . y) z) binds x to (head z) and y to (tail z). This leads to a "forced match" in which z is decomposed into a head and a tail even if z is not a pair.

(:let (nil . y) z) binds y to (tail z) and makes no further bindings. In general, occurrences of nil makes no bindings.

(:let (x . x) y) binds x to (tail y) and also generates a compile time warning about the first x being defined but not used.

(:let (1 . x) y) attempts to bind 1 to (head y) yielding a compile time warning.



(:mlet pattern form &rest ignored)

Convert the multiple values returned by "form" into a list and then bind the pattern to that list.

Example:

(:mlet (x y) (floor 10 3)) binds x to the quotient 3 and y to the remainder 1.



(:catch pattern body)

If the next units in the enclosing body raises an exception then bind the pattern to the value of the exception and execute the body.

Example. The following function returns (3 . 2):

(deff f ()
 (:catch (x y) (cons y x))
 1
 (raise 2 3)
 4)

|#

(defmacro deff (name args &rest body)
 (let* (
   (args (add-ignore args))
   (pair (deff-body body name))
   (body (head pair))
   (defs (tail pair))
   (body (deff-trace name nil body)) ; this line can be omitted
   (def `(defun ,name ,@args ,@body))
   (defs (cons def defs)))
 `(progn ,@defs)))

(defmacro exec (&rest body)
 (let* (
   (pair (deff-body body "EXEC"))
   (body (head pair))
   (defs (tail pair)))
  (when defs (error ":io not allowed in exec block"))
 `(progn ,@body)))

; convert e.g. (u :v w :x) into ((u v w x) (declare (ignore x v)))
(defun add-ignore (parm*)
 (add-ignore1 parm* nil nil))

(defun add-ignore1 (parm* result ignores)
 (if (atom parm*) `(,(reverse result) (declare (ignore ,@ignores)))
  (let* (
    (parm (head parm*))
    (parm* (tail parm*)))
   (if (not (keywordp parm))
    (add-ignore1 parm* (cons parm result) ignores)
    (let* ((parm (gensym (symbol-name parm))))
     (add-ignore1 parm* (cons parm result) (cons parm ignores)))))))

(defun deff-declaration-p (body)
 (cond
  ((atom body) nil)
  ((atom (car body)) nil)
  (t (equalp (caar body) 'declare))))

(defun deff-trace (name1 name2 body)
 (if (null *deff-trace-on*) body
  (if (deff-declaration-p body)
   (cons (car body) (deff-trace name1 name2 (cdr body)))
   `((let* ((deff-key (cons ',name1 ,name2))) (deff-count deff-key) ,@body)))))

(defun deff-trace-exp (name1 name2 exp)
 (if (null *deff-trace-on*) exp
  `(let* ((deff-key (cons ',name1 ,name2))) (deff-count deff-key) ,exp)))

(defun deff-body (body name)
 (if (atom body) (cons body nil)
  (let* (
    (unit (head body))
    (pair (deff-body (tail body) name))
    (body (head pair))
    (defs (tail pair)))
   (if (atom unit) (cons (cons unit body) defs)
    (let* ((args (tail unit)))
     (case (head unit)
      (:when   (deff-when   args body defs name))
      (:unless (deff-unless args body defs name))
      (:let    (deff-let    args body defs))
      (:mlet   (deff-mlet   args body defs))
      (:catch  (deff-catch  args body defs))
      (:io     (deff-io     args body defs name))
      (t       (cons (cons unit body) defs))))))))

(defun unite (body)
 (if (null (tail body))
  (head body)
  (cons 'progn body)))

(defun deff-if (condition unit1 unit2 defs1 defs2)
 (let* (
   (unit (list 'if condition unit1 unit2))
   (defs (append defs1 defs2)))
  (cons (list unit) defs)))

(defun deff-when (args body defs name)
 (let* (
   (condition (head args))
   (pair (deff-body (tail args) name))
   (body1 (head pair))
   (defs1 (tail pair))
   (unit1 (unite (deff-trace :when+ 'deff-key body1)))
   (unit2 (unite (deff-trace :when- 'deff-key body))))
  (deff-if condition unit1 unit2 defs defs1)))

(defun deff-unless (args body defs name)
 (let* (
   (condition (head args))
   (pair (deff-body (tail args) name))
   (body1 (head pair))
   (defs1 (tail pair))
   (unit1 (unite (deff-trace :unless+ 'deff-key body1)))
   (unit2 (unite (deff-trace :unless- 'deff-key body))))
  (deff-if condition unit2 unit1 defs defs1)))

(defun make-let (variable value body)
 (let* (
   (unit `(let ((,variable ,value)) ,@body)))
  (list unit)))

(defun make-head (x)
 `(head ,x))

(defun make-tail (x)
 `(tail ,x))

(defun deff-let (args body defs)
 (let* (
   (gensym 'deff-let-var) ; alternative: (gensym (gensym))
   (pattern (head args))
   (value (head (tail args)))
   (body1 (deff-let1 pattern gensym body)))
  (cons (make-let gensym value body1) defs)))

(defun deff-let1 (pattern value body)
 (if (constantp pattern) body
  (if (atom pattern) (make-let pattern value body)
   (deff-let1 (car pattern) (make-head value)
    (deff-let1 (cdr pattern) (make-tail value) body)))))

(defun deff-mlet (args body defs)
 (let* (
   (gensym 'deff-mlet-var) ; alternative: (gensym (gensym))
   (pattern (head args))
   (value (head (tail args)))
   (body1 (deff-let1 pattern gensym body)))
  (cons (make-let gensym `(multiple-value-list ,value) body1) defs)))

(defstruct :exception (value))

(defun deff-catch (args body defs)
 (cons (deff-catch1 args body) defs))

(defun deff-catch1 (args body)
 (let* (
   (gensym 'deff-catch-var) ; alternative: (gensym (gensym))
   (pattern (head args))
   (body1 (tail args)))
  `((let ((,gensym (catch :exception ,@body)))
     (if (not (exception-p ,gensym)) ,gensym
      ,(unite (deff-let1 pattern `(exception-value ,gensym) body1)))))))

(defun raise (&rest x)
 (throw :exception (make-exception :value x)))

(defun complain (format &rest args)
 (apply 'format t format args)
 (terpri)
 (raise))

(defun ignoring (&rest ignored)
 (declare (ignore ignored)))

(defmacro exceptional (x)
 `(exception-p (catch :exception ,x)))

(defun deff-io (args body defs name)
 (let* (
   (fct (intern (format nil "~a-~a" name (head args))))
   (args (tail args))
   (parm (add-ignore (head args)))
   (args (tail args))
   (body (deff-trace name nil body)) ; this line can be omitted
   (def `(defun ,fct ,@parm ,@body))
   (defs (cons def defs)))
  (cons `((list ',fct ,@args)) defs)))

#|
=============================================
String concatenation
=============================================
Put here by convenience because cat et al must be defined early.
|#

(defun cat (&rest string*)
 (apply 'concatenate 'string string*))

(defun unslash (string)
 (string-right-trim "/" string))

(defun slash (string)
 (cat (unslash string) "/"))

(etst (cat "ab" "cd" "ef") "abcdef")
(etst (unslash "abc") "abc")
(etst (unslash "abc/") "abc")
(etst (slash "abc") "abc/")
(etst (slash "abc/") "abc/")
