#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Server
=============================================
Socket interface
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Convert duration
=============================================
|#

(deff time2usec (time)
 (:let (m e) time)
 (:when (= e 6) m)
 (:when (<debug e 6) (* (expt 10 (- 6 e)) m))
 (round m (expt 10 (- e 6))))

(etst (time2usec '(123 0)) 123000000)
(etst (time2usec '(123 1)) 12300000)
(etst (time2usec '(123 2)) 1230000)
(etst (time2usec '(123 3)) 123000)
(etst (time2usec '(123 4)) 12300)
(etst (time2usec '(123 5)) 1230)
(etst (time2usec '(123 6)) 123)
(etst (time2usec '(123 7)) 12)
(etst (time2usec '(123 8)) 1)
(etst (time2usec '(123 9)) 0)

(deff time2triple (time)
 (:let usec (time2usec time))
 (:let (usec sec Msec) (card2card* 1000000 usec))
 (list (default 0 Msec) (default 0 sec) usec))

(etst (time2triple '(123456789 0)) '(123 456789 000000))
(etst (time2triple '(123456789 1)) '( 12 345678 900000))
(etst (time2triple '(123456789 2)) '(  1 234567 890000))
(etst (time2triple '(123456789 3)) '(  0 123456 789000))
(etst (time2triple '(123456789 4)) '(  0  12345 678900))
(etst (time2triple '(123456789 5)) '(  0   1234 567890))
(etst (time2triple '(123456789 6)) '(  0    123 456789))
(etst (time2triple '(123456789 7)) '(  0     12 345679))
(etst (time2triple '(123456789 8)) '(  0      1 234568))
(etst (time2triple '(123456789 9)) '(  0      0 123457))

#|
=============================================
Interface to foreign functions
=============================================
A Lisp function that calls a foreign functions typically has the
following format:

(deff caller (int ct)
(parmb)                    ; begin parameter transfer
(parmi int)                ; put integer parameter
(parms ct)                 ; put string parameter
(callee)
(:when (= (exiti) -1) die) ; get integer return value
(exits))                   ; get string return value
|#

(deff parmb ()
 (buffer_rewind))

; Integer parameter
(deff parmi (int)
 (int_put int))

; ct ('string') parameter
(deff parms (ct)
 (:let card* (ct2card* ct))
 (parmi (length card*))
 (dolist (card card*) (char_put card)))

; Integer return value
(deff exiti ()
 (int_get))

; card* ('string') return value
(deff exits ()
 (:let length (exiti))
 (:when (= length -2) :none)
 (:when (<debug length 0) :error)
 (exits1 length nil))

(deff exits1 (length result)
 (:when (equalp length 0) (reverse result))
 (exits1 (- length 1) (cons (char_get) result)))

; vector return value
(deff exitv ()
 (:let length (exiti))
 (:when (= length -2) :none)
 (:when (<debug length 0) :error)
 (:let vector (make-vector length))
 (dotimes (i length) (setf (aref vector i) (char_get)))
 vector)

#|
=============================================
Close file descriptor
=============================================
Close file descriptor.
fd_close (int fd) -> (int 0) 
Always succeeds, returns nil */
|#

(deff fd-close (fd)
 (:unless (integerp fd) nil)
 (parmb)
 (parmi fd)
 (fd_close)
 nil)

#|
=============================================
Translate host name to ip
=============================================
|#

(deff host-ip (hostname)
 (parmb)
 (parms (cons hostname 0))
 (host_ip)
 (exits))

#|
=============================================
Udp functions
=============================================
udp-open -> (int fd)
Open udp socket
Returns :error on error
Returns fd on success.

udp-bind (int fd) (ct ip) (int port) -> (int 0)
Bind udp socket
Returns :error on error.
Returns nil on success.

udp-recvfrom (int fd) -> (card* msg) (card* ip) . (int port)
Receive message from udp socket
Returns :error on error.
Returns :none when no message available.
Returns (msg . (ip . port)) when message available with ip as numbers and dots.

udp-sendto (int fd) (ct msg) ((ct domain) . (int port)) -> (int 0)
Send message from udp socket
Returns :error on error.
Returns nil on success.
|#

(deff udp-open ()
 (parmb)
 (udp_open)
 (:let fd (exiti))
 (:when (<debug fd 0) :error)
 fd)

(deff udp-bind (fd ip port)
 (parmb)
 (parmi fd)
 (parms (cons ip 0))
 (parmi port)
 (udp_bind)
 (if (<debug (exiti) 0) :error nil))
 
(deff udp-recvfrom (fd)
 (parmb)
 (parmi fd)
 (udp_recvfrom)
 (:let msg (exits))
 (:when (atom msg) msg)
 (:let ip (exits))
 (:let port (exiti))
 (list* msg ip port))

(deff udp-sendto (fd msg dest)
 (:let (domain . port) dest)
 (parmb)
 (parmi fd)
 (parms msg)
 (parms (cons domain 0))
 (parmi port)
 (udp_sendto)
 (if (<debug (exiti) 0) :error nil))

#|
=============================================
Tcp functions
=============================================
tcp-open (ct ip) (int port) (int backlog) -> (int fd)
Open tcp socket
Returns :error on error.

tcp-accept (int fd0) -> (int fd1) (char* ip)
Accept incomming connection
Returns :error on error.
Returns nil if there are no incomming connections.

tcp-recv (int fd) -> (char* msg)
Impatient read from connected socket
Returns :error on error.
Returns nil is nothing is received within 0.1 second

tcp-send (int fd) (ct msg) -> (int 0)
Impatient send to connected socket
Returns :error on error.
Returns nil on success.
Missing outgoing bandwidth is not treated as an error.

tcp-query (ct domain) (int port) (duration patience) (ct msg1) -> (vector msg2)
Impatiently sends msg1 to the given domain/port, then patiently reads the response and returns it. Halts on error.
|#

(deff tcp-open (ip port backlog)
 (parmb)
 (parms (cons ip 0))
 (parmi port)
 (parmi backlog)
 (tcp_open)
 (:let fd (exiti))
 (:when (<debug fd 0) :error)
 fd)

(deff tcp-accept (fd0)
 (parmb)
 (parmi fd0)
 (tcp_accept)
 (:let fd1 (exiti))
 (:when (<debug fd1 0) :error)
 (:when (= fd1 0) nil)
 (list fd1 (exits)))

(deff tcp-recv (fd)
 (parmb)
 (parmi fd)
 (tcp_recv)
 (exits))

(deff tcp-send (fd msg)
 (parmb)
 (parmi fd)
 (parms msg)
 (tcp_send)
 (if (<debug (exiti) 0) :error nil))

(deff tcp-query (domain port time msg)
 (:let (usec sec Msec) (time2triple time))
 (parmb)
 (parmi port)
 (parmi usec)
 (parmi sec)
 (parmi Msec)
 (parms msg)
 (parms (cons domain 0))
 (tcp_query)
 (:let result (exitv))
 (:when (equalp result :error) (error "Could not read referenced file"))
 result)

#|
=============================================
File functions
=============================================
(file-open filename) returns the opened file descriptor or :error.
(file-read fd) returns the bytes read. Returns NIL at EOF. Returns :none
if no input ready. Returns :error on error.
|#

(deff file-open (filename)
 (parmb)
 (parms (cons filename 0))
 (file_open)
 (:let fd (exiti))
 (:when (<debug fd 0) :error)
 fd)

(deff file-read (fd)
 (parmb)
 (parmi fd)
 (file_read)
 (exits))

#|
 (:let result (exiti))
 (:when (= result -2) :eof)
 (:when (= result 0) nil)
 (:when (<debug result 0) :error)
 (exits))
|#

#|
=============================================
Wait function
=============================================
(inet-select time fd*) waits for incomming messages or timeout. It returns nil on success and :error on error. There is no indication which fd's (if any) are ready for I/O. If this is needed later on, inet-select should  return a list of ready fd's (or nil if no fd's are ready).
|#

(deff inet-select (time fd*)
 (:let (Msec sec usec) (time2triple time))
 (parmb)
 (parmi usec)
 (parmi sec)
 (parmi Msec)
 (parmi (length fd*))
 (dolist (fd fd*) (parmi fd))
 (inet_select)
 (if (<debug (exiti) 0) :error nil))

#|
=============================================
Open tcp and udp with unwind protect
=============================================
(socket-protect var opener &rest forms) binds 'var' to the result of 'opener' and then executes the given forms. The 'opener' can be a udp-open, a tcp-open, or a tcp-accept. The opened socket is closed on exit, even if the exit is via the debugger.

udp-open and tcp-open return a socket whereas tcp-accept returns a list whose first component is a socket. get-protected-socket extracts the socket in both cases.
|#

(deff get-protected-socket (socket)
 (if (atom socket) socket (car socket)))

(defmacro socket-protect (socket open &rest forms)
`(let ((,socket -1))
  (unwind-protect
   (progn (setq ,socket ,open) ,@forms)
   (fd-close (get-protected-socket ,socket)))))

#|
=============================================
ip -> string
=============================================
An "ip" is a list of four bytes, listed in netword byte order.
|#

(deff ip2string (ip)
 (:let (a b c d) ip)
 (format nil "~d.~d.~d.~d" a b c d))

(etst (ip2string '(127 0 0 1)) "127.0.0.1")

#|
=============================================
card* -> ip
=============================================
|#

(deff card*2ip (card*)
 (:let byte* (card*2ip1 card* 0 nil))
 (:when (unequal (length byte*) 4) nil)
 (reverse byte*))

(deff card*2ip1 (card* byte byte*)
 (:when (atom card*) (cons byte byte*))
 (:let (card . card*) card*)
 (:when (equalp card (char-code #\.)) (card*2ip1 card* 0 (cons byte byte*)))
 (:let digit (digit-char-p (code-char card)))
 (:when (null digit) nil)
 (card*2ip1 card* (+ digit (* byte 10)) byte*))

(etst (card*2ip (string2card* "127.0.0.1")) '(127 0 0 1))
(ntst (card*2ip (string2card* "127.0.0")))
(ntst (card*2ip (string2card* "127.0.0.a")))

#|
=============================================
Demonize
=============================================
|#

(deff demonize (log-file pid-file usr-name)
 (parmb)
 (parms (cons log-file 0))
 (parms (cons pid-file 0))
 (parms (cons usr-name 0))
 (unix_demonize))

#|
=============================================
Time function
=============================================
Function that returns the number of non-leap seconds that have elapsed since the Unix epoch (GRD-1970-01-01.00:00:00).
|#

(deff int-get (radix)
 (:let digit (exiti))
 (:when (<debug digit 0) 0)
 (+ digit (* radix (int-get radix))))

(deff unix-time ()
 (parmb)
 (unix_gettimeofday)
 (:let success (exiti))
 (:when (unequal success 0) (error "Could not read internal clock"))
 (:let exponent (exiti))
 (:let radix (exiti))
 (:let mantissa (int-get radix))
 (list mantissa exponent))






