#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Option handling
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Option addition
=============================================
(option-add key val option*) adds (key . val) to option* unless option* already associates some value to the given key.
|#

(deff option-add (key val option*)
 (:when (pool-get* option* key) (complain "Option ~a given more than once" key))
 (acons key val option*))

#|
=============================================
Pool layout
=============================================

:option :syntax -> syntax and default values for all known options.
:option :process -> order of processing of various sources of options.
:option :priority -> the priorities of various sources of options.
:option :string source name -> original strings that define options.
:option :value name -> value of option

=============================================
Option syntax
=============================================

The syntax of an option is defined by a list of (namelist initform translator) structures.

The namelist is the list of all names of the option. The first name in the list is the canonical name of the option whereas other names (if any) are aliases.

The initform is a form to be evaluated by (eval initform) to provide the compiled in default of the option.

All options have a 'no' version (like --noquit) which sets the option to the empty string. For Booleans, the empty string represents falsehood (the case insensitive strings "no", "false", and "f" also represent falsehood).

All options have a 'yes' version (like --quit with no arguments) which sets the option to the Lisp atom T. For Booleans, the Lisp atom T represents truth (the case insensitive strings "yes", "true", and "t" also represent truth).

If non-boolean options are given without argument then the T value will generate an error message.
|#

(deff pool-add-syntax (pool)
 (pool-add pool '(:option :syntax)
 '(
   (("bin")          ""             (opt-dir))
   (("cache")        ""             (opt-dir))
   (("conf")         *conf*         (opt-string))
   (("demonize")     "yes"          (opt-bool))
   (("dest")         ""             (opt-string))
   (("destmirror")   "lib/$64$"     (opt-string))
   (("destresubmit") "lib/$64$"     (opt-string))
   (("destsubmit")   "$LGS$/$DATE$" (opt-string))
   (("desttemp")     "$LGS$/undated"(opt-string))
   (("diagnose")     "no"           (opt-bool))
   (("dump")         ""             (opt-string))
   (("dumpconf")     ""             (opt-string))
   (("exec")         ""             (opt-bool))
   (("fifotcp0")     "10"           (opt-card))
   (("fifotcp1")     "10"           (opt-card))
   (("fifotcp2")     "10"           (opt-card))
   (("fifoudp1")     "10"           (opt-card))
   (("fifoudp2")     "10"           (opt-card))
   (("filter")       ""             (opt-stringlist))
   (("gc")           "no"           (opt-bool))
   (("gcpatience")   "0"            (opt-duration))
   (("gcroots")      "!-lib/*!+*/latest"
                                    (opt-stringlist))
   (("grammar")      "0"            (opt-card))
   (("header")       "warn"         (opt-enum "warn" "nowarn" "suggest" "html"))
   (("help" "h")     "no"           (opt-bool))
   (("httphost")     ""             (opt-string))
   (("interpreter")  "/usr/bin/lgwam"
                                    (opt-string))
   (("iterations")   "0"            (opt-card))
   (("jail")         "yes"          (opt-bool))
   (("keyword")      ""             (opt-stringlist))
   (("leap")         *leap*         (opt-leap))
   (("level")        "body"         (opt-enum "parse" "compile" "diagnose"
                                     "body" "codex" "all" "submit"))
   (("lgs")          ""             (opt-string))
   (("lgw")          ""             (opt-string))
   (("link")         ""             (opt-string))
   (("linkresubmit") ""             (opt-string))
   (("linksubmit")   ""             (opt-string))
   (("linktemp")     ""             (opt-string))
   (("localconf")    ""             (opt-string))
   (("localhome")    "/logiweb/"    (opt-dir))
   (("localrelay")   "/logiweb/server/relay/"
                                    (opt-dir))
   (("log")          ""             (opt-dir))
   (("logaccess")    "yes"          (opt-bool))
   (("logdelta")     "1 hour"       (opt-duration))
   (("loginit")      "no"           (opt-bool))
   (("logresponse")  "no"           (opt-bool))
   (("logschedule")  "no"           (opt-bool))
   (("machine")      ""             (opt-string))
   (("mirror")       "no"           (opt-bool))
   (("name")         ""             (opt-ignore))
   (("optidump")     ""             (opt-string))
   (("option")       "no"           (opt-bool))
   (("optionval")    ""             (opt-string))
   (("optionstr")    ""             (opt-string))
   (("org")          ""             (opt-ignore))
   (("pane")         ""             (opt-pane))
   (("pane1")        ""             (opt-ignore))
   (("pane2")        ""             (opt-ignore))
   (("patience")     "1"            (opt-duration))
   (("parenthesize") "no"           (opt-bool))
   (("pid")          ""             (opt-string))
   (("pntdist")      "3 hours"      (opt-duration))
   (("pntfew")       "5"            (opt-card))
   (("pntmany")      "8"            (opt-card))
   (("pntmax")       "10"           (opt-card))
   (("pntrisk")      "10m"          (opt-decadic))
   (("pnttries")     "3"            (opt-card))
   (("pntwait")      "1 minute"     (opt-duration))
   (("post")         ""             (opt-string))
   (("postkeys")     ""             (opt-stringlist))
   (("quit")         "yes"          (opt-bool))
   (("relaychain")   ""             (opt-chain))
   (("renderers")    "/text/binary/tex/latex/bibtex/makeindex/dvipdfm"
                                    (opt-stringlist))
   (("reqbaud")      "100k"         (opt-decadic))
   (("reqbyte")      "10"           (opt-card))
   (("reqfate")      ",1,1"         (opt-cardlist))
   (("reqheader")    "28"           (opt-card))
   (("reqid")        ",1,4,1"       (opt-cardlist))
   (("reqminfifo")   "1k"           (opt-decadic))
   (("reqsource")    ",9,1"         (opt-cardlist))
   (("reqtrust")     ",4,1,1"       (opt-cardlist))
   (("reqwait")      "100m"         (opt-duration))
   (("roots")        ""             (opt-roots))
   (("spy")          "no"           (opt-bool))
   (("spydepth")     "-2"           (opt-card -2))
   (("spylength")    "-2"           (opt-card -2))
   (("src")          "yes"          (opt-string))
   (("starters")     ""             (opt-serverlist))
   (("tcphost")      ""             (opt-string))
   (("tcpport")      "65535"        (opt-card 0 65535))
   (("tcpchain")     ""             (opt-chain))
   (("tcpip")        "0.0.0.0"      (opt-string))
   (("test")         "yes"          (opt-bool))
   (("timeexp")      "2"            (opt-card))
   (("tracecompile") ""             (opt-stringlist))
   (("trustref")     ""             (opt-bool))
   (("udphost")      ""             (opt-string))
   (("udpport")      "65535"        (opt-card 0 65535))
   (("udpchain")     ""             (opt-chain))
   (("udpip")        "0.0.0.0"      (opt-string))
   (("unfit")        "no"           (opt-bool))
   (("uninstall")    "no"           (opt-bool))
   (("url")          "*missing*"    (opt-url))
   (("user")         "logiweb"      (opt-string))
   (("userconf")     ""             (opt-string))
   (("varbin")       ""             (opt-dir))
   (("varconf")      ""             (opt-string))
   (("varcreate")    ""             (opt-string))
   (("varhome")      ""             (opt-string))
   (("varhttp")      ""             (opt-string))
   (("varinit")      ""             (opt-string))
   (("varjail")      ""             (opt-dir))
   (("varlgc")       ""             (opt-string))
   (("varlgwam")     ""             (opt-string))
   (("varlgwping")   ""             (opt-string))
   (("varlgwrelay")  ""             (opt-string))
   (("varlib")       ""             (opt-string))
   (("varlogiweb")   ""             (opt-string))
   (("varman")       ""             (opt-string))
   (("varrelay")     ""             (opt-string))
   (("varscript")    ""             (opt-string))
   (("varsubmit")    ""             (opt-string))
   (("verbose")      "0"            (opt-card -1 2))
   (("version" "v")  "no"           (opt-bool))
   (("wikisubmit")   ""             (opt-string)))))

#|
=============================================
Option processing order
=============================================
The processing order is represented by a list of (name action) structures.

The name is the name of an option source (e.g. 'conf' for options that come from the main configuration file).

The action indicates how options should be collected from the given source. Actions are collected by evaluating (apply (car action) pool (cdr action)) which is supposed to return a new pool enriched with the given actions at address (:option :source name)
|#

(deff pool-add-process (pool)
 (pool-add pool '(:option :process)
 '(("default"   (read-default))
   ("final"     (read-final))
   ("arg"       (read-arg))
   ("cmd"       (read-cmd))
   ("env"       (read-env))
   ("conf"      (read-config))
   ("userconf"  (read-config "HOME"))
   ("localconf" (read-config "PWD"))
   ("post"      (read-post)))))

#|
=============================================
Option priority
=============================================
The option priority indicates the priority among different sources of options. As an example, the list below states that options from "userconf" take precedence over options from "conf".
|#

(deff pool-add-priority (pool)
 (pool-add pool '(:option :priority)
  '("final" "arg" "cmd" "env" "post" "localconf" "userconf" "conf" "default")))

#|
=============================================
Option priority
=============================================
|#

(deff pool-add-option-info (pool)
 (:let pool (pool-add-syntax pool))
 (:let pool (pool-add-process pool))
 (:let pool (pool-add-priority pool))
 pool)

#|
=============================================
Get raw option
=============================================
|#

(deff option-string-get (name pool)
 (option-string-get1 name (pool-get* pool :option :priority) pool))

(deff option-string-get1 (name prio* pool)
 (:when (atom prio*) (error "Internal error. Option ~s not found" name))
 (:let (prio . prio*) prio*)
 (:let val (pool-get* pool :option :string prio name))
 (:when val val)
 (option-string-get1 name prio* pool))

(deff check-string (string name)
 (:when (stringp string) nil)
 (complain "Option ~a requires a string argument" name))

(deff checked-string-get (name pool)
 (:let string (option-string-get name pool))
 (check-string string name)
 string)

#|
=============================================
Get translated option
=============================================
|#

(deff option (name &optional (pool *pool*))
 (when (null (option-string-get name pool))
  (error "Internal check failed. Invalid option: ~s" name))
 (pool-get* pool :option :value name))

#|
=============================================
Get level option
=============================================
|#

(deff get-levels (pool)
 (:when (atom pool) (error "Internal error: Level not found"))
 (:let (((name) :init (:fct . arg*)) . pool) pool)
 (:when (equalp name "level") arg*)
 (get-levels pool))

(deff level (cmp string &optional (pool *pool*))
 (:let levels (get-levels (pool-get* *pool* :option :syntax)))
 (:let level (pool-get* pool :option :value "level"))
 (:let level1 (position level levels :test 'equalp))
 (:let level2 (position string levels :test 'equalp))
 (funcall cmp level1 level2))

#|
=============================================
Get verbose option
=============================================
|#

(deff verbose (cmp verbose2)
 (:let verbose1 (option "verbose"))
 (funcall cmp verbose1 verbose2))

#|
=============================================
Set options
=============================================
(read-options pool args) collects options from all sources, including 'args' which come from the invokation from the read-eval-print loop.

(option-add key val result) adds the given key/val pair to result unless the given key is already defined in result in which case option-add complains.
|#

(defc *read-eval-print-args* nil)

(deff init-pool (args)
 (setq *pool* (read-options (pool-add-option-info nil) args))
 (pool-check (pool-get* *pool* :option :string)))

(deff read-options (pool args)
 (setq *read-eval-print-args* args)
 (:let process (pool-get* pool :option :process))
 (:let pool (read-options1 process pool))
 (:let syntax (pool-get* pool :option :syntax))
 (:let pool (read-options2 syntax pool))
 pool)

(deff read-options1 (process pool)
 (:when (atom process) pool)
 (:let ((name (fct . arg*)) . process) process)
 (:let pool (apply fct pool name arg*))
 (read-options1 process pool))

(deff read-options2 (syntax pool)
 (:when (atom syntax) pool)
 (:let (((name) :init (fct . arg*)) . syntax) syntax)
 (:let pool (apply fct pool name arg*))
 (read-options2 syntax pool))

(deff pool-check (pool)
 (:when (atom pool) nil)
 (:let ((source . pool1) . pool) pool)
 (pool-check1 source pool1)
 (pool-check pool))

(deff pool-check1 (source pool)
 (:when (atom pool) nil)
 (:let ((name . :value) . pool) pool)
 (pool-check2 source name)
 (pool-check1 source pool))

(deff pool-check2 (source name)
 (:when (pool-check3 name (pool-get* *pool* :option :syntax)) nil)
 (format t "Warning. In ~a: Unknown option ~a~%" source name))

(deff pool-check3 (name pool)
 (:when (atom pool) nil)
 (:let ((name* :default :syntax) . pool) pool)
 (:when (member name name* :test 'equalp) t)
 (pool-check3 name pool))

#|
=============================================
Set default options
=============================================
Default values of options are set to the values of the init-forms given in the option syntax.

|#

(deff read-default (pool name)
 (:catch () (complain "Internal error: Error when scanning default values"))
 (:let syntax (pool-get* pool :option :syntax))
 (:let option* (read-default1 syntax nil))
 (pool-put* pool option* :option :string name))

(deff read-default1 (syntax result)
 (:when (atom syntax) result)
 (:let (((name) init) . syntax) syntax)
 (:let string (eval init))
 (:let result (option-add name string result))
 (read-default1 syntax result))

#|
=============================================
Set final options
=============================================
No final options are defined initially. Options such as 'userconf' who get their final value before all sources are processed get their value inserted in the list of final options at the time their value is fixed. The read-final function just clears the list of final options.
|#

(deff read-final (pool name)
 (pool-put* pool nil :option :string name))

#|
=============================================
Set options from read-eval-print loop
=============================================
|#

(deff read-arg (pool name)
 (:catch () (complain "The error was found in the function arguments"))
 (:let option* (read-arg1 *read-eval-print-args* nil))
 (pool-put* pool option* :option :string name))

(deff read-arg1 (arg* result)
 (:when (atom arg*) result)
 (:let (arg . arg*) arg*)
 (:when (keywordp arg) (read-arg-keyword arg arg* result))
 (:when (stringp arg) (read-arg-string arg arg* result))
 (complain "Neither string nor keyword: ~s" arg))

(deff read-arg-keyword (name arg* result)
 (:when (atom arg*) (complain "Keywords and values must occur pairwise"))
 (:let (value . arg*) arg*)
 (:let value (read-arg-value value))
 (:let name (symbol-name name))
 (:let result (option-add name value result))
 (read-arg1 arg* result))

(deff read-arg-value (value)
 (:when (equalp value t) t)
 (:when (equalp value nil) "")
 (format nil "~a" value))

(deff read-arg-string (value arg* result)
 (:when (pool-get* result "lgs") (complain "More than one lgs file: ~a" value))
 (:let result (option-add "lgs" value result))
 (read-arg1 arg* result))

#|
=============================================
Set options from command line arguments
=============================================
|#

(deff read-cmd (pool name)
 (:catch () (complain "Error was found when scanning command line options"))
 (:let option* (read-cmd1 *args* nil))
 (pool-put* pool option* :option :string name))

(deff read-cmd1 (arg* result)
 (:when (atom arg*) result)
 (:let (arg . arg*) arg*)
 (:when (string-prefix "--no" arg)
  (:let name (subseq arg 4))
  (:let value "")
  (:let result (option-add name value result))
  (read-cmd1 arg* result))
 (:when (string-prefix "-no" arg)
  (:let name (subseq arg 3))
  (:let value "")
  (:let result (option-add name value result))
  (read-cmd1 arg* result))
 (:when (string-prefix "--" arg) (read-cmd2 (subseq arg 2) arg* result))
 (:when (string-prefix "-" arg) (read-cmd2 (subseq arg 1) arg* result))
 (:let position (position #\= arg))
 (:when position
  (:let name (subseq arg 0 position))
  (:let value (subseq arg (+ position 1)))
  (:let result (option-add name value result))
  (read-cmd1 arg* result))
 (:let result (option-add "lgs" arg result))
 (read-cmd1 arg* result))

(deff read-cmd2 (name arg* result)
 (:when (atom arg*) (option-add name t result))
 (:let (value) arg*)
 (:when (string-prefix "-" value) (read-cmd1 arg* (option-add name t result)))
 (:when (position #\= value) (read-cmd1 arg* (option-add name t result)))
 (:let (:arg . arg*) arg*)
 (read-cmd1 arg* (option-add name value result)))

#|
=============================================
Set options from environment variables
=============================================
|#

(deff read-env (pool name)
 (:catch () (complain "Error was found when scanning environment variables"))
 (:let option* (read-env1 (getenv) nil))
 (pool-put* pool option* :option :string name))

(deff read-env1 (arg* result)
 (:let prefix "logiweb_")
 (:when (atom arg*) result)
 (:let ((key . val) . arg*) arg*)
 (:unless (string-prefix prefix key) (read-env1 arg* result))
 (:let key (subseq key (length prefix)))
 (option-add key val result))

#|
=============================================
Set options from configuration file
=============================================
|#

(deff read-config (pool name &optional env)
 (:catch () (complain "Error was found when processing option ~a" name))
 (:let filename (checked-string-get name pool))
 (:let final (pool-get* pool :option :string "final"))
 (:let final (option-add name filename final))
 (:let pool (pool-put* pool final :option :string "final"))
 (:when (equalp filename "") pool)
 (:let env (when env (getenv env)))
 (:let filename (if env (cat (slash env) filename) filename))
 (:let option* (read-config1 filename))
 (pool-put* pool option* :option :string name))

(deff read-config1 (filename)
 (:catch () (complain "Error was found when reading config file ~a" filename))
 (with-open-file
  (stream filename :if-does-not-exist nil)
  (when stream (read-config2 stream nil))))

(deff read-line* (stream result)
 (:let line (read-line stream nil nil))
 (:when (null line) (reverse result))
 (read-line* stream (cons line result)))

(deff read-config2 (stream line*)
 (:let line (read-line stream nil nil))
 (:when (null line) (read-config3 (reverse line*) nil))
 (:let line (string-trim *spaces* line))
 (:when (equalp line "") (read-config2 stream line*))
 (:when (equalp (aref line 0) #\;) (read-config2 stream line*))
 (:unless (equalp (aref line 0) #\-) (read-config2 stream (cons line line*)))
 (:let line (subseq line 1))
 (:let line (string-left-trim *spaces* line))
 (:when (null line*) (complain "File starts with continuation line"))
 (:let (line1 . line*) line*)
 (:let line (cat line1 (coerce '(#\Newline) 'string) line))
 (read-config2 stream (cons line line*)))

(deff read-config3 (line* result)
 (:when (atom line*) result)
 (:let (line . line*) line*)
 (:let position (position #\= line))
 (:when (null position) (complain "Line without equal sign: ~a~%" line))
 (:let key (string-right-trim *spaces* (subseq line 0 position)))
 (:let val (string-left-trim *spaces* (subseq line (+ position 1))))
 (:let result (option-add key val result))
 (read-config3 line* result))

#|
=============================================
Set options from post option
=============================================
|#

(deff read-post (pool name)
 (:catch () (complain "Error was found when scanning option ~a" name))
 (:let post (checked-string-get name pool))
 (:when (equalp post "") pool)
 (:let postkeys (checked-string-get "postkeys" pool))
 (:let postkeys (string2list postkeys))
 (:let option* (read-post1 post 0 postkeys nil))
 (pool-put* pool option* :option :string name))

(deff read-post1 (string pos postkeys result)
 (:let position (position #\& string :start pos))
 (:when (null position) (read-post2 (subseq string pos) postkeys result))
 (:let result (read-post2 (subseq string pos position) postkeys result))
 (read-post1 string (+ position 1) postkeys result))

(defc *post-keys*
'("org" "name" "level" "date" "test" "verbose" "pane1" "pane2"))

(deff read-post2 (string postkeys result)
 (:let position (position #\= string))
 (:when (null position) (complain "Option without equal sign: ~a" string))
 (:let key (subseq string 0 position))
 (:let val (subseq string (+ position 1)))
 (:when (member key postkeys :test 'equalp) (option-add key val result))
 (complain "Invalid key in post string. Valid keys are: ~a" postkeys))

(etst (read-post1 "org=xx" 0 '("name" "org") nil)
 '(("org" . "xx")))
(etst (read-post1 "org=xx&name=y" 0 '("name" "org") nil)
 '(("name" . "y") ("org" . "xx")))
(etst (read-post1 "org=&name=" 0 '("name" "org") nil)
 '(("name" . "") ("org" . "")))
(xtst (read-post1 "org=&name=" 0 '("name") nil))

#|
=============================================
Translate url-encoded string
=============================================
(decode-url string) translates the given url-encoded string. As an example,
(decode-url "ab+%41cd") returns "ab Acd". Returns :error on error.
|#

(deff add-char (char result)
 (:let code (char-code char))
 (:when (and (<debug 31 code) (<debug code 127)) (cons char result))
 (:when (= code 10) (cons char result))
 (:when (= code 9) (cons #\Space result))
 result)

(deff decode-url (string)
 (decode-url1 string 0 nil))

(deff decode-url1 (string pos result)
 (:when (>= pos (length string)) (coerce (reverse result) 'string))
 (:let char (char string pos))
 (:when (equal char #\+)
  (:let char #\Space)
  (:let result (add-char char result))
  (decode-url1 string (+ 1 pos) result))
 (:when (unequal char #\%)
  (:let result (add-char char result))
  (decode-url1 string (+ 1 pos) result))
 (:when (>= (+ pos 2) (length string)) :error)
 (:let char1 (digit-char-p (char string (+ pos 1)) 16))
 (:let char2 (digit-char-p (char string (+ pos 2)) 16))
 (:when (null char1) :error)
 (:when (null char2) :error)
 (:let char (code-char (+ char2 (* 16 char1))))
 (:let result (add-char char result))
 (decode-url1 string (+ pos 3) result))

(etst (decode-url "abc") "abc")
(etst (decode-url "ab+cd") "ab cd")
(etst (decode-url "ab+%41cd") "ab Acd")
(etst (decode-url "%41%42%43") "ABC")
(etst (decode-url "%41%42%4") :error)
(etst (decode-url "%41%42%") :error)

#|
=============================================
Test validity of file name from post formula
=============================================
(invalid-name string) returns the first offending character if the given string is is not a safe file name. As an example, "/etc/passwd" is not a safe file name since it names a file outside the current directory. (invalid-name string) is used for validating file names entered on web forms.
|#

(deff valid-char (char)
 (or
  (alphanumericp char)
  (member char '(#\- #\+ #\%) :test 'equalp)))

(deff invalid-name (string)
 (:let position (position-if-not 'valid-char string))
 (:when (null position) nil)
 (aref string position))

(etst (invalid-name "/etc/passwd") #\/)
(etst (invalid-name "abc.def") #\.)
(ntst (invalid-name "Aa19+-"))

(deff check-name (string name)
 (:let char (invalid-name string))
 (:when char (complain "Invalid character in ~a: ~s" name char)))

#|
=============================================
Translate duration
=============================================
(string2duration string) translates a string like "2 minutes" to (120 0) (i.e. the given duration measured in seconds). The function returns nil if the string does not represent a duration.
|#

(defc seconds-per-minute 60)                 ; seconds per minute
(defc seconds-per-hour   (* 60 seconds-per-minute))
(defc seconds-per-day    (* 24 seconds-per-hour))
(defc seconds-per-week   (*  7 seconds-per-day))
(defc days-per-400-years (+ 97 (* 400 365)))
(defc seconds-per-year   (floor (* seconds-per-day days-per-400-years) 400))
(defc seconds-per-month  (floor seconds-per-year 12))
(defc yocto              (* 3 8))
(defc zepto              (* 3 7))
(defc atto               (* 3 6))
(defc femto              (* 3 5))
(defc pico               (* 3 4))
(defc nano               (* 3 3))
(defc micro              (* 3 2))
(defc milli              (* 3 1))
(defc unit               (expt 1000 0))
(defc kilo               (expt 1000 1))
(defc mega               (expt 1000 2))
(defc giga               (expt 1000 3))
(defc tera               (expt 1000 4))
(defc peta               (expt 1000 5))
(defc exa                (expt 1000 6))
(defc zeta               (expt 1000 7))
(defc yota               (expt 1000 8))

(defc suffix-assoc-decadic
 (list
  (list "y"       1                  yocto)
  (list "z"       1                  zepto)
  (list "a"       1                  atto )
  (list "f"       1                  femto)
  (list "p"       1                  pico )
  (list "n"       1                  nano )
  (list "u"       1                  micro)
  (list "m"       1                  milli)
  (list ""        unit               0    )
  (list "U"       unit               0    )
  (list "k"       kilo               0    )
  (list "M"       mega               0    )
  (list "G"       giga               0    )
  (list "T"       tera               0    )
  (list "P"       peta               0    )
  (list "E"       exa                0    )
  (list "Z"       zeta               0    )
  (list "Y"       yota               0    )))

(defc suffix-assoc-duration
 (append suffix-assoc-decadic
  (list
   (list "second"  1                  0    )
   (list "seconds" 1                  0    )
   (list "minute"  seconds-per-minute 0    )
   (list "minutes" seconds-per-minute 0    )
   (list "hour"    seconds-per-hour   0    )
   (list "hours"   seconds-per-hour   0    )
   (list "day"     seconds-per-day    0    )
   (list "days"    seconds-per-day    0    )
   (list "week"    seconds-per-week   0    )
   (list "weeks"   seconds-per-week   0    )
   (list "month"   seconds-per-month  0    )
   (list "months"  seconds-per-month  0    )
   (list "year"    seconds-per-year   0    )
   (list "years"   seconds-per-year   0    ))))

; One letter suffixes are case sensitive. Longer suffixes are not.
(deff suffix-equal (x y)
 (:when (unequal x y) nil)
 (:when (> (length x) 1) t)
 (equal x y))

(deff string2duration (string)
 (string2duration1 string suffix-assoc-duration))

(deff string2decadic (string)
 (string2duration1 string suffix-assoc-decadic))

(deff string2duration1 (string suffix-assoc)
 (:unless (stringp string) nil)
 (:mlet (value length) (parse-integer string :junk-allowed t))
 (:unless (numberp value) nil)
 (:when (<debug value 0) nil)
 (:let suffix (subseq string length))
 (:let suffix (string-trim *spaces* suffix))
 (:let (:suffix m e) (assoc suffix suffix-assoc :test 'suffix-equal))
 (:when (null m) nil)
 (list (* value m) e))

(ntst (string2duration "abc"))
(ntst (string2duration "123x"))
(ntst (string2duration ""))
(ntst (string2duration "-"))
(ntst (string2duration "-1"))
(etst (string2duration "123")       '(    123 0))
(etst (string2duration "0")         '(      0 0))
(etst (string2duration "2 seconds") '(      2 0))
(etst (string2duration "2 minutes") '(    120 0))
(etst (string2duration "1 hour")    '(   3600 0))
(etst (string2duration "1 day")     '(  86400 0))
(etst (string2duration "1 week")    '( 604800 0))
(etst (string2duration "1week")     '( 604800 0))
(etst (string2duration "1 Week")    '( 604800 0))
(etst (string2duration "1 WEEK")    '( 604800 0))
(etst (string2duration "2m")        '(      2 3))
(etst (string2duration "2M")        '(2000000 0))

#|
=============================================
Translate string option
=============================================
|#

(deff opt-string (pool name)
 (:let string (checked-string-get name pool))
 (pool-put* pool string :option :value name))

#|
=============================================
Translate directory option
=============================================
|#

(deff opt-dir (pool name)
 (:let string (checked-string-get name pool))
 (pool-put* pool (slash string) :option :value name))

#|
=============================================
Translate boolean option
=============================================
|#

(deff opt-bool (pool name)
 (:let string (option-string-get name pool))
 (:let value (opt-bool1 string name))
 (pool-put* pool value :option :value name))

(deff opt-bool1 (string name)
 (:when (member string '("yes" "true" "t" t) :test 'equalp) t)
 (:when (member string '("no" "false" "f" "") :test 'equalp) nil)
 (complain "Option ~a is not a boolean: ~s" name string))

#|
=============================================
Translate option of enumerated type
=============================================
|#

(deff opt-enum (pool name &rest list)
 (:let string (checked-string-get name pool))
 (:unless (member string list :test 'equalp)
  (complain "Value ~s of option ~s does not belong to ~s" string name list))
 (pool-put* pool string :option :value name))

#|
=============================================
Translate duration option
=============================================
|#

(deff opt-duration (pool name)
 (:let string (checked-string-get name pool))
 (:let value (opt-duration1 string name))
 (pool-put* pool value :option :value name))

(deff opt-duration1 (string name)
 (:let value (string2duration string))
 (:when value value)
 (complain "Option ~a is not a duration: ~s" name string))

#|
=============================================
Translate decadic option
=============================================
|#

(deff opt-decadic (pool name)
 (:let string (checked-string-get name pool))
 (:let value (opt-decadic1 string name))
 (pool-put* pool value :option :value name))

(deff opt-decadic1 (string name)
 (:let value (string2decadic string))
 (:when value value)
 (complain "Option ~a is not a decadic value: ~s" name string))

#|
=============================================
Translate ignored option
=============================================
|#

(deff opt-ignore (pool :name)
 pool)

#|
=============================================
Translate cardinal option
=============================================
|#

(deff opt-card (pool name &optional (min 0) (max nil))
 (:let string (checked-string-get name pool))
 (:let value (opt-card1 name string min max))
 (pool-put* pool value :option :value name))

(deff opt-card1 (name string min max)
 (:mlet (card length) (parse-integer string :junk-allowed t))
 (:when (<debug length (length string))
  (complain "Value of ~a not a cardinal: ~a" name string))
 (:when (> min card) (complain "Value of ~a out of range: ~a" name string))
 (:when (null max) card)
 (:when (<debug max card) (complain "Value of ~a out of range: ~a" name string))
 card)

#|
=============================================
Translate page option
=============================================

opt-pane is non-general. It can only process the "pane1" and "pane2" options.

opt-pane decodes and concatenates the "pane1" and "pane2" options into a single "pane" option.
|#

(deff opt-pane (pool :name)
 (:let pane1 (checked-string-get "pane1" pool))
 (:let pane2 (checked-string-get "pane2" pool))
 (:let pane1 (decode-url pane1))
 (:let pane2 (decode-url pane2))
 (:let pane (cat pane1 pane2))
 (pool-put* pool pane :option :value "pane"))

#|
=============================================
Translate url option
=============================================

opt-url is non-general. It can only process the "url", "org", and "name" options. Furthermore, it uses the 'roots' option which happens to be
processed before 'url' (because 'roots' is before 'url' in the
alphabet). This is not very general.
|#

(deff string2url (string pool)
 (:unless (string-prefix "file:" string) string)
 (:let url (filename2url1 (subseq string 5) (option "roots" pool)))
 (:when url url)
 string)

(deff filename2url1 (filename assoc)
 (:when (atom assoc) nil)
 (:let ((filename1 . url) . assoc) assoc)
 (:let suffix (suffix filename1 filename))
 (:when (null suffix) (filename2url1 filename assoc))
 (cat url suffix))

(deff opt-url (pool :name)
 (:let post (checked-string-get "post" pool))
 (:let url (checked-string-get "url" pool))
 (:let url (string2url url pool))
 (:when (equalp post "") (pool-put* pool (slash url) :option :value "url"))
 (:let org (checked-string-get "org" pool))
 (:let name (checked-string-get "name" pool))
 (check-name org "org")
 (check-name name "name")
 (:when (equalp org "") (complain "Empty org"))
 (:when (equalp name "") (complain "Empty name"))
 (:let url (cat (slash url) (slash org) (slash name)))
 (pool-put* pool url :option :value "url"))

#|
=============================================
Translate cardlist option
=============================================
|#

(deff opt-list (pool name translator)
 (:catch () (complain "Error found when processing ~s option" name))
 (:let string (checked-string-get name pool))
 (:let string* (string2list string))
 (:let list (map 'list translator string*))
 (pool-put* pool list :option :value name))

(deff translate-card (string)
 (:mlet (card length) (parse-integer string :junk-allowed t))
 (:when (<debug length (length string))
  (complain "Value not a cardinal: ~a" string))
 card)

(deff opt-cardlist (pool name)
 (opt-list pool name 'translate-card))

(deff opt-stringlist (pool name)
 (opt-list pool name 'identity))

#|
=============================================
Translate string option
=============================================
|#

(deff vector2server (vector)
 (:let (:length . card*) vector)
 (card*2server card*))

(deff card*2server (card*)
 (:let string (card*2string card*))
 (string2server string))

(deff string2server (string)
 (:let pos1 (position #\/ string))
 (:when (null pos1) "Too few slashes")
 (:let string1 (subseq string 0 pos1))
 (:let protocol (cdr (assoc string1 '(("udp" . :udp)) :test 'equalp)))
 (:when (null protocol) "Malformed protocol")
 (:let pos1 (+ 1 pos1))
 (:let pos2 (position #\/ string :start pos1))
 (:when (null pos2) "Too few slashes")
 (:let server (subseq string pos1 pos2))
 (:let pos2 (+ 1 pos2))
 (:let pos3 (position #\/ string :start pos2))
 (:when (null pos3) "Too few slashes")
 (:let port (subseq string pos2 pos3))
 (:mlet (card length) (parse-integer port :junk-allowed t))
 (:when (<debug length (length port)) "Invalid port number")
 (:when (null card) "Invalid port number")
 (:unless (<= 0 card 65535) "Port number out of range")
 (:let relay (subseq string (+ 1 pos3)))
 (list* relay protocol server card))

(deff server2vector (string)
 (:let string (slash string))
 (:let msg (string2server string))
 (:when (atom msg) (complain "~a: ~s" msg string))
 (m-ct2vector string))

(etst (vector2server (cons :garbage (ct2ct "udp/logiweb.eu/65535/abc")))
 '("abc" :udp "logiweb.eu" . 65535))
(etst (vector2server (cons :garbage (ct2ct "udp/logiweb.eu/65536/abc")))
 "Port number out of range")
(etst (server2vector "udp/logiweb.eu/65535/abc")
 (m-ct2vector "udp/logiweb.eu/65535/abc/"))
(xtst (server2vector "xyz/logiweb.eu/65535/abc"))

(deff opt-serverlist (pool name)
 (opt-list pool name 'server2vector))

#|
=============================================
Translate roots option
=============================================
|#

(deff slash* (string*)
 (map 'list 'slash string*))

(deff list2assoc (list)
 (:let (key value . list) list)
 (:when (null value) nil)
 (acons value key (list2assoc list)))

(etst (list2assoc '(1 2 3 4 5 6)) '((2 . 1) (4 . 3) (6 . 5)))
(etst (list2assoc '(1 2 3 4 5)) '((2 . 1) (4 . 3)))

(deff opt-roots (pool name)
 (:let string (checked-string-get name pool))
 (:let list (string2list string))
 (:let list (slash* list))
 (:let assoc (list2assoc list))
 (pool-put* pool assoc :option :value name))

#|
=============================================
Translate leap option
=============================================
|#

(deff opt-leap (pool name)
 (:catch () (complain "Error found when processing ~s option" name))
 (:let string (checked-string-get name pool))
 (:let list (string2list string))
 (:let leap* (string*2leap* list))
 (:let pool (pool-put* pool leap* :option :value name))
 (pool-put-leap pool leap*))

#|
=============================================
Translate chain option
=============================================
|#

(deff string2tuple (string)
 (string2tuple1 string 0))

(deff string2tuple1 (string index)
 (:let index1 (position-if 'digit-char-p string :start index))
 (:when (null index1) (list (string-trim *spaces* (subseq string index))))
 (:mlet (card index2) (parse-integer string :start index1 :junk-allowed t))
 (:let substring (subseq string index index1))
 (:let substring (string-trim *spaces* substring))
 (list* substring card (string2tuple1 string index2)))

(etst (string2tuple "a1b") '("a" 1 "b"))
(etst (string2tuple "ab12cd") '("ab" 12 "cd"))
(etst (string2tuple "12") '("" 12 ""))
(etst (string2tuple " [ 1 , 2 , 3 ] ") '("[" 1 "," 2 "," 3 "]"))

(deff card2mask (card)
 (:unless (<= 0 card 32) nil)
 (:when (= card 0) (list 0 0 0 0))
 (:let bit* (append (repeat (- 32 card) 0) (repeat card 1)))
 (:let card (card*2card 2 bit*))
 (:let card* (card2card* 256 card))
 (reverse card*))

(etst (card2mask 0) '(0 0 0 0))
(etst (card2mask 1) '(128 0 0 0))
(etst (card2mask 2) '(192 0 0 0))
(etst (card2mask 8) '(255 0 0 0))
(etst (card2mask 9) '(255 128 0 0))
(etst (card2mask 24) '(255 255 255 0))
(etst (card2mask 25) '(255 255 255 128))
(etst (card2mask 32) '(255 255 255 255))

(deff string2mask (string)
 (:let (slash mask1 . rest) (string2tuple string))
 (:when (unequal slash "/") nil)
 (:when (equal rest '(""))(card2mask mask1))
 (:let (dot1 mask2 dot2 mask3 dot3 mask4 . rest) rest)
 (:when (unequal dot1 ".") nil)
 (:when (unequal dot2 ".") nil)
 (:when (unequal dot3 ".") nil)
 (:when (unequal rest '("")) nil)
 (list mask1 mask2 mask3 mask4))

(etst (string2mask "/1.2.3.4") '(1 2 3 4))
(etst (string2mask "/1") '(128 0 0 0))
(ntst (string2mask "//1"))
(ntst (string2mask "/1,2,3,4"))
(ntst (string2mask "/1.2.3"))
(ntst (string2mask "/1.2.3.4."))
(ntst (string2mask "/1.2.3.4.5"))

(deff string2trust (string)
 (:let index1 (position #\: string))
 (:when (null index1) nil)
 (:let index2 (position #\/ string :start index1))
 (:let (empty1 trust . empty2) (string2tuple (subseq string 0 index1)))
 (:when (unequal empty1 "") nil)
 (:when (unequal empty2 '("")) nil)
 (:let mask (if index2 (string2mask (subseq string index2)) '(255 255 255 255)))
 (:when (null mask) nil)
 (:let host (string-trim *spaces* (subseq string (+ index1 1) index2)))
 (:let ip (host-ip host))
 (:when (equalp ip :error) nil)
 (:let ip (string2tuple (ct2string ip)))
 (:let (:empty ip1 :dot ip2 :dot ip3 :dot ip4 :empty) ip)
 (:let ip (list ip1 ip2 ip3 ip4))
 (list trust ip mask))

(etst (string2trust "1: 2.3.4.5") '(1 (2 3 4 5) (255 255 255 255)))
(etst (string2trust "1: 2.3.4.5/1") '(1 (2 3 4 5) (128 0 0 0)))
(etst (string2trust "1: 2.3.4.5/24") '(1 (2 3 4 5) (255 255 255 0)))
(etst (string2trust "1: 2.3.4.5/6.7.8.9") '(1 (2 3 4 5) (6 7 8 9)))
(ntst (string2trust "1; 2.3.4.5"))
(ntst (string2trust "1: 2.3.4.5//1"))
(ntst (string2trust "1: 2.3.4.5/1,2,3,4"))
(ntst (string2trust "1: 2.3.4.5/1.2.3"))
(ntst (string2trust "1: 2.3.4.5/1.2.3.4."))
(ntst (string2trust "1: 2.3.4.5/1.2.3.4.5"))

(deff string*2chain (string*)
 (:when (atom string*) nil)
 (:let (string . string*) string*)
 (:let trust (string2trust string))
 (:when (null trust) (complain "Malformed chain item: ~s" string))
 (cons trust (string*2chain string*)))

(deff opt-chain (pool name)
 (:catch () (complain "Error found when processing ~s option" name))
 (:let string (checked-string-get name pool))
 (:let string* (string2list string))
 (:let chain (string*2chain string*))
 (pool-put* pool chain :option :value name))

#|
=============================================
Translate url to corresponding file name
=============================================
|#

(deff url2filename (url)
 (url2filename1 url (option "roots")))

(deff url2filename1 (url assoc)
 (:when (atom assoc) nil)
 (:let ((filename . url1) . assoc) assoc)
 (:let suffix (suffix url1 url))
 (:when (null suffix) (url2filename1 url assoc))
 (cat filename suffix))

#|
=============================================
Default host name handling
=============================================
|#

(deff option-default (x y)
 (if (equalp y "") x y))

(deff option-udphost ()
 (option-default (option "httphost") (option "udphost")))

(deff option-tcphost ()
 (option-default (option "httphost") (option "tcphost")))

(deff url-relay (&rest ct)
 (ct2string (list* "http://" (option "httphost") (option "localrelay") ct)))

(deff url-home (&rest ct)
 (ct2string (list* "http://" (option "httphost") (option "localhome") ct)))

(deff url-localrelay (&rest ct)
 (ct2string (list* (option "localrelay") ct)))

(deff url-localhome (&rest ct)
 (ct2string (list* (option "localhome") ct)))

#|
=============================================
Print options
=============================================
|#

(deff print-options (source format &rest arg*)
 (apply 'format t format arg*)
 (terpri)
 (terpri)
 (:let list (pool-get* *pool* :option :string source))
 (:when (atom list) (format t "(none)~%~%"))
 (dolist (item list) (format t "~9a = ~a~%" (head item) (tail item)))
 (terpri))

(deff print-option ()
 (print-options "arg"       "Funcall options")
 (print-options "cmd"       "Command line options")
 (print-options "env"       "Environment options")
 (print-options "post"      "Post options")
 (print-options "localconf" "From local configuration file (in $PWD)")
 (print-options "userconf"  "From user configuration file (in $HOME)")
 (print-options "conf"      "From site configuration file")
 (print-options "default"   "Compiled in"))

(deff print-optionval (name)
 (format t "~s~%" (option name)))

(deff print-optionstr (name)
 (format t "~a~%" (option-string-get name *pool*)))

#|
=============================================
Determine *conf*
=============================================
|#

(defc init-args '(:localconf "" :userconf ""))
;(defc init-args nil)

(deff determine-conf ()
 (:when (boundp '*conf*) nil)
 (setq *conf* (cat *main-dir* "logiweb.conf"))
;(format t "Processing ~s~%" *conf*)
 (init-pool init-args)
 (:let varconf (option "varconf"))
 (:when (null (probe-file varconf)) nil)
 (setq *conf* varconf)
;(format t "Processing ~s~%" *conf*)
 (init-pool init-args))

#|
=============================================
Set options for the sake of test and make
=============================================
|#

(determine-conf)













