#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb System
=============================================
Clisp make file
=============================================
Usage:
clisp make                 = make all
clisp make code            = compile everything
clisp make enhancement     = pre-load standard pages into Logiweb compiler
clisp make most            = make code + make enhancement
clisp make all             = make code + re-render pages + make enhancement
clisp make newpages        = re-compile pages (old timestamps are lost forever)
clisp make clean           = clean up after make most
clisp make distclean       = clean up after make all
clisp make manual          = translate man pages to html using rman
clisp make wrap            = add standard wrappers to html documentation
clisp make maintainerclean = clean up after make all + make manual
clisp make install         = install everything
clisp make uninstall       = remove installation
clisp make dist            = make distclean + make distribution tar-ball
clisp make newdist         = set version number
clisp make test            = run test suite (Logiweb must be installed first)
clisp make logiweb         = make logiweb server
clisp make lgc             = make logiweb compiler
|#

(in-package "COMMON-LISP-USER")
(load     "common/test")
(testload "common/def"   :prompt nil :test nil)
(testload "common/maker" :prompt nil :test nil)
(testload "common/conf"  :prompt nil :test nil)

#|
=============================================
clisp make all
=============================================
|#

(deff make-all ()
 (make-code)
 (make-pages)
 (format t "~%")
 (format t "------------------------------------------------------------~%")
 (format t "~%")
 (format t "To install:~%")
 (format t "> su~%")
 (format t "> make install~%")
 (format t "~%")
 (format t "If installing as root is too frightening, omit 'su', execute~%")
 (format t "'make install', and skip each step. That will show what the~%")
 (format t "installer intends to do.~%")
 (format t "~%"))

#|
=============================================
clisp make most
=============================================
Recompile and generate enhanced lgc
|#

(deff make-most ()
 (make-code)
 (make-enhancement))

#|
=============================================
clisp make code
=============================================
|#

(deff make-code ()
 (mk-option-load)
 (:unless (check-needed-options) nil)
;(make-jail)
 (make-logiweb)
 (make-lgc))

#|
=============================================
clisp make lgc-only
=============================================
|#

(deff make-lgc-only ()
 (mk-option-load)
 (:unless (check-needed-options) nil)
 (in-directory "lgc" (run-clisp "make" "lgc-only")))

#|
=============================================
clisp make logiweb
=============================================
|#

(deff make-logiweb ()
 (in-directory "server" (run-clisp "make")))

#|
=============================================
clisp make lgc
=============================================
|#

(deff make-lgc ()
 (in-directory "lgc" (run-clisp "make")))

#|
=============================================
clisp make jail
=============================================
|#

(deff make-jail ()
 (in-directory "jail" (run-clisp "make")))

#|
=============================================
clisp make install
=============================================
|#

(deff redist (release-dir gz release)
 (:let gzname (cat "logiweb-" *logiweb-version* ".tar.gz"))
 (:let tarname (cat "logiweb-" *logiweb-version* ".tar"))
 (when (probe-file release)
  (run-cp release release-dir)
  (in-directory release-dir
   (when gz (run-gzip tarname))
   (run-rm "-f" "latest.tar.gz")
   (run-ln "-s" gzname "latest.tar.gz"))
  (simple-html-dir (slash release-dir))))

(deff install-link1 (dir user group)
 (install-link2 (cat dir "page/base/") user group)
 (install-link2 (cat dir "page/check/") user group)
 (install-link2 (cat dir "page/Peano/") user group))

(deff install-link2 (dir user group)
 (install-link3 dir "fixed" "latest" user group)
 (install-link3 dir "lgwdir.html" "index.html" user group))

(deff install-link3 (dir path link user group)
 (:let path (cat dir path))
 (:let link (cat dir link))
 (run-rm "-f" link)
 (run-ln "-s" path link)
 (when user (set-owner link user group)))

(deff install-doc (docdir)
 (format t "Installing, have patience ...~%")
 (:let user (option "user"))
 (:let group user)
 (:let release-dir (cat docdir "release/download"))
 (unless (probe-directory docdir) (install-dir1 docdir "755" user group))
 (install-file-tree1 "doc/" docdir "u=rwX,go=rX" user group)
 (install-file1 "README" (cat docdir "release/readme") "644" user group)
 (install-link1 docdir user group)
 (:let mask (if (equalp (option "varsubmit") "") "755" "777"))
 (run-chmod mask (cat docdir "wiki/"))
 (run-chmod mask (cat docdir "cache/"))
 (redist release-dir   t (cat "logiweb-" *logiweb-version* ".tar"))
 (redist release-dir nil (cat "logiweb-" *logiweb-version* ".tar.gz"))
 (redist release-dir   t (cat "../logiweb-" *logiweb-version* ".tar"))
 (redist release-dir nil (cat "../logiweb-" *logiweb-version* ".tar.gz")))

(deff make-install ()
 (:catch () (quit))
 (mk-option-load)
 (:unless (check-needed-options) nil)
 (:let user (option "user"))
 (:when (equalp 1 (run-program "id" :output nil :arguments (list user)))
  (format t "Create user ~s or change user= in ~s" user *conf*))
 (prompt-info "install of Logiweb" "root")
 (format t "~%")
 (format t "------------------------------------------------------------~%")
 (install-file
  "Install Logiweb abstract machine"
  "lgc/build/lgwam" "varlgwam" "755" "root")
 (install-file
  "Install lgc compiler"
  "lgc/lgc2" "varlgc" "755" "root")
 (install-file
  "Install shared object needed by compiler and server"
  "common/logiweb.so" "varlib" "755" "root")
 (install-run "Stop logiweb server" "/sbin/service" "logiweb" "stop")
 (install-file
  "Install Logiweb server"
  "server/logiweb" "varlogiweb" "755" "root")
 (install-file
  "Install init script for managing server by /sbin/service and /sbin/chkconfig"
  "server/boot" "varinit" "755" "root")
 (if (probe-file (option "varconf"))
  (progn
   (format t "~%")
   (format t "   Config file ~a exists and will not been touched~%" *conf*)
   (format t "   If needed, edit it manually or replace it by logiweb.conf~%"))
  (install-file
   "Install site configuration file for server"
   "logiweb.conf" "varconf" "644" "root"))
 (:let docdir (slash (option "varhome")))
 (format t "~%")
 (format t "   Install online documentation in ~a~%" docdir)
 (unless (skipped-make) (install-doc docdir))
 (:let logdir (slash (option "log")))
 (if (probe-directory logdir)
  (progn
   (format t "~%")
   (format t "   Log file directory ~a~%" logdir)
   (format t "   exists and will not been touched~%" *conf*))
  (install-dir
   "Install log file directory for Logiweb server"
   logdir "755" user))
 (install-run
  "Start logiweb server"
  "/sbin/service" "logiweb" "start")
 (install-run
  "Make logiweb server start at next boot"
  "/sbin/chkconfig" "logiweb" "reset")
 (install-file
  "Install lgwping command that allows to ping the Logiweb server"
  "server/lgwping" "varlgwping" "755" "root")
#|
 (install-file*
  "Install programs for rendering in a chroot jail." "jail/"
  '("lgwlatex" "lgwbibtex" "lgwmakeindex" "lgwdvipdfm" "lgwmkjail" "lgwrmjail")
  "varbin" "6755" "root")
 (clean-install-file-tree
  "Install chroot jail template"
  "jail/jail/" "varjail" #o022)
|#
 (install-file
  "Install the CGI script that provides the user interface of the server"
  "server/relay" "varrelay" "755" "root")
 (install-file
  "Install command used by the CGI script"
  "server/lgwrelay" "varlgwrelay" "755" "root")
 (install-file
  "Install the CGI script for Logiwiki submission"
  "lgc/submit" "varsubmit" "755" "root")
 (install-file
  "Install the CGI script for Logiwiki header creation"
  "lgc/create" "varcreate" "755" "root")
 (install-file-tree
  "Install man pages"
  "doc/troff/" "varman" "u=rwX,go=rX" "root")
 (install-file
  "Install configuration file for http server"
  "server/http" "varhttp" "644" "root")
 (install-run "Reinitialize http server" "/sbin/service" "httpd" "reload")
 (format t "~%")
 (format t "------------------------------------------------------------~%")
 (format t "~%")
 (format t "If you have plenty of time:~%")
 (format t "> su yourself~%")
 (format t "> make test~%")
 (format t "~%")
 (format t "Otherwise just look up ~a~%" (url-home))
 (format t "Have fun~%")
 (format t "~%"))

#|
=============================================
clisp make uninstall
=============================================
Once sources are gone, one can still uninstall using 'lgc uninstall=true'.
|#

(deff make-uninstall ()
 (mk-option-load)
 (:unless (check-needed-options) nil)
 (logiweb-uninstall))

#|
=============================================
clisp make manual
=============================================
|#

(deff make-manual ()
 (run-rman* "doc/troff/" "../man/")
 (simple-html-dir "doc/man/"))

#|
=============================================
clisp make wrap
=============================================
|#

(deff make-wrap ()
 (testload "common/auxil" :prompt nil :test nil)
 (testload "common/wrap" :prompt nil :test nil)
 (add-formats "doc"))

#|
=============================================
clisp make site
=============================================
|#

(deff confirm ()
 (format t "Answer yes to confirm~%")
 (format t "> ")
 (:let confirm (read-line))
 (:let confirm (string-trim " " confirm))
 (equalp "yes" confirm))

(defc *pagelist* '("base" "check" "Peano" "combinations" "multzero"))

(deff make-pages ()
 (mk-pages :lgw))

(deff mk-newpages (page)
 (:let from (cat "doc/page/"  page "/fixed/vector/page.lgw"))
 (:let to (cat "lgc/" page ".lgw"))
 (run-cp "-f" from to))

(deff make-newpages ()
 (make-most)
 (mk-pages :lgs))

(deff local-lgc-options ()
 (list
  "dest=$PAGE$/fixed"
  "level=submit"
  "cache=doc/cache"
  "roots=+file:doc/+doc/"
  "url=file:doc/page"
  "varhome=../../../../.."
  "link="))

(deff run-local-lgc (source name options)
 (:let lgs (cat "lgs=lgc/" name))
 (:let lgw (cat "lgw=file:lgc/" name))
 (:let src (cat "src=file:lgc/" name))
 (:when (equalp source :lgs)
  (run-program "./lgc/lgc" :arguments (list* lgs options)))
 (run-program "./lgc/lgc" :arguments (list* lgw src options)))

(deff mk-pages (source)
 (:catch () (quit))
 (mk-option-load)
 (:unless (check-needed-options) nil)
 (:let options (local-lgc-options))
 (mk-pages1 source *pagelist* options)
 (format t "~%Translating page ~s~%~%" "dump")
 (make-enhancement))

(deff mk-pages1 (source pages options)
 (:when (atom pages) nil)
 (:let (page . pages) pages)
 (format t "~%Translating page ~s~%~%" page)
 (:let source (if (probe-file (cat "lgc/" page ".lgw")) source :lgs))
 (run-local-lgc source page options)
 (when (equalp source :lgs) (mk-newpages page))
 (mk-pages1 source pages options))

#|
=============================================
clisp make dist
=============================================
|#

(deff make-enhancement ()
 (:let lgs "lgs=lgc/dump")
 (:let dump "dump=lgc/lgc1")
 (:let options (local-lgc-options))
 (run-program "./lgc/lgc" :arguments (list* lgs dump options))
 (run-mv "-f" "lgc/lgc1" "lgc/lgc"))

#|
(deff make-enhancement ()
 (apply 'run-lgc "lgc=lgc/dump" "dump=lgc/lgc1" (local-lgc-options))
 (run-mv "-f" "lgc/lgc1" "lgc/lgc"))
|#

#|
=============================================
clisp make dist
=============================================
|#

(deff make-dist ()
 (make-distclean)
 (:let dir (cat "logiweb-" *logiweb-version*))
 (:let tar (cat dir ".tar"))
 (run-ln "-s" "." dir)
 (:let self (cat dir "/" dir))
 (run-tar "-cf" tar "-h" "--exclude" tar "--exclude" self dir)
 (run-gzip tar)
 (run-rm dir))

#|
=============================================
clisp make newdist
=============================================
|#

(deff mk-version (version)
 (ct2file "common/conf.lisp"
  (list
   "; Logiweb, Copyright (C) 2004-2010 Klaus Grue" 10
   "; Available under GNU GPL" 10
   "; Logiweb comes with ABSOLUTELY NO WARRANTY" 10
   "(defc *logiweb-version* \"" version "\")" 10)))

(deff make-newdist ()
 (:catch () (quit))
 (mk-option-load)
 (format t "~%")
 (format t "Current version: ~a~%" *logiweb-version*)
 (format t "Enter new version number or hit return~%")
 (format t "> ")
 (:let version (read-line))
 (:let version (string-trim " " version))
 (when (unequal version "") (mk-version version))
 (format t "~%")
 (format t "------------------------------------------------------------~%")
 (format t "~%")
 (format t "Now edit CHANGELOG and do 'make dist'~%"))

#|
=============================================
clisp make clean
=============================================
|#

(deff make-clean ()
 (delete-patterns "*.fas" "*.lib" "*.so" "*.h" "*.tar" "*.gz")
 (in-directory "lgc" (run-clisp "make" "clean"))
 (in-directory "server" (run-clisp "make" "clean"))
;(in-directory "jail" (run-clisp "make" "clean"))
 (in-directory "common"
  (delete-patterns "*.fas" "*.lib" "*.so" "*.h")))

(deff make-distclean ()
 (make-clean)
 (run-rm "-f" "doc/page/index.html")
 (delete-patterns
  "doc/dll/*"
  "doc/page/*"
  "doc/page/*/"
  "doc/cache/*/"
  "doc/man/*"
  "doc/man/*/")
 (ext:shell "rm -f doc/cache/*"))

#|
=============================================
Make this a make file
=============================================
(exec-make) turns a shell command like "clisp make clean" into an
invokation of (make-clean) inside clisp
|#

(exec-make)
