package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut


__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8*
M``3\KU$``````````/``+@(+`@(7`$(```#@````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````"``0``!```180!``,``(``
M`"`````````0```````````0````````$``````````````0`````"`!`*<!
M````,`$`D`L```!``0#((P```.```.P!``````````````!P`0`P````````
M````````````````````````````````````````````````````````````
M_#(!`)@"```````````````````````````````````N=&5X=````.!!````
M$````$(````$``````````````````!@`%!@+F1A=&$```!H`````&`````"
M````1@``````````````````0`!@P"YR9&%T80``G&(```!P````9````$@`
M`````````````````$``4$`N<&1A=&$``.P!````X`````(```"L````````
M``````````!``#!`+GAD871A``#$`0```/`````"````K@``````````````
M````0``P0"YB<W,``````!(``````0```````````````````````````(``
M8,`N961A=&$``*<!````(`$```(```"P``````````````````!``#!`+FED
M871A``"0"P```#`!```,````L@``````````````````0``PP"YR<W)C````
MR",```!``0``)````+X``````````````````$``,,`N<F5L;V,``#``````
M<`$```(```#B``````````````````!``#!"````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0V5/P``
MZ%`W``!%,<`QTC')Z&0W``!%,<`QTC')Z&@W``!%,<`QTC')Z&PW``!%,<`Q
MTC')2(/$*.EL-P``D)"0D)"0D)"0D)"054B)Y5W#9BX/'X0``````%5(B>5(
M@^P@2(,]\$\```!T0$B-#8=?``#_%?TD`0!(A<!(C17'____=!-(C15\7P``
M2(G!_Q7I)`$`2(G"2(72=`](C0VV3P``2(/$(%U(_^)(@\0@7<,/'X0`````
M`%5(B>5=PY"0D)"0D)"0D)!!5T%6055!5%575E-(@>R(`@``13'`3(U9.$F)
MX0\?`$N+%`-(B=!)B=)(P>`(2<'B&"4``/\`18G23`G028G22<'J"$&!X@#_
M``!,"=!)B=))P>H810^VTDP)T$F)TDG!ZB!+B01!3(G02<'B&$C!X`A%B=(E
M``#_`$P)T$F)TDC!ZCA)P>HH08'B`/\``$P)T$@)T$N)1$$(28/`"$F#^$`/
MA7C___])C5%H28'!:`(``$B+0MA(,P)(@\((2#-"H$@S0I!)B<!(`<!)P>@?
M20G`08/@_TR)0A!,.<IUTTB+$4R+:2!,BUD(2(M9$$R+81A-B>A(B=!,`P0D
M28G12,'H&TR)W4G!X05(P>T"38GB3`G(3`-4)`A)B=&)P$GWT4F)UDF-M`"9
M>8):3(G828G82/?032'822'93"'@2<'N`DB)WTP)P$@#?"002`'&3(G82,'@
M'D&)\$@)Q8GM2(GH2"'03`G(38G!38V4`IEY@EI,B<!)P>D;2,'@!4D)P4B)
MT$C!X!Y%B<E)"<9,B<!-`<I%B?9(]]!%B=%,(?9((>A("?!(C;0'F7F"6DR)
MR$R)STC!YP5(P>@;2`GX3(G'2<'@'HG`2,'O`D@!QDP)QT@#;"0838G(B?^)
M\$GWT$DA^DTA\$P#="0@30G03HV4!9EY@EI)B<!(B<5(P>4%2<'H&TD)Z$R)
MS4G!X1Y(P>T"18G`3`G-28G!30'"B>U)]]%%B=!)(?E((>Y(`WPD*$D)\4N-
MM`Z9>8):38G!38G&2<'F!4G!Z1M-"?%)B<9(P>`>2<'N`D6)R4D)QDR)P$P!
MSD6)]DCWT$&)\4TA\D@AZ$P)T$R-E`>9>8):3(G(3(G/2,'G!4C!Z!M("?A,
MB<=)P>`>B<!(P>\"20'"3`G'2`-L)#!-B<B)_T2)T$GWT$@A_DTA\$P#="0X
M20GP2HVT!9EY@EI)B<!(B<5(P>4%2<'H&TD)Z$R)S4G!X1Y(P>T"18G`3`G-
M28G!3`'&B>U)]]%!B?!)(?E)(>I(`WPD0$T)T4^-E`Z9>8):38G!38G&2<'F
M!4G!Z1M-"?%)B<9(P>`>2<'N`D6)R4D)QDR)P$T!RD6)]DCWT$6)T4PA]D@A
MZ$@)\$B-M`>9>8):3(G(3(G/2,'G!4C!Z!M("?A,B<=)P>`>B<!(P>\"2`'&
M3`G'2`-L)$A-B<B)_XGP2??022'Z32'P3`-T)%!-"=!.C90%F7F"6DF)P$B)
MQ4C!Y05)P>@;20GH3(G-2<'A'DC![0)%B<!,"<U)B<%-`<*)[4GWT46)T$DA
M^4@A[D@#?"1820GQ2XVT#IEY@EI-B<%-B<9)P>8%2<'I&TT)\4F)QDC!X!Y)
MP>X"18G)20G&3(G`3`'.18GV2/?008GQ32'R2"'H3`G03(V4!YEY@EI,B<A,
MB<](P><%2,'H&T@)^$R)QTG!X!Z)P$C![P))`<),"<=(`VPD8$V)R(G_1(G0
M2??02"'^32'P3`-T)&A)"?!*C;0%F7F"6DF)P$B)Q4C!Y05)P>@;20GH3(G-
M2<'A'DC![0)%B<!,"<U)B<%,`<:)[4GWT4&)\$DA^4DAZD@#?"1P30G13XV4
M#IEY@EI-B<%-B<9)P>8%2<'I&TT)\4F)QDC!X!Y)P>X"18G)20G&3(G`30'*
M18GV2/?018G13"'V2"'H2`GP3(G.2(V\!YEY@EI,B<A(P>8%2,'H&T@)\$R)
MQDG!X!Z)P$C![@)(`<=,"<9(`VPD>$V)R(GVB?A)]]!)(?)-(?!,`[0D@```
M`$T)T$Z-E`69>8):28G`2(G%2,'E!4G!Z!M)">A,B<U)P>$>2,'M`D6)P$P)
MS4F)P4T!PHGM2??118G022'Q2"'O2`.T)(@```!)"?E+C;P.F7F"6DV)P4V)
MQDG!Y@5)P>D;30GQ28G&2,'@'DG![@)%B<E)"<9,B<!,`<]%B?9(]]!!B?E-
M(?)((>A,"=!,C90&F7F"6DR)R$R)SDC!Y@5(P>@;2`GP3(G&2<'@'HG`2,'N
M`DD!PDP)QD@#K"20````38G(B?9$B=!)]]!((?=-(?!,`[0DF````$D)^$B)
MQTJ-K`69>8):28G`2,'G!4G!Z!M)"?A%B<!,`<5-B<A)P>$>2<'H`HGO30G(
M28G!18G'2??132'Z22'Q2`.T)*````!-"=%)B?I,`[PDJ````$^-M`Z9>8):
M28GY2<'B!4G!Z1M-"=%%B<E-`<Y)B<%(P>`>2<'I`D6)\DD)P4R)T$TQR$C!
MZ!M$,<5-B=!)P>`%2(VL+J'KV6Y,"<!)B?A(P><>2<'H`HG`20GX2`'%3(G`
MB>Y,,<A(B?=%B<E!,<9(B?!(P><%2,'H&T^-M#>AZ]EN3`.,)+````!("?B)
MP$D!QDR)T$G!XAY(P>@"1(GW3`G028G"33'"18G`3`.$)+@```!$,=5)B?I)
MC:PIH>O9;DF)^4G!X@5)P>D;30G118G)3`'-28GQ2,'F'DG!Z0)!B>I)"?%,
MB<Y(,<:)P$@#A"3`````03'V3(G63XVT,*'KV6Y-B=!(P>8%2<'H&TD)\$6)
MP$T!QDF)^$C!YQY)P>@"1(GV20GX3(G'3#'/18G),?U(B?=(C:PHH>O9;DB)
M\$C!Z!M(P><%3`.,),@```!("?B)P$@!Q4R)T$G!XAY(P>@"B>],"=!)B<)-
M,<)%B<!,`X0DT````$4QUDF)^D^-M#&AZ]EN28GY2<'B!4G!Z1M-"=%%B<E-
M`<Y)B?%(P>8>2<'I`D6)\DD)\4R)SD@QQHG`2`.$)-@````Q]4R)UDF-K"BA
MZ]EN38G02,'F!4G!Z!M)"?!%B<!,`<5)B?A(P><>2<'H`HGN20GX3(G'3#'/
M18G)03'^2(GW3HVT,*'KV6Y(B?!(P><%2,'H&T@)^(G`20'&3(G02,'H`DG!
MXAY,`XPDX````$P)T$2)]TF)PDTQPD6)P$P#A"3H````1#'528GZ28VL*:'K
MV6Y)B?E)P>(%2<'I&TT)T46)R4P!S4F)\4C!YAY)P>D"08GJ20GQ3(G.2#'&
MB<!(`X0D\````$$Q]DR)UD^-M#"AZ]EN38G02,'F!4G!Z!M)"?!%B<!-`<9)
MB?A(P><>2<'H`D2)]DD)^$R)QTPQST6)R3']2(GW2(VL**'KV6Y(B?!(P><%
M2,'H&T@)^(G`2`'%3(G02<'B'DC!Z`*)[TP)T$P#C"3X````28G"33'"18G`
M3`.$)``!``!%,=9)B?I/C;0QH>O9;DF)^4G!X@5)P>D;30G118G)30'.28GQ
M2,'F'DG!Z0)%B?))"?%,B<Y(,<:)P$@#A"0(`0``,?5,B=9)C:PHH>O9;DV)
MT$C!Y@5)P>@;20GP18G`3`'%28GX2,'G'DG!Z`*)[DD)^$R)QTPQST6)R4P#
MC"00`0``03'^2(GW3HVT,*'KV6Y(B?!(P><%2,'H&T@)^(G`20'&3(G02<'B
M'DC!Z`)$B?=,"=!)B<)-,<)%B<!$,=5)B?I)C:PIH>O9;DF)^4G!Z1M)P>(%
M3`.$)!@!``!-"=%%B<E,`<U)B?%(P>8>2<'I`D&)ZDD)\4R)SD@QQHG`2`.$
M)"`!``!!,?9,B=9/C;0PH>O9;DV)T$C!Y@5)P>@;20GP18G`30'&28GX2,'G
M'DG!Z`)$B?9)"?A,B<=,,<]%B<E,`XPD*`$``#']2(GW2(VL**'KV6Y(B?!(
MP><%2,'H&T@)^(G`2`'%3(G02<'B'DC!Z`*)[TP)T$F)PDTQPD6)P$4QUDF)
M^D^-O#&AZ]EN28GY2<'B!4G!Z1M-"=%)B?)%B<E-`<](P>8>2<'J`DD)\DP#
MA"0P`0``18G^38G123'!B<!(`X0D.`$``$0QS4V)\4F-M"BAZ]EN38GP2<'A
M!4G!Z!M-"<A)B?E(P><>2<'I`D6)P$D)^4P!QDV)R(GU33'018G213''28GH
M3HV\.*'KV6Y(B>A)P>`%2,'H&TP)P$V)\$G!YAZ)P$G!Z`))`<>XW+P;CTT)
M\$F)QDP#M"1``0``1(G_30'R38G&30G.3"'V38G&32'.18G)1`GV38TT,DF)
M^DB)_DC!Y@5)P>H;20GR18G230'628GJ2,'E'DG!Z@)$B?9)">I(B<5(`ZPD
M2`$``$D!Z4R)U4P)Q4DA[TR)U4PAQ46)P$$)[TB)]4T!STF)\4C!Y05)P>D;
M20GI18G)30'/28GY2,'G'DG!Z0)$B?U)"?E(B<=(`[PD4`$``$D!^$R)STP)
MUTDA_DR)STPAUT6)TD$)_DB)[TT!QDF)Z$C!YP5)P>@;20GX18G`30'&28GP
M2,'F'DG!Z`)$B?=)"?!(B<9(`[0D6`$``$D!\DR)QDP)SDDA]TR)QDPASD6)
MR4$)]TB)_DT!UTF)^DC!Y@5)P>H;20GR18G230'728GJ2,'E'DG!Z@)$B?Y)
M">I(B<5(`ZPD8`$``$D!Z4R)U4P)Q4DA[DR)U4PAQ46)P$$)[DB)]4T!SDF)
M\4C!Y05)P>D;20GI18G)30'.28GY2,'G'DG!Z0)$B?5)"?E(B<=(`[PD:`$`
M`$D!^$R)STP)UTDA_TR)STPAUT6)TD$)_TB)[TT!QTF)Z$C!YP5)P>@;20GX
M18G`30''28GP2,'F'DG!Z`)$B?])"?!(B<9(`[0D<`$``$D!\DR)QDP)SDDA
M]DR)QDPASD6)R4$)]DB)_DT!UDF)^DC!Y@5)P>H;20GR18G230'628GJ2,'E
M'DG!Z@)$B?9)">I(B<5(`ZPD>`$``$D!Z4R)U4P)Q4DA[TR)U4PAQ46)P$$)
M[TB)]4T!STF)\4C!Y05)P>D;20GI18G)30'/28GY2<'I`DC!YQY$B?U)"?E(
MB<=(`[PD@`$``$D!^$R)STP)UTDA_DR)STPAUT6)TD$)_DB)[TT!QDF)Z$C!
MYP5)P>@;20GX18G`30'&28GP2,'F'DG!Z`)$B?=)"?!(B<9(`[0DB`$``$D!
M\DR)QDP)SDDA]TR)QDPASD6)R4$)]TB)_DT!UTF)^DC!Y@5)P>H;20GR18G2
M30'728GJ2,'E'DG!Z@)$B?Y)">I(B<5(`ZPDD`$``$D!Z4R)U4P)Q4DA[DR)
MU4PAQ46)P$$)[DB)]4T!SDF)\4C!Y05)P>D;20GI18G)30'.28GY2,'G'DG!
MZ0)$B?5)"?E(B<=(`[PDF`$``$D!^$R)STP)UTDA_TR)STPAUT6)TD$)_TB)
M[TT!QTF)Z$C!YP5)P>@;20GX18G`30''28GP2,'F'DG!Z`)$B?])"?!(B<9(
M`[0DH`$``$D!\DR)QDP)SDDA]DR)QDPASD6)R4$)]DB)_DT!UDF)^DG!ZAM(
MP>8%20GR18G230'628GJ2,'E'DG!Z@)$B?9)">I(B<5(`ZPDJ`$``$D!Z4R)
MU4P)Q4DA[TR)U4PAQ46)P$$)[TB)]4T!STF)\4C!Y05)P>D;20GI18G)30'/
M28GY2,'G'DG!Z0)$B?U)"?E(B<=(`[PDL`$``$D!^$R)STP)UTDA_DR)STPA
MUT6)TD$)_DB)[TT!QDF)Z$C!YP5)P>@;20GX18G`30'&28GP2<'H`DC!YAY$
MB?=)"?!(B<9(`[0DN`$``$D!\DR)QDP)SDDA]TR)QDPASD6)R4$)]TB)_DT!
MUTF)^DC!Y@5)P>H;20GR18G230'728GJ2,'E'DG!Z@)$B?Y)">I(B<5(`ZPD
MP`$``$D!Z4R)U4P)Q4DA[DR)U4PAQ46)P$$)[DB)]4T!SDF)\4C!Y05)P>D;
M20GI18G)30'.28GY2,'G'DG!Z0)$B?5)"?E(B<=(`[PDR`$``$D!^$R)STP)
MUTDA_TR)STPAUT6)TD$)_TB)[TT!QTF)Z$C!YP5)P>@;20GX18G`30''28GP
M2,'F'DG!Z`)$B?])"?!(B<9(`[0DT`$``$@#A"38`0``20'R3(G&3`G.22'V
M3(G&3"'.18G)00GV2(G^3`'(30'628GZ2,'F!4G!ZAM)"?)%B=)-`=9)B>I(
MP>4>2<'J`D2)]DD)ZDV)T4T)P4TASTV)T4TAP46)P$4)STF)\4D!QTB)\$C!
MZ!M)P>$%3`G(28GY2,'G'HG`2<'I`DD!Q[C6P6+*20GY2(G'2`.\).`!``!$
MB?U)`?A,B<],,==%B=)!,?Y(B>]-`<9)B>A(P><%2<'H&TD)^$6)P$T!QDF)
M\$C!YAY)P>@"1(GW20GP2(G&2`.T).@!``!)`?),B<9,,<Y%B<E!,?=(B?Y-
M`==)B?I(P>8%2<'J&TD)\D6)TDT!UTF)ZDC!Y1Y)P>H"1(G^20GJ2(G%2`.L
M)/`!``!)`>E,B=5,,<5%B<!!,>Y(B?5-`<Y)B?%(P>4%2<'I&TD)Z46)R4T!
MSDF)^4C!YQY)P>D"1(GU20GY2(G'2`.\)/@!``!)`?A,B<],,==%B=)!,?](
MB>]-`<=)B>A(P><%2<'H&TD)^$6)P$T!QTF)\$C!YAY)P>@"1(G_20GP2(G&
M2`.T)``"``!)`?),B<9,,<Y%B<E!,?9(B?Y-`=9)B?I(P>8%2<'J&TD)\D6)
MTDT!UDF)ZDC!Y1Y)P>H"1(GV20GJ2(G%2`.L)`@"``!)`>E,B=5,,<5%B<!!
M,>](B?5-`<])B?%(P>4%2<'I&TD)Z46)R4T!STF)^4C!YQY)P>D"1(G]20GY
M2(G'2`.\)!`"``!)`?A,B<],,==%B=)!,?Y(B>]-`<9)B>A(P><%2<'H&TD)
M^$6)P$T!QDF)\$C!YAY)P>@"1(GW20GP2(G&2`.T)!@"``!)`?),B<9,,<Y%
MB<E!,?=(B?Y-`==)B?I(P>8%2<'J&TD)\D6)TDT!UTF)ZDG!Z@)(P>4>1(G^
M20GJ2(G%2`.L)"`"``!)`>E,B=5,,<5%B<!!,>Y(B?5-`<Y)B?%(P>4%2<'I
M&TD)Z46)R4T!SDF)^4C!YQY)P>D"1(GU20GY2(G'2`.\)"@"``!)`?A,B<],
M,==%B=)!,?](B>]-`<=)B>A(P><%2<'H&TD)^$6)P$T!QTF)\$C!YAY)P>@"
M1(G_20GP2(G&2`.T)#`"``!)`?),B<9,,<Y%B<E!,?9(B?Y-`=9)B?I)P>H;
M2,'F!4D)\D6)TDT!UDF)ZDC!Y1Y)P>H"1(GV20GJ2(G%2`.L)#@"``!)`>E,
MB=5,,<5%B<!!,>](B?5-`<])B?%(P>4%2<'I&TD)Z46)R4T!STF)^4C!YQY)
MP>D"1(G]20GY2(G'2`.\)$`"``!)`?A,B<],,==%B=)!,?Y(B>]-`<9)B>A(
MP><%2<'H&TD)^$6)P$T!QDF)\$C!YAY)P>@"1(GW20GP2(G&2`.T)$@"``!)
M`?),B<9,,<Y%B<E!,?=(B?Y-`==)B?I(P>8%2<'J&TD)\D6)TDT!UTF)ZDC!
MY1Y)P>H"1(G^20GJ2(G%2`.L)%`"``!)`>E,B=5,,<5%B<!!,>Y(B?5-`<Y)
MB?%(P>4%2<'I&TD)Z46)R4T!SDF)^4C!YQY)P>D"1(GU20GY2(G'2`.\)%@"
M``!)`?A,B<],,==%B=)!,?](B>]-`<=)B>A(P><%2<'H&TD)^$6)P$T!QTF)
M\$C!YAY)P>@"1(G_20GP2(G&2`.T)&`"``!)`?),B<9,,<Y%B<E!,?9(B?Y-
M`=9)B?I(P>8%2<'J&TD)\D6)TDT!UDF)ZDC!Y1Y)P>H"1(GV20GJ2(G%2`.L
M)&@"``!)`>E,B=5,,<5%B<!!,>](B?5-`<])B?%(P>4%2<'I&TD)Z46)R4T!
MSTF)^4C!YQY)P>D"1(G]20GY2(G'2`.\)'`"``!)`?A,B<],,==%B=)!,?Y+
MC3PP28GH28GN2<'F!4G!Z!M-"?!$`<=)B?!)P>@"2,'F'D@#A"1X`@``20GP
M2(GN2,'E'DC![@)!`?M(">Y(B?U,`=!-B<)(P>4%33'*`?-%`>!%,==%`>E.
MC10X2(GX2,'H&T@)Z$0!T`'"2(D13(E9"$B)61!,B4$83(E)($B!Q(@"``!;
M7E]=05Q!74%>05_##Q]``$%505155U932(/L*$B%R4B)S4F)U71-2(L%<J<`
M`$R+($V%Y'0^#[8!/#UU1C'_28L<)$B%VW0L2&/_3(GF9I!)B?A(B>I(B=GH
M4A\``(7`=0:`/#L]=#A(@\8(2(L>2(7;==PQP$B#Q"A;7E]=05Q!7<.$P'2V
M2(G/2(/'`0^V!SP]=`2$P'7Q*>_KH@\?`$PIYDB-1#L!2,'^`T&)=0!(@\0H
M6UY?74%<05W##Q\`5U932(/L,$B+/=*F``!(C5PD+$B)SDB)VDB)\>@O____
M2(7`=$IF+@\?A```````2&-4)"Q(BP=(C0303(M`"$V%P$R)`'3.#Q^$````
M``!(@\`(3(M`"$V%P$R)`'7P2(G:2(GQZ.7^__](A<!UP$B#Q#!;7E_##Q^$
M``````!!5D%505155U932(/L,#'`@#H]2(G32(G/#Y3`2`'#2(G9Z%D>``!(
MC50D+$B)^4B)Q>B9_O__2(7`2(G&#X3M````2(G!Z#4>``")ZD@YT`^#O```
M`(MT)"Q,BR7_I0``#[8'/#T/A'0!``"$P`^$;`$``$B)^NL,#Q^``````$6$
MP'0.2(/"`40/M@)!@/@]=>TI^DF+!"2-3"H"2(TT\.CJ'0``2(7`2(D&#X1!
M`0``BU0D+$F+!"1,BP30#[8'2(U7`83`08@`=1?K&6:02(/"`0^V0O])@\`!
MA,!!B`!T!#P]=>E!Q@`],<`/'P`/MA0#08A4``%(@\`!A-)U[S'`2(/$,%M>
M7UU!7$%=05[#2(/#`0^V0_](@\8!A,"(1O]U[>O:9@\?A```````3(LE*:4`
M`$V++"1)@WT```^$H0```$V)Z#'V#Q]$``!)@\`(@\8!28,X`'7SC58"2&/2
M2,'B`XL%,^4``(7`="],B>GH)QT``$B%P$F)!"1T<4QC]DF)P4O'1/$(````
M`(ET)"SIQ_[__V8/'T0``$B)T<<%\^0```$```#HWAP``$B%P'0\3&/V3(GJ
M2(G!3HT$]0````#HXQP``$F)P4F)!"3KLF8N#Q^$```````QTNFN_O__NA``
M```Q]NET____N/_____I!O___P\?`$%7059!54%455=64TB![*@```!(A<E(
MB<MT!8`Y`'442('$J````%M>7UU!7$%=05Y!7\/HAQP``$B%P$B)Q73?3(UT
M)"!!O`(```!!O0,```!(B>GH=AP``$B%P`^$G@```$R->!A(C3UB1```3(GA
M3(G^\Z9TVDB-/5-$``!,B?Y,B>GSIG3)2(G9Z.\;``!,B?E(B<;HY!L``(U,
M!@)(8\GHZ!L``$B-%29$``!(B<9-B?E)B=A(B<'H(!P``$B)\4R)\N@E'```
M@_C_2(GQ=!"+1"0P)0#P```]`$```'0]Z!@<``!(B?'H(!P``$B)Z>C8&P``
M2(7`#X5B____2(GIZ!<<``!(B=GH'QP``.D*____9BX/'X0``````.C;_O__
MZ\%F#Q^$``````!75E-(@^P@2(L=TJ(``$B+"^CZ&P``2(L+2(N`0`(``$B+
M>`CHYQL``$B+"P^VL.P$``#HV!L``$B+0`B#YG\/MT`@P>`'"?!FB4<"2(L+
MZ+P;``!(BPM(BX!``@``2(MP".BI&P``2(T5/4,``$4QP$B)P>A'&0``2(L+
M2(E&*.B+&P``2(L+2(N`0`(``$B+<`CH>!L``$B+@*@&``!(BPM(B48PZ&4;
M``!(BPM(BX!``@``2(MP".A2&P``,=)(@[@@"@```'062(L+Z#X;``!(BX`@
M"@``2(N0J````$B+"TB)5CCH)!L``$B+@$`"``!(BPM(BT`(2,=`0`````#H
M"1L``$B+@$`"``!(BPM(BT`(2,=`(`````#H[AH``$B+"TB+@$`"``!(BW`(
MZ-L:``!(BX!0`@``2(E&2$B#Q"!;7E_##Q^$``````!64TB#["A(BQV#H0``
M2(G.2(L+Z*@:``!(C9"8````2(GQZ*D:``"%P'0),<!(@\0H6U[#2(L+Z(0:
M``"+@*@````E`/```#T`0```=!I(BPOH:AH``(N`J````"4`\```/0"@``!U
MP[H"````2(GQZ&L:``"%P`^4P`^VP$B#Q"A;7L-F9F8N#Q^$``````!!5+@P
M$```55=64^B@&```2"G$2(L]%J$``$&X_P\``$B-7"0@2(G-2(T-G4$``$F)
MU$B)VDB+!TB)A"0H$```,<#H&1H``(7`?FF-2/](8_&`?#0@`'182&/P28GP
MNB\```!(B=GH!1H``$B%P'1$2(L%B:```$B+".BQ&0``28GQ28G82(GJ2(G!
MZ%@7``!(BX0D*!```$@S!W4V2('$,!```%M>7UU!7,,/'T0``(7)=:=(BP5%
MH```2(L(Z&T9``!-B>!(B>I(B<'H'Q<``.N]Z-`7``"0#Q^``````$B#["A(
MBP45H```2(L(Z#T9``!,C0WA0```3(T%]Q8``$B-%=U```!(B<%(@\0HZ>P6
M``!F9F8N#Q^$``````!32(/L,$B+'=2?``!(BPOH_!@``$B)P>BD____2(L+
MZ.P8``!,C0V`0```3(T%_A$``$B-%:A```#'1"0H`````$B)P4R)3"0@Z)H6
M``"02(/$,%O#9F8N#Q^$```````QP,-F9F9F+@\?A```````05>X6!```$%6
M055!5%575E/H"A<``$@IQ$R+-8"?``!(B<Y(B==(C0U90```2(U4)#Q)BP9(
MB80D2!```#'`Z.SW__^Z+P```$B)\4B)P^B<&```2(7`=#=(C0TO0```2(GR
MZ/CX__](B?!(B[0D2!```$DS-@^%1P$``$B!Q%@0``!;7E]=05Q!74%>05_#
M2(GQZ$@7``!(C17]/P``2(GY08G%Z%88``!(A<!)B<=TITB++;>>``!,C25P
M/P``2(U\)$!!@#\`30]$_$B%VW033(GZ2(G9Z#,8``"%P`^$D@```$R)^>CS
M%@``08G"2)A-C4P'_T&`.2]U%0\?`$'&`0!)@^D!08/J`4&`.2]T[D.-1!4!
M/?\/```/CS?___],C0UU/P``2(T5<#\``$V)^$B)^4B)="0@Z``7``!(BTT`
MZ%<7``!(C9"8````2(GYZ/@6``"%P'4;2(M-`.@[%P``BX"H````)0#P```]
M`(```'0?2(T5'C\``#')Z'L7``!(A<!)B<</A33____IP_[__[H!````2(GY
MZ!T7``"%P'702(T-X3X``$B)^NBJ]___2(GYZ&(7``#IJ/[__^A0%0``D`\?
M@`````!$#[8!2(G(183`=!E(C5$!08#X+T0/M@)(#T3"2(/"`46$P'7K\\-F
M#Q^$``````!32(/L($B%R4B)RP^$D````(`Y``^$AP```.C!%0``3(U$`_],
M.<-R$.LE#Q]$``!)@^@!23G8='E!@#@O=/'K!D&`."]T"4F#Z`%,.<-R\4DY
MV'41ZUMF#Q^$``````!!@#@O=0E)@^@!3#G#<O%)*=@QP$F-4`)(@?H`$```
M=QI(C0UXS0``28/``4B)VNB<%@``2(T%9<T``$B#Q"!;PTB-#8@]``!(@\0@
M6^EN%@``08`X+W3I2(T-WST``$B#Q"!;Z5<6```/'X``````55=64TB#[&A(
MBRW!G```2(T-P3T``$B-7"0L2(M%`$B)1"18,<!(BP5[/@``2(E$)#!(BP5W
M/@``2(E$)#A(BP5S/@``2(E$)$!(BP5O/@``2(E$)$@/MP5K/@``9HE$)%#H
MJ?7__TB-#7(]``#HG?7__TB-#6X]``#HD?7__TB-#6L]``#HA?7__TB-#68]
M``#H>?7__TB-#6H]``#H;?7__TB-#0P]``#H8?7__TB-#5X]``#H5?7__TB-
M#5P]``#H2?7__TB-#5H]``#H/?7__TB-#>4\``#H,?7__TB-#4P]``#H)?7_
M_TB-#4D]``#H&?7__TB-#4@]``!(B=KH6O3__TB%P'0/2(T-##T``$B)PNAV
M]?__2(T--CT``$B)VN@W]/__2(7`=`](C0TT/0``2(G"Z%/U__](C0TP/0``
M2(G:Z!3T__](A<!T/TB-#5X\``!(B<+H,/7__TB-%3D]``!(C0V*/```Z!WU
M__](BT0D6$@S10!U7TB#Q&A;7E]=PV8/'X0``````$B-#>T\``!(B=KHP?/_
M_TB%P'012(T-:3P``$B)PNC=]/__ZZM(C70D04B-/=0\``"Y"@```/.F=99(
MC50D2TB-#3X\``#HM?3__^N#Z&82``"09F9F9BX/'X0``````$B#[#B+!38K
M``"#^/]T!4B#Q#C#2(U4)"Q(C0T$/```Z$OS__](A<!T)@^V$#'`A-)T"#'`
M@/HP#Y7`B07_*@``2(/$.,-F+@\?A```````,<#KYV9F9BX/'X0``````$B#
M["CH%Q0``(G!Z"`4``!(A<!T"TB+`$B#Q"C##Q\`,<#K]&9F9BX/'X0`````
M`$%6055!5%575E-(@\2`2(T%/3P``$B--2X\``!(C2TO/```28G,2,=$)'``
M````2(U\)%!(B40D0$B-!0H\``!,C6PD/$B)1"1(2(T%$3P``$B)1"102(T%
M$CP``$B)1"182(T%"SP``$B)1"1@2(T%$3P``$B)1"1HZR,/'T``2(G!Z(@3
M``!(A<!T4TB%]G0U2(L'2(GU2(/'"$B)QDR)ZDB)Z>@V\O__2(7`2(G#3(GB
M=`6`.`!UQDB)Z>A.\___2(7V=<M(@^R`6UY?74%<05U!7L-F+@\?A```````
M3(GAZ*@1``!(B=E)B<;HG1$``$F-3`8"Z*,1``!,C0U(.@``2(T513H``$F)
MQDB)7"0@38G@2(G!Z-(1``!,B?)(B>GHY_+__^N7#Q]$``!!5[BX@0``059!
M54%455=64^B*$```2"G$2(L%`)D``$F)S4B-#=PY``!(QT0D<`````!(QX0D
MR`````````!(QX0DD`````````!(BSA(B;PDJ($``#'_2(T%#CL``$B-?"1<
M2(E$)&!(C05`.0``2(GZ2(E$)&A(C04Z.@``2(F$)*````!(C07D.@``2(F$
M)*@```!(C07<.@``2(F$)+````!(C075.@``2(F$)+@```!(C07+.@``2(F$
M),````!(C067.@``2(F$)(````!(C06Q.@``2(F$)(@```#HRO#__TB%P'0)
M@#@`#X5,!```Z*?]__](A<!(B<,/A%H$``!(B=GH4Q```$B-3``!Z%D0``"`
M.P!)B<9(B<9T($0/M@-(C15F.@``2(GQ2(/#`4B#Q@+H@Q```(`[`'7@2(VT
M)*@```!(C0UH.0``ZPU(@\8(2(M.^$B%R70Q2(GZZ$CP__](A<!(B<-TXX`X
M`'3>2(G!Z./U__^%P'322(G9Z!<1``!(A<!(B<-U6$B+7"1@2(7;#X21`0``
M@#L`O@$```!U).F"`0``9I!(8\9(BUS$8$B%VP^$;P$``(/&`8`[``^$8P$`
M`$B)V>B+]?__A<!TUTB)V>B_$```2(7`2(G#=,=(B=B+"$B#P`2-D?_^_O[W
MT2'*@>*`@("`=.B)T<'I$/?"@(````]$T4B-2`)(#T3!`-),B?%(@]@#2"G8
M1(L!2(/!!$&-D/_^_OY!]]!$(<*!XH"`@(!TY$&)T$'!Z!#WPH"```!!#T30
M3(U!`DD/1,@`TDB#V0-,*?&-M`@$!```2&/V2(GQZ/H.``!(B<5(C04A.0``
M3(T-ES<``$B-%0HY``!)B=A(B>E,B70D*$B)1"0@Z!T/``"ZP`$``$B)Z>A`
M$```@_C_#X1^````2(L=,)8``$B+"^A8#P``2(V0F````$B)Z>A9#P``A<`/
MA)4```!)BUT`Z!@0``!(BT@82(T5+3D``$F)Z4F)V.@2$```,<!(BST9E@``
M2(N4)*B!``!(,Q</A1@(``!(@<2X@0``6UY?74%<05U!7D%?PP\?0``QV^FR
M_O__Z.0/``"#.!&0#X1S____Z-4/``!)BUT`BS#HJ@\``$B+2!A(C15'.```
M28GIB70D($F)V.B@#P``,<#KC$B+"^BD#@``BX"H````)0#P```]`$````^%
M3?___TB+"^B&#@``1(NPL````.@:#P``03G&#X4P____2(L+Z&D.``"+@*@`
M```E_P$``#W``0``#X42____2(GQZ)L-``!(C0UC-P``2(GZ2(G#Z,GM__](
MA<!(B<$/A$('``#HJ`X``$F+30!(B<+H?/7__TB%P$F)Q@^$)0<``$R)\>A(
M#0``@_@'?C-(8]!(C3V\-P``N0D```!)C706^/.F=1N#^`@/A!D&``"#Z`A(
MF$&`?`;_+P^$"`8``)#HZ_G__X7`=1FZ```!`$R)\>C*#@``A<")1"1`#X5&
M`0``2(T5=S8``$B-#>0U``!(C3TT-0``,?;H4N[__^BM#@``3(T-A#4``$B-
M%5\W``!(B7PD*(E$)"!)B>A(B=GH"PT``.M(9@\?A```````Z%L.``"#.!%U
M1TB-/>HT``"#Q@'H9PX``$R-#3XU``!(C14G-P``2(E\)#")="0HB40D($F)
MZ$B)V>C!#```NL`!``!(B=GHY`T``(/X_W2O2(GIZ-<,``!(C0WF-```2(G:
MZ+CM__](B=GHH/G__TB)V.G(_?__#Q^$``````!(B<%(B40D0.B#^?__2(M$
M)$!(B<'H1@T``.FA_?__2(T-^S4``+X!````ZQ8/'P!(8\:#Q@%(BXS$@```
M`$B%R70G2(GZZ"7L__](A<!TX(`X`'3;2(G!Z`,-``!(A<!(B<-TR^E;^___
ML0U(C1VS-0``Z5K[__],C;0DH`$``$&X`@```$C'PN[___^)QXG!Z'H-``!!
MN`8```!,B?*)^>AZ#0``@+PDH`$```!U#H"\)*$!``!##X3-!```BTPD0$4Q
MP#'22(VT)"`!``#H/`T``+B)J\WO2,>$)"`!```!(T5G2,>$)#@!``!V5#(0
M2(F$)"@!``"X_MRZF$C'A"1(`0```````$B)A"0P`0``N/#ATL-(QX0D4`$`
M``````!(B80D0`$``$B-1CC'A"28`0```````$B)1"1(D(M,)$!!N`"```!,
MB?+HS@P``(7`28G%B<</CET!``!(BXPD2`$``$ECQ8T4P4@YT4B+C"10`0``
M=@1(@\$!2,'H'4B)E"1(`0``2`'(2(F$)%`!``!(8X0DF`$``(7`#X0/`0``
MOT````!(C4P&.$R)\BG'03G]00]._4QCYTV)X.AJ#```BX0DF`$```'X@_A`
MB80DF`$```^%8?___T2)Z$B)\4^-+"8I^(G'Z![3__^#_S\/CIT```!!B?]-
MB>Q)BPPD08/O0$F#Q$!(B8PD6`$``$F+3"3(2(F,)&`!``!)BTPDT$B)C"1H
M`0``28M,)-A(B8PD<`$``$F+3"3@2(F,)'@!``!)BTPDZ$B)C"2``0``28M,
M)/!(B8PDB`$``$F+3"3X2(F,))`!``!(B?'HF-+__T&#_S]_@X/O0(GZP>H&
MB=#!X@9(@\`!*==(P>`&20'%2(M,)$A,8\=,B>KHB`L``(F\))@!``#IC/[_
M_TV)]>DV____BTPD0.A["P``3(NL)$@!``!,B[0D4`$``$R)Z$C!Z`.#X#^-
M4`%(8\C&A`Q8`0``@(/Z.`^/7`(``$ACTD&X-P```$$IP$B-3!8X,=)-8\#H
M/PL``$R)\$B)\42(M"23`0``2,'H&$2(K"27`0``2(V\)-$```"(A"20`0``
M3(GP2,'H$(B$))$!``!,B?!,C;0D\````$C!Z`B(A"22`0``3(GH3(GV2,'H
M&(B$))0!``!,B>A(P>@0B(0DE0$``$R)Z$V-;BA(P>@(B(0DE@$``.A\T?__
M2(N$)"`!``!(B<*(A"33````2,'J&$0/ML*(E"30````2(G"2,'J$(B4)-$`
M``!(B<)(BX0D*`$``$C!Z@B(E"32````2(G"B(0DUP```$C!ZAB(E"34````
M2(G"2,'J$(B4)-4```!(B<)(BX0D,`$``$C!Z@B(E"36````2(G"B(0DVP``
M`$C!ZAB(E"38````2(G"2,'J$(B4)-D```!(B<)(BX0D.`$``$C!Z@B(E"3:
M````2(G"B(0DWP```$C!ZAB(E"3<````2(G"2,'J$(B4)-T```!(B<)(BX0D
M0`$``$C!Z@B(E"3>````2(G"B(0DXP```$C!ZAB(E"3@````2(G"2,'J$(B4
M).$```!(B<)(P>H(B)0DX@```.L19@\?A```````1`^V!TB#QP%(C16$,0``
M2(GQ2(/&`NBE!P``3#GN=>#&A"08`0```$B-!8PO``!,C0WH+P``2(T5M#$`
M`$R)="0@28GH2(G92(E$)"CH;@<``.FZ^O__38MM"$V%[0^$[/G__TR)Z>CT
M!@``2(/X`P^&VOG__TF-=`7\2(T]:C$``+D%````\Z9-#T3UZ;[Y__](8])!
MN#\```!(C4P6.$$IP#'236/`Z.,(``!(B?'HF\___TB-5CBY!P```#'`2(G7
M\TBKZ8;]__^`O"2B`0``00^%)?O__X"\)*,!``!##X47^___@+PDI`$``$@/
MA0G[__^`O"2E`0``10^%^_K__XM\)$!!N`(```!(Q\+&____B?GH-0@``$&X
M*0```$R)\HGYZ#4(``#I^/[__^AC!0``38MU`.G2^/__9I!!54%455=64TB#
M[#A(BRV=C0``3(UL)"Q(BTT`Z+\&``!(C0W@+@``QH#M!````4R)ZN@YYO__
M2(7`=`F`.#$/A%L"``#HQO#__TB+30#HC08``$B-%=DP``!!N0L```!!N`$`
M``!(B<'H4@0``$B%P`^$J@```(,]TLT```%(BT`03(M@*`^.E0```+X(````
M,=OK+$B+5Q!(BP6HS0``2(L,,.BG!@``A<!U8X/#`8L%FLT``$B#Q@B-4P$Y
MPGU52(M-`.@6!@``13')08G83(GB2(G!Z.T#``!(A<!T,$B+./9'#01UK$B+
M30#H[@4``$B)^D&Y`@```$4QP$B)P>C*`P``2(G"ZXX/'T0``(L%.LT``(U0
M_SG:#X0R`0``2(M-`.BV!0``2(T5!S```$&Y!````$&X`0```$B)P>A[`P``
M2(7`2(G#=#-(BT`02(M-`$B+,.B#!0``3(T%UB\``$B)P4B)\N@Q`P``2(M#
M$$B+&/9##D`/A04!``!(BTT`Z%<%``!(C16O+P``0;D$````0;@!````2(G!
MZ!P#``!(A<!(B<-T.4B-#1PM``!,B>KHM>3__TB%P$B)Q@^$60$``$B+0Q!(
MBTT`2(LXZ`D%``!)B?!(B<%(B?KHNP(``$B+30#H\@0``$B+30#&@.T$````
MZ.($``!(C0W!+```2,>`<`<```````!,B>KH6.3__TB%P$B)!3;,```/A(T`
M``"ZP`$``$B)P>A[!0``@_@1=`R-4`&#^@$/AUH!``!(@\0X6UY?74%<05W#
M@_@!#X[%_O__NP$```!(BTT`@\,!Z',$``!,B>)(B<'H8`(``#D=TLL``'_A
MZ9S^__](BTT`Z%($``!(B=I(B<'H1P(``.GB_O__9I"`>`$`#X2@_?__9@\?
M1```Z9#]__](BPV,RP``Z/_Q__](A<!(B06-RP``#X3#````2(M#$$B+/6S+
M``!(BS#V1@T$=&5(BTX02(GZZ&\%``")QDB+0Q!(BQCV0PT$=&],BT,02(L%
M/<L``$B-#5XN``!!B?%(BQ#HRP$```\?`$B+!2'+``!(BTT`2(LX2(M#$$B+
M,.BF`P``2(GZ2(GQZ&OI___IF_[__TB+30#HC0,``$&Y`@```$B)P44QP$B)
M\NAI`0``2(G!Z7K___](BTT`Z&@#``!%,<!!N0(```!(B=I(B<'H1`$``$F)
MP.EP____2(T-K2T``.A(`0``08G!2(L%GLH``$R+!:?*``!(C0W@+0``2(L0
MZ"@!``"09F9F9F9F+@\?A```````5E-(@^PH2(G.Z&+L__]$#[8`2(G#183`
M=!M(C5`!9I!!@/@O1`^V`D@/1-I(@\(!183`=>OH]N[__X7`=!Y(A?9T&8`^
M`'042(T50BP``$B)V>AY`P``2#G8=`=(@\0H6U[#2(GQ2(/$*%M>Z5_E__]F
M9F9F9F8N#Q^$``````!32(/L($B+'52)``!(BPOH?`(``+K_____2(G!Z'\`
M``!(BPOH9P(``$B-%5#F__]%,<!(B<%(@\0@6^EH````_R66[@``D)#_)<;N
M``"0D/\EMNX``)"0_R7.[@``D)#_)7[N``"0D/\E?NX``)"0_R56[@``D)#_
M)2[N``"0D/\E?NX``)"0_R4F[@``D)#_)3[N``"0D/\E'NX``)"0_R4>[@``
MD)#_)4[N``"0D/\E9NX``)"0_R7F[0``D)#_)=;M``"0D/\E9NX``)"0_R5F
M[@``D)#_)1;N``"0D/\E;NX``)"0_R56[@``D)#_)5;N``"0D/\E)NX``)"0
M_R56[@``D)#_)6;M``"0D%%02#T`$```2(U,)!AR&4B!Z0`0``!(@PD`2"T`
M$```2#T`$```=^=(*<%(@PD`6%G#D)"0D)"0D)"0D)"0D)!(@^PH,=+HY0(`
M`/\5H^L``)"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0D)"0
MD)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)#_)4;K
M``"0D)"0D)"0D)"0_R5^[```D)"0D)"0D)"0D/\E9NP``)"0D)"0D)"0D)#_
M);;K``"0D)"0D)"0D)"0_R4.[```D)"0D)"0D)"0D/\EKNL``)"0D)"0D)"0
MD)#_);[K``"0D)"0D)"0D)"0_R7.ZP``D)"0D)"0D)"0D/\EWNL``)"0D)"0
MD)"0D)#_)=;K``"0D)"0D)"0D)"0_R4.[```D)"0D)"0D)"0D/\E!NL``)"0
MD)"0D)"0D)#_);;J``"0D)"0D)"0D)"0_R6&ZP``D)"0D)"0D)"0D/\E3NL`
M`)"0D)"0D)"0D)#_)>[J``"0D)"0D)"0D)"0_R5>Z@``D)"0D)"0D)"0D/\E
M-NL``)"0D)"0D)"0D)#_)<[J``"0D)"0D)"0D)"0_R4^ZP``D)"0D)"0D)"0
MD/\E9NL``)"0D)"0D)"0D)#_)2;K``"0D)"0D)"0D)"0_R4>ZP``D)"0D)"0
MD)"0D/\E)NL``)"0D)"0D)"0D)#_)4[J``"0D)"0D)"0D)"0_R4VZ@``D)"0
MD)"0D)"0D/\E_NH``)"0D)"0D)"0D)#_)5[J``"0D)"0D)"0D)"0_R5^Z0``
MD)"0D)"0D)"0D/\EWND``)"0D)"0D)"0D)#_)5;I``"0D)"0D)"0D)"0_R4F
MZ@``D)"0D)"0D)"0D/\EOND``)"0D)"0D)"0D)#_)<;I``"0D)"0D)"0D)"0
M_R46Z@``D)"0D)"0D)"0D/\EQND``)"0D)"0D)"0D)#_)2[I``"0D)"0D)"0
MD)"0_R6VZ0``D)"0D)"0D)"0D/\E/ND``)"0D)"0D)"0D)#_);;I``"0D)"0
MD)"0D)"0_R4FZ0``D)"0D)"0D)"0D%564TB)Y4B#["!(B=-(B<ZY"````.B6
M!```2(7;#X0M`@``2(ES0$B--0+]__](C0TK!P``2(L5D.@``$B%P,=#"$@!
M``!(B;.`````2(TU[/S__TB)2TA(C0T9!P``QT,,[P,``,=#$!,```!(B;.(
M````2(TUU?S__TB)4QA(B4M02(T-9L4``$B+50!(B;.0````2(TUQ/S__\>#
M\`````````!(B4LX2(T-;_W__\>#]`````H!``!(B;.8````2(TUU_S__TB)
M$TB)2RA(C0WI`P``2(ES($B--<[\__](B4MX2(ES,`^$Y0```$B+D.````!(
MA=(/A-4```"+!?K$``"%P`^%QP```$B#/;H4````#X2&`@``2(,]M!0````/
MA&@"``!(@SVN%`````^$2@(``$B#/:@4````#X0L`@``2(,]HA0````/A`X"
M``!(@SV<%`````^$\`$``$B#/984````#X32`0``2(,]D!0````/A+0!``!(
MBP5+%```2(D"2(L%210``$B)0@A(BP5&%```2(E"$$B+!4,4``!(B4(82(L%
M0!0``$B)0B!(BP4]%```2(E"*$B+!3H4``!(B4(P2(L%-Q0``$B)0C@QR>AD
M`P``2(F#Z````$B-!<83```QR4B)0UA(C04A%```2(E#8$B-!:ZS``!(B4-H
M2(T%H\4``$B)0W!(C07XA0``2(F#(`$``$B-!2:&``!(B8,H`0``2(T%?+/_
M_TB)@S`!``#HD`(``+@!````2(/$(%M>7<,/'P!(C14)!0``2(N(0`$``$B)
M<$!(C377^O__QT`(2`$``,=`#.\#``!(B5!(2(T5]@0``,=`$!,```!(B;"(
M````2(TUR?K__\>`\`````````!(B5!02(L5(.8``$B)PTB)L)@```!(C35C
M^___QX#T````"@$``$B)"DB-%5_Z__](C0W(^O__2(F0@````$B-%6KZ__](
MB9"0````2(T5_,(``$B)4#A(BU4`2(E(($B-#;D!``!(B7`H2(D02(T5F_K_
M_TB)2'A(B5`P2(N0X````$B%T@^%T_W__^FC_O__2(M".$B)!<L2``#I//[_
M_TB+0C!(B06S$@``Z1[^__](BT(H2(D%FQ(``.D`_O__2(M"($B)!8,2``#I
MXOW__TB+0AA(B05K$@``Z<3]__](BT(02(D%4Q(``.FF_?__2(M""$B)!3L2
M``#IB/W__TB+`DB)!202``#I:_W__Y"0D)"0D)"0D)"0D)"0D$R+!4G"``"+
M%4O"``!(BPU,P@``Z0<!```/'X``````2(/L.(/Z`71G<C6#^@)T$X/Z`W4<
MN@,```!(@\0XZ=T```"Z`@```$B#Q#CISP```+@!````2(/$.,,/'T0``#'2
MZ+D```"%P'3K2(L-UL$``(E$)"SHM0```$C'!<+!``#_____BT0D+.O*#Q]`
M`#'`2(T59____TV%P`^4P$B)#;K!``!,B06CP0``QP6AP0```0```(D%@\$`
M`.A^````2(/X_TB)!7O!```/E<`/ML!(@\0XPP\?A```````Z3O___^0D)"0
MD)"0D)"0D/\E1N0``)"0D)"0D)"0D)#_)1;D``"0D)"0D)"0D)"0PY"0D)"0
MD)"0D)"0D)"0D+@!````PY"0D)"0D)"0D)#_)?[C``"0D)"0D)"0D)"04TB#
M["!(B<M(B=%(C14>P0``Z.GZ__](C142P0``2(G92(/$(%OI!0```)"0D)"0
M_R7.XP``D)"0D)"0D)"0D/\EYN4``)"0_R7FY0``D)!64TB#[$B)3"1@2(E4
M)&A,B40D<.@'^/__2(U4)&A,C40D<$B-3"1@Z//V__],C07T]O__2(T-]?;_
M_TR)PNA-^O__2(L%=G\``$B+'7]_``"`.`!U,^C=]O__2(7`2(D%&\````^$
M<@$``$B)P>C-]O__2(L=5G\``$B+"^A^^/__QH#T!````$B+"TB--68C``#H
M:/C__X"(8P4```)(BPOH6?C__X"(8P4```&+1"1@C4@%2&/)2,'A`^B']O__
M28G!2(D%G;\``$B+1"1H1(M4)&#'!9&_```$````2(L028EQ"$B--0\C``!!
M@_H!28EQ&$F)$4B+%6D/``!)B5$0?B%!C5+^3(T$U0@````QTDB+3!`(28E,
M$2!(@\((3#G"=>U!C4($2(L-3[\``$6-0@-(C16TWO__2,=$)"``````2)A)
MQT3!^`````#H^_7__X7`B<-T/TB+#1Z_``!(C70D/.CL]?__2(T-Q1\``$B)
M\N@5U___2(7`="I(BPWYO@``Z-3U___HU_7__XG82(/$2%M>PTB+#=^^``#H
MRO7__XG#Z[%(BPW'O@``2(7)=`?H-?3__^O#2(T-(A\``$B)\NC$UO__2(7`
M2(D%HKX``'2H2(G!Z]FY`0```.BY^/__D)"0D)"0D)"054B)Y5WIQK[__Y"0
MD)"0D%5(B>5=Z1:___^0D)"0D)#__________Y!10``!``````````````#_
M_________Z!10``!````````````````````````````````````````````
M`````````````/____\`````H'-```$`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````8WEG9V-J+3$T+F1L;`!?2G9?
M4F5G:7-T97)#;&%S<V5S````````````````````+@`N+@`E<R\E<P``+W!R
M;V,O<V5L9B]E>&4`<&5R;'AS:2YC`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO
M861E<@!);G1E<FYA;',Z.E!!4CHZ0D]/5`!005)?5$5-4`!005)?4%)/1TY!
M344`.@`O`"5S)7,E<P!015),-4Q)0@!015),3$E"`%!%4DPU3U!4`%!%4DQ)
M3P!005)?24Y)5$E!3$E:140`4$%27U-005=.140`4$%27T-,14%.`%!!4E]$
M14)51P!005)?0T%#2$4`4$%27T%21T,`4$%27T%21U9?,`!005)?1TQ/0D%,
M7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!
M3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`4$%27T-,14%./0`Q`````````%]?
M14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@``!,24)0051(`$Q$7TQ)
M0E)!4EE?4$%42`!,24)205)97U!!5$@`4$%42`!$64Q$7TQ)0E)!4EE?4$%4
M2`!32$Q)0E]0051(`%5315(`4UE35$5-`"]T;7``5$U01$E2`%1%35!$25(`
M5$5-4`!435``55-%4DY!344`)3`R>``E<R5S)7,E<P!P87(M````)7,Z(&-R
M96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R
M;F\])6DI"@!P87)L+F5X90`N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP
M+25U)7,`)7,E<W1E;7`M)74M)74E<P``)7,Z('!R:79A=&4@<W5B9&ER96-T
M;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@
M>6]U<B!O<&5R871I;VXI"@!!4D=6`!@`<&5R;``P``````````!5;F%B;&4@
M=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``````````)7,Z(&5X96-U=&EO
M;B!O9B`E<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!C<F5A=&EO;B!O9B!P
M<FEV871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I
M*0H`+64`+2T``",A+W5S<B]B:6XO<&5R;`IE=F%L("=E>&5C("]U<W(O8FEN
M+W!E<FP@("U3("0P("1[,2LB)$`B?2<*("`@(&EF(#`[(",@;F]T(')U;FYI
M;F<@=6YD97(@<V]M92!S:&5L;`H*<&%C:V%G92!?7W!A<E]P;#L*"B,@+2TM
M(%1H:7,@<V-R:7!T(&UU<W0@;F]T('5S92!A;GD@;6]D=6QE<R!A="!C;VUP
M:6QE('1I;64@+2TM"B,@=7-E('-T<FEC=#L*"B-L:6YE(#$U.`H*;7D@*"1P
M87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92D["D5.1"![(&EF("@D14Y6
M>U!!4E]#3$5!3GTI('L*("`@(')E<75I<F4@1FEL93HZ5&5M<#L*("`@(')E
M<75I<F4@1FEL93HZ0F%S96YA;64["B`@("!R97%U:7)E($9I;&4Z.E-P96,[
M"B`@("!M>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A
M<E]T96UP*3L*("`@(&]U=',H<7%[4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?
M=&5M<")]*3L*("`@($9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD
M("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PH@("`@<FUD:7(@
M)'!A<E]T96UP.PH@("`@(R!$;VXG="!R96UO=F4@=&]P9&ER(&)E8V%U<V4@
M=&AI<R!C875S97,@82!R86-E('=I=&@@;W1H97(@87!P<PH@("`@(R!T:&%T
M(&%R92!T<GEI;F<@=&\@<W1A<G0N"@H@("`@:68@*"UD("1P87)?=&5M<"`F
M)B`D7D\@;F4@)TU35VEN,S(G*2!["B`@("`@("`@(R!3;VUE=&AI;F<@=V5N
M="!W<F]N9R!U;FQI;FMI;F<@=&AE('1E;7!O<F%R>2!D:7)E8W1O<GDN("!4
M:&ES"B`@("`@("`@(R!T>7!I8V%L;'D@:&%P<&5N<R!O;B!P;&%T9F]R;7,@
M=&AA="!D:7-A;&QO=R!U;FQI;FMI;F<@<VAA<F5D"B`@("`@("`@(R!L:6)R
M87)I97,@86YD(&5X96-U=&%B;&5S('1H870@87)E(&EN('5S92X@56YL:6YK
M('=I=&@@82!B86-K9W)O=6YD"B`@("`@("`@(R!S:&5L;"!C;VUM86YD('-O
M('1H92!F:6QE<R!A<F4@;F\@;&]N9V5R(&EN('5S92!B>2!T:&ES('!R;V-E
M<W,N"B`@("`@("`@(R!$;VXG="!D;R!A;GET:&EN9R!O;B!7:6YD;W=S(&)E
M8V%U<V4@;W5R('!A<F5N="!P<F]C97-S('=I;&P*("`@("`@("`C('1A:V4@
M8V%R92!O9B!C;&5A;FEN9R!T:&EN9W,@=7`N"@H@("`@("`@(&UY("1T;7`@
M/2!N97<@1FEL93HZ5&5M<"@*("`@("`@("`@("`@5$5-4$Q!5$4@/3X@)W1M
M<%A86%A8)RP*("`@("`@("`@("`@1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID
M:7)N86UE*"1T;W!D:7(I+`H@("`@("`@("`@("!3549&25@@/3X@)RYC;60G
M+`H@("`@("`@("`@("!53DQ)3DL@/3X@,"P*("`@("`@("`I.PH*("`@("`@
M("!P<FEN="`D=&UP("(C(2]B:6XO<V@*>#TQ.R!W:&EL92!;(%PD>"`M;'0@
M,3`@73L@9&\*("`@<FT@+7)F("<D<&%R7W1E;7`G"B`@(&EF(%L@7"$@+60@
M)R1P87)?=&5M<"<@73L@=&AE;@H@("`@("`@8G)E86L*("`@9FD*("`@<VQE
M97`@,0H@("!X/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R(@+B`D=&UP+3YF
M:6QE;F%M92`N("(G"B(["B`@("`@("`@("`@(&-H;6]D(#`W,#`L)'1M<"T^
M9FEL96YA;64["B`@("`@("`@;7D@)&-M9"`]("1T;7`M/F9I;&5N86UE("X@
M)R`^+V1E=B]N=6QL(#(^)C$@)B<["B`@("`@("`@8VQO<V4@)'1M<#L*("`@
M("`@("!S>7-T96TH)&-M9"D["B`@("`@("`@;W5T<RAQ<2A3<&%W;F5D(&)A
M8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("D*("`@("`@
M("`@("`@("X@)'1M<"T^9FEL96YA;64I.PH@("`@?0I]('T*"D)%1TE.('L*
M("`@($EN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N
M86QS.CI005(Z.D)/3U0["@H@("`@979A;"!["@I?<&%R7VEN:71?96YV*"D[
M"@II9B`H97AI<W1S("1%3E9[4$%27T%21U9?,'T@86YD("1%3E9[4$%27T%2
M1U9?,'T@*2!["B`@("!`05)'5B`](&UA<"`D14Y6>R)005)?05)'5E\D7R)]
M+"`H,2`N+B`D14Y6>U!!4E]!4D=#?2`M(#$I.PH@("`@)#`@/2`D14Y6>U!!
M4E]!4D=67S!].PI]"F5L<V4@>PH@("`@9F]R("AK97ES("5%3E8I('L*("`@
M("`@("!D96QE=&4@)$5.5GLD7WT@:68@+UY005)?05)'5E\O.PH@("`@?0I]
M"@IM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PH*(R!F:7@@)'!R;V=N
M86UE(&EF(&EN=F]K960@9G)O;2!0051("FUY("5#;VYF:6<@/2`H"B`@("!P
M871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*
M("`@(%]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG
M=VEN*2\@/R`G+F5X92<@.B`G)RDL"B`@("!?9&5L:6T@("`@("`]/B`H)%Y/
M(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PH*7W-E=%]P<F]G
M;F%M92@I.PI?<V5T7W!A<E]T96UP*"D["@HC($UA9VEC('-T<FEN9R!C:&5C
M:VEN9R!A;F0@97AT<F%C=&EN9R!B=6YD;&5D(&UO9'5L97,@>WM["FUY("@D
M<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI["B`@("!L;V-A;"`D4TE'>U]?5T%2
M3E]??2`]('-U8B![?3L*"B`@("`C($-H96-K(&9I;&4@='EP92P@9V5T('-T
M87)T(&]F(&1A=&$@<V5C=&EO;B![>WL*("`@(&]P96X@7T9(+"`G/"<L("1P
M<F]G;F%M92!O<B!L87-T.PH@("`@8FEN;6]D92A?1D@I.PH*("`@(&UY("1B
M=68["B`@("!S965K(%]&2"P@+3@L(#(["B`@("!R96%D(%]&2"P@)&)U9BP@
M.#L*("`@(&QA<W0@=6YL97-S("1B=68@97$@(EQN4$%2+G!M7&XB.PH*("`@
M('-E96L@7T9(+"`M,3(L(#(["B`@("!R96%D(%]&2"P@)&)U9BP@-#L*("`@
M('-E96L@7T9(+"`M,3(@+2!U;G!A8VLH(DXB+"`D8G5F*2P@,CL*("`@(')E
M860@7T9(+"`D8G5F+"`T.PH*("`@("1D871A7W!O<R`]("AT96QL(%]&2"D@
M+2`T.PH@("`@(R!]?7T*"B`@("`C($5X=')A8W1I;F<@96%C:"!F:6QE(&EN
M=&\@;65M;W)Y('M[>PH@("`@;7D@)7)E<75I<F5?;&ES=#L*("`@('=H:6QE
M("@D8G5F(&5Q(")&24Q%(BD@>PH@("`@("`@(')E860@7T9(+"`D8G5F+"`T
M.PH@("`@("`@(')E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*
M"B`@("`@("`@;7D@)&9U;&QN86UE(#T@)&)U9CL*("`@("`@("!O=71S*'%Q
M*%5N<&%C:VEN9R!F:6QE("(D9G5L;&YA;64B+BXN*2D["B`@("`@("`@;7D@
M)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@
M)#$@.B!U;F1E9CL*("`@("`@("!M>2`H)&)A<V5N86UE+"`D97AT*2`]("@D
M8G5F(#U^(&U\*#\Z+BHO*3\H+BHI*%PN+BHI?"D["@H@("`@("`@(')E860@
M7T9(+"`D8G5F+"`T.PH@("`@("`@(')E860@7T9(+"`D8G5F+"!U;G!A8VLH
M(DXB+"`D8G5F*3L*"B`@("`@("`@:68@*&1E9FEN960H)&5X="D@86YD("1E
M>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["B`@("`@("`@("`@(&UY
M("@D;W5T+"`D9FEL96YA;64I(#T@7W1E;7!F:6QE*"1E>'0L("1C<F,I.PH@
M("`@("`@("`@("!I9B`H)&]U="D@>PH@("`@("`@("`@("`@("`@8FEN;6]D
M92@D;W5T*3L*("`@("`@("`@("`@("`@('!R:6YT("1O=70@)&)U9CL*("`@
M("`@("`@("`@("`@(&-L;W-E("1O=70["B`@("`@("`@("`@("`@("!C:&UO
M9"`P-S4U+"`D9FEL96YA;64["B`@("`@("`@("`@('T*("`@("`@("`@("`@
M)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE
M.PH@("`@("`@("`@("`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M
M97T@/2`D9G5L;&YA;64["B`@("`@("`@?0H@("`@("`@(&5L<VEF("@@)&9U
M;&QN86UE(#U^(&U\7B\_<VAL:6(O?"!A;F0@9&5F:6YE9"`D14Y6>U!!4E]4
M14U0?2`I('L*("`@("`@("`@("`@(R!S:&]U;&0@8F4@;6]V960@=&\@7W1E
M;7!F:6QE*"D*("`@("`@("`@("`@;7D@)&9I;&5N86UE(#T@(B1%3E9[4$%2
M7U1%35!]+R1B87-E;F%M921E>'0B.PH@("`@("`@("`@("!O=71S*")32$Q)
M0CH@)&9I;&5N86UE7&XB*3L*("`@("`@("`@("`@;W!E;B!M>2`D;W5T+"`G
M/B<L("1F:6QE;F%M92!O<B!D:64@)"$["B`@("`@("`@("`@(&)I;FUO9&4H
M)&]U="D["B`@("`@("`@("`@('!R:6YT("1O=70@)&)U9CL*("`@("`@("`@
M("`@8VQO<V4@)&]U=#L*("`@("`@("!]"B`@("`@("`@96QS92!["B`@("`@
M("`@("`@("1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2`]"B`@("`@("`@("`@
M("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PH@("`@
M("`@("`@("`@("`@8G5F(#T^("1B=68L"B`@("`@("`@("`@("`@("!C<F,@
M/3X@)&-R8RP*("`@("`@("`@("`@("`@(&YA;64@/3X@)&9U;&QN86UE+`H@
M("`@("`@("`@("!].PH@("`@("`@('T*("`@("`@("!R96%D(%]&2"P@)&)U
M9BP@-#L*("`@('T*("`@(",@?7U]"@H@("`@;&]C86P@0$E.0R`]("AS=6(@
M>PH@("`@("`@(&UY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*"B`@("`@("`@
M<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PH*("`@("`@("!M
M>2`D9FEL96YA;64@/2!D96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?2!\
M?"!D;R!["B`@("`@("`@("`@(&UY("1K97D["B`@("`@("`@("`@(&9O<F5A
M8V@@*&ME>7,@)7)E<75I<F5?;&ES="D@>PH@("`@("`@("`@("`@("`@;F5X
M="!U;FQE<W,@+UQ1)&UO9'5L95Q%)"\["B`@("`@("`@("`@("`@("`D:V5Y
M(#T@)%\[(&QA<W0["B`@("`@("`@("`@('T*("`@("`@("`@("`@9&5L971E
M("1R97%U:7)E7VQI<W1[)&ME>7T@:68@9&5F:6YE9"@D:V5Y*3L*("`@("`@
M("!](&]R(')E='5R;CL*"B`@("`@("`@)$E.0WLD;6]D=6QE?2`]("(O;&]A
M9&5R+R1F:6QE;F%M92\D;6]D=6QE(CL*"B`@("`@("`@:68@*"1%3E9[4$%2
M7T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*("`@("`@
M("`@("`@;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I92`D
M(3L*("`@("`@("`@("`@8FEN;6]D92@D9F@I.PH@("`@("`@("`@("!P<FEN
M="`D9F@@)&9I;&5N86UE+3Y[8G5F?3L*("`@("`@("`@("`@<V5E:R@D9F@L
M(#`L(#`I.PH@("`@("`@("`@("!R971U<FX@)&9H.PH@("`@("`@('T*("`@
M("`@("!E;'-E('L*("`@("`@("`@("`@;7D@*"1O=70L("1N86UE*2`](%]T
M96UP9FEL92@G+G!M)RP@)&9I;&5N86UE+3Y[8W)C?2D["B`@("`@("`@("`@
M(&EF("@D;W5T*2!["B`@("`@("`@("`@("`@("!B:6YM;V1E*"1O=70I.PH@
M("`@("`@("`@("`@("`@<')I;G0@)&]U="`D9FEL96YA;64M/GMB=69].PH@
M("`@("`@("`@("`@("`@8VQO<V4@)&]U=#L*("`@("`@("`@("`@?0H@("`@
M("`@("`@("!O<&5N(&UY("1F:"P@)SPG+"`D;F%M92!O<B!D:64@)"$["B`@
M("`@("`@("`@(&)I;FUO9&4H)&9H*3L*("`@("`@("`@("`@<F5T=7)N("1F
M:#L*("`@("`@("!]"@H@("`@("`@(&1I92`B0F]O='-T<F%P<&EN9R!F86EL
M960Z(&-A;FYO="!F:6YD("1M;V1U;&4A7&XB.PH@("`@?2P@0$E.0RD["@H@
M("`@(R!.;W<@;&]A9"!A;&P@8G5N9&QE9"!F:6QE<R![>WL*"B`@("`C(&EN
M:71I86QI>F4@<VAA<F5D(&]B:F5C="!P<F]C97-S:6YG"B`@("!R97%U:7)E
M(%A33&]A9&5R.PH@("`@<F5Q=6ER92!005(Z.DAE879Y.PH@("`@<F5Q=6ER
M92!#87)P.CI(96%V>3L*("`@(')E<75I<F4@17AP;W)T97(Z.DAE879Y.PH@
M("`@4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PH*("`@(",@;F]W
M(&QE="=S('1R>2!G971T:6YG(&AE;'!E<B!M;V1U;&5S(&9R;VT@=VET:&EN
M"B`@("!R97%U:7)E($E/.CI&:6QE.PH*("`@(",@;&]A9"!R97-T(&]F('1H
M92!G<F]U<"!I;@H@("`@=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME
M>7,@)7)E<75I<F5?;&ES="E;,%TI('L*("`@("`@("`C;&]C86P@)$E.0WLG
M0W=D+G!M)WT@/2!?7T9)3$5?7R!I9B`D7D\@;F4@)TU35VEN,S(G.PH@("`@
M("`@('5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O
M0E-$4$%.+RD@>PH@("`@("`@("`@("`C(')E<75I<F4@;6]D=6QE<RP@9&\@
M;W1H97(@97AE8W5T86)L92!F:6QE<PH@("`@("`@("`@("!I9B`H)&9I;&5N
M86UE(#U^("]<+G!M8S\D+VDI('L*("`@("`@("`@("`@("`@(')E<75I<F4@
M)&9I;&5N86UE.PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<V4@>PH@
M("`@("`@("`@("`@("`@(R!3:VEP($%C=&EV95-T871E)W,@<VET96-U<W1O
M;6EZ92YP;"!F:6QE.@H@("`@("`@("`@("`@("`@9&\@)&9I;&5N86UE('5N
M;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["B`@("`@
M("`@("`@('T*("`@("`@("!]"B`@("`@("`@9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*("`@('T*"B`@("`C('U]?0H*("`@(&QA<W0@=6YL
M97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PH@("`@)'-T87)T7W!O<R`]("AT
M96QL(%]&2"D@+2`T.PI]"B,@?7U]"@HC($%R9W5M96YT('!R;V-E<W-I;F<@
M>WM["FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PH*9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@
M<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"@HD
M<75I970@/2`P('5N;&5S<R`D14Y6>U!!4E]$14)51WT["B,@1&]N)W0@<W=A
M;&QO=R!A<F=U;65N=',@9F]R(&-O;7!I;&5D(&5X96-U=&%B;&5S('=I=&AO
M=70@+2UP87(M;W!T:6]N<PII9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=
M(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*("`@(&UY("5D:7-T
M7V-M9"`]('%W*`H@("`@("`@('`@("!B;&EB7W1O7W!A<@H@("`@("`@(&D@
M("!I;G-T86QL7W!A<@H@("`@("`@('4@("!U;FEN<W1A;&Q?<&%R"B`@("`@
M("`@<R`@('-I9VY?<&%R"B`@("`@("`@=B`@('9E<FEF>5]P87(*("`@("D[
M"@H@("`@(R!I9B!T:&4@87!P(&ES(&EN=F]K960@87,@(F%P<&YA;64@+2UP
M87(M;W!T:6]N<R`M+7)E=7-E(%!23T=204T@0%!23T=?05)'5B(L"B`@("`C
M('5S92!T:&4@87!P('1O(')U;B!T:&4@9VEV96X@<&5R;"!C;V1E(&EN<W1E
M860@;V8@86YY=&AI;F<@9G)O;2!T:&4*("`@(",@87!P(&ET<V5L9B`H8G5T
M('-T:6QL('-E="!U<"!T:&4@;F]R;6%L(&%P<"!E;G9I<F]N;65N="!A;F0@
M0$E.0RD*("`@(&EF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G
M*2!["B`@("`@("`@<VAI9G0@0$%21U8["B`@("`@("`@)$5.5GM005)?05!0
M7U)%55-%?2`]('-H:69T($!!4D=6.PH@("`@?0H@("`@96QS92![(",@;F]R
M;6%L('!A<FP@8F5H879I;W5R"@H@("`@("`@(&UY($!A9&1?=&]?:6YC.PH@
M("`@("`@('=H:6QE("A`05)'5BD@>PH@("`@("`@("`@("`D05)'5ELP72`]
M?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T.PH*("`@("`@
M("`@("`@:68@*"0Q(&5Q("=))RD@>PH@("`@("`@("`@("`@("`@<'5S:"!`
M861D7W1O7VEN8RP@)#(["B`@("`@("`@("`@('T*("`@("`@("`@("`@96QS
M:68@*"0Q(&5Q("=-)RD@>PH@("`@("`@("`@("`@("`@979A;"`B=7-E("0R
M(CL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E;'-I9B`H)#$@97$@)T$G
M*2!["B`@("`@("`@("`@("`@("!U;G-H:69T($!P87)?87)G<RP@)#(["B`@
M("`@("`@("`@('T*("`@("`@("`@("`@96QS:68@*"0Q(&5Q("=/)RD@>PH@
M("`@("`@("`@("`@("`@)&]U="`]("0R.PH@("`@("`@("`@("!]"B`@("`@
M("`@("`@(&5L<VEF("@D,2!E<2`G8B<I('L*("`@("`@("`@("`@("`@("1B
M=6YD;&4@/2`G<VET92<["B`@("`@("`@("`@('T*("`@("`@("`@("`@96QS
M:68@*"0Q(&5Q("=")RD@>PH@("`@("`@("`@("`@("`@)&)U;F1L92`]("=A
M;&PG.PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<VEF("@D,2!E<2`G
M<2<I('L*("`@("`@("`@("`@("`@("1Q=6EE="`](#$["B`@("`@("`@("`@
M('T*("`@("`@("`@("`@96QS:68@*"0Q(&5Q("=,)RD@>PH@("`@("`@("`@
M("`@("`@;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92`B6%A8.B!#86YN
M;W0@;W!E;B!L;V<Z("0A(CL*("`@("`@("`@("`@?0H@("`@("`@("`@("!E
M;'-I9B`H)#$@97$@)U0G*2!["B`@("`@("`@("`@("`@("`D8V%C:&5?;F%M
M92`]("0R.PH@("`@("`@("`@("!]"@H@("`@("`@("`@("!S:&EF="A`05)'
M5BD["@H@("`@("`@("`@("!I9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI
M('L*("`@("`@("`@("`@("`@(&1E;&5T92`D14Y6>R=005)?5$5-4"=].PH@
M("`@("`@("`@("`@("`@:6YI=%]I;F,H*3L*("`@("`@("`@("`@("`@(')E
M<75I<F4@4$%2.CI$:7-T.PH@("`@("`@("`@("`@("`@)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B`@("`@("`@("`@("`@("`F>R)0
M05(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PH@("`@("`@("`@("`@
M("`@97AI=#L*("`@("`@("`@("`@?0H@("`@("`@('T*"B`@("`@("`@=6YS
M:&EF="!`24Y#+"!`861D7W1O7VEN8SL*("`@('T*?0H*(R!86%@@+2T@861D
M("TM<&%R+61E8G5G('-U<'!O<G0A"@HC('U]?0H*(R!/=71P=70@;6]D92`H
M+4\I(&AA;F1L:6YG('M[>PII9B`H)&]U="D@>PH@("`@>PH@("`@("`@("-L
M;V-A;"`D24Y#>R=#=V0N<&TG?2`](%]?1DE,15]?(&EF("1>3R!N92`G35-7
M:6XS,B<["B`@("`@("`@<F5Q=6ER92!)3SHZ1FEL93L*("`@("`@("!R97%U
M:7)E($%R8VAI=F4Z.EII<#L*("`@('T*"B`@("!M>2`D<&%R(#T@<VAI9G0H
M0$%21U8I.PH@("`@;7D@)'II<#L*"@H@("`@:68@*&1E9FEN960@)'!A<BD@
M>PH@("`@("`@(&]P96X@;7D@)&9H+"`G/"<L("1P87(@;W(@9&EE(")#86YN
M;W0@9FEN9"`G)'!A<B<Z("0A(CL*("`@("`@("!B:6YM;V1E*"1F:"D["B`@
M("`@("`@8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*"B`@("`@("`@)'II<"`]
M($%R8VAI=F4Z.EII<"T^;F5W.PH@("`@("`@("@@)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M("D*("`@("`@("`@("`@;W(@9&EE(")296%D("<D<&%R)R!E<G)O<CH@)"$B
M.PH@("`@?0H*"B`@("!M>2`E96YV(#T@9&\@>PH@("`@("`@(&EF("@D>FEP
M(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2![
M"B`@("`@("`@("`@("1M971A(#U^(',O+BI><&%R.B0O+VUS.PH@("`@("`@
M("`@("`D;65T82`]?B!S+UY<4RXJ+R]M<SL*("`@("`@("`@("`@)&UE=&$@
M/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PH@("`@("`@('T*("`@('T["@H@
M("`@(R!/<&5N(&EN<'5T(&%N9"!O=71P=70@9FEL97,@>WM["B`@("!L;V-A
M;"`D+R`](%PT.PH*("`@(&EF("AD969I;F5D("1P87(I('L*("`@("`@("!O
M<&5N(%!!4BP@)SPG+"`D<&%R(&]R(&1I92`B)"$Z("1P87(B.PH@("`@("`@
M(&)I;FUO9&4H4$%2*3L*("`@("`@("!D:64@(B1P87(@:7,@;F]T(&$@4$%2
M(&9I;&4B('5N;&5S<R`\4$%2/B!E<2`B4$M<,#`S7#`P-"(["B`@("!]"@H@
M("`@0W)E871E4&%T:"@D;W5T*2`["B`@("`*("`@(&UY("1F:"`]($E/.CI&
M:6QE+3YN97<H"B`@("`@("`@)&]U="P*("`@("`@("!)3SHZ1FEL93HZ3U]#
M4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)3SHZ1FEL93HZ3U]4
M4E5.0R@I+`H@("`@("`@(#`W-S<L"B`@("`I(&]R(&1I92`D(3L*("`@(&)I
M;FUO9&4H)&9H*3L*"B`@("`D+R`]("AD969I;F5D("1D871A7W!O<RD@/R!<
M)&1A=&%?<&]S(#H@=6YD968["B`@("!S965K(%]&2"P@,"P@,#L*("`@(&UY
M("1L;V%D97(@/2!S8V%L87(@/%]&2#X["B`@("!I9B`H(21%3E9[4$%27U9%
M4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*("`@
M("`@("!R97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*("`@("`@("!0
M05(Z.D9I;'1E<CHZ4&]D4W1R:7`M/FYE=RT^87!P;'DH7"1L;V%D97(L("0P
M*0H@("`@?0H@("`@9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)65N=BD@
M>PH@("`@("`@(&UY("1V86P@/2`D96YV>R1K97E](&]R(&YE>'0["B`@("`@
M("`@)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>6R<B72\["B`@("`@
M("`@;7D@)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B7U\B
M.PH@("`@("`@(&UY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N("(])'9A
M;"(["B`@("`@("`@)&QO861E<B`]?B!S>R1M86=I8R@@*RE]>PH@("`@("`@
M("`@("`D;6%G:6,@+B`D<V5T("X@*"<@)R!X("AL96YG=&@H)#$I("T@;&5N
M9W1H*"1S970I*2D*("`@("`@("!]96<["B`@("!]"B`@("`D9F@M/G!R:6YT
M*"1L;V%D97(I.PH@("`@)"\@/2!U;F1E9CL*("`@(",@?7U]"@H@("`@(R!7
M<FET92!B=6YD;&5D(&UO9'5L97,@>WM["B`@("!I9B`H)&)U;F1L92D@>PH@
M("`@("`@(')E<75I<F4@4$%2.CI(96%V>3L*("`@("`@("!005(Z.DAE879Y
M.CI?:6YI=%]D>6YA;&]A9&5R*"D["B`@("`@("`@:6YI=%]I;F,H*3L*"B`@
M("`@("`@<F5Q=6ER95]M;V1U;&5S*"D["@H@("`@("`@(&UY($!I;F,@/2!S
M;W)T('L*("`@("`@("`@("`@;&5N9W1H*"1B*2`\/3X@;&5N9W1H*"1A*0H@
M("`@("`@('T@9W)E<"!["B`@("`@("`@("`@("$O0E-$4$%.+PH@("`@("`@
M('T@9W)E<"!["B`@("`@("`@("`@("@D8G5N9&QE(&YE("=S:71E)RD@;W(*
M("`@("`@("`@("`@*"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP
M?2!A;F0*("`@("`@("`@("`@("1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV
M;&EB97AP?2D["B`@("`@("`@?2!`24Y#.PH*("`@("`@("`C($9I;&4@97AI
M<W1S('1E<W0@861D960@=&\@9FEX(%)4(",T,3<Y,#H*("`@("`@("`C($9U
M;FYY+"!N;VXM97AI<W1I;F<@96YT<GD@:6X@7SPN+BXN875T;R]#;VUP<F5S
M<R]287<O6FQI8B]A=71O<W!L:70N:7@N"B`@("`@("`@(R!4:&ES(&ES(&$@
M8F%N9"UA:60@9FEX('=I=&@@;F\@9&5E<&5R(&=R87-P(&]F('1H92!I<W-U
M92X*("`@("`@("`C(%-O;65B;V1Y('!L96%S92!G;R!T:')O=6=H('1H92!P
M86EN(&]F('5N9&5R<W1A;F1I;F<@=VAA="=S(&AA<'!E;FEN9RP*("`@("`@
M("`C($D@9F%I;&5D+B`M+2!3=&5F9F5N"B`@("`@("`@;7D@)69I;&5S.PH@
M("`@("`@("]>7SPH+BLI)"\@86YD("UE("0Q(&%N9"`D9FEL97-[)#%]*RL@
M9F]R(&ME>7,@)3HZ.PH@("`@("`@("1F:6QE<WLD7WTK*R!F;W(@=F%L=65S
M("5)3D,["@H@("`@("`@(&UY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG
M>VQI8E]E>'1].PH@("`@("`@(&UY("5W<FET=&5N.PH*("`@("`@("!F;W)E
M86-H("AS;W)T(&ME>7,@)69I;&5S*2!["B`@("`@("`@("`@(&UY("@D;F%M
M92P@)&9I;&4I.PH*("`@("`@("`@("`@9F]R96%C:"!M>2`D9&ER("A`:6YC
M*2!["B`@("`@("`@("`@("`@("!I9B`H)&YA;64@/2`D4$%2.CI(96%V>3HZ
M1G5L;$-A8VAE>R1??2D@>PH@("`@("`@("`@("`@("`@("`@("1F:6QE(#T@
M)%\["B`@("`@("`@("`@("`@("`@("`@;&%S=#L*("`@("`@("`@("`@("`@
M('T*("`@("`@("`@("`@("`@(&5L<VEF("@O7BA<421D:7)<15PO*"XJ6UY#
M8UTI*5Q:+VDI('L*("`@("`@("`@("`@("`@("`@("`H)&9I;&4L("1N86UE
M*2`]("@D,2P@)#(I.PH@("`@("`@("`@("`@("`@("`@(&QA<W0["B`@("`@
M("`@("`@("`@("!]"B`@("`@("`@("`@("`@("!E;'-I9B`H;2%>+VQO861E
M<B];7B]=*R\H+BI;7D-C72E<6B$I('L*("`@("`@("`@("`@("`@("`@("!I
M9B`H;7D@)')E9B`]("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD,7TI('L*
M("`@("`@("`@("`@("`@("`@("`@("`@*"1F:6QE+"`D;F%M92D@/2`H)')E
M9BP@)#$I.PH@("`@("`@("`@("`@("`@("`@("`@("!L87-T.PH@("`@("`@
M("`@("`@("`@("`@('T*("`@("`@("`@("`@("`@("`@("!E;'-I9B`H+68@
M(B1D:7(O)#$B*2!["B`@("`@("`@("`@("`@("`@("`@("`@("@D9FEL92P@
M)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PH@("`@("`@("`@("`@("`@("`@
M("`@("!L87-T.PH@("`@("`@("`@("`@("`@("`@('T*("`@("`@("`@("`@
M("`@('T*("`@("`@("`@("`@?0H*("`@("`@("`@("`@;F5X="!U;FQE<W,@
M9&5F:6YE9"`D;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE?2LK.PH@("`@
M("`@("`@("!N97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<
M421L:6)?97AT7$4D+SL*("`@("`@("`@("`@;W5T<R@@:F]I;B`B(BP*("`@
M("`@("`@("`@("`@('%Q*%!A8VMI;F<@(BDL(')E9B`D9FEL92`_("1F:6QE
M+3Y[;F%M97T@.B`D9FEL92P*("`@("`@("`@("`@("`@('%Q*"(N+BXI"B`@
M("`@("`@("`@("D["@H@("`@("`@("`@("!M>2`D8V]N=&5N=#L*("`@("`@
M("`@("`@:68@*')E9B@D9FEL92DI('L*("`@("`@("`@("`@("`@("1C;VYT
M96YT(#T@)&9I;&4M/GMB=69].PH@("`@("`@("`@("!]"B`@("`@("`@("`@
M(&5L<V4@>PH@("`@("`@("`@("`@("`@;W!E;B!&24Q%+"`G/"<L("1F:6QE
M(&]R(&1I92`B0V%N)W0@;W!E;B`D9FEL93H@)"$B.PH@("`@("`@("`@("`@
M("`@8FEN;6]D92A&24Q%*3L*("`@("`@("`@("`@("`@("1C;VYT96YT(#T@
M/$9)3$4^.PH@("`@("`@("`@("`@("`@8VQO<V4@1DE,13L*"B`@("`@("`@
M("`@("`@("!005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/FYE=RT^87!P;'DH7"1C
M;VYT96YT+"`D9FEL92D*("`@("`@("`@("`@("`@("`@("!I9B`A)$5.5GM0
M05)?5D520D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD[
M"@H@("`@("`@("`@("`@("`@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^
M;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["B`@("`@("`@
M("`@('T*"B`@("`@("`@("`@(&]U=',H<7$H5W)I='1E;B!A<R`B)&YA;64B
M*2D["B`@("`@("`@("`@("1F:"T^<')I;G0H(D9)3$4B*3L*("`@("`@("`@
M("`@)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N9W1H*"1N86UE*2`K(#DI*3L*
M("`@("`@("`@("`@)&9H+3YP<FEN="AS<')I;G1F*`H@("`@("`@("`@("`@
M("`@(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N
M=&5N="DL("1N86UE"B`@("`@("`@("`@("DI.PH@("`@("`@("`@("`D9F@M
M/G!R:6YT*'!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2D["B`@("`@("`@
M("`@("1F:"T^<')I;G0H)&-O;G1E;G0I.PH@("`@("`@('T*("`@('T*("`@
M(",@?7U]"@H@("`@(R!.;W<@=W)I=&4@;W5T('1H92!005(@86YD(&UA9VEC
M('-T<FEN9W,@>WM["B`@("`D>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I
M(&EF("1Z:7`["@H@("`@)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA
M;64L(#`L(#0P.PH@("`@:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE
M(#T@*'-T870H)&]U="DI6SE=*2!["B`@("`@("`@;7D@)&-T>"`](&5V86P@
M>R!R97%U:7)E($1I9V5S=#HZ4TA!.R!$:6=E<W0Z.E-(02T^;F5W*#$I('T*
M("`@("`@("`@("`@?'P@979A;"![(')E<75I<F4@1&EG97-T.CI32$$Q.R!$
M:6=E<W0Z.E-(03$M/FYE=R!]"B`@("`@("`@("`@('Q\(&5V86P@>R!R97%U
M:7)E($1I9V5S=#HZ340U.R!$:6=E<W0Z.DU$-2T^;F5W('T["@H@("`@("`@
M(",@5V]R:V%R;W5N9"!F;W(@8G5G(&EN($1I9V5S=#HZ4TA!(#4N,S@@86YD
M(#4N,SD*("`@("`@("!M>2`D<VAA7W9E<G-I;VX@/2!E=F%L('L@)$1I9V5S
M=#HZ4TA!.CI615)324].('T@?'P@,#L*("`@("`@("!I9B`H)'-H85]V97)S
M:6]N(&5Q("<U+C,X)R!O<B`D<VAA7W9E<G-I;VX@97$@)S4N,SDG*2!["B`@
M("`@("`@("`@("1C='@M/F%D9&9I;&4H)&]U="P@(F(B*2!I9B`H)&-T>"D[
M"B`@("`@("`@?0H@("`@("`@(&5L<V4@>PH@("`@("`@("`@("!I9B`H)&-T
M>"!A;F0@;W!E;BAM>2`D9F@L("(\)&]U="(I*2!["B`@("`@("`@("`@("`@
M("!B:6YM;V1E*"1F:"D["B`@("`@("`@("`@("`@("`D8W1X+3YA9&1F:6QE
M*"1F:"D["B`@("`@("`@("`@("`@("!C;&]S92@D9F@I.PH@("`@("`@("`@
M("!]"B`@("`@("`@?0H*("`@("`@("`D8V%C:&5?;F%M92`]("1C='@@/R`D
M8W1X+3YH97AD:6=E<W0@.B`D;71I;64["B`@("!]"B`@("`D8V%C:&5?;F%M
M92`N/2`B7#`B('@@*#0Q("T@;&5N9W1H("1C86-H95]N86UE*3L*("`@("1C
M86-H95]N86UE("X](")#04-(12(["B`@("`D9F@M/G!R:6YT*"1C86-H95]N
M86UE*3L*("`@("1F:"T^<')I;G0H<&%C:R@G3B<L("1F:"T^=&5L;"`M(&QE
M;F=T:"@D;&]A9&5R*2DI.PH@("`@)&9H+3YP<FEN="@B7&Y005(N<&U<;B(I
M.PH@("`@)&9H+3YC;&]S93L*("`@(&-H;6]D(#`W-34L("1O=70["B`@("`C
M('U]?0H*("`@(&5X:70["GT*(R!]?7T*"B,@4')E<&%R92`D<')O9VYA;64@
M:6YT;R!005(@9FEL92!C86-H92![>WL*>PH@("`@;&%S="!U;FQE<W,@9&5F
M:6YE9"`D<W1A<G1?<&]S.PH*("`@(%]F:7A?<')O9VYA;64H*3L*"B`@("`C
M($YO=R!L;V%D('1H92!005(@9FEL92!A;F0@<'5T(&ET(&EN=&\@4$%2.CI,
M:6)#86-H92![>WL*("`@(')E<75I<F4@4$%2.PH@("`@4$%2.CI(96%V>3HZ
M7VEN:71?9'EN86QO861E<B@I.PH*"B`@("!["B`@("`@("`@(VQO8V%L("1)
M3D-[)T-W9"YP;2=](#T@7U]&24Q%7U\@:68@)%Y/(&YE("=-4U=I;C,R)SL*
M("`@("`@("!R97%U:7)E($9I;&4Z.D9I;F0["B`@("`@("`@<F5Q=6ER92!!
M<F-H:79E.CI::7`["B`@("!]"B`@("!M>2`D>FEP(#T@07)C:&EV93HZ6FEP
M+3YN97<["B`@("!M>2`D9F@@/2!)3SHZ1FEL92T^;F5W.PH@("`@)&9H+3YF
M9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE("(D(3H@)$`B.PH@("`@
M)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+*"D@;W(@9&EE("(D(3H@)$`B.PH*("`@('!U
M<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B`@("`D4$%2.CI,:6)#86-H97LD
M<')O9VYA;65](#T@)'II<#L*"B`@("`D<75I970@/2`A)$5.5GM005)?1$5"
M54=].PH@("`@;W5T<RAQ<2A<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB*2D["@H@("`@:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@
M(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"B`@("`@("`@9F]R96%C
M:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*("`@("`@("`@("`@
M;F5X="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*("`@("`@("`@("`@;7D@
M)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64["B`@("`@("`@("`@
M(&YE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PH@("`@("`@("`@("`@
M("`@7@H@("`@("`@("`@("`@("`@+S]S:&QI8B\*("`@("`@("`@("`@("`@
M("@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PH@("`@("`@("`@("`@
M("`@*#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PH@("`@("`@("`@
M("`@("`@*%M>+UTK*0H@("`@("`@("`@("`@("`@)`H@("`@("`@("`@("!]
M>#L*("`@("`@("`@("`@;7D@)&5X=')A8W1?;F%M92`]("0Q.PH@("`@("`@
M("`@("!M>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6
M>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["B`@("`@("`@("`@(&EF("@M
M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D
M4VEZ92@I*2!["B`@("`@("`@("`@("`@("!O=71S*'%Q*%-K:7!P:6YG("(D
M;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S
M=%]N86UE(BDI.PH@("`@("`@("`@("!](&5L<V4@>PH@("`@("`@("`@("`@
M("`@;W5T<RAQ<2A%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S
M=%]N86UE(BDI.PH@("`@("`@("`@("`@("`@)&UE;6)E<BT^97AT<F%C=%1O
M1FEL94YA;65D*"1D97-T7VYA;64I.PH@("`@("`@("`@("`@("`@8VAM;V0H
M,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*("`@("`@("`@
M("`@?0H@("`@("`@('T*("`@('T*("`@(",@?7U]"GT*(R!]?7T*"B,@268@
M=&AE<F4G<R!N;R!M86EN+G!L('1O(')U;BP@<VAO=R!U<V%G92![>WL*=6YL
M97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["B`@("!D:64@/#P@
M(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M
M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@
M70H@("`@("`@)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@H@
M("`@)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF
M="A`05)'5BD["GT*(R!]?7T*"G-U8B!#<F5A=&50871H('L*("`@(&UY("@D
M;F%M92D@/2!`7SL*("`@(`H@("`@<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*
M("`@(&UY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N
M86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*("`@(`H@("`@<F5Q
M=6ER92!&:6QE.CI0871H.PH@("`@"B`@("!&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"@IS=6(@<F5Q=6ER95]M;V1U;&5S('L*("`@("-L;V-A;"`D24Y#
M>R=#=V0N<&TG?2`](%]?1DE,15]?(&EF("1>3R!N92`G35-7:6XS,B<["@H@
M("`@<F5Q=6ER92!L:6(["B`@("!R97%U:7)E($1Y;F%,;V%D97(["B`@("!R
M97%U:7)E(&EN=&5G97(["B`@("!R97%U:7)E('-T<FEC=#L*("`@(')E<75I
M<F4@=V%R;FEN9W,["B`@("!R97%U:7)E('9A<G,["B`@("!R97%U:7)E($-A
M<G`["B`@("!R97%U:7)E($-A<G`Z.DAE879Y.PH@("`@<F5Q=6ER92!%<G)N
M;SL*("`@(')E<75I<F4@17AP;W)T97(Z.DAE879Y.PH@("`@<F5Q=6ER92!%
M>'!O<G1E<CL*("`@(')E<75I<F4@1F-N=&P["B`@("!R97%U:7)E($9I;&4Z
M.E1E;7`["B`@("!R97%U:7)E($9I;&4Z.E-P96,["B`@("!R97%U:7)E(%A3
M3&]A9&5R.PH@("`@<F5Q=6ER92!#;VYF:6<["B`@("!R97%U:7)E($E/.CI(
M86YD;&4["B`@("!R97%U:7)E($E/.CI&:6QE.PH@("`@<F5Q=6ER92!#;VUP
M<F5S<SHZ6FQI8CL*("`@(')E<75I<F4@07)C:&EV93HZ6FEP.PH@("`@<F5Q
M=6ER92!005(["B`@("!R97%U:7)E(%!!4CHZ2&5A=GD["B`@("!R97%U:7)E
M(%!!4CHZ1&ES=#L*("`@(')E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PH@("`@<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PH@("`@
M<F5Q=6ER92!A='1R:6)U=&5S.PH@("`@979A;"![(')E<75I<F4@0W=D('T[
M"B`@("!E=F%L('L@<F5Q=6ER92!7:6XS,B!].PH@("`@979A;"![(')E<75I
M<F4@4V-A;&%R.CI5=&EL('T["B`@("!E=F%L('L@<F5Q=6ER92!!<F-H:79E
M.CI5;GII<#HZ0G5R<W0@?3L*("`@(&5V86P@>R!R97%U:7)E(%1I93HZ2&%S
M:#HZ3F%M961#87!T=7)E('T["B`@("!E=F%L('L@<F5Q=6ER92!097)L24\[
M(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*?0H*(R!4:&4@0R!V97)S:6]N
M(&]F('1H:7,@8V]D92!A<'!E87)S(&EN(&UY;&1R+VUK=&UP9&ER+F,*(R!4
M:&ES(&-O9&4@86QS;R!L:79E<R!I;B!005(Z.E-E='5P5&5M<"!A<R!S971?
M<&%R7W1E;7!?96YV(0IS=6(@7W-E=%]P87)?=&5M<"!["B`@("!I9B`H9&5F
M:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO*2!["B`@("`@("`@)'!A<E]T96UP(#T@)#$["B`@("`@("`@<F5T=7)N
M.PH@("`@?0H*("`@(&9O<F5A8V@@;7D@)'!A=&@@*`H@("`@("`@("AM87`@
M)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@
M5$U0("DI+`H@("`@("`@('%W*"!#.EQ<5$5-4"`O=&UP("X@*0H@("`@*2![
M"B`@("`@("`@;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A
M=&@@86YD("UW("1P871H.PH@("`@("`@(&UY("1U<V5R;F%M93L*("`@("`@
M("!M>2`D<'=U:60["B`@("`@("`@(R!D;V5S(&YO="!W;W)K(&5V97)Y=VAE
M<F4Z"B`@("`@("`@979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@
M9&5F:6YE9"`D/CM].PH*("`@("`@("!I9B`H(&1E9FEN960H)E=I;C,R.CI,
M;V=I;DYA;64I("D@>PH@("`@("`@("`@("`D=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*("`@("`@("!]"B`@("`@("`@96QS:68@*&1E9FEN960@
M)'!W=6ED*2!["B`@("`@("`@("`@("1U<V5R;F%M92`]("1P=W5I9#L*("`@
M("`@("!]"B`@("`@("`@96QS92!["B`@("`@("`@("`@("1U<V5R;F%M92`]
M("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["B`@
M("`@("`@?0H@("`@("`@("1U<V5R;F%M92`]?B!S+UQ7+U\O9SL*"B`@("`@
M("`@;7D@)'-T;7!D:7(@/2`B)'!A=&@D0V]N9FEG>U]D96QI;7UP87(M(BYU
M;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*("`@("`@("!M:V1I<B`D<W1M<&1I
M<BP@,#<U-3L*("`@("`@("!I9B`H(21%3E9[4$%27T-,14%.?2!A;F0@;7D@
M)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;.5TI('L*("`@("`@("`@("`@
M;W!E;B`H;7D@)&9H+"`B/"(N("1P<F]G;F%M92D["B`@("`@("`@("`@('-E
M96L@)&9H+"`M,3@L(#(["B`@("`@("`@("`@('-Y<W)E860@)&9H+"!M>2`D
M8G5F+"`V.PH@("`@("`@("`@("!I9B`H)&)U9B!E<2`B7#!#04-(12(I('L*
M("`@("`@("`@("`@("`@('-E96L@)&9H+"`M-3@L(#(["B`@("`@("`@("`@
M("`@("!S>7-R96%D("1F:"P@)&)U9BP@-#$["B`@("`@("`@("`@("`@("`D
M8G5F(#U^(',O7#`O+V<["B`@("`@("`@("`@("`@("`D<W1M<&1I<B`N/2`B
M)$-O;F9I9WM?9&5L:6U]8V%C:&4M(B`N("1B=68["B`@("`@("`@("`@('T*
M("`@("`@("`@("`@96QS92!["B`@("`@("`@("`@("`@("!M>2`D8W1X(#T@
M979A;"![(')E<75I<F4@1&EG97-T.CI32$$[($1I9V5S=#HZ4TA!+3YN97<H
M,2D@?0H@("`@("`@("`@("`@("`@("`@('Q\(&5V86P@>R!R97%U:7)E($1I
M9V5S=#HZ4TA!,3L@1&EG97-T.CI32$$Q+3YN97<@?0H@("`@("`@("`@("`@
M("`@("`@('Q\(&5V86P@>R!R97%U:7)E($1I9V5S=#HZ340U.R!$:6=E<W0Z
M.DU$-2T^;F5W('T["@H@("`@("`@("`@("`@("`@(R!7;W)K87)O=6YD(&9O
M<B!B=6<@:6X@1&EG97-T.CI32$$@-2XS."!A;F0@-2XS.0H@("`@("`@("`@
M("`@("`@;7D@)'-H85]V97)S:6]N(#T@979A;"![("1$:6=E<W0Z.E-(03HZ
M5D524TE/3B!]('Q\(#`["B`@("`@("`@("`@("`@("!I9B`H)'-H85]V97)S
M:6]N(&5Q("<U+C,X)R!O<B`D<VAA7W9E<G-I;VX@97$@)S4N,SDG*2!["B`@
M("`@("`@("`@("`@("`@("`@)&-T>"T^861D9FEL92@D<')O9VYA;64L(")B
M(BD@:68@*"1C='@I.PH@("`@("`@("`@("`@("`@?0H@("`@("`@("`@("`@
M("`@96QS92!["B`@("`@("`@("`@("`@("`@("`@:68@*"1C='@@86YD(&]P
M96XH;7D@)&9H+"`B/"1P<F]G;F%M92(I*2!["B`@("`@("`@("`@("`@("`@
M("`@("`@(&)I;FUO9&4H)&9H*3L*("`@("`@("`@("`@("`@("`@("`@("`@
M)&-T>"T^861D9FEL92@D9F@I.PH@("`@("`@("`@("`@("`@("`@("`@("!C
M;&]S92@D9F@I.PH@("`@("`@("`@("`@("`@("`@('T*("`@("`@("`@("`@
M("`@('T*"B`@("`@("`@("`@("`@("`D<W1M<&1I<B`N/2`B)$-O;F9I9WM?
M9&5L:6U]8V%C:&4M(B`N("@@)&-T>"`_("1C='@M/FAE>&1I9V5S="`Z("1M
M=&EM92`I.PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&-L;W-E*"1F:"D[
M"B`@("`@("`@?0H@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`D14Y6>U!!
M4E]#3$5!3GT@/2`Q.PH@("`@("`@("`@("`D<W1M<&1I<B`N/2`B)$-O;F9I
M9WM?9&5L:6U]=&5M<"TD)"(["B`@("`@("`@?0H*("`@("`@("`D14Y6>U!!
M4E]414U0?2`]("1S=&UP9&ER.PH@("`@("`@(&UK9&ER("1S=&UP9&ER+"`P
M-S4U.PH@("`@("`@(&QA<W0["B`@("!]"@H@("`@)'!A<E]T96UP(#T@)#$@
M:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI
M+SL*?0H*<W5B(%]T96UP9FEL92!["B`@("!M>2`H)&5X="P@)&-R8RD@/2!`
M7SL*("`@(&UY("@D9F@L("1F:6QE;F%M92D["@H@("`@)&9I;&5N86UE(#T@
M(B1P87)?=&5M<"\D8W)C)&5X="(["@H@("`@:68@*"1%3E9[4$%27T-,14%.
M?2D@>PH@("`@("`@('5N;&EN:R`D9FEL96YA;64@:68@+64@)&9I;&5N86UE
M.PH@("`@("`@('!U<V@@0'1M<&9I;&4L("1F:6QE;F%M93L*("`@('T*("`@
M(&5L<V4@>PH@("`@("`@(')E='5R;B`H=6YD968L("1F:6QE;F%M92D@:68@
M*"UR("1F:6QE;F%M92D["B`@("!]"@H@("`@;W!E;B`D9F@L("<^)RP@)&9I
M;&5N86UE(&]R(&1I92`D(3L*("`@(&)I;FUO9&4H)&9H*3L*("`@(')E='5R
M;B@D9F@L("1F:6QE;F%M92D["GT*"B,@<V%M92!C;V1E(&QI=F5S(&EN(%!!
M4CHZ4V5T=7!0<F]G;F%M93HZ<V5T7W!R;V=N86UE"G-U8B!?<V5T7W!R;V=N
M86UE('L*("`@(&EF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@
M)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PH@("`@("`@("1P<F]G
M;F%M92`]("0Q.PH@("`@?0H*("`@("1P<F]G;F%M92!\?#T@)#`["@H@("`@
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B`@("`@("`@)'!R;V=N86UE(#T@<W5B<W1R
M*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]
M*2`K(#$I.PH@("`@?0H*("`@(&EF("@A)$5.5GM005)?4%)/1TY!345](&]R
M(&EN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`^/2`P*2!["B`@
M("`@("`@:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PH@("`@
M("`@("`@("!R971U<FX@:68@+7,@)&9H.PH@("`@("`@('T*("`@("`@("!I
M9B`H+7,@(B1P<F]G;F%M921#;VYF:6=[7V5X97TB*2!["B`@("`@("`@("`@
M("1P<F]G;F%M92`N/2`D0V]N9FEG>U]E>&5].PH@("`@("`@("`@("!R971U
M<FX["B`@("`@("`@?0H@("`@?0H*("`@(&9O<F5A8V@@;7D@)&1I<B`H<W!L
M:70@+UQ1)$-O;F9I9WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*("`@
M("`@("!N97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E
M<2`D14Y6>U!!4E]414U0?3L*("`@("`@("`D9&ER(#U^(',O7%$D0V]N9FEG
M>U]D96QI;7U<120O+SL*("`@("`@("`H*"1P<F]G;F%M92`]("(D9&ER)$-O
M;F9I9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I9WM?97AE?2(I+"!L87-T*0H@
M("`@("`@("`@("!I9B`M<R`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M
M921#;VYF:6=[7V5X97TB.PH@("`@("`@("@H)'!R;V=N86UE(#T@(B1D:7(D
M0V]N9FEG>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*("`@("`@("`@("`@
M:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64B.PH@("`@?0I]
M"@IS=6(@7V9I>%]P<F]G;F%M92!["B`@("`D,"`]("1P<F]G;F%M92!\?#T@
M)$5.5GM005)?4%)/1TY!345].PH@("`@:68@*&EN9&5X*"1P<F]G;F%M92P@
M)$-O;F9I9WM?9&5L:6U]*2`\(#`I('L*("`@("`@("`D<')O9VYA;64@/2`B
M+B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["B`@("!]"@H@("`@(R!86%@@
M+2!H86-K('1O(&UA:V4@4%=$('=O<FL*("`@(&UY("1P=V0@/2`H9&5F:6YE
M9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C=V0H*0H@("`@("`@("`@("`@
M("`@.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W
M9"@I(#H@8'!W9&`I.PH@("`@8VAO;7`H)'!W9"D["B`@("`D<')O9VYA;64@
M/7X@<R]>*#\]7"Y<+C]<421#;VYF:6=[7V1E;&EM?5Q%*2\D<'=D)$-O;F9I
M9WM?9&5L:6U]+SL*"B`@("`D14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA
M;64["GT*"G-U8B!?<&%R7VEN:71?96YV('L*("`@(&EF("@@)$5.5GM005)?
M24Y)5$E!3$E:141]*RL@/3T@,2`I('L*("`@("`@("!R971U<FX["B`@("!]
M(&5L<V4@>PH@("`@("`@("1%3E9[4$%27TE.251)04Q)6D5$?2`](#(["B`@
M("!]"@H@("`@9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!
M0TA%(%!23T=.04U%($%21T,@05)'5E\P("D@*2!["B`@("`@("`@9&5L971E
M("1%3E9[)U!!4E\G+B1??3L*("`@('T*("`@(&9O<B`H<7<O(%1-4$1)4B!4
M14U0($-,14%.($1%0E5'("\I('L*("`@("`@("`D14Y6>R=005)?)RXD7WT@
M/2`D14Y6>R=005)?1TQ/0D%,7R<N)%]](&EF(&5X:7-T<R`D14Y6>R=005)?
M1TQ/0D%,7R<N)%]].PH@("`@?0H*("`@(&UY("1P87)?8VQE86X@/2`B7U]%
M3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@("`B.PH*("`@(&EF("@D14Y6
M>U!!4E]414U0?2D@>PH@("`@("`@(&1E;&5T92`D14Y6>U!!4E]#3$5!3GT[
M"B`@("!]"B`@("!E;'-I9B`H(65X:7-T<R`D14Y6>U!!4E]'3$]"04Q?0TQ%
M04Y]*2!["B`@("`@("`@;7D@)'9A;'5E(#T@<W5B<W1R*"1P87)?8VQE86XL
M(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*("`@("`@("`D14Y6>U!!4E]#3$5!
M3GT@/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%04X]*%Q3*RDO.PH@("`@
M?0I]"@IS=6(@;W5T<R!["B`@("!R971U<FX@:68@)'%U:65T.PH@("`@:68@
M*"1L;V=F:"D@>PH@("`@("`@('!R:6YT("1L;V=F:"`B0%]<;B(["B`@("!]
M"B`@("!E;'-E('L*("`@("`@("!P<FEN="`B0%]<;B(["B`@("!]"GT*"G-U
M8B!I;FET7VEN8R!["B`@("!R97%U:7)E($-O;F9I9SL*("`@('!U<V@@0$E.
M0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H
M"B`@("`@("`@87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*("`@("`@("!V96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`H@
M("`@*3L*?0H*(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C"B,@5&AE(&UA
M:6X@<&%C:V%G92!F;W(@<V-R:7!T(&5X96-U=&EO;@H*<&%C:V%G92!M86EN
M.PH*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R
M.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PH*9&EE('%Q*'!A<BYP;#H@0V%N
M)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE
M(&]R(&1I<F5C=&]R>5QN*0H@("`@=6YL97-S("UE("1P<F]G;F%M93L*"F1O
M("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)
M5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PH*?3L*"B0Z.E]?15)23U(@
M/2`D0"!I9B`D0#L*?0H*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@
M/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D
M.CI?7T524D]2.PH*,3L*"B-L:6YE(#$P,3$*"E]?14Y$7U\*`"!/``````(`
M``````````"<-$$``0``````````````I#1!``$``````````````-!&0``!
M``````````````#(1D```0``````````````C#1!``$``````````````&A&
M0``!``````````````!<,T$``0``````````````1T-#.B`H1TY5*2`T+CDN
M,"`R,#$S,#,R-2`H97AP97)I;65N=&%L*0``````````1T-#.B`H1TY5*2`T
M+C@N,0````````````````````!'0T,Z("A'3E4I(#0N."XQ````````````
M`````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S,C4@*&5X<&5R:6UE;G1A
M;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S,C4@*&5X<&5R:6UE
M;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S,C4@*&5X<&5R
M:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S,C4@*&5X
M<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S,C4@
M*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q,S`S
M,C4@*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@,C`Q
M,S`S,C4@*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY+C`@
M,C`Q,S`S,C4@*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@-"XY
M+C`@,C`Q,S`S,C4@*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.52D@
M-"XY+C`@,C`Q,S`S,C4@*&5X<&5R:6UE;G1A;"D``````````$=#0SH@*$=.
M52D@-"XX+C$`````````````````````1T-#.B`H1TY5*2`T+C@N,0``````
M``````````````````````````$```"<-`$`8,\``$````",-`$`H,\``$``
M``"D-`$`<,\``$````!<,P$`P,\``$``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````0``!$$````/```%`0``!6$```"/``
M`&`0``"X$```$/```,`0``#&$```'/```-`0```L*```)/```#`H``#=*```
M//```.`H``!8*0``4/```&`I``!=*P``7/```&`K``"'+```</```)`L``#8
M+0``B/```.`M``!D+@``E/```'`N``!)+P``H/```%`O``"$+P``M/```)`O
M``#E+P``O/```/`O``#S+P``Q/`````P``#),0``R/```-`Q``#W,0``X/``
M```R``#),@``Y/```-`R``"S-```[/```,`T```4-0``_/```"`U``!$-0``
M!/$``%`U``![-@``#/$``(`V``"^00``(/$``,!!``"!10``./$``)!%```!
M1@``3/$``!!&``!01@``6/$``&!'``!R1P``;/$``(!'``"!1P``=/$``)!'
M``"11P``>/$``*!'``"A1P``?/$``+!'``"Q1P``@/$``%!*```!3@``A/$`
M`!!.```I3@``E/$``#!.``#H3@``F/$``/!.``#U3@``H/$``"!/```A3P``
MI/$``#!/```V3P``J/$``%!/``![3P``K/$``*!/``"(40``8/$``)!1``":
M40``M/$``*!1``"J40``O/$``````````````````````````````00!``1"
M```!!`(%!`,!4`$(`P4(,@0#`5````$$`@4$`P%0`1,*`!,!40`,,`M@"G`)
M4`C`!M`$X`+P`0P'``Q""#`'8`9P!5`$P`+0```!!P0`!U(#,`)@`7`!#@@`
M#E(*,`E@"'`'4`;`!-`"X`$3"@`3`14`##`+8`IP"5`(P`;0!.`"\`$'!``'
M,@,P`F`!<`$&`P`&0@(P`6````$3!P`3`08""S`*8`EP"%`"P````00!``1"
M```!!0(`!5(!,`$````!&0H`&0$+`A$P$&`/<`Y0#<`+T`G@`O`!`````04"
M``4R`3`!"`4`",($,`-@`G`!4````00!``1B```!!`$`!$(```$."``.\@HP
M"6`(<`=0!L`$T`+@`1D*`!D!-Q`1,!!@#W`.4`W`"]`)X`+P`0P'``QB"#`'
M8`9P!5`$P`+0```!!@,`!D(",`%@```!!0(`!3(!,`$&`P`&@@(P`6````$$
M`0`$0@```0````$````!`````0````$*!04*,@8#`S`"8`%0```!`````00!
M``1B```!`````0````$````!!0(`!3(!,`$$`@4$`P%0`00"!00#`5``````
M````````````````````````````````````````````````````````````
M````````````````````!/RO40````"^(`$``0````\````/````*"`!`&0@
M`0"@(`$`P$$``!!&```@-0``H$\```A@``#0,0``D$4``/`O````,@``P#0`
M```P``#0,@``@#8``%`U``!0+P``QB`!`-P@`0#X(`$`$R$!`!@A`0`B(0$`
M+R$!`#LA`0!,(0$`6"$!`&8A`0!S(0$`@"$!`(TA`0"?(0$````!``(``P`$
M``4`!@`'``@`"0`*``L`#``-``X`<&%R+F5X90!84U]);G1E<FYA;'-?4$%2
M7T)/3U0`6%-?26YT97)N86QS7U!!4E]#3$5!4E-404-+`&=E=%]U<V5R;F%M
M95]F<F]M7V=E='!W=6ED`&UA:6X`;7E?<&%R7W!L`'!A<E]B87-E;F%M90!P
M87)?8VQE86YU<`!P87)?8W5R<F5N=%]E>&5C`'!A<E]D:7)N86UE`'!A<E]E
M;G9?8VQE86X`<&%R7V9I;F1P<F]G`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D
M:7(`<&%R7W-E='5P7VQI8G!A=&@`>'-?:6YI=```````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````&0P`0```````````-@Z`0#\
M,@$`[#$!````````````[#H!`(0T`0`$,@$```````````!H.P$`G#0!`.0R
M`0```````````(`[`0!\-0$```````````````````````````"4-0$`````
M`*`U`0``````L#4!``````"\-0$``````,@U`0``````V#4!``````#D-0$`
M`````/`U`0``````^#4!```````$-@$``````!@V`0``````+#8!```````\
M-@$``````$@V`0``````5#8!``````!<-@$``````&@V`0``````<#8!````
M``!\-@$``````(@V`0``````E#8!``````"<-@$``````*0V`0``````L#8!
M``````"\-@$``````,@V`0``````U#8!``````#@-@$``````.@V`0``````
M\#8!``````#\-@$``````!`W`0``````*#<!```````P-P$``````#PW`0``
M````2#<!``````!4-P$``````%PW`0``````:#<!``````!P-P$``````'PW
M`0``````B#<!``````"4-P$``````*`W`0``````K#<!``````"X-P$`````
M`,0W`0``````T#<!`````````````````-PW`0``````\#<!````````````
M``````0X`0``````%#@!```````D.`$``````#@X`0``````4#@!``````!@
M.`$``````'`X`0``````B#@!``````"8.`$``````*PX`0``````O#@!````
M``#,.`$``````-PX`0``````\#@!```````(.0$``````"0Y`0``````.#D!
M``````!(.0$``````%PY`0``````<#D!``````"`.0$``````)0Y`0``````
MI#D!``````"X.0$``````,@Y`0``````U#D!``````#D.0$`````````````
M````\#D!```````$.@$`````````````````E#4!``````"@-0$``````+`U
M`0``````O#4!``````#(-0$``````-@U`0``````Y#4!``````#P-0$`````
M`/@U`0``````!#8!```````8-@$``````"PV`0``````/#8!``````!(-@$`
M`````%0V`0``````7#8!``````!H-@$``````'`V`0``````?#8!``````"(
M-@$``````)0V`0``````G#8!``````"D-@$``````+`V`0``````O#8!````
M``#(-@$``````-0V`0``````X#8!``````#H-@$``````/`V`0``````_#8!
M```````0-P$``````"@W`0``````,#<!```````\-P$``````$@W`0``````
M5#<!``````!<-P$``````&@W`0``````<#<!``````!\-P$``````(@W`0``
M````E#<!``````"@-P$``````*PW`0``````N#<!``````#$-P$``````-`W
M`0````````````````#<-P$``````/`W`0`````````````````$.`$`````
M`!0X`0``````)#@!```````X.`$``````%`X`0``````8#@!``````!P.`$`
M`````(@X`0``````F#@!``````"L.`$``````+PX`0``````S#@!``````#<
M.`$``````/`X`0``````"#D!```````D.0$``````#@Y`0``````2#D!````
M``!<.0$``````'`Y`0``````@#D!``````"4.0$``````*0Y`0``````N#D!
M``````#(.0$``````-0Y`0``````Y#D!`````````````````/`Y`0``````
M!#H!``````````````````\`7U]E<G)N;P```!,`7U]G971R965N=``````:
M`%]?;6%I;@````!$`%]D;&Q?8W)T,`!)`%]I;7!U<F5?<'1R````60!A8V-E
M<W,`````G0!C86QL;V,`````R@!C;&]S90#+`&-L;W-E9&ER``#Q`&-Y9W=I
M;E]D971A8VA?9&QL`/,`8WEG=VEN7VEN=&5R;F%L````!`%D;&Q?9&QL8W)T
M,````!\!96YV:7)O;@```#@!97AE8W9P`````#H!97AI=```B@%F<')I;G1F
M````D0%F<F5E``#Q`6=E='!I9`````#\`6=E='!W=6ED```)`F=E='5I9```
M``"?`FQS965K`*$";'-T870`I0)M86QL;V,`````M0)M96UC:'(`````MP)M
M96UC<'D`````N0)M96UM;W9E````O`)M96US970`````O0)M:V1I<@#P`F]P
M96X``/0";W!E;F1I<@```!4#<'1H<F5A9%]A=&9O<FL`````00-P=&AR96%D
M7V=E='-P96-I9FEC````B0-R96%D``"*`W)E861D:7(```",`W)E861L:6YK
M``"/`W)E86QL;V,```"X`W)M9&ER`#<$<W!R:6YT9@```#X$<W1A=```1@1S
M=')C:'(`````2`1S=')C;7``````3`1S=')D=7``````4P1S=')L96X`````
M5P1S=')N8VUP````6`1S=')N8W!Y````801S=')S='(`````901S=')T;VL`
M````L@1U;FQI;FL`````"`!?7W-T86-K7V-H:U]F86EL```*`%]?<W1A8VM?
M8VAK7V=U87)D``H`4$Q?9&]?=6YD=6UP``!$`%!,7W1H<E]K97D`````.@%0
M97)L7V%T9F]R:U]L;V-K```[`5!E<FQ?871F;W)K7W5N;&]C:P````!#`5!E
M<FQ?879?9F5T8V@`2P%097)L7V%V7W-H:69T`*`!4&5R;%]C<F]A:U]N;V-O
M;G1E>'0``.H!4&5R;%]D;W5N=VEN9``Z`E!E<FQ?9W9?9F5T8VAP=@```!D#
M4&5R;%]M9U]S970```!S`U!E<FQ?;F5W4U9P=@``@`-097)L7VYE=UA3````
M`($#4&5R;%]N97=84U]F;&%G<P``$@5097)L7W-A9F5S>7-M86QL;V,`````
M(`5097)L7W-A=F5?9&5S=')U8W1O<E]X`````'$%4&5R;%]S=E\R<'9?9FQA
M9W,`U05097)L7W-V7W-E='!V`-T%4&5R;%]S=E]S971P=FX`````!09097)L
M7W-Y<U]I;FET,P`````&!E!E<FQ?<WES7W1E<FT`@`9B;V]T7T1Y;F%,;V%D
M97(```"@!G!E<FQ?86QL;V,`````H@9P97)L7V-O;G-T<G5C=`````"C!G!E
M<FQ?9&5S=')U8W0`I`9P97)L7V9R964`I09P97)L7W!A<G-E`````*8&<&5R
M;%]R=6X``"0"1V5T36]D=6QE2&%N9&QE00``5@)'9710<F]C061D<F5S<P``
M````,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!
M```P`0``,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!```P`0``,`$`
M`#`!```P`0``,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!```P`0``
M,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!```P`0``,`$``#`!```P
M`0``,`$``#`!```P`0!C>6=W:6XQ+F1L;``4,`$`%#`!`&-Y9W-S<"TP+F1L
M;``````H,`$`*#`!`"@P`0`H,`$`*#`!`"@P`0`H,`$`*#`!`"@P`0`H,`$`
M*#`!`"@P`0`H,`$`*#`!`"@P`0`H,`$`*#`!`"@P`0`H,`$`*#`!`"@P`0`H
M,`$`*#`!`"@P`0`H,`$`*#`!`"@P`0!C>6=P97)L-5\Q-"YD;&P`/#`!`#PP
M`0!+15).14PS,BYD;&P`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````"
M_*]1````````!``#````,```@`X```#```"`$````/```(`8````(`$`@```
M```"_*]1````````!``!````8```@`(```!X``"``P```)```(`$````J```
M@``````"_*]1`````````0`)!```8`$````````"_*]1`````````0`)!```
M<`$````````"_*]1`````````0`)!```@`$````````"_*]1`````````0`)
M!```D`$````````"_*]1``````$```!0`0"`V```@``````"_*]1````````
M`0`)!```H`$````````"_*]1`````````0`!````"`$`@``````"_*]1````
M`````0`)!```L`$````````"_*]1`````````0`!````.`$`@``````"_*]1
M`````````0`)!```P`$```8`5P!)`$X`10!8`$4```#000$`:`4`````````
M````.$<!`*@(`````````````.!/`0!H!0````````````!(50$`J`@`````
M````````\%T!`#X``````````````#!>`0!8`@````````````"(8`$`/@,`
M````````````*````!`````@`````0`(`````````0```````````````0``
M``$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN;@"!@8$`L;&Q`/___P!S
M<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7`!86%@#:VMH`EI:6`(Z.
MC@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$``P,#`!$1$0"3DY,`L[.S
M`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'!P<`<7%Q`+6UM0#U]?4`
MB8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8V`#0T-``'1T=``P,#`"/
MCX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9`!D9&0!-34T`1$1$`%!0
M4``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````!04%!04%!
M04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!!0$\%Q<!`1,!`0$!`0$!
M`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!`0$!(A(0,0\%`2`!`0$!
M`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!`0$!)1`0$!`0$!`0$!=%
M`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0$`$!`0$2$!`3'Q`0$!`0
M$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040$!`Q`0$!`0$!`0$!`0$!
M004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!``#@`P``\`,``/`#``#P
M`P``X`,``,`!``"``0``@`$``(`#``"``P````<````'``".#P``_Q\``"@`
M```@````0`````$`"`````````0```````````````$````!`````````0$!
M``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)"0D`"@H*``L+"P`,#`P`
M#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04%``5%14`%A86`!<7%P`8
M&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?`"`@(``A(2$`(B(B`",C
M(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`*RLK`"PL+``M+2T`+BXN
M`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V-C8`-S<W`#@X.``Y.3D`
M.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!00!"0D(`0T-#`$1$1`!%
M144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,`$U-30!.3DX`3T]/`%!0
M4`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`6%A8`%E960!:6EH`6UM;
M`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C8V,`9&1D`&5E90!F9F8`
M9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN;@!O;V\`<'!P`'%Q<0!R
M<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY`'IZ>@![>WL`?'Q\`'U]
M?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`A86%`(:&A@"'AX<`B(B(
M`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0D)``D9&1`)*2D@"3DY,`
ME)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;FP"<G)P`G9V=`)Z>G@"?
MGY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F`*>GIP"HJ*@`J:FI`*JJ
MJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`LK*R`+.SLP"TM+0`M;6U
M`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]O;T`OKZ^`+^_OP#`P,``
MP<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(R`#)R<D`RLK*`,O+RP#,
MS,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3`-34U`#5U=4`UM;6`-?7
MUP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`W]_?`.#@X`#AX>$`XN+B
M`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#JZNH`Z^OK`.SL[`#M[>T`
M[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U]0#V]O8`]_?W`/CX^`#Y
M^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T/#P\/#P\/#P\/#P\/#P\
M/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34U-34U-34U-2T-%$$!`0$
M!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$!`0$9,3D[/3T]/3T]/3T
M]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SLY.3DY.SLY.SD[.SL[.2\
M-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SLW+PT45%145$$9+S<Y.SD
MQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SDY.SDY%&$=.2,;*2<;(R\
M=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QTS-RD4<3DW+PT!`0$!`0$
M9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$!`1<O-S<Y.3DY-2<I%R4
MC,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL++RTS%K$9)QLE*3<U+0T
M!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,M#0$!`0$!`1DM-3<W-SD
MW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<S'R,A+3\S,3\]+3\O&3,
MK(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SLM.RL?)11K#0$!`0$!`1<
MK,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$!%RLQ+Q\Y/S\_/S\_/S\
M_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\_/S\_/S\_/STO%'$K#0$
M!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D-`0$!`0$!%RLQ*11_/S\
MM)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L_/S,9+3L_/S\_/S\_-1L
MK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SDC%',S,2D-`0$!`0$!%RD
M;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$7*2L9%QDI*11K,S$=)3D
M]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL47Q\E&Q\O,S$Q+R<+`0$
M!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT!`0$!`0$5)RTQ,3$Q,3$
MQ,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\O+R\O+R\O+R\O+R\O+R\
MO+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2TM+2TK)PL!`0$!`0$5(24
MG)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$5%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P````<````#`````0``````
M```_````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_*````!`````@`````0`(`````````0```````````````0``
M``$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN;@"!@8$`L;&Q`/___P!S
M<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7`!86%@#:VMH`EI:6`(Z.
MC@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$``P,#`!$1$0"3DY,`L[.S
M`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'!P<`<7%Q`+6UM0#U]?4`
MB8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8V`#0T-``'1T=``P,#`"/
MCX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9`!D9&0!-34T`1$1$`%!0
M4``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````!04%!04%!
M04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!!0$\%Q<!`1,!`0$!`0$!
M`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!`0$!(A(0,0\%`2`!`0$!
M`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!`0$!)1`0$!`0$!`0$!=%
M`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0$`$!`0$2$!`3'Q`0$!`0
M$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040$!`Q`0$!`0$!`0$!`0$!
M004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!``#@`P``\`,``/`#``#P
M`P``X`,``,`!``"``0``@`$``(`#``"``P````<````'``".#P``_Q\``"@`
M```@````0`````$`"`````````0```````````````$````!`````````0$!
M``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)"0D`"@H*``L+"P`,#`P`
M#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04%``5%14`%A86`!<7%P`8
M&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?`"`@(``A(2$`(B(B`",C
M(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`*RLK`"PL+``M+2T`+BXN
M`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V-C8`-S<W`#@X.``Y.3D`
M.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!00!"0D(`0T-#`$1$1`!%
M144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,`$U-30!.3DX`3T]/`%!0
M4`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`6%A8`%E960!:6EH`6UM;
M`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C8V,`9&1D`&5E90!F9F8`
M9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN;@!O;V\`<'!P`'%Q<0!R
M<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY`'IZ>@![>WL`?'Q\`'U]
M?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`A86%`(:&A@"'AX<`B(B(
M`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0D)``D9&1`)*2D@"3DY,`
ME)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;FP"<G)P`G9V=`)Z>G@"?
MGY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F`*>GIP"HJ*@`J:FI`*JJ
MJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`LK*R`+.SLP"TM+0`M;6U
M`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]O;T`OKZ^`+^_OP#`P,``
MP<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(R`#)R<D`RLK*`,O+RP#,
MS,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3`-34U`#5U=4`UM;6`-?7
MUP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`W]_?`.#@X`#AX>$`XN+B
M`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#JZNH`Z^OK`.SL[`#M[>T`
M[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U]0#V]O8`]_?W`/CX^`#Y
M^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T/#P\/#P\/#P\/#P\/#P\
M/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34U-34U-34U-2T-%$$!`0$
M!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$!`0$9,3D[/3T]/3T]/3T
M]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SLY.3DY.SLY.SD[.SL[.2\
M-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SLW+PT45%145$$9+S<Y.SD
MQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SDY.SDY%&$=.2,;*2<;(R\
M=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QTS-RD4<3DW+PT!`0$!`0$
M9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$!`1<O-S<Y.3DY-2<I%R4
MC,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL++RTS%K$9)QLE*3<U+0T
M!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,M#0$!`0$!`1DM-3<W-SD
MW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<S'R,A+3\S,3\]+3\O&3,
MK(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SLM.RL?)11K#0$!`0$!`1<
MK,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$!%RLQ+Q\Y/S\_/S\_/S\
M_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\_/S\_/S\_/STO%'$K#0$
M!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D-`0$!`0$!%RLQ*11_/S\
MM)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L_/S,9+3L_/S\_/S\_-1L
MK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SDC%',S,2D-`0$!`0$!%RD
M;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$7*2L9%QDI*11K,S$=)3D
M]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL47Q\E&Q\O,S$Q+R<+`0$
M!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT!`0$!`0$5)RTQ,3$Q,3$
MQ,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\O+R\O+R\O+R\O+R\O+R\
MO+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2TM+2TK)PL!`0$!`0$5(24
MG)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$5%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P````<````#`````0``````
M```_````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_```!``0`$!````$`"`!H!0```0`@(````0`(`*@(```"`!`0
M```!``@`:`4```,`("````$`"`"H"```!````%@"-````%8`4P!?`%8`10!2
M`%,`20!/`$X`7P!)`$X`1@!/``````"]!._^```!````````````````````
M```_``````````0`!``!````````````````````N`$```$`4P!T`'(`:0!N
M`&<`1@!I`&P`90!)`&X`9@!O````E`$```$`,``P`#``,``P`#0`0@`P````
M)``"``$`0P!O`&T`<`!A`&X`>0!.`&$`;0!E```````@````+``"``$`1@!I
M`&P`90!$`&4`<P!C`'(`:0!P`'0`:0!O`&X``````"`````P``@``0!&`&D`
M;`!E`%8`90!R`',`:0!O`&X``````#``+@`P`"X`,``N`#`````D``(``0!)
M`&X`=`!E`'(`;@!A`&P`3@!A`&T`90```"`````H``(``0!,`&4`9P!A`&P`
M0P!O`'``>0!R`&D`9P!H`'0````@````+``"``$`3`!E`&<`80!L`%0`<@!A
M`&0`90!M`&$`<@!K`',``````"`````L``(``0!/`'(`:0!G`&D`;@!A`&P`
M1@!I`&P`90!N`&$`;0!E````(````"0``@`!`%``<@!O`&0`=0!C`'0`3@!A
M`&T`90``````(````#0`"``!`%``<@!O`&0`=0!C`'0`5@!E`'(`<P!I`&\`
M;@```#``+@`P`"X`,``N`#````!$`````0!6`&$`<@!&`&D`;`!E`$D`;@!F
M`&\``````"0`!````%0`<@!A`&X`<P!L`&$`=`!I`&\`;@````````"P!#P_
M>&UL('9E<G-I;VX](C$N,"(@96YC;V1I;F<](E541BTX(B!S=&%N9&%L;VYE
M/2)Y97,B/SX-"CQA<W-E;6)L>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S
M;V9T+6-O;3IA<VTN=C$B(&UA;FEF97-T5F5R<VEO;CTB,2XP(CX-"B`@("`\
M87-S96UB;'E)9&5N=&ET>2!V97)S:6]N/2(P+C`N,"XP(B!T>7!E/2)W:6XS
M,B(@;F%M93TB4$%2+5!A8VME<BY!<'!L:6-A=&EO;B(@+SX-"B`@("`\9&5S
M8W)I<'1I;VX^4$%2+5!A8VME<B!!<'!L:6-A=&EO;CPO9&5S8W)I<'1I;VX^
M#0H@("`@/&1E<&5N9&5N8WD^#0H@("`@("`@(#QD97!E;F1E;G1!<W-E;6)L
M>3X-"B`@("`@("`@("`@(#QA<W-E;6)L>4ED96YT:71Y('1Y<&4](G=I;C,R
M(B!N86UE/2)-:6-R;W-O9G0N5VEN9&]W<RY#;VUM;VXM0V]N=')O;',B('9E
M<G-I;VX](C8N,"XP+C`B('!U8FQI8TME>51O:V5N/2(V-3DU8C8T,30T8V-F
M,61F(B!L86YG=6%G93TB*B(@<')O8V5S<V]R07)C:&ET96-T=7)E/2(J(B`O
M/@T*("`@("`@("`\+V1E<&5N9&5N=$%S<V5M8FQY/@T*("`@(#PO9&5P96YD
M96YC>3X-"B`@("`\=')U<W1);F9O('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R
M;W-O9G0M8V]M.F%S;2YV,R(^#0H@("`@("`@(#QS96-U<FET>3X-"B`@("`@
M("`@("`@(#QR97%U97-T9610<FEV:6QE9V5S/@T*("`@("`@("`@("`@("`@
M(#QR97%U97-T961%>&5C=71I;VY,979E;"!L979E;#TB87-);G9O:V5R(B!U
M:4%C8V5S<STB9F%L<V4B("\^#0H@("`@("`@("`@("`\+W)E<75E<W1E9%!R
M:79I;&5G97,^#0H@("`@("`@(#PO<V5C=7)I='D^#0H@("`@/"]T<G5S=$EN
M9F\^#0H\+V%S<V5M8FQY/@T*````````````````````````````````````
M``````````````````````````````````````````!0```,````N*'0H0!@
M```,````"*````#````8````8*]PKX"OD*^@K["OP*\`````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
#````
