// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: gffont.C,v 1.20 91/02/22 15:58:04 hmgr Exp $";

// read font data from the METAFONT GF (generic font) files
//
// by Parag Patel

#include "defs.h"


// these are for "drawing" an image of a character in a font
const int WHITE = FALSE;
const int BLACK = TRUE;


// return a string containing the magnification sub-directory for the
// requested magnification
// 
char *gfmagdir(Dirlist &dirlist, long mag)
{
    // look for the closest magnification value to the one desired
    int ent = -1;
    long diff = MAXLONG;

    for (int i = 0; i < dirlist.size(); i++)
    {
	long d = mag - dirlist[i].val;
	long newdiff = d < 0 ? -d : d;	// absolute value
	if (newdiff >= diff)
	    continue;
	diff = newdiff;
	ent = i;
    }

    // we should be within 1/32 (.03%) of the desired mag value
    if (ent < 0 || diff > (mag >> 5))
	return "";

    // return the new value as a string
    return dirlist[ent].name;
}


// setup a GF font file in memory
//
void setupgffont(font &f, FILE *fp)
{
    // skip the comment
    int len = (int)getuval(1, fp);
    while (len-- > 0)
	(void)getuval(1, fp);

    // now we jump to the end to find the postamble
    if (fseek(fp, 0, SEEK_END) != 0)
	quit("Cannot seek to end of GF file %s", f.path);

    register int op = GFILLER;
    long floc = ftell(fp);

    // skip the filler bytes at the end of the file
    while (op == GFILLER && floc > 0)
    {
	(void)fseek(fp, --floc, SEEK_SET);
	op = (int)getuval(1, fp);
    }
    if (op != GFID)
	quit("Incorrect GF version for %s", f.path);

    // now we get the file-pointer to the start of the postamble
    (void)fseek(fp, floc -= 4, SEEK_SET);
    long postloc = getuval(4, fp);

    // verify that we have a POST byte
    (void)fseek(fp, postloc, SEEK_SET);
    if (getuval(1, fp) != FPOST)
	quit("Expected FPOST");

    long fsize = getuval(4, fp);// pointer to last EOC in file
    debug(3, "GF last EOC=%ld", fsize);

    // get and verify that the design size is ok - the GF file stores
    // this value as a "fix_word" value while the DVI file keeps it as
    // a "scaled-point" value - also note that the "basename" pointer
    // is NULL only for the "dumpfont" main program

    long dsize = getsval(4, fp);
    debug(4, "GF designsize=%ld", dsize);
    if (f.basename != NULL && dsize >> 4 != f.designsize)
	if (dochecksum && dsize != 0 && f.designsize != 0)
	    quit("Designsize in DVI and GF file %s does not match", f.path);
	else
	    warn("Designsize in DVI and GF file %s does not match", f.path);

    // check the checksum (!)
    long check = getuval(4, fp);
    debug(5, "GF checksum=%ld", check);
    if (f.basename != NULL && check != f.checksum)
	if (dochecksum && check != 0 && f.checksum != 0)
	    quit("Checksums in DVI and GF file %s do not match", f.path);
	else
	    warn("Checksums in DVI and GF file %s do not match", f.path);

    // get the horizontal and vertical pixels per point values
    f.hppp = getuval(4, fp);
    f.vppp = getuval(4, fp);

    // get the size ranges of the characters in this font
    f.minm = getsval(4, fp);
    f.maxm = getsval(4, fp);
    f.minn = getsval(4, fp);
    f.maxn = getsval(4, fp);

    debug(3,"mag=%ld  hppp=%ld vppp=%ld  minm=%ld maxm=%ld  minn=%ld, maxn=%ld",
	    f.mag, f.hppp, f.vppp, f.minm, f.maxm, f.minn, f.maxn);

    // now initialize the info for each character in this font - note
    // that there may only be CHARLOC or NOOPs in this part of the file
    while ((op = (int)getuval(1, fp)) != FPOSTPOST)
    {
	if (op == FNOOP)
	    continue;

	// assume that the next byte is a character "residue" (id % 256)
	int c = (int)getuval(1, fp);
	if (c >= MAXCHARS)
	    quit("Char %d too big in GF file %s", c, f.path);
	fontchar & g = f.chr[c];

	// get the values for delta-x, delta-y, width, and file-pointer
	// for this character
	switch (op)
	{
	Case CHARLOC:
	    g.dx = getuval(4, fp);
	    g.dy = getuval(4, fp);
	    g.width = Getsval(4, fp);
	    g.floc = getsval(4, fp);

	Case CHARLOC0:
	    g.dx = getuval(1, fp) << 16;
	    g.dy = 0;
	    g.width = Getsval(4, fp);
	    g.floc = getsval(4, fp);

	Default:
	    quit("Illegal opcode %d in GF postamble", op);
	}

	g.width = g.width / double(16) * double(f.designsize) /
	double(65536L) * double(f.mag) / 1000.0;
    }
}


// load the specified character "ch" into the bitmap - we read this
// character's GF paint commands to paint this font
// 
void getgfchar(font &f, fontchar &g, int ch)
{
    // go to the file where this character is defined
    if (fseek(f.fp, g.floc, SEEK_SET) < 0)
	quit("Cannot fseek to start of font in %s", f.path);

    register int op = (int)getuval(1, f.fp);

    // we had better get some flavor of BOC here...
    switch (op)
    {
    Case BOC:
	// verify the character (Just In Case)
	if (ch != getsval(4, f.fp))
	    quit("Character definition for %d does not match", ch);

	(void)getuval(4, f.fp);	// pointer to next character % 256

	// get the size (in pixels) of this character
	g.minm = getsval(4, f.fp);
	g.maxm = getsval(4, f.fp);
	g.minn = getsval(4, f.fp);
	g.maxn = getsval(4, f.fp);

    Case BOC1:
	if (ch != getuval(1, f.fp))
	    quit("Character definition for %d does not match", ch);

	// decode the size (in pixels) of this character
	g.minm = getuval(1, f.fp);
	g.maxm = getuval(1, f.fp);
	g.minm = g.maxm - g.minm;

	g.minn = getuval(1, f.fp);
	g.maxn = getuval(1, f.fp);
	g.minn = g.maxn - g.minn;

    Default:
	quit("Expected BOC* (not %d) in font file %s for char %d",
		op, f.path, ch);
    }

    {
	int len;
	const char *x = dev_char2dev(ch, len);
	debug(5, "char=%d(%s)  minm=%ld maxm=%ld  minn=%ld maxn=%ld",
		ch, x, g.minm, g.maxm, g.minn, g.maxn);
	debug(5, "    dx=%ld dy=%ld  dx/=%ld dy/=%ld  width=%f", g.dx, g.dy,
		g.dx >> 16, g.dy >> 16, g.width);
    }

    // initialize the character painting variables
    register long m = 0;
    register long n = g.maxn - g.minn;
    int p = WHITE;

    long width = g.maxm - g.minm + 1;
    long height = g.maxn - g.minn + 1;

    // clear the global fontbits for the bitmap
    register long d;
    for (d = f.maxn - f.minn; d >= 0; d--)
	fontbits[d]->clear();

    // paint the fontbits to build the character
    while ((op = (int)getuval(1, f.fp)) != EOC)
    {
	if (op > PAINT0 && op <= PAINT63)
	{
	    // paint with the current color, advance X
	    if (p == BLACK)
		while (op-- > 0)
		    *fontbits[n] += m++;
	    else
		m += op;
	    p = !p;
	    continue;
	}
	if (op >= NEWROW0 && op <= NEWROW164)
	{
	    // advance Y, paint to black
	    n--;
	    m = op - NEWROW0;
	    p = BLACK;
	    continue;
	}

	switch (op)
	{
	    // paint with current color, advange X
	Case PAINT0:
	    p = !p;
	Case PAINT1:
	    d = getuval(1, f.fp);
	    if (p == BLACK)
		while (d-- > 0)
		    *fontbits[n] += m++;
	    else
		m += d;
	    p = !p;
	Case PAINT2:
	    d = getuval(2, f.fp);
	    if (p == BLACK)
		while (d-- > 0)
		    *fontbits[n] += m++;
	    else
		m += d;
	    p = !p;
	Case PAINT3:
	    d = getuval(3, f.fp);
	    if (p == BLACK)
		while (d-- > 0)
		    *fontbits[n] += m++;
	    else
		m += d;
	    p = !p;

	    // advance Y, paint to white
	Case SKIP0:
	    n--;
	    m = 0;
	    p = WHITE;
	Case SKIP1:
	    n -= getuval(1, f.fp) + 1;
	    m = 0;
	    p = WHITE;
	Case SKIP2:
	    n -= getuval(2, f.fp) + 1;
	    m = 0;
	    p = WHITE;
	Case SKIP3:
	    n -= getuval(3, f.fp) + 1;
	    m = 0;
	    p = WHITE;

	    // special opcodes - just ignore for now
	Case FXXX1:
	    skipbytes(getuval(1, f.fp), f.fp);
	Case FXXX2:
	    skipbytes(getuval(2, f.fp), f.fp);
	Case FXXX3:
	    skipbytes(getuval(3, f.fp), f.fp);
	Case FXXX4:
	    skipbytes(getuval(4, f.fp), f.fp);

	    // special numeric opcode - also ignored
	Case YYY:
	    (void)getuval(4, f.fp);

	Case FNOOP:

	Default:
	    quit("Illegal GF opcode %ld in file %f", d, f.path);
	}
    }
}
