#include <win32dll/win32lib.h>
#include "wdvipsres.h"
#include <malloc.h>
#include <string.h>
#include <mbstring.h>
#include <stdio.h>

#define WM_DVIPS_PAGE (WM_USER + 100)

typedef struct {
  HANDLE hPrinter;
  FILE *fp;
  LPSTR szPrinterName;
  DWORD nPrinterDpi;
  LPSTR szDviFileName;
  LPSTR szOutputFileName;
  LONG nDviPages;
  LONG nCurrentPage;
  LONG nFromPage, nToPage;
  DWORD nPaperSize;
  DWORD nCopies;
  
  LPSTR szMessage;
  LONG nMessageLength;
  LONG nMessageMax;
  HWND hProgressWnd;

  BOOL bManualFeed;
  BOOL bCollate;
  BOOL bDuplex;
} PrintInfo;

HINSTANCE g_hInstance;

UINT ErrorMessage(LPCSTR msg, UINT flags)
{
  return MessageBox(0, msg, NULL, flags ? flags : (MB_OK | MB_ICONSTOP));
}

UINT FileErrorMessage(LPCSTR msg, LPCSTR str, UINT flags)
{
  UINT code;
  char* s = malloc(strlen(msg)+strlen(str)+10);
  strcpy(s, str);
  strcat(s, "\n\n");
  strcat(s, msg);
  code = ErrorMessage(s, flags);
  free(s);
  return code;
}

static LPSTR GetDviFileName(LPCSTR defaultFileName)
{
  OPENFILENAME ofn;
  char fileName[1000];
  
  memset(&ofn, 0, sizeof(ofn));
  if (defaultFileName) strcpy(fileName, defaultFileName);
  else fileName[0] = 0;
  ofn.lStructSize = sizeof(ofn);
  ofn.hwndOwner = NULL;
  ofn.hInstance = g_hInstance;
  ofn.lpstrFilter = "DVI Files (*.dvi)\0*.dvi\0All Files (*.*)\0*\0";
  ofn.nFilterIndex = 1;
  ofn.lpstrFile = fileName;
  ofn.nMaxFile = sizeof(fileName)/sizeof(char);
  ofn.lpstrDefExt = "dvi";
  ofn.Flags = OFN_FILEMUSTEXIST | OFN_NONETWORKBUTTON | OFN_PATHMUSTEXIST |
    OFN_HIDEREADONLY;
  if (!GetOpenFileName(&ofn))
    return NULL;
  return strdup(fileName);
}

static LPSTR GetOutputFileName(LPCSTR defaultFileName)
{
  OPENFILENAME ofn;
  char fileName[1000];
  
  memset(&ofn, 0, sizeof(ofn));
  strcpy(fileName, defaultFileName ? defaultFileName : "output.ps");
  ofn.lStructSize = sizeof(ofn);
  ofn.hwndOwner = NULL;
  ofn.hInstance = g_hInstance;
  ofn.lpstrFilter = "PostScript Files (*.ps)\0*.ps\0All Files (*.*)\0*\0";
  ofn.nFilterIndex = 1;
  ofn.lpstrFile = fileName;
  ofn.nMaxFile = sizeof(fileName)/sizeof(char);
  ofn.lpstrDefExt = "ps";
  ofn.Flags = OFN_HIDEREADONLY | OFN_NONETWORKBUTTON;
  if (!GetSaveFileName(&ofn))
    return NULL;
  return strdup(fileName);
}

static LRESULT CALLBACK ProgressDlgProc(HWND hWnd, UINT uMsg,
					WPARAM wParam, LPARAM lParam)
{
  return 0;
}

static LRESULT CALLBACK MessageDlgProc(HWND hWnd, UINT uMsg,
				       WPARAM wParam, LPARAM lParam)
{
  switch (uMsg) {
  case WM_INITDIALOG:
    {
      PrintInfo* pInfo = (PrintInfo*)lParam;
      SetDlgItemText(hWnd, IDC_MESSAGE, pInfo->szMessage);
    }
    return 1;
  case WM_COMMAND:
    switch (LOWORD(wParam)) {
    case IDOK:
    case IDCANCEL:
      EndDialog(hWnd, LOWORD(wParam));
      return 1;
    }
    break;
  }
  return 0;
}

static VOID PopupProgress(PrintInfo* pPrt)
{
  pPrt->hProgressWnd =
    CreateDialog(g_hInstance, MAKEINTRESOURCE(IDD_PROGRESS),
		 0, (DLGPROC)ProgressDlgProc);
}

static VOID PopdownProgress(PrintInfo* pPrt)
{
  if (pPrt->hProgressWnd) {
    DestroyWindow(pPrt->hProgressWnd);
    pPrt->hProgressWnd = NULL;
  }
}

static LONG CALLBACK PrinterCallback(TeXDLLMessage msg,
				     DWORD param1, DWORD param2, LPVOID pv)
{
  PrintInfo* pInfo = (PrintInfo*)pv;
  switch (msg) {
  case TeXDLL_STDOUT_WRITE:
    if (pInfo->hPrinter) {
      DWORD nWritten;
      WritePrinter(pInfo->hPrinter, (LPVOID)param1, param2, &nWritten);
    }
    else {
      fwrite((LPVOID)param1, param2, 1, pInfo->fp);
    }
    break;
  case TeXDLL_CONSOLE_WRITE:
#define MESSAGE_BLOCK_SIZE 256
    {
      LPCSTR p = (LPCSTR)param1;
      LPSTR q = pInfo->szMessage;
      int n = pInfo->nMessageLength;
      while (*p) {
	if (n + param2 + 2 >= pInfo->nMessageMax) {
	  LPSTR tmp;
	  pInfo->nMessageMax =
	    (pInfo->nMessageMax + param2 + MESSAGE_BLOCK_SIZE + 2) /
	      MESSAGE_BLOCK_SIZE * MESSAGE_BLOCK_SIZE;
	  tmp = malloc(pInfo->nMessageMax);
	  if (q) {
	    strncpy(tmp, q, n);
	    free(q);
	  }
	  q = tmp;
	}
	if (*p == '\n') q[n++] = '\r';
	q[n++] = *p++;
      }
      q[n] = 0;
      pInfo->nMessageLength = n;
      pInfo->szMessage = q;
    }
    break;
  case TeXDLL_NOTIFY_PAGE_BEGIN:
    if (pInfo->hPrinter) StartPagePrinter(pInfo->hPrinter);
    if (pInfo->hProgressWnd) {
      char buf[50];
      sprintf(buf, "%4lu / %4lu page",
	      ++pInfo->nCurrentPage, pInfo->nToPage - pInfo->nFromPage + 1);
      SetDlgItemText(pInfo->hProgressWnd, IDC_PAGE, buf);
    }
    break;
  case TeXDLL_NOTIFY_PAGE_END:
    if (pInfo->hPrinter) EndPagePrinter(pInfo->hPrinter);
    break;
  }
  return 0;
}

static LPSTR GetPaper(DWORD n)
{
  switch (n) {
  case DMPAPER_A3: return "A3";
  case DMPAPER_A4:
  case DMPAPER_A4SMALL:
    return "A4";
  case DMPAPER_A5: return "A5";
  case DMPAPER_B4: return "B4";
  case DMPAPER_B5: return "B5";
  case DMPAPER_LEDGER: return "Ledger";
  case DMPAPER_LEGAL: return "Legal";
  case DMPAPER_LETTER: return "Letter";
  case DMPAPER_NOTE: return "Note";

  case DMPAPER_10X14: return "Sheet10x14";
  case DMPAPER_11X17: return "Sheet11x17";
  case DMPAPER_CSHEET: return "SheetC";
  case DMPAPER_DSHEET: return "SheetD";
  case DMPAPER_ENV_9: return "Env9";
  case DMPAPER_ENV_10: return "Env10";
  case DMPAPER_ENV_11: return "Env11";
  case DMPAPER_ENV_12: return "Env12";
  case DMPAPER_ENV_14: return "Env14";
  case DMPAPER_ENV_B4: return "EnvB4";
  case DMPAPER_ENV_B5: return "EnvB5";
  case DMPAPER_ENV_B6: return "EnvB6";
  case DMPAPER_ENV_C3: return "EnvC3";
  case DMPAPER_ENV_C4: return "EnvC4";
  case DMPAPER_ENV_C5: return "EnvC5";
  case DMPAPER_ENV_C6: return "EnvC6";
  case DMPAPER_ENV_C65: return "EnvC65";
  case DMPAPER_ENV_DL: return "EnvDL";
  case DMPAPER_ENV_ITALY: return "EnvItaly";
  case DMPAPER_ENV_MONARCH: return "EnvMonarch";
  case DMPAPER_ENV_PERSONAL: return "EnvPersonal";
  case DMPAPER_ESHEET: return "SheetE";
  case DMPAPER_EXECUTIVE: return "Executive";
  case DMPAPER_FANFOLD_LGL_GERMAN: break;
  case DMPAPER_FANFOLD_STD_GERMAN: break;
  case DMPAPER_FANFOLD_US: break;
  case DMPAPER_FOLIO: return "Folio";
  case DMPAPER_LETTERSMALL: return "LetterSmall";
  case DMPAPER_QUARTO: return "Quarto";
  case DMPAPER_STATEMENT: return "Statement";
  case DMPAPER_TABLOID: return "Tabloid";
  case DMPAPER_USER: break;
  }
  return "A4";
}

UINT WINAPI DVIPSThreadFunc(LPVOID pv)
{
  PrintInfo* pInfo = (PrintInfo*)pv;
  static LPCSTR dll_name = "dvips.dll";
  static LPCSTR dll_main = "DVIPSMain";
  HANDLE hDLL;
  BOOL code = FALSE;

  if (pInfo->hProgressWnd)
    SetDlgItemText(pInfo->hProgressWnd, IDC_FILENAME,
		   win32_basename(pInfo->szDviFileName));
  
  hDLL = win32_load_tex_library(dll_name);
  if (hDLL) {
    LPFDVIPSMainProc Main;
    (FARPROC)Main = GetProcAddress(hDLL, dll_main);
    if (Main) {
      int argc = 0;
      LPSTR argv[30];
      char pagenum[30];
      char dpi[30];
      char copies[30];
      char papername[30];
      LPSTR dviDir = strdup(pInfo->szDviFileName);
      {
	LPSTR p = (LPSTR)win32_basename(dviDir);
	LPBYTE s;
	*p = 0;
	s = _mbsrchr(p, '\\');
	if (s - dviDir == strlen(dviDir)-1 && strlen(dviDir) > 3)
	  *s = 0; /* remove trailing backslash */
      }
      argv[argc++] = "dvips";
      argv[argc++] = "-h";
      argv[argc++] = "wdvips.pro";
      argv[argc++] = "-P";
      argv[argc++] = "wdvips";
      argv[argc++] = "-o";
      argv[argc++] = "-";
      argv[argc++] = "-pp";
      argv[argc++] = pagenum;
      sprintf(pagenum, "%lu-%lu",
	      (ULONG)pInfo->nFromPage, (ULONG)pInfo->nToPage);
      if (pInfo->nPrinterDpi > 0) {
	argv[argc++] = "-D";
	argv[argc++] = dpi;
	sprintf(dpi, "%lu", (ULONG)pInfo->nPrinterDpi);
      }
      strcpy(papername, GetPaper(pInfo->nPaperSize));
      if (pInfo->bDuplex) strcat(papername, "Duplex");
      argv[argc++] = "-t";
      argv[argc++] = papername;
      if (pInfo->nCopies) {
	argv[argc++] = pInfo->bCollate ? "-C" : "-c";
	argv[argc++] = copies;
	sprintf(copies, "%lu", (ULONG)pInfo->nCopies);
      }
      argv[argc] = win32_strdup(pInfo->szDviFileName);
      win32_dos2unix(argv[argc]);
      argc++;
      SetCurrentDirectory(dviDir);
      if (!(*Main)(PrinterCallback, pInfo, NULL, argc, argv))
	code = TRUE;
    }
    else {
      FileErrorMessage("Could not resolve the entry point.", dll_main, 0);
    }
    FreeLibrary(hDLL);
  }
  else {
    FileErrorMessage("Could not load DLL.", dll_name, 0);
  }
  return code;
}

BOOL DoDVIPS(PrintInfo* pInfo)
{
  HANDLE hThread;
  DWORD idThread;
  DWORD result = FALSE;
  PopupProgress(pInfo);
  hThread = (HANDLE)_beginthreadex(NULL, 0, DVIPSThreadFunc, pInfo,
				   0, &idThread);
  if (hThread) {
    for ( ; ; ) {
      MSG msg;
      while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
	if (pInfo->hProgressWnd &&
	    !IsDialogMessage(pInfo->hProgressWnd, &msg)) {
	  TranslateMessage(&msg);
	  DispatchMessage(&msg);
	}
      }
      if (MsgWaitForMultipleObjects(1, &hThread, FALSE,
				    INFINITE, QS_ALLINPUT) == WAIT_OBJECT_0)
	break;
    }
    GetExitCodeThread(hThread, &result);
    CloseHandle(hThread);
  }
  PopdownProgress(pInfo);
  return result;
}

static BOOL DoPrint(PrintInfo* pInfo)
{
  DOC_INFO_1 di;
  BOOL code;
  
  if (!OpenPrinter(pInfo->szPrinterName, &pInfo->hPrinter, NULL)) {
    FileErrorMessage("Could not open the printer.", pInfo->szPrinterName, 0);
    return FALSE;
  }
  di.pDatatype = "RAW";
  di.pDocName = (LPSTR)win32_basename(pInfo->szDviFileName);
  di.pOutputFile = 0;
  if (StartDocPrinter(pInfo->hPrinter, 1, (LPBYTE)&di) == SP_ERROR) {
    ErrorMessage("Print Error.", 0);
    AbortPrinter(pInfo->hPrinter);
    return FALSE;
  }
  code = DoDVIPS(pInfo);
  EndDocPrinter(pInfo->hPrinter);
  ClosePrinter(pInfo->hPrinter);
  return code;
}

static BOOL DoFile(PrintInfo* pInfo)
{
  BOOL code;
  FILE *fp = pInfo->fp = fopen(pInfo->szOutputFileName, "w");
  if (!fp) {
    FileErrorMessage("Could not open.", pInfo->szOutputFileName, 0);
    return FALSE;
  }
  code = DoDVIPS(pInfo);
  fclose(fp);
  return code;
}

static BOOL CheckDvi(PrintInfo* pInfo)
{
  int i;
  int c;
  LONG post;
  FILE* fp = fopen(pInfo->szDviFileName, "rb");
  
  if (!fp) {
    FileErrorMessage("Could not open.", pInfo->szDviFileName, 0);
    return FALSE;
  }
  
  c = getc(fp);
  if (c != 247) {
  not_dvi:
    FileErrorMessage("File does not seem DVI file.",
		     pInfo->szDviFileName, 0);
    fclose(fp);
    return FALSE;
  }
  c = getc(fp);
  if (c != 2) goto not_dvi;

  if (fseek(fp, -1, SEEK_END) != 0)
    goto not_dvi;
  
  for ( ; ; ) {
    c = getc(fp);
    if (c != 223) break;
    if (fseek(fp, -2, SEEK_CUR) != 0) goto not_dvi;
  }
  if (fseek(fp, -6, SEEK_CUR) != 0) goto not_dvi;
  c = getc(fp);
  if (c != 249) goto not_dvi;

  post = 0;
  for (i=0;i<4;i++)
    post = (post << 8) | getc(fp);
  if (fseek(fp, post, SEEK_SET) != 0) goto not_dvi;
  c = getc(fp);
  if (c != 248) goto not_dvi;
  if (fseek(fp, 26, SEEK_CUR) != 0) goto not_dvi;
  pInfo->nDviPages = 0;
  pInfo->nDviPages = (getc(fp) << 8);
  pInfo->nDviPages |= getc(fp);
  
  fclose(fp);

  if (pInfo->nDviPages == 0) {
    FileErrorMessage("No page in DVI file.",pInfo->szDviFileName , 0);
    return FALSE;
  }
  
  return TRUE;
}

static BOOL MaybePostScript(DEVMODE* pMode, DEVNAMES* pNames)
{
  if (!(pMode->dmFields & DM_TTOPTION) ||
      pMode->dmTTOption != DMTT_SUBDEV)
    return FALSE;
#if 0
  if (!(pMode->dmFields & DM_ICMMETHOD) ||
      !(pMode->dmICMMethod & DMICMMETHOD_DEVICE))
    return FALSE;
#endif
  return TRUE;
}

UINT WINAPI PrintDlgHook(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  return 0;
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrev,
		   LPSTR szCmdLine, int nCmdShow)
{
  BOOL code = TRUE;
  PrintInfo* pPrt;
  PRINTDLG dlg;
  DEVMODE* pMode;
  DEVNAMES* pNames;

  g_hInstance = hInstance;
  
  win32_init_module(hInstance);
  
  if (__argc > 2) {
    ErrorMessage("Too many arguments.", 0);
    return 0;
  }

  pPrt = malloc(sizeof(PrintInfo));
  memset(pPrt, 0, sizeof(PrintInfo));
  
  if (__argc <= 1) {
    if (!(pPrt->szDviFileName = GetDviFileName(NULL)))
      return 0;
  }
  else {
    pPrt->szDviFileName = __argv[1];
  }

  if (!CheckDvi(pPrt))
    return 0;
  
  memset(&dlg, 0, sizeof(dlg));
  dlg.lStructSize = sizeof(PRINTDLG);
  dlg.Flags = PD_NOSELECTION | PD_USEDEVMODECOPIESANDCOLLATE;
  dlg.nFromPage = dlg.nToPage = 1;
  dlg.nMinPage = 1;
  dlg.nMaxPage = pPrt->nDviPages;
  dlg.nCopies = 1;
  dlg.hInstance = g_hInstance;
  dlg.lpPrintTemplateName = MAKEINTRESOURCE(IDD_PRINT);
  
  if (!PrintDlg(&dlg)) return 0;
  pMode = GlobalLock(dlg.hDevMode);
  pNames = GlobalLock(dlg.hDevNames);
  
  if (dlg.Flags & PD_PAGENUMS) {
    pPrt->nFromPage = dlg.nFromPage;
    pPrt->nToPage = dlg.nToPage;
  }
  else {
    pPrt->nFromPage = dlg.nMinPage;
    pPrt->nToPage = dlg.nMaxPage;
  }
  pPrt->szPrinterName = strdup((LPCSTR)pNames + pNames->wDeviceOffset);
  pPrt->nCopies = dlg.nCopies;
  if ((pMode->dmFields & DM_PRINTQUALITY) && pMode->dmPrintQuality > 0) {
    pPrt->nPrinterDpi = pMode->dmPrintQuality;
  }
  if (pMode->dmFields & DM_PAPERSIZE) {
    pPrt->nPaperSize = pMode->dmPaperSize;
  }
  if (pMode->dmFields & DM_COLLATE)
    pPrt->bCollate = (pMode->dmCollate == DMCOLLATE_TRUE);
  else
    pPrt->bCollate = FALSE;
  if (pMode->dmFields & DM_DUPLEX)
    pPrt->bDuplex = !(pMode->dmCollate == DMDUP_SIMPLEX);
  else
    pPrt->bDuplex = FALSE;
  if ((dlg.Flags & PD_PRINTTOFILE) ||
      strcmp("FILE:", (LPCSTR)pNames + pNames->wOutputOffset) == 0) {
    pPrt->szOutputFileName =
      GetOutputFileName(win32_make_suffix(pPrt->szDviFileName, "ps"));
    if (pPrt->szOutputFileName == NULL)
      code = FALSE;
  }
  if (code && !pPrt->szOutputFileName && !MaybePostScript(pMode, pNames)) {
    if (FileErrorMessage("Printer does not seem a PostScript device.\n"
			 "Continue ?",
			 pMode->dmDeviceName, MB_YESNO | MB_ICONWARNING) ==
	IDNO)
      code = FALSE;
  }
  GlobalUnlock(dlg.hDevMode);
  GlobalUnlock(dlg.hDevNames);
  
  if (!code) return code;
  
  code = pPrt->szOutputFileName ? DoFile(pPrt) : DoPrint(pPrt);
  
  DialogBoxParam(g_hInstance, MAKEINTRESOURCE(IDD_MESSAGE),
		 0, (DLGPROC)MessageDlgProc, (LONG)pPrt);

  free(pPrt);
  
  return code;
}
