/* Copyright (C) 1985, 86, 87, 93, 94, 96 Free Software Foundation, Inc.
   Copyright (C) 2001, 2010 Ben Wing.

This file is part of XEmacs.

XEmacs is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs.  If not, see <http://www.gnu.org/licenses/>. */

/* Synced with FSF 20.2 */

#include <config.h>
#include "lisp.h"

#include "buffer.h"
#include <paths.h>

#include "sysdir.h"
#include "sysfile.h"
#include "sysproc.h" /* for qxe_getpid() */
#include "syspwd.h"
#include "syssignal.h" /* for kill. */

Lisp_Object Qask_user_about_supersession_threat;
Lisp_Object Qask_user_about_lock;
Boolint inhibit_clash_detection;

#ifdef CLASH_DETECTION

/* The strategy: to lock a file FN, create a symlink .#FN# in FN's
   directory, with link data `user@host.pid'.  This avoids a single
   mount (== failure) point for lock files.

   When the host in the lock data is the current host, we can check if
   the pid is valid with kill.

   Otherwise, we could look at a separate file that maps hostnames to
   reboot times to see if the remote pid can possibly be valid, since we
   don't want Emacs to have to communicate via pipes or sockets or
   whatever to other processes, either locally or remotely; rms says
   that's too unreliable.  Hence the separate file, which could
   theoretically be updated by daemons running separately -- but this
   whole idea is unimplemented; in practice, at least in our
   environment, it seems such stale locks arise fairly infrequently, and
   Emacs' standard methods of dealing with clashes suffice.

   We use symlinks instead of normal files because (1) they can be
   stored more efficiently on the filesystem, since the kernel knows
   they will be small, and (2) all the info about the lock can be read
   in a single system call (readlink).  Although we could use regular
   files to be useful on old systems lacking symlinks, nowadays
   virtually all such systems are probably single-user anyway, so it
   didn't seem worth the complication.

   Similarly, we don't worry about a possible 14-character limit on
   file names, because those are all the same systems that don't have
   symlinks.

   Originally we used a name .#FN without the final #; this may have been
   compatible with the locking scheme used by Interleaf (which has
   contributed this implementation for Emacs), and was designed by Ethan
   Jacobson, Kimbo Mundy, and others.

   --karl@cs.umb.edu/karl@hq.ileaf.com.  */

/* NOTE:  We added the final # in the name .#FN# so that programs
   that e.g. search for all .c files, such as etags, or try to
   byte-compile all .el files in a directory (byte-recompile-directory),
   won't get tripped up by the bogus symlink file. --ben */


/* Here is the structure that stores information about a lock.  */

typedef struct
{
  Ibyte *user;
  Ibyte *host;
  pid_t pid;
} lock_info_type;

/* Free the two dynamically-allocated pieces in PTR.  */
#define FREE_LOCK_INFO(i) do {			\
    xfree ((i).user);			\
    xfree ((i).host);			\
  } while (0)

/* Write the name of the lock file for FN into LFNAME.  Length will be that
   of FN plus two more for the leading `.#' plus one for the trailing #
   plus one for the null.  */
#define MAKE_LOCK_NAME(lock, file) \
  (lock = alloca_ibytes (XSTRING_LENGTH (file) + 2 + 1 + 1), \
   fill_in_lock_file_name (lock, file))

static void
fill_in_lock_file_name (Ibyte *lockfile, Lisp_Object fn)
{
  Ibyte *file_name = XSTRING_DATA (fn);
  Ibyte *p;
  size_t dirlen;

  for (p = file_name + XSTRING_LENGTH (fn) - 1;
       p > file_name && !IS_ANY_SEP (p[-1]);
       p--)
    ;
  dirlen = p - file_name;

  memcpy (lockfile, file_name, dirlen);
  p = lockfile + dirlen;
  *(p++) = '.';
  *(p++) = '#';
  memcpy (p, file_name + dirlen, XSTRING_LENGTH (fn) - dirlen);
  p += XSTRING_LENGTH (fn) - dirlen;
  *(p++) = '#';
  *p = '\0';
}

/* Lock the lock file named LFNAME.
   If FORCE is nonzero, we do so even if it is already locked.
   Return 1 if successful, 0 if not.  */

static int
lock_file_1 (Ibyte *lfname, int force)
{
  /* Does not GC. */
  int err;
  Ibyte *lock_info_str;
  Bytecount lock_info_size;
  Ibyte *host_name;
  Ibyte *user_name = user_login_name (NULL);

  if (user_name == NULL)
    user_name = (Ibyte *) "";

  if (STRINGP (Vsystem_name))
    host_name = XSTRING_DATA (Vsystem_name);
  else
    host_name = (Ibyte *) "";

  lock_info_size = qxestrlen (user_name) + sizeof ("@.") +
    qxestrlen (host_name) + DECIMAL_PRINT_SIZE (pid_t);
  lock_info_str = alloca_ibytes (lock_info_size);

  emacs_snprintf (lock_info_str, lock_info_size,
                  "%s@%s.%d", user_name, host_name, (int) qxe_getpid ());

  err = qxe_symlink (lock_info_str, lfname);
  if (err != 0 && errno == EEXIST && force)
    {
      qxe_unlink (lfname);
      err = qxe_symlink (lock_info_str, lfname);
    }

  return err == 0;
}

/* Return 0 if nobody owns the lock file LFNAME or the lock is obsolete,
   1 if another process owns it (and set OWNER (if non-null) to info),
   2 if the current process owns it,
   or -1 if something is wrong with the locking mechanism.  */

static int
current_lock_owner (lock_info_type *owner, Ibyte *lfname)
{
  /* Does not GC. */
  Bytecount len;
  int local_owner = 0, ret;
  Ibyte *at, *dot;
  Ibyte *lfinfo = 0;
  size_t bufsize = 50;
  /* Read arbitrarily-long contents of symlink.  Similar code in
     file-symlink-p in fileio.c.  */
  do
    {
      bufsize *= 2;
      lfinfo = (Ibyte *) xrealloc (lfinfo, bufsize);
      len = qxe_readlink (lfname, lfinfo, bufsize);
    }
  while (len >= (Bytecount) bufsize);

  /* If nonexistent lock file, all is well; otherwise, got strange error. */
  if (len == -1)
    {
      xfree (lfinfo);
      return errno == ENOENT ? 0 : -1;
    }

  /* Link info exists, so `len' is its length.  Null terminate.  */
  lfinfo[len] = 0;

  /* Even if the caller doesn't want the owner info, we still have to
     read it to determine return value, so allocate it.  */
  if (!owner)
    {
      owner = alloca_new (lock_info_type);
      local_owner = 1;
    }

  /* Parse USER@HOST.PID.  If can't parse, return -1.  */
  /* The USER is everything before the first @.  */
  at = qxestrchr (lfinfo, '@');
  dot = qxestrrchr (lfinfo, '.');
  if (!at || !dot) {
    xfree (lfinfo);
    return -1;
  }
  len = at - lfinfo;
  owner->user = xnew_ibytes (len + 1);
  qxestrncpy (owner->user, lfinfo, len);
  owner->user[len] = 0;

  /* The PID is everything after the last `.'.  */
  owner->pid = atoi ((CIbyte *) dot + 1);

  /* The host is everything in between.  */
  len = dot - at - 1;
  owner->host = xnew_ibytes (len + 1);
  qxestrncpy (owner->host, at + 1, len);
  owner->host[len] = 0;

  /* We're done looking at the link info.  */
  xfree (lfinfo);

  /* On current host?  */
  if (STRINGP (Fsystem_name ())
      && qxestrcmp (owner->host, XSTRING_DATA (Fsystem_name ())) == 0)
    {
      if (owner->pid == qxe_getpid ())
        ret = 2; /* We own it.  */
      else if (owner->pid > 0
               && (kill (owner->pid, 0) >= 0 || errno == EPERM))
        ret = 1; /* An existing process on this machine owns it.  */
      /* The owner process is dead or has a strange pid (<=0), so try to
         zap the lockfile.  */
      else if (qxe_unlink (lfname) < 0)
        ret = -1;
      else
	ret = 0;
    }
  else
    { /* If we wanted to support the check for stale locks on remote machines,
         here's where we'd do it.  */
      ret = 1;
    }

  /* Avoid garbage.  */
  if (local_owner || ret <= 0)
    {
      FREE_LOCK_INFO (*owner);
    }
  return ret;
}

/* Lock the lock named LFNAME if possible.
   Return 0 in that case.
   Return positive if some other process owns the lock, and info about
     that process in CLASHER.
   Return -1 if cannot lock for any other reason.  */

static int
lock_if_free (lock_info_type *clasher, Ibyte *lfname)
{
  /* Does not GC. */
  if (lock_file_1 ((Ibyte *) lfname, 0) == 0)
    {
      int locker;

      if (errno != EEXIST)
	return -1;

      locker = current_lock_owner (clasher, lfname);
      if (locker == 2)
        {
          FREE_LOCK_INFO (*clasher);
          return 0;   /* We ourselves locked it.  */
        }
      else if (locker == 1)
        return 1;  /* Someone else has it.  */

      return -1; /* Something's wrong.  */
    }
  return 0;
}

/* lock_file locks file FN,
   meaning it serves notice on the world that you intend to edit that file.
   This should be done only when about to modify a file-visiting
   buffer previously unmodified.
   Do not (normally) call this for a buffer already modified,
   as either the file is already locked, or the user has already
   decided to go ahead without locking.

   When this returns, either the lock is locked for us,
   or the user has said to go ahead without locking.

   If the file is locked by someone else, this calls
   ask-user-about-lock (a Lisp function) with two arguments,
   the file name and info about the user who did the locking.
   This function can signal an error, or return t meaning
   take away the lock, or return nil meaning ignore the lock.  */

void
lock_file (Lisp_Object fn)
{
  /* This function can GC.  GC checked 7-11-00 ben */
  /* dmoore - and can destroy current_buffer and all sorts of other
     mean nasty things with pointy teeth.  If you call this make sure
     you protect things right. */
  /* Somebody updated the code in this function and removed the previous
     comment.  -slb */

  REGISTER Lisp_Object attack, orig_fn;
  REGISTER Ibyte *lfname;
  lock_info_type lock_info;
  struct gcpro gcpro1, gcpro2, gcpro3, gcpro4;
  Lisp_Object old_current_buffer, subject_buf = Qnil, locker = Qnil;

  if (inhibit_clash_detection)
    return;

  old_current_buffer = wrap_buffer (current_buffer);
  GCPRO4 (fn, subject_buf, old_current_buffer, locker);
  orig_fn = fn;
  fn = Fexpand_file_name (fn, Qnil);

  /* Create the name of the lock-file for file fn */
  MAKE_LOCK_NAME (lfname, fn);

  /* See if this file is visited and has changed on disk since it was
     visited.  */
  {
    subject_buf = get_truename_buffer (orig_fn);
    if (!NILP (subject_buf)
	&& NILP (Fverify_visited_file_modtime (subject_buf))
	&& !NILP (Ffile_exists_p (fn)))
      call1_in_buffer (XBUFFER (subject_buf),
		       Qask_user_about_supersession_threat, fn);
  }

  /* Try to lock the lock. */
  if (current_buffer != XBUFFER (old_current_buffer)
      || lock_if_free (&lock_info, lfname) <= 0)
    /* Return now if we have locked it, or if lock creation failed
     or current buffer is killed. */
    goto done;

  /* Else consider breaking the lock */
  locker = emacs_sprintf_string ("%s@%s (pid %d)", lock_info.user,
                                 lock_info.host, (int) lock_info.pid);
  FREE_LOCK_INFO (lock_info);

  attack = call2_in_buffer (BUFFERP (subject_buf) ? XBUFFER (subject_buf) :
			    current_buffer, Qask_user_about_lock , fn,
			    locker);
  if (!NILP (attack) && current_buffer == XBUFFER (old_current_buffer))
    /* User says take the lock */
    {
      lock_file_1 (lfname, 1);
      goto done;
    }
  /* User says ignore the lock */
 done:
  UNGCPRO;
}

void
unlock_file (Lisp_Object fn)
{
  /* This can GC */
  REGISTER Ibyte *lfname;
  struct gcpro gcpro1;

  GCPRO1 (fn);

  fn = Fexpand_file_name (fn, Qnil);

  MAKE_LOCK_NAME (lfname, fn);

  if (current_lock_owner (0, lfname) == 2)
    qxe_unlink (lfname);

  UNGCPRO;
}

void
unlock_all_files (void)
{
  REGISTER Lisp_Object tail;

  for (tail = Vbuffer_alist; CONSP (tail); tail = XCDR (tail))
    {
      struct buffer *b = XBUFFER (XCDR (XCAR (tail)));
      if (STRINGP (b->file_truename)
          && buf_tick_arithcompare (BUF_SAVE_MODIFF (b), BUF_MODIFF (b)) < 0)
	unlock_file (b->file_truename);
    }
}

DEFUN ("lock-buffer", Flock_buffer,   0, 1, 0, /*
Lock FILE, if current buffer is modified.
FILE defaults to current buffer's visited file,
or else nothing is done if current buffer isn't visiting a file.
*/
       (file))
{
  if (NILP (file))
    file = current_buffer->file_truename;
  CHECK_STRING (file);
  if (buf_tick_arithcompare (BUF_SAVE_MODIFF (current_buffer),
                             BUF_MODIFF (current_buffer)) < 0
      && !NILP (file))
    lock_file (file);
  return Qnil;
}

DEFUN ("unlock-buffer", Funlock_buffer, 0, 0, 0, /*
Unlock the file visited in the current buffer,
if it should normally be locked.
*/
       ())
{
  /* This function can GC */
  /* dmoore - and can destroy current_buffer and all sorts of other
     mean nasty things with pointy teeth.  If you call this make sure
     you protect things right. */

  if (buf_tick_arithcompare (BUF_SAVE_MODIFF (current_buffer),
                             BUF_MODIFF (current_buffer)) < 0
      && STRINGP (current_buffer->file_truename))
    unlock_file (current_buffer->file_truename);
  return Qnil;
}

/* Unlock the file visited in buffer BUFFER.  */


void
unlock_buffer (struct buffer *buffer)
{
  /* This function can GC */
  /* dmoore - and can destroy current_buffer and all sorts of other
     mean nasty things with pointy teeth.  If you call this make sure
     you protect things right. */
  if (buf_tick_arithcompare (BUF_SAVE_MODIFF (buffer),
                             BUF_MODIFF (buffer)) < 0
      && STRINGP (buffer->file_truename))
    unlock_file (buffer->file_truename);
}

DEFUN ("file-locked-p", Ffile_locked_p, 0, 1, 0, /*
Return nil if the FILENAME is not locked,
t if it is locked by you, else a string of the name of the locker.
*/
       (filename))
{
  Lisp_Object ret;
  REGISTER Ibyte *lfname;
  int owner;
  lock_info_type locker;
  struct gcpro gcpro1;

  GCPRO1 (filename);

  filename = Fexpand_file_name (filename, Qnil);

  MAKE_LOCK_NAME (lfname, filename);

  owner = current_lock_owner (&locker, lfname);
  if (owner <= 0)
    ret = Qnil;
  else if (owner == 2)
    ret = Qt;
  else
    ret = build_istring (locker.user);

  if (owner > 0)
    FREE_LOCK_INFO (locker);

  UNGCPRO;

  return ret;
}


/* Initialization functions.  */

void
syms_of_filelock (void)
{
  /* This function can GC */
  DEFSUBR (Funlock_buffer);
  DEFSUBR (Flock_buffer);
  DEFSUBR (Ffile_locked_p);

  DEFSYMBOL (Qask_user_about_supersession_threat);
  DEFSYMBOL (Qask_user_about_lock);
}

void
vars_of_filelock (void)
{
  DEFVAR_BOOL ("inhibit-clash-detection", &inhibit_clash_detection /*
Non-nil inhibits creation of lock file to detect clash.
*/);
  inhibit_clash_detection = 0;
}

#endif /* CLASH_DETECTION */
