#!/usr/bin/perl

use strict;
use warnings;
use Test::More;

use Business::ES::CodigoPostal 'validate_cp';

BEGIN { use_ok('Business::ES::CodigoPostal') }

{
    my $result = validate_cp('28001');
    ok($result->{valid}, 'Código válido: 28001');
    is($result->{provincia}, 'Madrid', 'Provincia correcta');
    is($result->{iso_3166_2}, 'ES-M', 'ISO 3166-2 correcto');
}

{
    my $result = validate_cp('99999');
    ok(!$result->{valid}, 'Código inválido: 99999');
    like($result->{error}, qr/no asignado/, 'Error correcto para código no asignado');
}

{
    my $result = validate_cp('abcde');
    ok(!$result->{valid}, 'Código inválido: abcde');
    like($result->{error}, qr/5 dígitos/, 'Error correcto para formato inválido');
}

{
    my $result = validate_cp(undef);
    ok(!$result->{valid}, 'Código undefined');
    like($result->{error}, qr/no definido/, 'Error correcto para undefined');
}

{
    my $cp = Business::ES::CodigoPostal->new(codigo => '18001');
    ok($cp->is_valid, 'Objeto válido creado');
    is($cp->codigo, '18001', 'Código almacenado correctamente');
    is($cp->provincia, 'Granada', 'Provincia correcta');
    is($cp->iso_3166_2, 'ES-GR', 'ISO correcto');
}

{
    my $cp = Business::ES::CodigoPostal->new(
        codigo => ' 080-01 ',
        strict => 0
    );
    ok($cp->is_valid, 'Normalización funciona');
    is($cp->codigo, '08001', 'Código normalizado correctamente');
}

{
    my $cp = Business::ES::CodigoPostal->new(
        codigo => ' 080-01 ',
        strict => 1
    );
    ok(!$cp->is_valid, 'Modo strict rechaza formato incorrecto');
}

{
    my $cp = Business::ES::CodigoPostal->new();
    ok($cp->set('41001'), 'Set con código válido');
    is($cp->provincia, 'Sevilla', 'Provincia actualizada correctamente');
    
    ok(!$cp->set('99999'), 'Set con código inválido');
    ok(!$cp->is_valid, 'Objeto inválido después de set fallido');
}

{
    my $cp1 = Business::ES::CodigoPostal->new(codigo => '50001');
    my $cp2 = Business::ES::CodigoPostal->new({ codigo => '50001' });
    
    is($cp1->codigo, $cp2->codigo, 'Ambos constructores funcionan igual');
}

{
    ok(defined Business::ES::CodigoPostal::ERROR_DIGITS5, 'Constante ERROR_DIGITS5 existe');
    ok(defined Business::ES::CodigoPostal::ERROR_DEFINED, 'Constante ERROR_DEFINED existe');
    ok(defined Business::ES::CodigoPostal::ERROR_ASSIGN, 'Constante ERROR_ASSIGN existe');
}

{
    my $result1 = validate_cp('01000');
    ok($result1->{valid}, 'Código límite inferior válido');
    
    my $result2 = validate_cp('52999');
    ok($result2->{valid}, 'Código límite superior válido');
    
    my $result3 = validate_cp('00999');
    ok(!$result3->{valid}, 'Código por debajo del límite');
    
    my $result4 = validate_cp('53000');
    ok(!$result4->{valid}, 'Código por encima del límite');
}

{
    my @codigos_test = qw(01001 28001 08001 41001 51001 52001);
    
    for my $codigo (@codigos_test) {
        my $result = validate_cp($codigo);
        ok($result->{valid}, "Código $codigo es válido");
        ok(defined $result->{provincia}, "Provincia definida para $codigo");
        ok(defined $result->{iso_3166_2}, "ISO definido para $codigo");
    }
}

done_testing();
