package Proc::NiceSleep;

#############################################################################
# Proc::NiceSleep - quasi-intelligent sleeping library
# Copyright (c) 2002 Josh Rabinowitz, see COPYRIGHT below
# $Id: NiceSleep.pm,v 1.23 2002/02/23 17:10:53 root Exp root $ 
# originally generated by joshr 20020216
#############################################################################
use 5.004;	# tested this far back and up to 5.6.1...
use strict;	# please
#use warnings;	# doesn't exist in 5.004

require Exporter;
#use AutoLoader qw(AUTOLOAD);	# we don't use this yet

# We do 'use vars' like this so we can work nice in old perls
use vars qw($VERSION);
$VERSION = '0.52';

# these are 'public'
use vars qw ( %EXPORT_TAGS @EXPORT_OK @ISA );

@ISA = qw(Exporter);

# Items to export into callers namespace by default. 

# This allows declaration	use Proc::NiceSleep ':all';
# Note to self: If we do not need this, moving things directly 
# into @EXPORT or @EXPORT_OK will save memory.
#our 
%EXPORT_TAGS = ( 'all' => [ qw(
	nice maybesleep maxload sleepfactor minruntime minsleeptime 
) ] );

#our 
@EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

# these are private. 
use vars qw ( $_sleepfactor $_minruntime $_minsleeptime $_maxload 
		$_totalruntime $_totalsleeptime $_lastsleeptime $_lastloadchecktime 
		$_havetimehires $_havetimehires $_havesetpriority 
		$_haveprocprocesstable $_havesyscpuload ); 
# variables prefixed by _ are intended to be private
#$_lastsleeptime;	# the last time we slept, from time()
#$_lastloadchecktime;	# the last time we checked load()
#$_sleepfactor;	# 1.0 means to sleep 1.0 times as long as we 'run'
				# 0.0 means don't sleep based on fraction of tmie
#$_minruntime;	# how long we run before considering yielding
#$_minsleeptime;	# minimum time to sleep, if we do
#$_maxload;		# the maximum 1-minute avg system load we yield at,
				# if supported and Sys::CpuLoad works
#$_totalruntime;	# how long we were running, in seconds;
#					# does not include time sleeping in maybesleep()
#$_totalsleeptime;	# how long we slept in maybesleep(), 
#						# in apparent wallclock seconds
#$_havetimehires;		# do we have Time::HiRes ?
#$_haveprocprocesstable;	# do we have Proc::ProcessTable?
#$_havesyscpuload;			# do we have Sys::CpuLoad?
#$_havesetpriority;		# do we have a setpriority() call?

# all through we use Time::HiRes or built-in versions of
# time() and sleep(), and get microsecond res ... or not.

#############################################################################
# Preloaded methods go here.
#############################################################################

# nice() this renices the process, like /bin/nice, if it can.
# if passed an integer parameter (between -20 to 20 inclusive)
#   it attempts to set the priority and returns the priority 
#   it tried to set the process to.
# if called without a parameter, returns what we think the priority is
# does not work on win32 (always should return 0); use maybesleep() ! :)
sub nice {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $param = shift;
	# 'man setpriority' on rh7.2: The setpriority call returns 0 if there 
	# is no error, or -1 if there is.
	if (defined($param)) {
		$param = int($param);	# pass me an int, holmes
		if ($_havesetpriority && setpriority(0,0,$param) != -1) {
			# even though man page says the above, setpriority(0,0,5) returns 1
			# on RH7.2
			return $param;
		} else {
			return 0;
		} 
	} 
	return ($_havesetpriority ? getpriority(0,0) : 0); 
	# no param, return what we think the nice value is.
} 

# checks to see if we should sleep.
# returns how long we think we slept if we did, 0 otherwise
sub maybesleep {	
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $t1 = ($_havetimehires ? Time::HiRes::time() : CORE::time());
	my $timepassed = $t1 - $_lastsleeptime;
	my ($timetosleep, $timeslept) = (0, 0);
	if ($_minruntime && $timepassed < $_minruntime) { return 0; }
	if ($_sleepfactor) {
		$timetosleep = $_sleepfactor * $timepassed;
	} elsif ($_havesyscpuload && $_maxload && 
	  ($t1 - $_lastloadchecktime > 0.9)) {
		# we only check the load a max of about once per second
		$_lastloadchecktime = $t1;
		my @loads = Sys::CpuLoad::load();	# (1minavg, 5minavg, 15minavg)
		if ($loads[0] > $_maxload) {	
			# sleep at least 4 seconds if load is too high
			$timetosleep = MAX(4, $_minsleeptime);	
		} 
	}
	if ($timetosleep) {	# we should sleep... snore....
		if ($_minsleeptime && $timetosleep < $_minsleeptime) {
			$timetosleep = $_minsleeptime;
		}
		if($_havetimehires) {
			Time::HiRes::sleep($timetosleep);	# yield the system via sleep
		} else {
			$timetosleep = int($timetosleep + .5);	# round off.
			if ($timetosleep <= 0) { $timetosleep = 1; } # can't be neg or 0
			CORE::sleep($timetosleep);	# actually yield the system via sleep
		}
		my $t2 = ($_havetimehires ? Time::HiRes::time() : CORE::time());
		my $actualsleeptime = $t2 - $t1;
		$_totalruntime += ($t1 - $_lastsleeptime);
		$_totalsleeptime += $actualsleeptime;	# how long we slept
		$_lastsleeptime = $t2;					# record this
		$timeslept = $actualsleeptime;				# for return
	}
	return $timeslept;	# in case they wonder. this is how long we slept
}
# sets or gets, depending on whether it gets param or not
sub sleepfactor {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $param = shift;
	if (defined($param)) { 
		$param = 0 if ($param < 0);	# don't allow negative sleepfactor
		$_sleepfactor = $param; 
	} 
	else { return $_sleepfactor; }
} 
# sets or gets, depending on whether it gets param or not
sub minsleeptime {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $param = shift;
	if (defined($param)) { 
		$param = 0 if ($param < 0);	# don't allow negative value
		$_minsleeptime = $param; 
	} 
	else { return $_minsleeptime; } 
}
# sets or gets, depending on whether it gets param or not
sub minruntime {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $param = shift;
	if (defined($param)) { 
		$param = 0 if ($param < 0);	# don't allow negative value
		$_minruntime = $param; 
	} 
	else { return $_minruntime; } 
}
# sets or gets, depending on whether it gets param or not
sub maxload {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $param = shift;
	if (defined($param)) { 
		$param = 0 if ($param < 0);	# don't allow negative value
		$_maxload = $param; 
	} 
	else { return $_maxload; } 
}
# returns a ref to a hash with data about the progress...
# for informational purposes only. return values subject to change.
sub Dump {
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my %hash = (
		HAVE_TIME_HIRES => $_havetimehires,
		HAVE_PROC_PROCESSTABLE => $_haveprocprocesstable,
		HAVE_SYS_CPULOAD => $_havesyscpuload,
		HAVE_SETPRIORITY => $_havesetpriority,
		LAST_LOAD_CHECK_TIME => scalar(localtime($_lastloadchecktime)),
		LAST_SLEEP_TIME => scalar(localtime($_lastsleeptime)),
		MAX_LOAD => $_maxload,
		MIN_RUN_TIME => $_minruntime,
		MIN_SLEEP_TIME => $_minsleeptime,
		SLEEP_FACTOR => $_sleepfactor,
		TOTAL_RUN_TIME => $_totalruntime, 	
		TOTAL_SLEEP_TIME => $_totalsleeptime, # extra comma here is ok, cool!  
	);
	return \%hash;
}
# this is for informational purposes only. Data and its output subject to change
# written to remove dependence on Data::Dumper in our examples
sub DumpText { 
	# a convenient method to ascii-ify return of Dump() nicely for reporting.
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	my $hashref = Dump();
	my $str = "";
	for my $e (sort keys(%$hashref)) {
		$str .= sprintf("  %-23s: %s\n", $e, $$hashref{$e});
	}
	return $str;	# returns a nice, ascii text page of the name/vals :)
}

# time() and sleep() are so test programs don't have to test for Time::HiRes
# they do hi-res if possible. They are also shown used in example.pl, 
# but are not yet documented as public... should they be, kind reader?
sub time  { ($_havetimehires ? Time::HiRes::time()    : CORE::time());    }
sub sleep { ($_havetimehires ? Time::HiRes::sleep(@_) : CORE::sleep(@_)); }

#############################################################################
#  THINGS AFTER HERE (until perldocs) ARE PRIVATE METHODS !!!
#############################################################################
sub MAX { my ($a, $b) = @_; return (($a < $b) ? $b : $a); }	# private util
sub init {		# intended to be private
	# try to load Time::HiRes and ProcessTable

	eval("use Time::HiRes");
	if ($@) { $_havetimehires = 0; } else { $_havetimehires = 1; }
	# eval alone can't seem to import sleep() and time() from Time::HiRes.
	# 'use Time::HiRes qw(sleep time);' from here doesn't seem to get 
	# sleep() and time() imported outside this function, either.

	eval("use Proc::ProcessTable");  # we don't use this.... yet.
	if ($@) { $_haveprocprocesstable = 0 } else { $_haveprocprocesstable = 1 }

	eval("use Sys::CpuLoad");  
	if ($@) { $_havesyscpuload = 0 } else { $_havesyscpuload = 1 }

	eval('my $pri=getpriority(0,0); setpriority(0,0,$pri);');  
		# check for setpriority() and setpriority() with a (hopefully) no-op
	if ($@) { $_havesetpriority = 0 } else { $_havesetpriority = 1 }

	$_sleepfactor = .1;
	$_minruntime = 1;	# can be shorter if we have Time::HiRes
	$_minsleeptime = 0;	# no 'minimum' time to sleep by default
	$_maxload = 0;		# 0 means don't watch loads
	$_totalruntime = 0;	
	$_totalsleeptime = 0;
	$_lastloadchecktime = 0;
	$_lastsleeptime = Proc::NiceSleep::time();
}

# Invariant(): attempt to check that the vars are self-consistent.
# returns 1 if OK, 0 if object 'bad'. Not intended to be called often
sub Invariant {	# intended to be private.  Used in tests
	unless (defined($_lastsleeptime)) { init(); }	# autoinit on first use
	# check obvious things:
	# can we load a method/func from each mod we loaded?
	if ($_havetimehires) { 	# this will die if we can't load func
		my $t = Time::HiRes::time(); 
		Time::HiRes::sleep(0.0001);
	}
	if ($_havesyscpuload) { 	# this will die if we can't load func
		my @l = Sys::CpuLoad::load();
	} 
	# if we think we have Time::HiRes, is time() fractional? Inverse?
	# we used to test that we did or didn't get fractional times, but
	# it turns out that just cause you have Time::HiRes doesn't mean you
	# get fractional times and sleeps.
	if ($_havetimehires) { # could still be integer-based 
		#my ($t1, $t2) = (time(), time());	# at least ONE shouldn't be int
		#return 0 if ($t1 == int($t1) && $t2 == int($t2));
	} else {
		# we assume no version of perl has a sub-second time() in CORE (!)
		my ($t1, $t2) = (CORE::time(), CORE::time());	# both should be ints
		return 0 if ($t1 != int($t1) || $t2 != int($t2));
		# but really, even if we fail this test, everything is probably ok
	}
	return 1;	# that's all we test... seems ok!
}

# Autoload methods go after =cut, and are processed by the autosplit program.
# we have none ... yet.
#############################################################################
1;

__END__

# Below is documentation for Proc::NiceSleep 

=head1 NAME

Proc::NiceSleep - yield system in a quasi-intelligent fashion

=head1 SYNOPSIS

  use Proc::NiceSleep qw( sleepfactor minruntime maybesleep nice ); 
  nice(5);                 # lower our priority, if our OS supports it 
  maxload(1.1);            # max load we allow, if Sys::CpuLoad found
  sleepfactor(.5);         # sleep 50% as long as we run
  minruntime(2);           # run at least 2 seconds without sleep
  while($somecondition) {
    dosomething();
    $slept = maybesleep(); # sleep some amount of time if appropriate 
  }

=head1 DESCRIPTION

Proc::NiceSleep is a Perl module which defines subroutines
to allow a process to yield use of the system in a method consistent 
with the configured policy.  
Proc::NiceSleep is intended for use in situations where the 
operating system does not support priorities, or where using the 
operating system's built-in priorities does not yield the system 
sufficiently. 

By default Proc::NiceSleep expects to yield the process for an amount
of time equal to one tenth the amount of time the runs without sleeping. 
This is expressed by the default Sleep Factor of 0.10.
Proc::NiceSleep can also be configured to attempt to keep the 
average system load below a certain threshhold through use of the
maxload() function.

A convenient nice() function, which acts much like the shell 
command and executable of the same name, is also provided 
for easy, platform independent access to your system's 
priorities (if available). 

If Proc::NiceSleep autodetects the presence of the Time::HiRes 
module and your operating system supports it then timing and yielding
operations will occur with sub-second granularity.
If not, no warning or error will be issued but Proc::NiceSleep operations 
will occur with a granularity of about one second.  

The following functions can be imported from this module.

=over 4

=item maybesleep ()

Checks to see if this process should yield use of the system by
issuing some kind of sleep at this point, and if so, does so 
for an appropriate amount of time.  Returns 0 if no sleep was 
performed, otherwise returns the amount of seconds we think
maybesleep() actually slept for.

=item maxload ()

Set or gets the maximum 1-minute average load allowed to occur 
before a sleep call will be issued by maybesleep().  The default value
of 0 disables this feature; setting the maximum load will only have 
an effect if Sys::CpuLoad is successfully loaded.  This module will 
check the system load no more than about once per second.  
If both sleepfactor() and maxload() are used then maybesleep() 
will yield the system if either condition is met. 

=item sleepfactor ()

Sets or gets the sleep factor depending on whether a number is 
passed or not.  A sleep factor of 1 means to sleep an equal amount 
of time as we run, 2 means to sleep twice as long, and so on. The default
value is 0.1. If the sleep factor is set to zero, then this feature is
disabled. If both sleepfactor() and maxload() are used then maybesleep()
will yield the system if either condition is met.

=item nice ()

Sets or gets the priority of the process, as understood by the operating system.
If passed an integer, nice() attempts to set priority of the process to the 
value specified, and returns that value.  If no parameter is passed, 
nice() attempts to query the operating system for the priority of the 
process and return it.  If your OS doesn't support priorities then 
nice() will likely always return 0.  

The exact nice() values returned and recognized, and their meanings 
to the system, are system dependent but usually range from about 
-20 (highest priority) to 20 (lowest priority, 'nicest').  

=item minruntime ()

Sets or gets the minimum run time, in seconds, depending on whether 
a number is passed or not. The minumum run time is the least amount of time 
that Proc::NiceSleep will allow the process to run between sleeps. 
The default value is 1 second.

=item minsleeptime ()

Sets or gets the minimum amount time, in seconds, that maybesleep() will
sleep for if it detects that a sleep is appropriate. Setting the minimum
sleep time to zero (which is also the default value) will disable this feature.

=item DumpText ()

Returns a string (intended for display) containing multiple lines 
with internal information about Proc::NiceSleep's runtime configuration 
and statistics. The format and contents of the returned string are 
intended for informational and debugging use and are subject to change.

=item Dump ()

Returns a reference to a hash with internal information about Proc::NiceSleep
configuration and statistics. The names and presence of the returned hash 
names and values are for informational and debugging purposes only and 
are subject to change. Modifying the returned hash will have no effect on 
the operation of Proc::NiceSleep.

=back

Proc::NiceSleep is loosely modeled on Lincoln Stein's CGI.pm, and 
on D. Wegscheid and other's Time::HiRes.pm.  

=head1 EXPORT

None by default.  

=head1 AUTHOR

Josh Rabinowitz, E<lt>joshr-proc-nicesleep@joshr.comE<gt>

=head1 CAVEATS

sleepfactor() and minruntime() require numeric parameters if present,
but no check is made that the passed scalar is a number.  

Uncoordinated use of sleep() (and possibly of signal() and alarm()) in 
your perl program may cause your program to yield the system more or 
less than specified via Proc::NiceSleep policies.

=head1 SEE ALSO

L<Time::HiRes>, L<Sys::CpuLoad>, L<Proc::Queue>, L<Proc::Swarm>

=head1 COPYRIGHT

  Copyright (c) 2002 Josh Rabinowitz
All rights reserved. This program is free software; you can
redistribute it and/or modify it under the same terms as Perl itself.  

=head1 ACKNOWLEDGEMENTS

Thanks to D. Wegscheid and others for Time::HiRes.pm.  Thanks also
to Michael G Schwern, Terrence Brannon, and David Alban for their valuable 
input.

=cut

