/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <GrFileSink.h>
#include <cstdio>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdexcept>

// should be handled via configure
#ifdef O_LARGEFILE
#define	OUR_O_LARGEFILE	O_LARGEFILE
#else
#define	OUR_O_LARGEFILE 0
#endif

GrFileSink::GrFileSink (size_t itemsize, const char *filename)
  : GrSink (itemsize), d_itemsize (itemsize), d_fp (0)
{
  // we use "open" to get access to the O_LARGEFILE flag.

  int fd;
  if ((fd = open (filename,
		  O_WRONLY | O_CREAT | O_TRUNC | OUR_O_LARGEFILE, 0664)) < 0){
    perror (filename);
    throw std::runtime_error ("can't open file");
  }

  if ((d_fp = fdopen (fd, "wb")) == NULL){
    perror (filename);
    throw std::runtime_error ("can't open file");
  }
}

GrFileSink::~GrFileSink ()
{
  fclose ((FILE *) d_fp);
}

int 
GrFileSink::work3 (VrSampleRange output,
		   VrSampleRange inputs[], void *ai[])
{
  sync (output.index);

  if (getNumberInputs () != 1)
    throw std::runtime_error ("GrFileSink: number of inputs must be 1");
  
  char *inbuf = ((char **) ai)[0];
  unsigned int  nwritten = 0;

  while (nwritten < output.size){
    int count = fwrite (inbuf, d_itemsize, output.size - nwritten, (FILE *) d_fp);
    if (count == 0)
      break;
    nwritten += count;
    inbuf += count * d_itemsize;
  }
  return nwritten;
}
