package Map::Tube::KualaLumpur;

# Pragmas.
use strict;
use warnings;
use 5.006;

# Modules.
use File::Share ':all';
use Moo;
use namespace::clean;

# Version.
our $VERSION = 0.02;

# Get XML.
has xml => (
	'is' => 'ro',
	'default' => sub {
		return dist_file('Map-Tube-KualaLumpur', 'kuala_lumpur-map.xml');
	},
);

with 'Map::Tube';

1;

__END__

=encoding utf8

=head1 NAME

Map::Tube::KualaLumpur - Interface to the KualaLumpur urban monorail system map.

=head1 SYNOPSIS

 use Map::Tube::KualaLumpur;
 my $obj = Map::Tube::KualaLumpur->new;
 my $route = $obj->get_shortest_route($from, $to);
 my $xml_file = $obj->xml;
 my $routes_ar = $obj->get_all_routes($from, $to);
 my $lines_ar = $obj->get_lines;
 my $station = $obj->get_node_by_id($station_id);
 my $station = $obj->get_node_by_name($station_name);
 my $route = $obj->get_shortest_route($from, $to);
 my $stations_ar = $obj->get_stations($line);
 my $metro_name = $obj->name;
 my $xml_file = $obj->xml;

=head1 DESCRIPTION

It currently provides functionality to find the shortest route between the two
given nodes.

For more information about KualaLumpur Map, click L<here|https://https://en.wikipedia.org/wiki/Rapid_Rail> and L<here|https://en.wikipedia.org/wiki/KL_Monorail>.

=head1 METHODS

=over 8

=item C<new()>

 Constructor.

=item C<get_all_routes($from, $to)> [EXPERIMENTAL]

 Get all routes from station to station.
 Returns reference to array with Map::Tube::Route objects.

=item C<get_lines()>

 Get lines in metro map.
 Returns reference to array with Map::Tube::Line objects.

=item C<get_node_by_id($station_id)>

 Get station node by id.
 Returns Map::Tube::Node object.

=item C<get_node_by_name($station_name)>

 Get station node by name.
 Returns Map::Tube::Node object.

=item C<get_line_by_id($line_id)>

 Get line by id.
 Returns Map::Tube::Line object.

=item C<get_line_by_name($line_name)>

 Get line by name.
 Returns Map::Tube::Line object.

=item C<get_shortest_route($from, $to)>

 Get shortest route between $from and $to node names. Node names in $from and $to are case insensitive.
 Returns Map::Tube::Route object.

=item C<get_stations($line)>

 Get list of stations for concrete metro line.
 Returns reference to array with Map::Tube::Node objects.

=item C<name()>

 Get metro name.
 Returns string with metro name.

=item C<xml()>

 Get XML specification of Kuala Lumpur metro.
 Returns string with XML.

=back

=head1 EXAMPLE1

 # Pragmas.
 use strict;
 use warnings;

 # Modules.
 use Encode qw(decode_utf8 encode_utf8);
 use Map::Tube::KualaLumpur;

 # Object.
 my $obj = Map::Tube::KualaLumpur->new;

 # Get route.
 my $route = $obj->get_shortest_route('Kuang', 'Subang Jaya');

 # Print out type.
 print "Route: ".encode_utf8($route)."\n";

 # Output:
 # TODO

=head1 EXAMPLE2

 # Pragmas.
 use strict;
 use warnings;

 # Modules.
 use Map::Tube::KualaLumpur;

 # Object.
 my $obj = Map::Tube::KualaLumpur->new;

 # Get XML file.
 my $xml_file = $obj->xml;

 # Print out XML file.
 print "XML file: $xml_file\n";

 # Output like:
 # XML file: .*/kuala_lumpur-map.xml

=head1 DEPENDENCIES

L<File::Share>,
L<Map::Tube>,
L<Moo>,
L<namespace::clean>.

=head1 SEE ALSO

L<Map::Tube>,
L<Task::Map::Tube>,
L<Task::Map::Tube::Metro>.

=head1 REPOSITORY

L<https://github.com/tupinek/Map-Tube-KualaLumpur>

=head1 AUTHOR

Michal Špaček L<mailto:skim@cpan.org>

L<http://skim.cz>

=head1 LICENSE AND COPYRIGHT

 © 2014-2015 Michal Špaček
 Artistic License
 BSD 2-Clause License

=head1 VERSION

0.02

=cut
