#!/usr/bin/env python

"""
Construct free/busy records for a user, either recording that user's own
availability schedule or the schedule of another user (using details provided
when scheduling events with that user).

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from os.path import abspath, split
import sys

# Find the modules.

try:
    import imiptools
except ImportError:
    parent = abspath(split(split(__file__)[0])[0])
    if split(parent)[1] == "imip-agent":
        sys.path.append(parent)

from codecs import getwriter
from imiptools.config import settings
from imiptools.client import Client
from imiptools.data import get_window_end, Object
from imiptools.dates import get_default_timezone, to_utc_datetime
from imiptools.period import FreeBusyCollection, FreeBusyGroupCollection, \
                             FreeBusyGroupPeriod
from imiptools.stores import get_store, get_publisher, get_journal

def make_freebusy(client, participants, storage, store_and_publish,
    include_needs_action, reset_updated_list, verbose):

    """
    Using the given 'client' representing a user, make free/busy details for the
    records of the user, generating details for 'participants' if not indicated
    as None; otherwise, generating free/busy details concerning the given user.

    The 'storage' is the specific store or journal object used to access data.

    If 'store_and_publish' is set, the stored details will be updated;
    otherwise, the details will be written to standard output.

    If 'include_needs_action' is set, details of objects whose participation
    status is set to "NEEDS-ACTION" for the participant will be included in the
    details.

    If 'reset_updated_list' is set, all objects will be inspected for periods;
    otherwise, only those in the stored free/busy providers file will be
    inspected.

    If 'verbose' is set, messages will be written to standard error.
    """

    user = client.user
    journal = client.get_journal()
    publisher = client.get_publisher()
    preferences = client.get_preferences()

    tzid = preferences.get("TZID") or get_default_timezone()

    # Get the size of the free/busy window.

    try:
        window_size = int(preferences.get("window_size"))
    except (TypeError, ValueError):
        window_size = 100
    window_end = get_window_end(tzid, window_size)

    providers = []

    # Iterate over participants, with None being a special null participant
    # value.

    for participant in participants or [None]:

        # Get identifiers for uncancelled events either from a list of events
        # providing free/busy periods at the end of the given time window, or from
        # a list of all events.

        all_events = not reset_updated_list and storage.get_freebusy_providers(user, window_end)

        if not all_events:
            all_events = storage.get_all_events(user)
            if storage is journal:
                fb = FreeBusyGroupCollection()
            else:
                fb = FreeBusyCollection()

        # With providers of additional periods, append to the existing collection.

        else:
            if participants is None:
                fb = storage.get_freebusy_for_update(user)
            else:
                fb = storage.get_freebusy_for_other_for_update(user, participant)

        # Obtain event objects.

        objs = []
        for uid, recurrenceid in all_events:
            if verbose:
                print >>sys.stderr, uid, recurrenceid
            event = storage.get_event(user, uid, recurrenceid)
            if event:
                objs.append(Object(event))

        # Build a free/busy collection for the given user.

        for obj in objs:
            recurrenceids = not obj.get_recurrenceid() and storage.get_recurrences(user, obj.get_uid())

            # Obtain genuine attendees.

            if storage is journal:
                attendees = storage.get_delegates(user)
            else:
                attendees = [participant]

            # Generate records for each attendee (applicable to consolidated
            # journal data).

            for attendee in attendees:
                partstat = obj.get_participation_status(attendee)

                # Only include objects where the attendee actually participates.

                if obj.get_participation(partstat, include_needs_action):

                    # Add each active period to the collection.

                    for p in obj.get_active_periods(recurrenceids, tzid, window_end):

                        # Obtain a suitable period object.

                        fbp = obj.get_freebusy_period(p, partstat == "ORG")

                        if storage is journal:
                            fbp = FreeBusyGroupPeriod(*fbp.as_tuple(), attendee=attendee)

                        fb.insert_period(fbp)

        # Store and publish the free/busy collection.

        if store_and_publish:

            # Set the user's own free/busy information.

            if participant is None:
                storage.set_freebusy(user, fb)

                if client.is_sharing() and client.is_publishing():
                    publisher.set_freebusy(user, fb)

            # Set free/busy information concerning another user.

            else:
                storage.set_freebusy_for_other(user, fb, participant)

            # Update the list of objects providing periods on future occasions.

            if participant is None or storage is journal:
                providers += [obj for obj in objs if obj.possibly_active_from(window_end, tzid)]

        # Alternatively, just write the collection to standard output.

        else:
            f = getwriter("utf-8")(sys.stdout)
            for item in fb:
                print >>f, "\t".join(item.as_tuple(strings_only=True))

    # Update free/busy providers if storing.

    if store_and_publish:
        storage.set_freebusy_providers(user, to_utc_datetime(window_end, tzid), providers)

# Main program.

if __name__ == "__main__":

    # Interpret the command line arguments.

    participants = []
    args = []
    store_type = []
    store_dir = []
    publishing_dir = []
    journal_dir = []
    preferences_dir = []
    ignored = []

    # Collect user details first, switching to other arguments when encountering
    # switches.

    l = participants

    for arg in sys.argv[1:]:
        if arg in ("-n", "-s", "-v", "-r", "-q"):
            args.append(arg)
            l = ignored
        elif arg == "-T":
            l = store_type
        elif arg == "-S":
            l = store_dir
        elif arg == "-P":
            l = publishing_dir
        elif arg == "-j":
            l = journal_dir
        elif arg == "-p":
            l = preferences_dir
        else:
            l.append(arg)

    try:
        user = participants[0]
    except IndexError:
        print >>sys.stderr, """\
Usage: %s <user> [ <other user> ... ] [ <options> ]

Need a user and optional participants (if different from the user),
along with the -s option if updating the store and the published details.

Specific options:

-q  Access quotas in the journal instead of users in the store
-s  Update the store and published details (write details to standard output
    otherwise)
-n  Include objects with PARTSTAT of NEEDS-ACTION
-r  Inspect all objects, not just those expected to provide details
-v  Show additional messages on standard error

General options:

-j  Indicates the journal directory location
-p  Indicates the preferences directory location
-P  Indicates the publishing directory location
-S  Indicates the store directory location
-T  Indicates the store type (the configured value if omitted)
""" % split(sys.argv[0])[1]
        sys.exit(1)

    # Define any other participant of interest plus options.

    participants = participants[1:]
    using_journal = "-q" in args
    store_and_publish = "-s" in args
    include_needs_action = "-n" in args
    reset_updated_list = "-r" in args
    verbose = "-v" in args

    # Override defaults if indicated.

    getvalue = lambda value, default=None: value and value[0] or default

    store_type = getvalue(store_type, settings["STORE_TYPE"])
    store_dir = getvalue(store_dir)
    publishing_dir = getvalue(publishing_dir)
    journal_dir = getvalue(journal_dir)
    preferences_dir = getvalue(preferences_dir)

    # Obtain store-related objects or delegate this to the Client initialiser.

    store = get_store(store_type, store_dir)
    publisher = get_publisher(publishing_dir)
    journal = get_journal(store_type, journal_dir)

    # Determine which kind of object will be accessed.

    if using_journal:
        storage = journal
    else:
        storage = store

    # Obtain a list of users for processing.

    if user in ("*", "all"):
        users = storage.get_users()
    else:
        users = [user]

    # Obtain a list of participants for processing.

    if participants and participants[0] in ("*", "all"):
        participants = storage.get_freebusy_others(user)

    # Provide a participants list to iterate over even if no specific
    # participant is involved. This updates a user's own records, but only for
    # the general data store.

    elif not participants:
        if not using_journal:
            participants = None
        else:
            print >>sys.stderr, "Participants must be indicated when updating quota records."
            sys.exit(1)

    # Process the given users.

    for user in users:
        if verbose:
            print >>sys.stderr, user
        make_freebusy(
            Client(user, None, store, publisher, journal, preferences_dir),
            participants, storage, store_and_publish, include_needs_action,
            reset_updated_list, verbose)

# vim: tabstop=4 expandtab shiftwidth=4
