/*-------------------------------------------------------------------------
 *
 * plpgpsm.h		- Definitions for the PL/pgPSM
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/plpgpsm.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef PLPGPSM_H
#define PLPGPSM_H

#include "postgres.h"

#include "fmgr.h"
#include "executor/spi.h"
#include "parser/parse_node.h"
#include "nodes/pg_list.h"
#include "lib/stringinfo.h"

/*
 * Abstract syntax tree node types
 *
 */
typedef enum
{
	PLPGPSM_SIMPLE_BLOCK,
	PLPGPSM_STMT_COMPOUND,
	PLPGPSM_STMT_ASSIGN,
	PLPGPSM_STMT_IF,
	PLPGPSM_STMT_CASE,
	PLPGPSM_CONDITIONAL_BLOCK,
	PLPGPSM_STMT_LOOP,
	PLPGPSM_STMT_WHILE,
	PLPGPSM_STMT_REPEAT_UNTIL,
	PLPGPSM_STMT_FOR,
	PLPGPSM_STMT_RETURN,
	PLPGPSM_STMT_PRINT,
	PLPGPSM_STMT_LEAVE,
	PLPGPSM_STMT_ITERATE,
	PLPGPSM_STMT_OPEN,
	PLPGPSM_STMT_CLOSE,
	PLPGPSM_STMT_FETCH,
	PLPGPSM_STMT_SIGNAL,
	PLPGPSM_STMT_RESIGNAL,
	PLPGPSM_STMT_DIAGNOSTICS,
	PLPGPSM_DECLARE_VARIABLE,
	PLPGPSM_DECLARE_PARAM,
	PLPGPSM_DECLARE_CURSOR,
	PLPGPSM_DECLARE_CONDITION,
	PLPGPSM_DECLARE_HANDLER,
	PLPGPSM_CONDITION,
	PLPGPSM_IDENT_VARIABLE,
	PLPGPSM_IDENT_TYPE,
	PLPGPSM_IDENT_CURSOR,
	PLPGPSM_REFERER,
	PLPGPSM_VARIABLE,
	PLPGPSM_ESQL,
	PLPGPSM_CURSOR,
	PLPGPSM_SIGNAL_INFO,
	PLPGPSM_DIAG_INFO
} PLpgPSM_astnode_enum;

/*
 * Result codes
 *
 */
typedef enum
{
	PLPGPSM_RC_OK,
	PLPGPSM_RC_LEAVE,
	PLPGPSM_RC_RETURN,
	PLPGPSM_RC_ITERATE,
	PLPGPSM_RC_UNDO
} PLpgPSM_result_code_enum;

/*
 * AST is based on two-way tree
 *
 */
typedef struct PLpgPSM_astnode
{
	PLpgPSM_astnode_enum		type;
	struct PLpgPSM_astnode		*parent;
	bool				is_visible;
} PLpgPSM_astnode;

/*
 * parser state data
 *
 */
typedef struct
{
	int	nvars;
	int	nargs;
	int	noutargs;
	bool		is_void;
	PLpgPSM_astnode *asttree;
} PLpgPSM_parser;

/*
 * holds function data and AST
 *
 */
typedef struct
{
	PLpgPSM_astnode		*astroot;
	PLpgPSM_parser		*parser;
} PLpgPSM_function;

typedef struct PLpgPSM_execstate *psm_estate;


/*
 * Basic type that holds expression/queries
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	char		*sqlstr;
	bool		is_expression;
	int		location;
	int		lineno;
	List			*var_refs;
	psm_estate curr_estate;

	SPIPlanPtr		plan;
} PLpgPSM_ESQL;

typedef struct
{
	PLpgPSM_astnode		astnode;
	char		*name;
	char		*scope;
	int		location;
} PLpgPSM_ident;

/*
 * Variable declaration
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		location;
	PLpgPSM_ident	*typename;
	PLpgPSM_ESQL	*expr;
	List		*variables;
} PLpgPSM_declare_variable;

typedef enum
{
	PLPGPSM_SPECIAL_NONE,
	PLPGPSM_SPECIAL_SQLCODE,
	PLPGPSM_SPECIAL_SQLSTATE
} PLpgPSM_special_enum;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		location;
	char		*name;
	int		offset;
	PLpgPSM_special_enum	special;
} PLpgPSM_variable;

typedef struct
{
	PLpgPSM_astnode *ref;
	int		nattr;
} PLpgPSM_variable_ref;

typedef struct
{
	PLpgPSM_astnode		astnode;
	PLpgPSM_variable	*variable;
	Oid			typeid;
	int			param_offset;
	char		param_mode;
} PLpgPSM_declare_param;

typedef enum
{
	PLPGPSM_HANDLER_CONTINUE,
	PLPGPSM_HANDLER_EXIT,
	PLPGPSM_HANDLER_UNDO,
} PLpgPSM_handler_type_enum;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		location;
	PLpgPSM_handler_type_enum	handler_type;
	List				*condition_list;
	PLpgPSM_astnode			*stmt;
} PLpgPSM_declare_handler;

typedef enum
{
	PLPGPSM_CURSOR_NOSCROLL,
	PLPGPSM_CURSOR_SCROLL
} PLpgPSM_cursor_option;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	char		*name;
	bool		force_name;
	int		offset;
} PLpgPSM_cursor;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		   location;
	PLpgPSM_cursor	*cursor;
	PLpgPSM_cursor_option	option;
	PLpgPSM_ESQL		*esql;
} PLpgPSM_declare_cursor;


typedef enum
{
	PLPGPSM_CONDITION_SQLSTATE,
	PLPGPSM_CONDITION_NOTFOUND,
	PLPGPSM_CONDITION_SQLWARNING,
	PLPGPSM_CONDITION_SQLEXCEPTION,
	PLPGPSM_CONDITION_NAMED
} PLpgPSM_condition_enum;

/*
 * Condition
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	PLpgPSM_condition_enum		type;
	int		lineno;
	int		location;
	int		sqlstate;
	char		*name;
} PLpgPSM_condition;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		location;
	PLpgPSM_condition	*condition;
} PLpgPSM_declare_condition;


/*
 * Enhanced Block statement
 *
 * This structure holds data of almost all block statements,
 * and when some other data are necessary, then it is used as
 * ancestor.
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int		location;
	char	*label;
	PLpgPSM_ESQL		*expr;			/* boolean expression or SQL query */
	bool			option;
	List		*stmts;
	PLpgPSM_declare_handler		*handler;
	char		*namespace;				/* namespace for FOR statement */
	PLpgPSM_cursor		*cursor;			/* implicit cursor for FOR statement */
} PLpgPSM_eb_stmt;

/*
 * RETURN expr
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_ESQL		*expr;
} PLpgPSM_stmt_return;

/*
 * PRINT expr, expr, ..
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	List		*expr_list;
} PLpgPSM_stmt_print;

/*
 * Join identifier and target variable
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	PLpgPSM_ident		*ident;
	PLpgPSM_astnode		*ref;
} PLpgPSM_referer;

/*
 * SET target = expr
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_ESQL	*expr;
	PLpgPSM_referer		*target;
	List			*target_list;
	int		extra_offset;		/* ptr to fmgr on cast func when it is necessary */
} PLpgPSM_stmt_assign;

/*
 * LEAVE label
 * ITERATE label
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	char		*label;
	PLpgPSM_astnode		*stop_node;
} PLpgPSM_stmt_leave_iterate;

/*
 * OPEN, CLOSE, FETCH
 *
 */
typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_referer		*cursor;
} PLpgPSM_stmt_open;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_referer		*cursor;
} PLpgPSM_stmt_close;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_referer		*cursor;
	List		    *target_list;
	PLpgPSM_declare_handler		*handler;
} PLpgPSM_stmt_fetch;

typedef enum
{
	PLPGPSM_SIGNAL_INFO_MESSAGE,
	PLPGPSM_SIGNAL_INFO_DETAIL,
	PLPGPSM_SIGNAL_INFO_HINT
} PLpgPSM_signal_info_enum;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_signal_info_enum	type;
	PLpgPSM_ESQL	*expr;
} PLpgPSM_signal_info;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_condition	   *condition;
	List			   *signal_info_list;
	PLpgPSM_declare_handler		*handler;
	int		elog_level;
} PLpgPSM_stmt_signal;


typedef enum
{
	PLPGPSM_DIAGINFO_RETURNED_SQLSTATE,
	PLPGPSM_DIAGINFO_RETURNED_SQLCODE,
	PLPGPSM_DIAGINFO_MESSAGE_TEXT,
	PLPGPSM_DIAGINFO_DETAIL_TEXT,
	PLPGPSM_DIAGINFO_HINT_TEXT,
	PLPGPSM_DIAGINFO_ROW_COUNT,
	PLPGPSM_DIAGINFO_CONDITION_IDENTIFIER
} PLpgPSM_diag_info_type;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_referer		*target;
	PLpgPSM_diag_info_type		type;
} PLpgPSM_diag_info;

typedef enum
{
	PLPGPSM_DIAGAREA_STACKED,
	PLPGPSM_DIAGAREA_CURRENT
} PLpgPSM_diagarea_type;

typedef struct
{
	PLpgPSM_astnode		astnode;
	int			lineno;
	int			location;
	PLpgPSM_diagarea_type		diag_area;
	List			*diag_info_list;
} PLpgPSM_stmt_diagnostics;

/*
 * Struct types used during parsing
 *
 */
typedef struct
{
	char	   *ident;			/* palloc'd converted identifier */
	bool		quoted;			/* Was it double-quoted? */
} PLword;

typedef struct
{
	List	   *idents;			/* composite identifiers (list of String) */
} PLcword;

/*
 * Structures used during execution
 *
 */
typedef struct
{
	char		*typename;
	Oid		typeid;
	int32		typmod;
	bool		typbyval;
	bool		typlen;
	Oid		typioparam;
	FmgrInfo	typinput;
	FmgrInfo	typoutput;
} PLpgPSM_typeinfo;


typedef enum
{
	PLPGPSM_DATUM_SCALAR,
	PLPGSPM_VARIABLE_RECORD,
	PLPGPSM_VARIABLE_ROW,
	PLPGPSM_DATUM_SQLCODE,
	PLPGPSM_DATUM_SQLSTATE,
	PLPGPSM_DATUM_CURSOR
} PLpgPSM_class_enum;

typedef struct
{
	SPITupleTable *tuptab;
	int		n;
	int	processed;
	bool		prefetch;
	int				nattrs;
	Datum			*values;
	bool			*nulls;
	HeapTuple	cursor_tuple;
	TupleDesc	cursor_tupdesc;
	bool		is_ready;
} PLpgPSM_fetch_extra;

typedef struct
{
	PLpgPSM_class_enum		class;
	PLpgPSM_typeinfo		*typeinfo;
	PLpgPSM_astnode			*variable;
	bool			   isnull;
	Datum			   value;
	void		*extra;
} PLpgPSM_datum;

typedef struct PLpgPSM_stacked_ErrorData
{
	ErrorData	*edata;
	struct PLpgPSM_stacked_ErrorData *next;
} PLpgPSM_stacked_ErrorData;

typedef struct
{
	PLpgPSM_function	*func;
	FunctionCallInfo	   fcinfo;
	PLpgPSM_typeinfo	*rtinfo;
	Datum			   retval;
	bool			   retisnull;
	TupleDesc		   rettupdesc;
	PLpgPSM_astnode		*curr_stmt;
	PLpgPSM_ESQL		*curr_esql;
	PLpgPSM_astnode			*stop_node;
	PLpgPSM_declare_handler			*handler;
	List				*used_tinfos;
	SPITupleTable		   *eval_tuptable;
	uint32			   eval_processed;
	int		   ndatums;
	PLpgPSM_datum	   *datums;
	int		nextras;
	void		**extras;
	MemoryContext		exec_cxt;
	int				sqlcode;
	char				sqlstate_buffer[10];
	ExprContext *eval_econtext;		/* for execution simple expressions */
	ErrorData		*edata;		/* current edata for current exception or warning */
	PLpgPSM_stacked_ErrorData *stacked_edata; /* stacked edata */
} PLpgPSM_execstate;


/*
 * Functions in handlers.c
 *
 */
extern void _PG_init(void);
extern Datum plpgpsm_call_handler(PG_FUNCTION_ARGS);
extern Datum plpgpsm_validator(PG_FUNCTION_ARGS);

/*
 * Functions in utils.c
 *
 */
extern bool plpgpsm_allow_cast_to_eb(PLpgPSM_astnode *astnode);
extern char *plpgpsm_astnode_tag(PLpgPSM_astnode *astnode);
extern void plpgpsm_elog_astnode(int level, PLpgPSM_astnode *astnode);
extern PLpgPSM_ESQL *plpgpsm_inner_esql(PLpgPSM_astnode *astnode);

/*
 * Scanner functions in scanner.c
 *
 */
extern int	plpgpsm_base_yylex(void);
extern int	plpgpsm_yylex(void);
extern void plpgpsm_push_back_token(int token);
extern void plpgpsm_append_source_text(StringInfo buf,
						   int startlocation, int endlocation);
extern int	plpgpsm_scanner_errposition(int location);
extern void plpgpsm_yyerror(const char *message);
extern int	plpgpsm_location_to_lineno(int location);
extern int	plpgpsm_latest_lineno(void);
extern void plpgpsm_scanner_init(const char *str);
extern void plpgpsm_scanner_finish(void);
extern void plpgpsm_peek2(int *tok1_p, int *tok2_p, int *tok1_loc, int *tok2_loc);

/*
 * Externs in gram.y
 *
 */
extern int	plpgpsm_yyparse(void);

/*
 * parser.c
 *
 */
extern PLpgPSM_parser parser;
extern PLpgPSM_function *plpgpsm_compile(FunctionCallInfo fcinfo, bool forValidator);
extern PLpgPSM_astnode *plpgpsm_search_variable(PLpgPSM_execstate *estate, PLpgPSM_ident *ident,
								    PLpgPSM_astnode *scope,
								    int *fnumber);
extern PLpgPSM_eb_stmt *plpgpsm_eb_stmt(PLpgPSM_astnode_enum type, PLpgPSM_astnode *parent,
								    int location, char *label,
								    PLpgPSM_ESQL *expr,
								    List	*stmts);
extern void plpgpsm_parser_setup(struct ParseState *pstate, PLpgPSM_ESQL *esql);
extern PLpgPSM_variable *plpgpsm_new_variable(char *name, int location);
extern PLpgPSM_astnode *plpgpsm_get_near_compound_stmt(PLpgPSM_astnode *astnode);
extern PLpgPSM_declare_handler *plpgpsm_search_handler(PLpgPSM_astnode *scope, PLpgPSM_condition *condition);


/*
 * Externs in interpreter.c
 *
 */
extern Datum plpgpsm_exec_function(PLpgPSM_function *func, FunctionCallInfo fcinfo);


/*
 * runtime
 *
 */
PLpgPSM_execstate *plpgpsm_init_execstate(PLpgPSM_execstate *estate, PLpgPSM_function *func,
								FunctionCallInfo fcinfo);
void plpgpsm_clean_execstate(PLpgPSM_execstate *estate);
void plpgpsm_eval_cleanup(PLpgPSM_execstate *estate);
void plpgpsm_set_datum(PLpgPSM_execstate *estate, PLpgPSM_datum *datum,
								Datum value,
								bool isnull,
								bool copy_datum);
void plpgpsm_store_result(PLpgPSM_execstate *estate, int attrn, PLpgPSM_datum *datum);
PLpgPSM_typeinfo *plpgpsm_make_typeinfo(PLpgPSM_execstate *estate, PLpgPSM_ident *ident);
PLpgPSM_typeinfo *plpgpsm_make_typeinfo_typeid(PLpgPSM_execstate *estate, Oid typeid, int32 typmod);
void plpgpsm_eval_expr(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql, PLpgPSM_typeinfo *typeinfo,
										bool recheck_single_value);
char *plpgpsm_eval_cstring(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql);
bool plpgpsm_eval_boolean(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql);
Datum plpgpsm_eval_datum(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql,
								PLpgPSM_typeinfo *tinfo,
								bool *isnull);
extern void plpgpsm_store_composed_result_to_datums(PLpgPSM_execstate *estate, List *datums);
extern void plpgpsm_store_result_to_datums(PLpgPSM_execstate *estate, List *datums);

extern Portal plpgpsm_open_cursor(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql, char *name, int cursor_options);
extern void plpgpsm_close_cursor(PLpgPSM_execstate *estate, Portal portal);
extern void plpgpsm_cursor_fetch(PLpgPSM_execstate *estate, Portal portal, int prefetch);
extern void plpgpsm_tuple_to_targets(PLpgPSM_execstate *estate, List *datums, HeapTuple tup, TupleDesc tupdesc);

extern Datum plpgpsm_cast(PLpgPSM_execstate *estate, Datum value, bool isnull,
								Oid	typeid, int32 typmod,
									    PLpgPSM_typeinfo *tinfo);

#endif		/* PLPGPSM_H */