/******************************************************
 *                                                    *
 * FPAC project.            FPAC PAD                  *
 *                                                    *
 * Parts of code from different sources of ax25-utils *
 *                                                    *
 * F6FBB 05-1997                                      *
 *                                                    *
 ******************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <ctype.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <arpa/inet.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "fpac.h"

enum conf_kw
{
	CALLSIGN,
	COMMAND,
	PATH,
	CITY,
	LOCATOR,
	ALTERNATE,
	DNIC,
	ADDRESS,
	COVERAGE,
	NODE,
	ROUTES,
	PORT,
	LUSER,
	USERPORT,
	DEFPORT,
	ADDPORT,
	ALIAS,
	PASSWORD,
	SYSOP,
	APPLICATION,
	INETPORT,
	OPTION,
	NOWP,
	END
};

typedef struct
{
	char keyword[20];
	int  ident;
} kw_t;

kw_t kw[] = 
{
	{ "l3call", CALLSIGN },
	{ "command", COMMAND },
	{ "path", PATH },
	{ "city", CITY },
	{ "locator", LOCATOR },
	{ "l2call", ALTERNATE },
	{ "dnic", DNIC },
	{ "address", ADDRESS },
	{ "coverage", COVERAGE },
	{ "node", NODE },
	{ "routes", ROUTES },
	{ "port", PORT },
	{ "user", LUSER },
	{ "userport", USERPORT },
	{ "defport", DEFPORT },
	{ "addport", ADDPORT },
	{ "alias", ALIAS },
	{ "password", PASSWORD },
	{ "sysop", SYSOP },
	{ "application", APPLICATION },
	{ "inetport", INETPORT },
	{ "option", OPTION },
	{ "nowp", NOWP },
	{ "end", END },
	{ "", -1}	/* End of the table */
};

static dniclu_t *dnic_list = NULL;
static int nb_dnic = 0;
#define DNICBLOCKSIZE	100

static long des2long(char *str)
{
	long val = 0L;
	int c;
	int nb = 0;
	
	while (*str)
	{
		if (nb == 3)
			return (0L);
		c = (islower(*str)) ? toupper(*str) : *str;
		val += (c << (nb * 8));
		++str;
		++nb;
	}
	return val;
}

static char *long2des(long val)
{
	static char des[4];
	
	des[0] = val & 0xff;
	des[1] = (val >> 8) & 0xff;
	des[2] = (val >> 16) & 0xff;
	des[3] = '\0';
	
	return des;
}

char *des2dnic(char *des)
{
	int i;
	long country;
	static char dnic[5];
	
	country = des2long(des);
	if (country == 0)
		return NULL;
		
	for (i = 0 ; i < nb_dnic ; i++)
	{
		if (dnic_list[i].country == country)
		{
			sprintf(dnic, "%d", dnic_list[i].dnic);
			return dnic;
		}
	}
	return NULL;
}
	
char *dnic2des(char *dnic)
{
	int d;
	int i;
	char ldnic[5];
	
	/* Get only the first 4 digits */
	memcpy(ldnic, dnic, 4);
	ldnic[4] = '\0';
	
	d = atoi(ldnic);
	if (d < 1000 || d > 9999)
		return NULL;
		
	for (i = 0 ; i < nb_dnic ; i++)
	{
		if (dnic_list[i].dnic == d)
		{
			return long2des(dnic_list[i].country);
		}
	}
	return NULL;
}
	
static void dnic_open(void)
{
	int max = 0;
	FILE *fptr;
	char line[256];
	char designator[256];
	int dnic;
	
	if ((fptr = fopen(FPACDNIC, "r")) == NULL)
		return;
		
	while (fgets(line, sizeof(line), fptr))
	{
		if (nb_dnic == max)
		{
			max += DNICBLOCKSIZE;
			if (dnic_list)
				dnic_list = realloc(dnic_list, max);
			else
				dnic_list = malloc(sizeof(dniclu_t) * max);
		}
		
		if ((sscanf(line, "%s %d", designator, &dnic) >= 2) && dnic > 0 && dnic < 10000)
		{
			if (*designator == '#')
				continue;
			
			dnic_list[nb_dnic].country = des2long(designator);
			dnic_list[nb_dnic].dnic = dnic;
			++nb_dnic;
		}
	}
	
	fclose(fptr);
}

/* Return a keyword identifier or -1 if not found */
static int kwid(char *keyword)
{
	int i;

	for (i = 0 ; kw[i].keyword[0] ; i++)
	{
		if (strcasecmp(keyword, kw[i].keyword) ==0)
			break;
	}
	return(kw[i].ident);
}

/* Cleans the beginning and the end of a string */
static char *strclean(char *str)
{
	int len;

	while (isspace(*str))
		++str;

	len = strlen(str);

	while ((len > 0) && (!isgraph(str[len-1])))
	{
		str[len-1] = '\0';
		--len;
	}

	return(str);
}

/* Returns the keyword and the value */
static int get_value(char *line, char *keyword, char *value)
{
	char *ptr;

	*keyword = *value = '\0';

	line = strclean(line);
	if ((*line == '\0') || (*line == '#'))
		return(0);

	ptr = strchr(line, '=');
	if (ptr)
	{
		*ptr++ = '\0';
		strcpy(value, strclean(ptr));
	}

	strcpy(keyword, strclean(line));

	return(1);	
}

static void strcpy_n(char *rec, char *src, int len)
{
	memset(rec, '\0', len);
	strncpy(rec, src, len-1);
}

/* Add a new application */
static void add_application(cfg_t *cfg, char *call, char *appli)
{
	appli_t *a;

	if (!strchr(call, '-'))
		strcat(call, "-0");

	a = calloc(1, sizeof(appli_t));
	strcpy_n(a->call, call, sizeof(a->call));
	strcpy_n(a->appli, appli, sizeof(a->appli));
	a->next = cfg->appli;
	cfg->appli = a;
}

int cfg_open(cfg_t *cfg)
{
	int mode = 0;
	int errnum;
	FILE *fptr;
	char *ptr;
	char *stmp;
	char line[256];
	char ax25_port[256];
	char keyword[80];
	char value[256];
	char full_addr[20];
	char dnic_cur[5];
	struct stat st;

	addrp_t	*d = NULL;
	alias_t *a = NULL;
	cmd_t	*c = NULL;
	luser_t	*l = NULL;
	node_t	*n = NULL;
	route_t *r = NULL;
	port_t	*p = NULL;
	cover_t	*o = NULL;

	memset(cfg, 0, sizeof(cfg_t));

	dnic_open();

	if (stat(FPACCONF, &st) == 0)
	{
		cfg->date = st.st_mtime;
	}

	fptr = fopen(FPACCONF, "r");
	if (fptr == NULL)
	{
		syslog(LOG_ERR, "Cannot open file %s\n", FPACCONF);
		return(1);
	}

	/* Application FPACNODE always exist */
	add_application(cfg, "NODE-*", FPACNODE);

	while (fgets(line, sizeof(line), fptr))
	{

		if (!get_value(line, keyword, value))
			continue;

		switch (mode)
		{
		case COMMAND :
			switch(kwid(keyword))
			{
			case END:
				mode = 0;
				break;
			default:
				/* New command */
				c = calloc(1, sizeof(cmd_t));
				if (c == NULL)
					return(1);
				strcpy_n(c->name, keyword, sizeof(c->name));
				c->cmd = malloc(strlen(value)+1);
				if (c->cmd == NULL)
					return(1);
				strcpy(c->cmd, value);
				c->next = cfg->cmd;
				cfg->cmd = c;
				break;
			}
			break;
		case SYSOP :
			switch(kwid(keyword))
			{
			case END:
				mode = 0;
				break;
			default:
				/* New command */
				c = calloc(1, sizeof(cmd_t));
				if (c == NULL)
					return(1);
				strcpy_n(c->name, keyword, sizeof(c->name));
				c->cmd = malloc(strlen(value)+1);
				if (c->cmd == NULL)
					return(1);
				strcpy(c->cmd, value);
				c->next = cfg->syscmd;
				cfg->syscmd = c;
				break;
			}
			break;
		case NODE:
			switch(kwid(keyword))
			{
			case PATH:
				strcpy_n(n->call, value, sizeof(n->call));
				break;
			case DNIC:
				strcpy_n(n->dnic, value, sizeof(n->dnic));
				break;
			case ADDRESS:
				strcpy_n(n->addr, value, sizeof(n->addr));
				break;
			case PORT:
				strcpy_n(n->port, value, sizeof(n->port));
				break;
			case NOWP:
				n->nowp = (*value != '0');
				break;
			case END:
				n->next = cfg->node;
				cfg->node = n;
				mode = 0;
				break;
			}
			break;
		case ADDPORT:
			switch(kwid(keyword))
			{
			case ADDRESS:
				strcpy_n(d->addr, value, sizeof(d->addr));
				break;
			case PORT:
				strcpy_n(d->port, value, sizeof(d->port));
				break;
			case END:
				d->next = cfg->addrp;
				cfg->addrp = d;
				mode = 0;
				break;
			}
			break;
		case APPLICATION:
			switch(kwid(keyword))
			{
			case END:
				mode = 0;
				break;
			default:
				add_application(cfg, keyword, value);
				break;
			}
			break;
		case LUSER:
			switch(kwid(keyword))
			{
			case PATH:
				strcpy_n(l->call, value, sizeof(l->call));
				break;
			case PORT:
				strcpy_n(l->port, value, sizeof(l->port));
				break;
			case END:
				l->next = cfg->luser;
				cfg->luser = l;
				mode = 0;
				break;
			}
			break;
		case ROUTES:
			switch(kwid(keyword))
			{
			case DNIC:
				strcpy_n(dnic_cur, value, sizeof(dnic_cur));
				break;
			case END:
				mode = 0;
				break;
			default :
				r = calloc(1, sizeof(route_t));
				full_addr[0] = '\0';
				if (strcmp(dnic_cur, "0") != 0)
					strcpy(full_addr, dnic_cur);
				keyword[10] = '\0';
				strcat(full_addr, keyword);
				strcpy_n(r->addr, full_addr, sizeof(r->addr));
				strcpy_n(r->nodes, value, sizeof(r->nodes));
				r->next = cfg->route;
				cfg->route = r;
				break;
			}
			break;
		case ALIAS:
			switch(kwid(keyword))
			{
			case PATH:
				ptr = value;
				while (*ptr)
				{
					if (*ptr == ',')
						*ptr = ' ';
					++ptr;
				}
				strcpy_n(a->path, value, sizeof(a->path));
				break;
			case END:
				a->next = cfg->alias;
				cfg->alias = a;
				mode = 0;
				break;
			}
			break;
		default:
			switch(kwid(keyword))
			{
			case OPTION:
				strcpy_n(cfg->option, value, sizeof(cfg->option));
				break;
			case PASSWORD:
				strcpy_n(cfg->password, value, sizeof(cfg->password));
				break;
			case CALLSIGN:
				strcpy_n(cfg->callsign, value, sizeof(cfg->callsign));
				break;
			case ALTERNATE:
				strcpy_n(cfg->alt_callsign, value, sizeof(cfg->alt_callsign));
				/* Application alias callsign always exist */
				add_application(cfg, value, FPACNODE);
				break;
			case CITY:
				strcpy_n(cfg->city, value, sizeof(cfg->city));
				break;
			case LOCATOR:
				strcpy_n(cfg->locator, value, sizeof(cfg->locator));
				break;
			case DNIC:
				strcpy_n(cfg->dnic, value, sizeof(cfg->dnic));
				break;
			case ADDRESS:
				strcpy_n(cfg->address, value, sizeof(cfg->address));
				break;
			case COVERAGE:
				ptr = strtok(value, " ,\t");
				while (ptr)
				{
					o = calloc(1, sizeof(cover_t));
					strcpy_n(o->addr, ptr, sizeof(o->addr));
					o->next = cfg->cover;
					cfg->cover = o;
					ptr = strtok(NULL, " ,\t");
				}
				break;
			case DEFPORT:
				strcpy_n(cfg->def_port, value, sizeof(cfg->def_port));
				break;
			case USERPORT:
				strcpy_n(ax25_port, value, sizeof(ax25_port));
				break;
			case INETPORT:
				cfg->inetport = atoi(value);
				break;
			case NODE:
				n = calloc(1, sizeof(node_t));
				strcpy_n(n->name, value, sizeof(n->name));
				mode = NODE;
				break;
			case LUSER:
				l = calloc(1, sizeof(luser_t));
				strcpy_n(l->name, value, sizeof(l->name));
				mode = LUSER;
				break;
			case ADDPORT:
				d = calloc(1, sizeof(addrp_t));
				d->fd = -1;
				strcpy_n(d->name, value, sizeof(d->name));
				mode = ADDPORT;
				break;
			case ALIAS:
				a = calloc(1, sizeof(alias_t));
				a->fd = -1;
				strcpy_n(a->alias, value, 9);
				mode = ALIAS;
				break;
			case COMMAND:
				mode = COMMAND;
				break;
			case APPLICATION:
				mode = APPLICATION;
				break;
			case SYSOP:
				mode = SYSOP;
				break;
			case ROUTES:
				strcpy(dnic_cur, cfg->dnic);
				mode = ROUTES;
				break;
			}
			break;
		}

	}

	/* Check for mandatory fields */
	if (*cfg->callsign == '\0')			errnum = CALLSIGN;
	else if (*cfg->alt_callsign == '\0')errnum = ALTERNATE;
	else if (*cfg->city == '\0')		errnum = CITY;
	else if (*cfg->locator == '\0')		errnum = LOCATOR;
	else if (*cfg->dnic == '\0')		errnum = DNIC;
	else if (*cfg->address == '\0')		errnum = ADDRESS;
	else errnum = -1;
	
	if (errnum != -1) {
		fprintf(stderr, "field \"%s\" not initalized in %s\n", kw[errnum].keyword, FPACCONF);
		syslog(LOG_ERR, "field \"%s\" not initalized in %s\n", kw[errnum].keyword, FPACCONF);
		return(2);
	}

	/* Build local full address */
	strcpy(cfg->fulladdr, cfg->dnic);
	strcat(cfg->fulladdr, cfg->address);

	sprintf(line, "FPAC_L2CALL=%s",  cfg->alt_callsign);
	putenv(line);
	sprintf(line, "FPAC_L3CALL=%s",  cfg->callsign);
	putenv(line);
	sprintf(line, "FPAC_ADDRESS=%s", cfg->fulladdr);
	putenv(line);
	sprintf(line, "FPAC_CITY=%s",    cfg->city);
	putenv(line);
	sprintf(line, "FPAC_LOCATOR=%s", cfg->locator);
	putenv(line);

	/* Create port list */

	stmp = strdup(ax25_port);
	ptr = strtok(stmp, " ,\t");
	while (ptr)
	{
		p = malloc(sizeof(port_t));
		p->fd_call = -1;
		p->fd_digi = -1;
		p->fd_appli = -1;
		p->alias = NULL;
		strcpy(p->name, ptr);
		for (a = cfg->alias ; (a != NULL) ; a = a->next)
		{
			alias_t *alias;

			alias = malloc(sizeof(alias_t));
			memcpy(alias, a, sizeof(alias_t));
			alias->next = p->alias;
			p->alias = alias;
		}
		p->next = cfg->port;
		cfg->port = p;
		ptr = strtok(NULL, " ,\t");
	}
	free(stmp);

	return(0);
}
