
/*
 *	linker.c
 *
 *	connection script system for creating outgoing links
 */
 
#include <errno.h>
#include <string.h>

#include "linker.h"
#include "csock.h"
#include "hmalloc.h"
#include "log.h"
#include "cfgfile.h"

struct linkjob_t *jobs = NULL;

/*
 *	Allocate a job structure
 */
 
struct linkjob_t *lj_alloc(void)
{
	struct linkjob_t *lj = jobs;
	struct linkjob_t **prevp = &jobs;
	
	while (lj) {
		prevp = &lj->next;
		lj = lj->next;
	}
	
	*prevp = lj = hmalloc(sizeof(struct linkjob_t));
	lj->next = NULL;
	lj->prevp = prevp;
	
	lj->f = NULL;
	lj->s = NULL;
	lj->rxfailure = 0;
	
	return lj;
}

/*
 *	Free a job structure
 */
 
void lj_free(struct linkjob_t *lj)
{
	*lj->prevp = lj->next;
	if (lj->next)
		lj->next->prevp = lj->prevp;
	
	if (lj->f)
		fclose(lj->f);
	if (lj->script)
		hfree(lj->script);
		
	hfree(lj);
}

/*
 *	Find a job structure for the socket
 */
 
struct linkjob_t *lj_find(struct csock_t *s)
{
	struct linkjob_t *lj = jobs;
	
	while ((lj) && (lj->s != s))
		lj = lj->next;
		
	return lj;
}

/*
 *	Read and parse a line from the script
 */
 
int read_scriptline(struct linkjob_t *lj)
{
	while (fgets(lj->line, CFGLINE_LEN, lj->f)) {
		lj->linec++;
		if ((lj->argc = parse_args(lj->argv, lj->line)) == 0 || *lj->argv[0] == '#')
			continue;
		hstrlwr(lj->argv[0]);
		return 0;
	}
	
	return -1;
}

/*
 *	Close a job
 */

void end_connect(struct linkjob_t *lj, int res)
{
	if ((res) && (lj->endfunc))
		(*lj->endfunc)(lj->ptr, res);
	
	lj_free(lj);
}

/*
 *	Outside has decided that the job is over (or aborted)
 */

void done_connect(struct csock_t *s)
{
	struct linkjob_t *lj;
	
	if (!(lj = lj_find(s)))
		return;
		
	end_connect(lj, 0);
}

/*
 *	Send a S line
 */
 
void linker_send(struct linkjob_t *lj)
{
	csprintf(lj->s, "%s%s", lj->argv[1], lj->s->eol);
}

/*
 *	Read and parse a line of script
 */

void linker_advance(struct linkjob_t *lj)
{
	if (read_scriptline(lj)) {
		log(L_LINK, "Linker: Script %s succeeded.",
			lj->script); 
		end_connect(lj, LJ_SUCCESS);
		return;
	}
	
	if (!strcmp(lj->argv[0], "s")) {
		if (lj->argc != 2) {
			log(L_ERR, "Linker: Script %s:%d: \"%s\" wants one parameter",
				lj->script, lj->linec, lj->argv[0]);
			end_connect(lj, LJ_ERROR);
		}
		linker_send(lj);
		linker_advance(lj);
	} else if (!strcmp(lj->argv[0], "r")) {
		if (lj->argc < 2) {
			log(L_ERR, "Linker: Script %s:%d: \"%s\" wants at least one parameter",
				lj->script, lj->linec, lj->argv[0]);
			end_connect(lj, LJ_ERROR);
		}
		return;
	} else {
		log(L_ERR, "Linker: Script %s:%d: Unknown command \"%s\"",
			lj->script, lj->linec, lj->argv[0]);
		end_connect(lj, LJ_ERROR);
		return;
	}
	
	return;
}

/*
 *	Receive data from a linking socket
 */

int linker_handler(struct csock_t *s, void *data, int len)
{
	struct linkjob_t *lj;
	char *p = (char *)data;
	int i;
	
	if (!(lj = lj_find(s))) {
		log(L_ERR, "Linker: linker_handler(): no job for the stream!");
		return 0;
	}
	
	//log(L_DEBUG, "linker_handler(): \"%s\"", p);
	
	if (strstr(p, lj->argv[1])) {
		linker_advance(lj);
		return 0;
	}
	
	for (i = 2; i < lj->argc; i++) {
		if (strstr(p, lj->argv[i])) {
			log(L_LINK, "Linker: Script %s failed: Received \"%s\"", lj->script, lj->argv[i]);
			lj->rxfailure = 1;
			sock_disconnect(lj->s);
			return -2;
		}
	}
	
	return 1;
}

/*
 *	The initial connect() has succeeded
 */
 
void linker_conn_handler(struct csock_t *s)
{
	struct linkjob_t *lj;
	
	if (!(lj = lj_find(s))) {
		log(L_ERR, "linker_conn_handler(): no job for the stream!");
		return;
	}
	
	linker_advance(lj);
	
	return;
}

/*
 *	We were disconnected
 */
 
void linker_disc_handler(struct csock_t *s)
{
	struct linkjob_t *lj;
	
	if (!(lj = lj_find(s))) {
		log(L_ERR, "linker_disc_handler(): no job for the stream!");
		return;
	}
	
	if (!lj->rxfailure)
		log(L_LINK, "Linker: Script %s failed: Disconnected (%s)", lj->script, strerror(s->cs_errno));
		
	end_connect(lj, LJ_FAILURE);
	return;
}

/*
 *	Create a new connection job
 */

struct csock_t *start_connect(struct stringlist_t *scripts,
			void (*endfunc)(void *ptr, int res), void *ptr,
			int timeout)
{
	struct linkjob_t *lj;
	int af_type;
	int compressed = 0;
	
	lj = lj_alloc();
	lj->linec = 0;
	lj->script = hstrdup(scripts->s);
	lj->endfunc = endfunc;
	lj->ptr = ptr;
	
	if (!(lj->f = fopen(lj->script, "r"))) {
		log(L_ERR, "Linker: Could not open script %s for reading: %s",
			lj->script, strerror(errno));
		lj_free(lj);
		return 0;
	}
	
	if (read_scriptline(lj)) {
		log(L_ERR, "Linker: Empty script file %s, no connection command found.",
			lj->script); 
		lj_free(lj);
		return 0;
	}
	
	if (!strcasecmp(lj->argv[0], "compressed")) {
		compressed = 1;
		if (read_scriptline(lj)) {
			log(L_ERR, "Linker: No connection command after \"compressed\" in %s",
				lj->script);
			lj_free(lj);
			return 0;
		}
	}
	
	if ((af_type = strtoaf(lj->argv[0])) == -1) {
		log(L_ERR, "Linker: Unknown AF type %s, confusing.", lj->argv[0]);
		lj_free(lj);
		return 0;
	}
	
	lj->s = sock_connect(af_type, compressed, lj->argc-1, &lj->argv[1]);
	
	if (!lj->s) {
		log(L_ERR, "Linker: sock_connect() failed");
		lj_free(lj);
		return 0;
	}
	
	lj->s->conn_handler = &linker_conn_handler;
	lj->s->disc_handler = &linker_disc_handler;
	lj->s->in_handler = &linker_handler;
	
	lj->s->eolmode = eol_raw;
	
	return lj->s;
}

