

/*
 *	luser.c
 *
 *	local user list handling functions
 */
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "config.h"
#include "csock.h"
#include "luser.h"
#include "cluster.h"
#include "command.h"
#include "log.h"
#include "hmalloc.h"
#include "net_user.h"
#include "f_luser.h"
#include "timer.h"
#include "ctime.h"

struct luser_t *lusers = NULL;

int luser_count;
int luser_max;

/*
 *	Allocate and initialize a local user structure
 */

struct luser_t *luser_alloc(struct csock_t *s)
{
	struct luser_t *lu = lusers, **plu = &lusers;
	
	while (lu) {
		plu = &lu->next;
		lu = lu->next;
	}
	
	lu = hmalloc(sizeof(struct luser_t));
	*plu = lu;
	
	lu->prevp = plu;
	lu->next = NULL;
	
	lu->sock = s;
	lu->f = NULL;
	lu->nu = NULL;
	lu->fl = NULL;
	
	lu->egid = 10;
	
	lu->m2 = M2_NONE;
	lu->m3 = M3_NONE;
	
	lu->str = NULL;
	lu->path[0] = '\0';
	lu->last_talk_from[0] = '\0';
	
	lu->illegal_cmds = 0;
	lu->su_tries = 0;
	lu->locked = 0;
	
	s->lu = lu;
	s->in_handler = luser_handler;
	s->disc_handler = luser_logout;
	
	luser_count++;
	if (luser_count > luser_max)
		luser_max = luser_count;
		
	return lu;
}

/*
 *	Free a local user structure
 */

void luser_free(struct luser_t *lu)
{
	*lu->prevp = lu->next;
	if (lu->next)
		lu->next->prevp = lu->prevp;
	
	hfree(lu->f);
	if (lu->fl)
		hfree(lu->fl);
	if (lu->str)
		hfree(lu->str);
	lu->sock->lu = NULL;
	hfree(lu);
	luser_count--;
}

/*
 *	Find a local user
 */

struct luser_t *get_luser(call_t *call)
{
	struct luser_t *p = lusers;

	while ((p) && (strcmp(p->nu->call, (char *)call) != 0))
		p = p->next;
	return p;
}

/*
 *	Send a message to lusers who are interested in it
 */

int send_us(int mtype, char *fmt, ...)
{
	struct luser_t *p;
	int i = 0;
	va_list args;
	char str[MAX_SENDLEN+1];
	char str_t[MAX_SENDLEN+1];
	char *cp, *ct;
	
	cp = fmt;
	ct = str;
	while (*cp) {
		if (*cp == '$') {
			cp++;
			switch (*cp) {
			case 't':
				*ct++ = ' ';
				strcpy(ct, timestr_s(now));
				ct += 4;
				*ct++ = 'Z';
				break;
			case '\0':
				cp--;
				break;
			default:
				*ct++ = '$';
			}
		} else {
			*ct++ = *cp;
		}
		cp++;
	}
	*ct++ = '\0';
	
	va_start(args, fmt);
	vsnprintf(str_t, MAX_SENDLEN, str, args);
	va_end(args);
	
	va_start(args, fmt);
	vsnprintf(str, MAX_SENDLEN, fmt, args);
	va_end(args);
	
	str[MAX_SENDLEN] = '\0';
	
	for (p = lusers; (p); p = p->next) {
		if ((p->f->messages & mtype) == mtype) {
		  	i++;
		  	if (p->f->flags & F_TIMESTAMP)
		  		csputs(p->sock, str_t);
		  	else
				csputs(p->sock, str);
			if (((p->f->flags & F_BEEPS) == F_BEEPS) /* Licence */
			  && ((p->f->beeps & mtype) == mtype)) /* to kill? */
			  	csputc(p->sock, '\007');
			csputc(p->sock, '\n');
		}
	}
	
	return i;
}

/*
 *	Create a new luserf structure for a new user
 */
 
struct luserf_t *new_luserf(call_t call)
{
	struct luserf_t *f;
	
	f = hmalloc(sizeof(struct luserf_t));
	memset(f, '\0', sizeof(struct luserf_t));
	strcpy(f->call, call);
	truncssid(&f->call);
	
	f->name[0] = '\0';
	f->qth[0] = '\0';
	f->personal[0] = '\0';
	f->locator[0] = '\0';
	f->loc.latitude = 0;
	f->loc.longitude = 0;
	f->group = 0;
	f->messages = M_DX|M_ANN|M_WWV|M_TALK|M_LUSER|M_LINK;
	f->beeps = 0;
	f->flags = F_TIMESTAMP|F_FILTER;
	f->login_act = 0;
	f->charset = 0;
	f->prompt = 0;
	f->page_length = 24;
	f->page_width = 78;
	f->language = 0;
	f->logins = 0;
	f->time = 0;
	
	return f;
}

/*
 *	Select an unique SSID for a new local user
 */

call_t *select_nucall(const call_t lcall)
{
	call_t call;
	static call_t trycall;
	int ssid = 0;
	
	strcpy(call, (char *)lcall);
	truncssid(&call);
	strcpy(trycall, call);
	
	while (get_nuser(&trycall) || get_node(&trycall)) {
		ssid++;
		sprintf((char *)&trycall, "%s-%d", call, ssid);
	}
	
	return &trycall;
}
 
/*
 *	Accept a user connection (or not)
 */

void luser_login(struct csock_t *s)
{
	struct luser_t *lu;
	int newuser = 0;
	struct link_t *li;
	call_t *ncall;
	
	if (getssid((call_t *)s->call) == 0)
		truncssid((call_t *)s->call);
	
	if ((li = get_link((call_t *)s->call)) && li->mode == lm_incoming) {
		link_login(s, li);
		return;
	}
	
	lu = luser_alloc(s);
	lu->f = read_luserf((call_t *)s->call);
	if (!lu->f) {
		newuser = 1;
		log(L_LUSER, "Local: %s is a new user, creating", s->call);
		lu->f = new_luserf(s->call);
		write_luserf(lu->f);
	}
	
	ncall = select_nucall(lu->f->call);
	strcpy(lu->f->call, (char *)ncall);
	
	log(L_LUSER, "Local: %s logged in", lu->f->call);
	lu->nu = lnuser_login(lu);
	
/*	OH7LZB Clusse v0.31 - DX Cluster node OH7RBA-1 - 54/6 users, 16/2 nodes */
	csprintf(s, "%s - DX Cluster node %s\n", SOFT_STRING, clucall);
	csprintf(s, "%d users, %d known, %d local, %d nodes, %d of %d links\n", kuser_count, nuser_count, luser_count, node_count, link_count, link_max);
	csprintf(s, "\nHeavily under construction. If it breaks, you get to keep both pieces.\n");
	
	if (newuser)
		csprintf(s, "Welcome, new user.\n");
	
	prompt(s);
}

/*
 *	A local user has disconnected
 */

void luser_logout(struct csock_t *s)
{
	log(L_LUSER, "Local: %s logged out", s->lu->f->call);
	net_userdel(s->lu->nu);
	luser_free(s->lu);
}

/*
 *	Handle input from a local user - this is where the commands come in
 */
 
int luser_handler(struct csock_t *s, void *data, int len)
{
	char *p = (char *)data;
	
	if (command(s, cluster_cmds, p) == -2)
		return -2;
		
	prompt(s);
	
	return 0;
}

