/* TCP control and status routines
 * Copyright 1991 Phil Karn, KA9Q
 */
#include <stdio.h>
#include "config.h"
#include "global.h"
#include "timer.h"
#include "mbuf.h"
#include "netuser.h"
#include "internet.h"
#include "tcp.h"
#include "cmdparse.h"
#include "commands.h"

#ifdef  TCPACCESS
static int doaccess __ARGS((int argc,char *argv[],void *p));
void addtaccess __ARGS((int32 target,unsigned int bits,int16 low,int16 high,int16 permit));
#endif
static int tcpirtt __ARGS((int argc,char *argv[],void *p));
static int dotcpirtt __ARGS((int argc,char *argv[],void *p));
static int tcpmss __ARGS((int argc,char *argv[],void *p));
static int dotcpmss __ARGS((int argc,char *argv[],void *p));
static int dortt __ARGS((int argc,char *argv[],void *p));
static int dotcpkick __ARGS((int argc,char *argv[],void *p));
static int dotcpreset __ARGS((int argc,char *argv[],void *p));
static int dotcpretry __ARGS((int argc,char *argv[],void *p));
static int tcpretries __ARGS((int argc,char *argv[],void *p));
static int dotcpstat __ARGS((int argc,char *argv[],void *p));
static int dotcptr __ARGS((int argc,char *argv[],void *p));
static int dotcpwindow __ARGS((int argc,char *argv[],void *p));
static int tcpwindow __ARGS((int argc,char *argv[],void *p));
static int dotcpsyndata __ARGS((int argc,char *argv[],void *p));
static int tcpsyndata __ARGS((int argc,char *argv[],void *p));
static int dotcpclose __ARGS((int argc,char *argv[],void *p));

#ifdef TCPACCESS
struct rtaccess *TCPaccess = NULLACCESS; /* access list */
#endif

/* TCP subcommand table */
int
dotcp(int argc,char *argv[],void *p) {
  struct cmds Tcpcmds[] = {
#ifdef  TCPACCESS
    "access",	doaccess,	0,	0,	NULLCHAR,
#endif
    "close",	dotcpclose, 	0,	2,	"tcp close <tcb>",
    "irtt",	dotcpirtt,	0,	0,	NULLCHAR,
    "kick",	dotcpkick,	0,	2,	"tcp kick <tcb>",
    "mss",	dotcpmss,	0,	0,	NULLCHAR,
    "reset",	dotcpreset,	0,	2,	"tcp reset <tcb>",
    "retry",	dotcpretry, 	0,	0,	NULLCHAR,
    "rtt",	dortt,		0,	3,	"tcp rtt <tcb> <val>",
    "status",	dotcpstat,	0,	0,	NULLCHAR,
    "syndata",	dotcpsyndata,  	0,	0, 	NULLCHAR,
    "trace",	dotcptr,	0,	0,	NULLCHAR,
    "window",	dotcpwindow,	0,	0,	NULLCHAR,
    NULLCHAR,
  };
  return subcmd(Tcpcmds,argc,argv,p);
}

/* Display status of TCBs */
static int
dotcpstat(int argc,char *argv[],void *p) {

  if(argc < 2) {
    /* Dump TCP stats and summary of all TCBs
    /*     &TCB Rcv-Q Snd-Q  Local socket           Remote socket          State
     *     1234     0     0  xxx.xxx.xxx.xxx:xxxxx  xxx.xxx.xxx.xxx:xxxxx  Established
     */
    int i, j;
    struct tcb *tcb;

    for(j = i = 1; i <= NUMTCPMIB; i++) {
      if(Tcp_mib[i].name == NULLCHAR)
	continue;
      tprintf("(%2u)tcp%-17s%10lu",i,Tcp_mib[i].name,Tcp_mib[i].value.integer);
      tputs((j++ % 2) ? "     " : "\n");
    }
    if((j % 2) == 0)
      tputs("\n");

    tputs("&TCB     Rcv-Q Snd-Q  Local socket           Remote socket          State\n");
    for(tcb = Tcbs; tcb != NULLTCB; tcb = tcb->next){
      tprintf("%8lx%6u%6u  ",ptol(tcb),tcb->rcvcnt,tcb->sndcnt);
      tprintf("%-23s",pinet(&tcb->conn.local));
      tprintf("%-23s%-s",pinet(&tcb->conn.remote),Tcpstates[tcb->state]);
      if(tcb->state == TCP_LISTEN && tcb->flags.clone)
	tputs(" (S)");
	  tputs("\n");
	}
  }
  else {
    struct tcb *tcb = (struct tcb *)ltop(htol(argv[1]));
    if(tcpval(tcb))
      st_tcp(tcb);
    else
      tputs(Notval);
  }
  return 0;
}

/* Set smoothed round trip time for specified TCB */
static int
dortt(int argc,char *argv[],void *p) {
  struct tcb *tcb = (struct tcb *)ltop(htol(argv[1]));

  if(!tcpval(tcb)){
    tputs(Notval);
    return 1;
  }
  tcb->srtt = atol(argv[2]);
  return 0;
}

/* Close a TCP connection */
static int
dotcpclose(int argc,char *argv[],void *p) {
  struct tcb *tcb = (struct tcb *)ltop(htol(argv[1]));

  if(!tcpval(tcb)) {
    tputs(Notval);
    return 1;
    }
  return close_tcp(tcb);
}

/* Eliminate a TCP connection */
static int
dotcpreset(int argc,char *argv[],void *p) {
  struct tcb *tcb = (struct tcb *)ltop(htol(argv[1]));

  if(!tcpval(tcb)){
    tputs(Notval);
    return 1;
  }
  close_self(tcb,RESET);
  return 0;
}

static int
dotcptr(int argc,char *argv[],void *p) {
  return setbool(&Tcp_trace,"TCP trace",argc,argv);
}

/* Force a retransmission */
static int
dotcpkick(int argc,char *argv[],void *p) {
  struct tcb *tcb = (struct tcb *)ltop(htol(argv[1]));

  if(kick_tcp(tcb) == -1){
    tputs(Notval);
    return 1;
  }
  return 0;
}

static int
dotcpirtt(int argc,char *argv[],void *p) {
  struct tcp_rtt *tp;

  tcpirtt(argc,argv,(void *)&Tcp_irtt);
  if(argc < 2){
    for(tp = &Tcp_rtt[0];tp < &Tcp_rtt[RTTCACHE];tp++) {
      if(tp->addr != 0) {
	if(tprintf("%s: srtt %ld mdev %ld\n",inet_ntoa(tp->addr),
	tp->srtt,tp->mdev) == EOF)
	  break;
      }
    }
  }
  return 0;
}

static int
dotcpmss(int argc,char *argv[],void *p) {
  return tcpmss(argc,argv,(void *)&Tcp_mss);
}

static int
dotcpretry(int argc,char *argv[],void *p) {
  extern int Tcp_retry;

  return tcpretries(argc,argv,(void *)&Tcp_retry);
}

static int
dotcpsyndata(int argc,char *argv[],void *p) {
  return tcpsyndata(argc,argv,(void *)&Tcp_syndata);
}

static int
dotcpwindow(int argc,char *argv[],void *p) {
  return tcpwindow(argc,argv,(void *)&Tcp_window);
}

/* Dump a TCP control block in detail */
void
st_tcp(struct tcb *tcb) {
  int32 sent, recvd;

  if(tcb == NULLTCB)
    return;

  /* Compute total data sent and received; take out SYN and FIN */
  sent = tcb->snd.una - tcb->iss;	/* Acknowledged data only */
  recvd = tcb->rcv.nxt - tcb->irs;

  switch(tcb->state){
    case TCP_LISTEN:
    case TCP_SYN_SENT:		/* Nothing received or acked yet */
      sent = recvd = 0;
      break;
    case TCP_SYN_RECEIVED:
      recvd--;			/* Got SYN, no data acked yet */
      sent = 0;
      break;
    case TCP_ESTABLISHED:	/* Got and sent SYN */
    case TCP_FINWAIT1:		/* FIN not acked yet */
      sent--;
      recvd--;
      break;
    case TCP_FINWAIT2:		/* Our SYN and FIN both acked */
      sent -= 2;
      recvd--;
      break;
    case TCP_CLOSE_WAIT:	/* Got SYN and FIN, our FIN not yet acked */
    case TCP_CLOSING:
    case TCP_LAST_ACK:
      sent--;
      recvd -= 2;
      break;
    case TCP_TIME_WAIT:		/* Sent and received SYN/FIN, all acked */
      sent -= 2;
      recvd -= 2;
      break;
    }
    tprintf("Local %s",pinet(&tcb->conn.local));
    tprintf(" Remote %s",pinet(&tcb->conn.remote));
    tprintf(" State: %s\n      Init seq    Unack     Next Resent CWind "
	    "Thrsh  Wind  MSS Queue      Total\n",Tcpstates[tcb->state]);
    tprintf("Send:%9lx%9lx%9lx",tcb->iss,tcb->snd.una,tcb->snd.nxt);
    tprintf("%7lu%6u%6u",tcb->resent,tcb->cwind,tcb->ssthresh);
    tprintf("%6u%5u%6u%11lu\n",tcb->snd.wnd,tcb->mss,tcb->sndcnt,sent);

    tprintf("Recv:%9lx%18lx%7lu",tcb->irs,tcb->rcv.nxt,tcb->rerecv);
    tprintf("%18u%11u%11lu\n",tcb->rcv.wnd,tcb->rcvcnt,recvd);

    if(tcb->backoff > 0)
      tprintf("Backoff %u ",tcb->backoff);
    if(tcb->flags.retran)
      tputs("Retrying ");
    tputs("Timer ");
    switch(tcb->timer.state){
      case TIMER_STOP:
	tputs("-");
	break;
      case TIMER_RUN:
	tprintf("%lu",read_timer(&tcb->timer));
	break;
      case TIMER_EXPIRE:
	tputs("E");
    }

    tprintf("/%lu ms SRTT %ld ms Mean dev %ld ms\n", dur_timer(&tcb->timer),tcb->srtt,tcb->mdev);

    if(tcb->reseq != (struct reseq *)NULL){
      struct reseq *rp;

      tputs("Reassembly queue:\n");
      for(rp = tcb->reseq;rp != (struct reseq *)NULL; rp = rp->next){
	if(tprintf("  seq x%lx %u bytes\n",rp->seg.seq,rp->length) == EOF)
	  return;
      }
    }
}

/* Set initial round trip time for new connections */
static int
tcpirtt(int argc,char *argv[],void *p) {
  return setlong((long*)p,"TCP default irtt",argc,argv);
}

/* Set default maximum segment size */
static int
tcpmss(int argc,char *argv[],void *p) {
  return setshort((unsigned short *)p,"TCP MSS",argc,argv);
}

static int
tcpretries(int argc,char *argv[],void *p) {
  return setint((int *)p,"TCP retry",argc,argv);
}

static int
tcpsyndata(int argc,char *argv[],void *p) {
  return setbool((int *)p,"TCP syn+data piggybacking",argc,argv);
}

/* Set default window size */
static int
tcpwindow(int argc,char *argv[],void *p) {
  return setshort((unsigned short *)p,"TCP window",argc,argv);
}

/* These are the interface dependent tcp parameters */
static int doiftcpirtt __ARGS((int argc,char *argv[],void *p));
static int doiftcpretries __ARGS((int argc,char *argv[],void *p));
static int doiftcpwindow __ARGS((int argc,char *argv[],void *p));
static int doiftcpsyndata __ARGS((int argc,char *argv[],void *p));
static int doiftcpmss __ARGS((int argc,char *argv[],void *p));

int doiftcp(int argc, char *argv[],void *p) {
  struct cmds Iftcpcmds[] = {
    "irtt",     doiftcpirtt,    0, 0, NULLCHAR,
    "mss",      doiftcpmss,     0, 0, NULLCHAR,
    "retry",	doiftcpretries, 0, 0, NULLCHAR,
    "syndata",  doiftcpsyndata, 0, 0, NULLCHAR,
    "window",   doiftcpwindow,  0, 0, NULLCHAR,
    NULLCHAR,
  };

  return subcmd(Iftcpcmds,argc,argv,p);
}

void init_iftcp(struct iftcp *tcp) {
  tcp->window = Tcp_window;
  tcp->mss = Tcp_mss;
  tcp->irtt = Tcp_irtt;
  tcp->retries = Tcp_retry;
  tcp->syndata = Tcp_syndata;
}

int doiftcpirtt(int argc,char *argv[],void *p) {
  struct iface *ifp = p;

  return tcpirtt(argc,argv,(void *)&ifp->tcp->irtt);
}

int doiftcpmss(int argc,char *argv[],void *p) {
  struct iface *ifp = p;

  return tcpmss(argc,argv,(void *)&ifp->tcp->mss);
}

int doiftcpretries(int argc,char *argv[],void *p) {
  struct iface *ifp = p;

  return tcpretries(argc,argv,(void *)&ifp->tcp->retries);
}

int doiftcpwindow(int argc,char *argv[],void *p) {
  struct iface *ifp = p;

  return tcpwindow(argc,argv,(void *)&ifp->tcp->window);
}

int doiftcpsyndata(int argc,char *argv[],void *p) {
  struct iface *ifp = p;

  return tcpsyndata(argc,argv,(void *)&ifp->tcp->syndata);
}


#ifdef  TCPACCESS
static int
doaccess(int argc,char *argv[],void *p) {
  int32 target;
  unsigned bits;
  char *bitp;
  int16 lport,hport,state;
  char *cp; /* for printing the table */
  struct rtaccess *tpacc;
  struct rtaccess *head;
  struct rtaccess *prev;

  if(argc == 1){ /* print out the table */
    tputs("IP Address      Mask  Low Port High Port State\n");
    for(tpacc = TCPaccess;tpacc != NULLACCESS;tpacc = tpacc->nxtbits){
      if(tpacc->target != 0)
	cp = inet_ntoa(tpacc->target);
      else
	cp = "all";
      tprintf("%-16s",cp);
      tprintf("%4u ",tpacc->bits);
      tprintf("%9u",tpacc->lowport);
      tprintf("%10u ",tpacc->highport);
      if(tpacc->status)
	cp = "deny";
      else
	cp = "permit";
      tprintf("%-6s\n",cp);
    }
    return 0;
  }

  if(strcmp(argv[1],"permit") == 0){
    state = 0;
  }
  else {
    if((strcmp(argv[1],"deny") == 0) || (strcmp(argv[1],"delete") == 0)){
      state = -1;
    }
    else {
      tputs("Usage: tcp access <permit|deny|delete> <dest addr>[/<bits>] [lowport [highport]]\n");
      return 1;
    }
  }
  if(strcmp(argv[2],"all") == 0){
    target = 0;
    bits = 0;
  } else {
    /* If IP address is followed by an optional slash and
     * a length field, (e.g., 128.96/16) get it;
     * otherwise assume a full 32-bit address
     */
    if((bitp = strchr(argv[2],'/')) != NULLCHAR){
      /* Terminate address token for resolve() call */
      *bitp++ = '\0';
      bits = atoi(bitp);
    }
    else
      bits = 32;

    if((target = resolve(argv[2])) == 0){
      tprintf(Badhost,argv[2]);
      return 1;
    }
  }

  if(argc > 3){
    if(strcmp(argv[3],"all") == 0){
      lport = 1;
      hport = 65534L;
    } else {
      lport = atoi(argv[3]);
      hport = lport;
    }
  } else {
    lport = 0;
    hport = 0;
  }
  if(argc > 4)
    hport = atoi(argv[4]);

  if(strcmp(argv[1],"delete") == 0){
    prev = NULLACCESS;
    head = tpacc = TCPaccess;
    while(tpacc != NULLACCESS){
      head = tpacc;
      tpacc = tpacc->nxtbits;
      if((head->target == target) &&
		(head->bits == bits)     &&
		(head->lowport == lport) &&
		(head->highport == hport)) { /*match*/

	/*now delete. watch for special cases*/
	if(head == TCPaccess) /* first in chain */
	  TCPaccess = head->nxtbits;
	else
	  /*
	   * sanity check: we cant get here with
	   * prev == NULLACCESS !!
	   */
	  prev->nxtbits = tpacc;
	xfree(head);
	return 0;
      }
      prev = head;
    }
    tputs("Not found.\n");
    return 1;
  }
  /* add the access */
  addtaccess(target,bits,lport,hport,state);
  return 0;
}

/* add an entry to the access control list */
/* not a lot of error checking 8-) */
void
addtaccess(target,bits,low,high,permit)
int32 target;           /* Target IP address prefix */
unsigned int bits;      /* Size of target address prefix in bits (0-32) */
int16 low;
int16 high;
int16 permit;
{
  struct rtaccess *tpacc; /*temporary*/
  struct rtaccess *holder; /*for the new record*/

  holder = (struct rtaccess *)cxallocw(1,sizeof(struct rtaccess));
  holder->nxtiface = NULLACCESS;
  holder->nxtbits = NULLACCESS;
  holder->target = target;
  holder->bits = bits;
  holder->lowport = low;
  holder->highport = high;
  holder->status = permit;
  if((tpacc = TCPaccess) == NULLACCESS){
    TCPaccess = holder;
  }
  else {
    while(tpacc->nxtbits != NULLACCESS)
      tpacc = tpacc->nxtbits;
    tpacc->nxtbits = holder;
  }
}

/* check to see if port is "authorized".  Returns 0 if matching permit record
   is found or no access records exists, -1 if not found or deny record found */
int
tcp_check(accptr,src,port)
struct rtaccess *accptr;
int32 src;
int16 port;
{
  unsigned long mask;

  if(accptr == NULLACCESS)
    return 0;               /* no access control */
  for(;accptr != NULLACCESS;accptr = accptr->nxtbits) {
    mask = ~0L << (32 - accptr->bits);
    if( (accptr->target == (mask & src)) &&
	 ( ((port >= accptr->lowport) && (port <= accptr->highport))
	   || (!accptr->lowport)) ){
      return (accptr->status);
    }
  }
  return -1; /* fall through to here if not found */
}
#endif /* TCPACCESS */
