#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <pwd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <linux/sockios.h>
#include <linux/if.h>
#include <linux/ax25.h>

#include "config.h"
#include "axutils.h"
#include "axconfig.h"
#include "version.h"

int routes(int s, int argc, char *argv[], ax25_address *callsign)
{
	struct ax25_routes_struct ax25_route;
	struct ax25_route_opt_struct ax25_opt;
	int i, j;
	int ip_mode = ' ';

	if (strcmp(argv[2], "add") == 0) {
		ax25_route.port_addr  = *callsign;
		ax25_route.digi_count = 0;

		if (strcmp(argv[4], "default") == 0) {
			ax25_route.dest_addr = null_ax25_address;
		} else {
			if (convert_call_entry(argv[4], (char *)&ax25_route.dest_addr) == -1)
				return 1;
		}

		for (i = 5, j = 0; i < argc && j < 6; i++) {
			if (strncmp(argv[i], "-i", 2) == 0) {
				if (++i == argc) {
					fprintf(stderr, "axparms: -i must have a parameter\n");
					return 1;
				}
				switch (*argv[i]) {
					case 'd':
					case 'D':
						ip_mode = 'D';
						break;
					case 'v':
					case 'V':
						ip_mode = 'V';
						break;
					default:
						ip_mode = ' ';
						break;
				}
			} else {
				if (convert_call_entry(argv[i], (char *)&ax25_route.digi_addr[j]) == -1)
					return 1;
				ax25_route.digi_count++;
				j++;
			}
		}

		if (ioctl(s, SIOCADDRT, &ax25_route) != 0) {
			perror("axparms: SIOCADDRT");
			return 1;
		}
		
		ax25_opt.port_addr = *callsign;
		ax25_opt.dest_addr = ax25_route.dest_addr;
		ax25_opt.cmd = AX25_SET_RT_IPMODE;
		ax25_opt.arg = ip_mode;
		
		if (ioctl(s, SIOCAX25OPTRT, &ax25_opt) != 0) {
			perror("axparms: SIOCAX25OPTRT");
			return 1;
		}
	}
	
	if (strcmp(argv[2], "del") == 0) {
		ax25_route.port_addr  = *callsign;
		ax25_route.digi_count = 0;

		if (strcmp(argv[4], "default") == 0) {
			ax25_route.dest_addr = null_ax25_address;
		} else {
			if (convert_call_entry(argv[4], (char *)&ax25_route.dest_addr) == -1)
				return 1;
		}

		if (ioctl(s, SIOCDELRT, &ax25_route) != 0) {
			perror("axparms: SIOCDELRT");
			return 1;
		}
	}

	return 0;
}

int setifcall(int s, char *ifn, char *name)
{
	char call[7];
	struct ifreq ifr;
	
	if (convert_call_entry(name, call) == -1)
		return 1;

	strcpy(ifr.ifr_name, ifn);
	memcpy(ifr.ifr_hwaddr.sa_data, call, 7);
	ifr.ifr_hwaddr.sa_family = AF_AX25;
	
	if (ioctl(s, SIOCSIFHWADDR, &ifr) != 0) {
		perror("axparms: SIOCSIFHWADDR");
		return 1;
	}

	return 0;
}

int bpqdev(int s, char *device, char *port)
{
	struct ax25_bpqaddr_struct bpqaddr;
	FILE *fp;
	char buffer[90], *p;
	int n = 0;

	if ((fp = fopen(CONF_AXPORTS_FILE, "r")) == NULL) {
		fprintf(stderr, "axparms: cannot open axports file\n");
		return 1;
	}

	while (fgets(buffer, 90, fp) != NULL) {
		n++;

		if ((p = strchr(buffer, '\n')) != NULL)
			*p = '\0';

		if (strlen(buffer) > 0 && *buffer == '#')
			continue;

		if ((p = strtok(buffer, " \t\r\n")) == NULL) {
			fprintf(stderr, "axparms: unable to parse line %d of the axports file\n", n);
			return 1;
		}

		if (strcmp(p, port) != 0)
			continue;

		if ((p = strtok(NULL, " \t\r\n")) == NULL) {
			fprintf(stderr, "axparms: unable to parse line %d of the axports file\n", n);
			return 1;
		}

		strcpy(bpqaddr.dev, device);
		if (convert_call_entry(p, bpqaddr.addr.ax25_call) == -1)
			return 1;

		if (ioctl(s, SIOCAX25BPQADDR, &bpqaddr) != 0) {
			perror("axparms: SIOCAX25BPQADDR");
			return 1;
		}

		printf("AX.25 port %s bound to device %s\n", port, device);

		fclose(fp);
		
		return 0;
	}

	fclose(fp);

	fprintf(stderr, "axparms: cannot find port %s in axports\n", port);

	return 1;
}
	
int associate(int s, int argc, char *argv[])
{
	char buffer[80], *u, *c;
	struct sockaddr_ax25 sax25;
	struct passwd *pw;
	int opt;
	FILE *fp;
	
	if (strcmp(argv[2], "show") == 0) {
		if ((fp = fopen(PROC_AX25_CALLS_FILE, "r")) == NULL) {
			fprintf(stderr, "axparms: associate: cannot open %s\n", PROC_AX25_CALLS_FILE);
			return 1;
		}

		fgets(buffer, 80, fp);

		printf("Userid     Callsign\n");

		while (fgets(buffer, 80, fp) != NULL) {
			u = strtok(buffer, " \t\n");
			c = strtok(NULL, " \t\n");
			if ((pw = getpwuid(atoi(u))) != NULL)
				printf("%-10s %s\n", pw->pw_name, c);
		}

		fclose(fp);

		return 0;
	}

	if (strcmp(argv[2], "policy") == 0) {
		if (strcmp(argv[3], "default") == 0) {
			opt = AX25_NOUID_DEFAULT;

			if (ioctl(s, SIOCAX25NOUID, &opt) == -1) {
				perror("axparms: SIOCAX25NOUID");
				return 1;
			}

			return 0;
		}

		if (strcmp(argv[3], "deny") == 0) {
			opt = AX25_NOUID_BLOCK;

			if (ioctl(s, SIOCAX25NOUID, &opt) == -1) {
				perror("axparms: SIOCAX25NOUID");
				return 1;
			}

			return 0;
		}

		fprintf(stderr, "axparms: associate: 'default' or 'deny' required\n");

		return 1;
	}
	
	
	if (convert_call_entry(argv[2], (char *)&sax25.sax25_call) == -1) {
		fprintf(stderr, "axparms: associate: invalid callsign %s\n", argv[2]);
		return 1;
	}

	if (strcmp(argv[3], "delete") == 0) {
		if (ioctl(s, SIOCAX25DELUID, &sax25) == -1) {
			perror("axparms: SIOCAX25DELUID");
			return 1;
		}

		return 0;
	}

	if ((pw = getpwnam(argv[3])) == NULL) {
		fprintf(stderr, "axparms: associate: unknown username %s\n", argv[3]);
		return 1;
	}

	sax25.sax25_uid = pw->pw_uid;
		
	if (ioctl(s, SIOCAX25ADDUID, &sax25) == -1) {
		perror("axparms: SIOCAX25ADDUID");
		return 1;
	}

	return 0;
}

int display_parms(int s, ax25_address *callsign)
{
	struct ax25_parms_struct ax25_parms;

	ax25_parms.port_addr = *callsign;

	if (ioctl(s, SIOCAX25GETPARMS, &ax25_parms) != 0) {
		perror("axparms: SIOCAX25GETPARMS");
		return 1;
	}

	printf("Default IP mode               : %s\n",
		(ax25_parms.values[AX25_VALUES_IPDEFMODE] == 'D') ? "Datagram" : "Virtual Circuit");
	printf("Default AX.25 mode            : %s\n",
		(ax25_parms.values[AX25_VALUES_AXDEFMODE] == 8) ? "Standard" : "Extended");

	printf("Is connected mode allowed ?   : %s\n",
		(ax25_parms.values[AX25_VALUES_CONMODE]) ? "Yes" : "No");
	printf("Is NET/ROM allowed ?          : %s\n",
		(ax25_parms.values[AX25_VALUES_NETROM]) ? "Yes" : "No");
	printf("Is AX.25 text mode allowed ?  : %s\n", 
		(ax25_parms.values[AX25_VALUES_TEXT]) ? "Yes" : "No");
	printf("Is in-band digi-ing allowed ? : %s\n",
		(ax25_parms.values[AX25_VALUES_DIGI] & AX25_DIGI_INBAND) ? "Yes" : "No");
	printf("Is x-band digi-ing allowed ?  : %s\n",
		(ax25_parms.values[AX25_VALUES_DIGI] & AX25_DIGI_XBAND) ? "Yes" : "No");

	printf("AX.25 MTU (PACLEN)            : %d\n", ax25_parms.values[AX25_VALUES_PACLEN]);
	printf("Standard AX.25 window size    : %d\n", ax25_parms.values[AX25_VALUES_WINDOW]);
	printf("Extended AX.25 window size    : %d\n", ax25_parms.values[AX25_VALUES_EWINDOW]);
	printf("Max. of enqueued IP frames    : %d * window size\n", ax25_parms.values[AX25_VALUES_IPMAXQUEUE]);

	printf("Initial T1 value (seconds)    : %d\n", ax25_parms.values[AX25_VALUES_T1]);
	printf("T2 value (seconds)            : %d\n", ax25_parms.values[AX25_VALUES_T2]);
	printf("T3 value (seconds)            : %d\n", ax25_parms.values[AX25_VALUES_T3]);
	printf("IDLE timeout value (minutes)  : %d\n", ax25_parms.values[AX25_VALUES_IDLE]);
	printf("N2 value                      : %d\n", ax25_parms.values[AX25_VALUES_N2]);

	return 0;
}

int assign_parms(int s, int argc, char *argv[], ax25_address *callsign)
{
	struct ax25_parms_struct ax25_parms;
	int i, n;

	ax25_parms.port_addr = *callsign;

	if (ioctl(s, SIOCAX25GETPARMS, &ax25_parms) != 0) {
		perror("axprms: SIOCAX25GETPARMS");
		return 1;
	}

	i = 3;
	
	while (i < argc) {
		if (strncmp(argv[i], "-ip", 3) == 0) {
			n = toupper(argv[i + 1][0]);
			if (n != 'D' && n != 'V') {
				fprintf(stderr, "axparms: parms: invalid default ip mode flag %s\n", argv[i + 1]);
			} else {
				ax25_parms.values[AX25_VALUES_IPDEFMODE] = n;
			}
			
			i++;
		}

		if (strncmp(argv[i], "-ax", 3) == 0) {
			n = toupper(argv[i + 1][0]);
			if (n != 'E' && n != 'S') {
				fprintf(stderr, "axparms: parms: invalid default AX.25 mode flag %s\n", argv[i + 1]);
			} else {
				if (n == 'S') {
					ax25_parms.values[AX25_VALUES_AXDEFMODE] = 8;
				} else {
					ax25_parms.values[AX25_VALUES_AXDEFMODE] = 128;
				}
			}
			
			i++;
		}

		if (strncmp(argv[i], "-con", 4) == 0) {
			n = toupper(argv[i + 1][0]);
			if (n != 'Y' && n != 'N') {
				fprintf(stderr, "axparms: parms: invalid connected mode allowed flag %s\n", argv[i + 1]);
			} else {
				ax25_parms.values[AX25_VALUES_CONMODE] = n == 'Y';
			}
			
			i++;
		}

		if (strncmp(argv[i], "-idig", 5) == 0) {
			switch (n = toupper(argv[i + 1][0])) {
				case 'N':
					ax25_parms.values[AX25_VALUES_DIGI] &= ~AX25_DIGI_INBAND;
					break;
				case 'Y':
					ax25_parms.values[AX25_VALUES_DIGI] |= AX25_DIGI_INBAND;
					break;
				default:
					fprintf(stderr, "axparms: parms: invalid in-band digi allowed flag %s\n", argv[i + 1]);
					break;
			}
			
			i++;
		}

		if (strncmp(argv[i], "-xdig", 5) == 0) {
			switch (n = toupper(argv[i + 1][0])) {
				case 'N':
					ax25_parms.values[AX25_VALUES_DIGI] &= ~AX25_DIGI_XBAND;
					break;
				case 'Y':
					ax25_parms.values[AX25_VALUES_DIGI] |= AX25_DIGI_XBAND;
					break;
				default:
					fprintf(stderr, "axparms: parms: invalid cross-band digi allowed flag %s\n", argv[i + 1]);
					break;
			}
			
			i++;
		}

		if (strncmp(argv[i], "-netr", 5) == 0) {
			n = toupper(argv[i + 1][0]);
			if (n != 'Y' && n != 'N') {
				fprintf(stderr, "axparms: parms: invalid NET/ROM allowed flag %s\n", argv[i + 1]);
			} else {
				ax25_parms.values[AX25_VALUES_NETROM] = n == 'Y';
			}
			
			i++;
		}

		if (strncmp(argv[i], "-text", 5) == 0) {
			n = toupper(argv[i + 1][0]);
			if (n != 'Y' && n != 'N') {
				fprintf(stderr, "axparms: parms: invalid AX.25 allowed flag %s\n", argv[i + 1]);
			} else {
				ax25_parms.values[AX25_VALUES_TEXT] = n == 'Y';
			}
			
			i++;
		}

		if (strncmp(argv[i], "-wind", 5) == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1 || n > 7) {
				fprintf(stderr, "axparms: parms: invalid standard window value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_WINDOW] = n;
			}
			
			i++;
		}

		if (strncmp(argv[i], "-ewind", 6) == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1 || n > 63) {
				fprintf(stderr, "axparms: parms: invalid extended window value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_EWINDOW] = n;
			}
			
			i++;
		}

		if (strcmp(argv[i], "-t1") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1) {
				fprintf(stderr, "axparms: parms: invalid t1 value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_T1] = n;
			}
			
			i++;
		}

		if (strcmp(argv[i], "-t2") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1) {
				fprintf(stderr, "axparms: parms: invalid t2 value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_T2] = n;
			}
			
			i++;
		}

		if (strcmp(argv[i], "-t3") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1) {
				fprintf(stderr, "axparms: parms: invalid t3 value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_T3] = n;

			}
			
			i++;
		}

		if (strcmp(argv[i], "-n2") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1 || n > 31) {
				fprintf(stderr, "axparms: parms: invalid n2 value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_N2] = n;
			}
			
			i++;
		}
		
		if (strcmp(argv[i], "-idle") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 0 || n > 100) {
				fprintf(stderr, "axparms: parms: invalid idle value %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_IDLE] = n;
			}
			i++;
		}
		
		if (strcmp(argv[i], "-paclen") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 16) {
				fprintf(stderr, "axparms: parms: invalid paclen %d\n", n);
			} else {
				if (n > 256)
					fprintf(stderr,"axparms: parms: WARNING: paclen %d is greater than 256\n", n);
				ax25_parms.values[AX25_VALUES_PACLEN] = n;
			}
		}
		
		if (strcmp(argv[i], "-maxq") == 0) {
			n = atoi(argv[i + 1]);
			if (n < 1) {
				fprintf(stderr, "axparms: parms: invalid IP maxqueue %d\n", n);
			} else {
				ax25_parms.values[AX25_VALUES_IPMAXQUEUE] = n;
			}
			i++;
		}

		i++;
	}

	if (ioctl(s, SIOCAX25SETPARMS, &ax25_parms) != 0) {
		perror("axparms: SIOCAX25SETPARMS");
		return 1;
	}

	return 0;
}

int main(int argc, char **argv)
{
	ax25_address callsign;
	int s, n;
	char *addr;

	if (argc == 1) {	
		fprintf(stderr, "usage: axparms -assoc|-dev|-parms|-route|-setcall|-version ...\n");
		return 1;
	}

	if (strncmp(argv[1], "-v", 2) == 0) {
		printf("axparms: %s\n", version);
		return 0;
	}

	if (strncmp(argv[1], "-a", 2) == 0) {
		if (argc == 2) {
			fprintf(stderr, "usage: axparms -assoc [callsign] [username]\n");
			fprintf(stderr, "usage: axparms -assoc [callsign] delete\n");
			fprintf(stderr, "usage: axparms -assoc policy default|deny\n");
			fprintf(stderr, "usage: axparms -assoc show\n");
			return 1;
		}

		if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			perror("axparms: socket");
			return 1;
		}

		n = associate(s, argc, argv);

		close(s);

		return n;
	}

	if (strncmp(argv[1], "-d", 2) == 0) {
		if (argc != 4) {
			fprintf(stderr, "usage: axparms -dev device port\n");
			return 1;
		}

		if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			perror("axparms: socket");
			return 1;
		}

		n = bpqdev(s, argv[2], argv[3]);

		close(s);

		return n;
	}

	if (strncmp(argv[1], "-p", 2) == 0) {
		if (argc == 2) {
			fprintf(stderr, "usage: axparms -parms port ...\n");
			return 1;
		}
	
		if (ax25_config_load_ports() == 0) {
			fprintf(stderr, "axparms: no AX.25 port data configured\n");
			return 1;
		}

		if ((addr = ax25_config_get_addr(argv[2])) == NULL) {
			fprintf(stderr, "axparms: invalid port name - %s\n", argv[2]);
			return 1;
		}

		if (convert_call_entry(addr, callsign.ax25_call) == -1)
			return 1;

		if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			perror("axparms: socket");
			return 1;
		}

		if (argc == 3) {
			printf("Parameters for port %s with callsign %s\n", argv[2], addr);
			n = display_parms(s, &callsign);
		} else {
			n = assign_parms(s, argc, argv, &callsign);
		}

		close(s);

		return n;
	}

	if (strncmp(argv[1], "-r", 2) == 0) {
		if (argc < 5) {
			fprintf(stderr, "usage: axparms -route add port callsign [digi ...] [-ipmode mode]\n");
			fprintf(stderr, "usage: axparms -route del port callsign\n");
			return 1;
		}

		if (strcmp(argv[2], "add") != 0 && strcmp(argv[2], "del") != 0) {
			fprintf(stderr, "usage: axparms -route add port callsign [digi ...]\n");
			fprintf(stderr, "usage: axparms -route del port callsign\n");
			return 1;
		}
		
		if (ax25_config_load_ports() == 0) {
			fprintf(stderr, "axparms: no AX.25 port data configured\n");
			return 1;
		}

		if ((addr = ax25_config_get_addr(argv[3])) == NULL) {
			fprintf(stderr, "axparms: invalid port name - %s\n", argv[3]);
			return 1;
		}

		if (convert_call_entry(addr, callsign.ax25_call) == -1)
			return 1;

		if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			perror("axparms: socket");
			return 1;
		}

		n = routes(s, argc, argv, &callsign);

		close(s);
		
		return n;
	}

	if (strncmp(argv[1], "-s", 2) == 0) {
		if (argc != 4) {
			fprintf(stderr, "usage: axparms -setcall interface callsign\n");
			return 1;
		}
	
		if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
			perror("axparms: socket");
			return 1;
		}

		n = setifcall(s, argv[2], argv[3]);

		close(s);

		return n;
	}

	fprintf(stderr, "usage: axparms -assoc|-dev|-parms|-route|-setcall|-version ...\n");

	return 1;
}
