/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include "qqueue.cpp"  // include entire template definition, to generate Qqueue<char> object
#include "buffers.h"

Display_buffer_with_scroll::Display_buffer_with_scroll(int a, int b, int c, int d):
    Qqueue<char>(a, b), display_cols(c), display_lines(d), scrollend_flag(0),
      last_scrolladr(0), buffer_read_scrollptr(0), firstline(FALSE) {
    display_length = display_cols*display_lines;
    buffer_read_ptr = new char[display_length + 1];
    if (!buffer_read_ptr) {
        cerr << "Memory allocation error in Display_buffer_with_scroll::Display_buffer_with_scroll()" << endl;
	exit(3);
    }
}
         
void Display_buffer_with_scroll::reset(void) {
    Qqueue<char>::reset();
    last_scrolladr = buffer_read_scrollptr = 0;
    scrollend_flag = 0;
}

int Display_buffer_with_scroll::add_item(char item) {
    int temp = Qqueue<char>::add_item(item);
    if (scrollend_flag) last_scrolladr = itembuffer_baseptr;
    else if (last_scrolladr == itembuffer_baseptr) scrollend_flag = 1;
    return temp;
}

int Display_buffer_with_scroll::front(Qqueue_enum::extract_mode mode) {
    int return_value;
    if (Qqueue<char>::is_empty()) return_value = -1;
    else {
        return_value = Qqueue<char>::front(mode);
	if (scrollend_flag) last_scrolladr = itembuffer_baseptr;
	else if (last_scrolladr == itembuffer_baseptr) scrollend_flag = 1;
    }
    return return_value;
}

int Display_buffer_with_scroll::back(Qqueue_enum::extract_mode mode) {
    int return_value;
    if (Qqueue<char>::is_empty()) return_value = -1;
    else {
        return_value = Qqueue<char>::back(mode);
	if (mode == remove && last_scrolladr == itembuffer_backptr) last_scrolladr = 0;
    }
    return return_value;
}

char* Display_buffer_with_scroll::goup_line(char* charadr) {
    int charcount = 0;
    int firstcharLF_firstpass_flag = FALSE;
    static int firstcharLF_secondpass_flag = FALSE;
  // if first line, find  its beginning or notional beginning
    if (firstline) {
        if (*charadr == '\n') {
	    firstcharLF_firstpass_flag = TRUE;
	    firstcharLF_secondpass_flag = TRUE;
	}
        while ((*charadr != '\n') && (charcount < display_cols) && (charadr != itembuffer_baseptr)) {
	    if (charadr > itembuffer) charadr--;
	    else charadr = max_itemptr;          // scroll charadr
	    charcount++;
	}
	if (charadr < max_itemptr) charadr++; // now go forward one letter
	else charadr = itembuffer;
	firstline = FALSE;
	charcount = 0;
    }
    // If the buffer is full and the back character is '\n', on the second pass, charadr
    // will point to itembuffer_baseptr - firstcharLFsecondpass_flag flags this and provides
    // a guard to stop the while loop test failing at the outset because of this
    else {                              // else pass over the lf (if any)
        for (;charcount < 2 && (firstcharLF_secondpass_flag || charadr != itembuffer_baseptr);
	                                                                        charcount++) {
	    firstcharLF_secondpass_flag = FALSE;
	    if (charadr > itembuffer) charadr--;
	    else charadr = max_itemptr;          // scroll charadr
	}
	charcount = 0;
    }

    // If the buffer is full and the back character is '\n', on the first pass, charadr will
    // point to itembuffer_baseptr - firstcharLF_firstpass_flag flags this and provides
    // a guard to stop the while loop test failing at the outset because of this
    // (firstcharLF_firstpass_flag also prevents the character in *itembuffer_baseptr
    // interfering if it happens to be '\n')
    while ((firstcharLF_firstpass_flag || *charadr != '\n') && charcount < (display_cols - 1)
	     && (firstcharLF_firstpass_flag || charadr != itembuffer_baseptr)) { 
        firstcharLF_firstpass_flag = FALSE;
        if (charadr > itembuffer) charadr--;
	else charadr = max_itemptr;          // scroll charadr
	charcount++;
    }
    if (charadr != itembuffer_baseptr) {  // got required full line, go back to next letter in buffer
 	if (charadr < max_itemptr) charadr++;
	else charadr = itembuffer;
    }
    else scrollend_flag = 1;
    return charadr;
}

char* Display_buffer_with_scroll::goup_page(scroll_mode mode) {
    int count;
    char* temp;
    int firstscroll = FALSE;
    int lines;
    if (mode == halfpage) lines = display_lines/2 + 1;
    else lines = display_lines + 1;
    if (!last_scrolladr) {                           // this is the first scroll
        temp = itembuffer_backptr; 
	// when the buffer is empty or full, itembuffer_backptr == itembuffer_baseptr.  We
	// now need to take temp to the buffer position before itembuffer_backptr, so it
	// points to the letter at the back (most recently added) of the buffer.  When
	// it reaches itembuffer_baseptr we will therefore know that we are at the
	// front (oldest) part of the buffer.  We will use temp to set last_scrolladr.
	if (temp > itembuffer) temp--;
	else temp = max_itemptr;
	firstscroll = TRUE;
    }
    else temp = last_scrolladr;                      // we are already scrolling
    for (count = 0; count < lines && !scrollend_flag; count++) {
        temp = goup_line(temp);
    }
    if (count < lines && firstscroll) {
        // if not enough lines on screen to scroll, we must now signal this so that
        // last_scrolladr will be set to null (and is_scrolling() will return null)
        // via the scrollback() method (which calls this method).
        temp = 0; 
	scrollend_flag = 0;
    }
    return temp;
}

void Display_buffer_with_scroll::scrollback(scroll_mode mode) {
        // check to see if there is a display buffer and something to the display
    if (display_length  && itemcount) {
        if (!last_scrolladr || (mode != line && !scrollend_flag)) {// if first scroll
	             // or we are not in line mode and not yet scrolled to start of buffer
	    if (!last_scrolladr) {
		firstline = TRUE;
		mode = page;
	    }
	    last_scrolladr = goup_page(mode);
	}
	else if (!scrollend_flag) {   // we have not yet scrolled to start of buffer
	                              // and we are in line mode
	    last_scrolladr = goup_line(last_scrolladr);
	}
    }
    else {                            // nothing to display - signal no scroll made
	last_scrolladr = 0;
    }
    buffer_read_scrollptr = last_scrolladr;
}

char* Display_buffer_with_scroll::godown_line(char* charadr) {
    int charcount = 0;
    while (*charadr != '\n' && charcount < display_cols
	      && (scrollend_flag || charadr != itembuffer_backptr)) {
	  // (include scrollend_flag in the last test so that a previous scroll to the
	  // front of a full buffer with scrollback() will not cause the loop to fail at
	  // the outset - with a full buffer itembuffer_backptr == itembuffer_baseptr)
        if (charadr < max_itemptr) charadr++;
	else charadr = itembuffer;          // scroll charadr
	scrollend_flag = 0;    // make sure that the loop will fail if
	charcount++;           // charadr == itembuffer on next time around loop
    }
    if (*charadr == '\n') {
                                   // got required LF - go to next letter in buffer
        if (charadr < max_itemptr) charadr++;
	else charadr = itembuffer;
	last_scrolladr = charadr;
    }
    if (charadr == itembuffer_backptr) {           // we are at the end of the buffer - signal it
	charadr = 0;
    }
    return charadr;
}

char* Display_buffer_with_scroll::godown_page(scroll_mode mode) {
    char* temp = last_scrolladr;
    int count;
    int lines;
    if (mode == halfpage) lines = display_lines/2;
    else lines = display_lines;
    for(count = 0; count < lines && temp; count++) {
        temp = godown_line(temp);
    }
    return temp;
}

void Display_buffer_with_scroll::scrollforward(scroll_mode mode) {
    if (display_length  && last_scrolladr) {  
                       // check to see if there is a display buffer
		       // and something to go forward from
        if (mode != line) last_scrolladr = godown_page(mode);
	    
	else last_scrolladr = godown_line(last_scrolladr);
	scrollend_flag = 0;
    }
    buffer_read_scrollptr = last_scrolladr;
}

void Display_buffer_with_scroll::exitscroll(void) {
    int count;
    char* temp;
    temp = itembuffer_backptr; 
	// when the buffer is empty or full, itembuffer_backptr == itembuffer_baseptr.  We
	// now need to take temp to the buffer position before itembuffer_backptr, so it
	// points to the letter at the back (most recently added) of the buffer.  When
	// it reaches itembuffer_baseptr we will therefore know that we are at the
	// front (oldest) part of the buffer.  We will use temp to set
        // buffer_read_scrollptr.
    if (temp > itembuffer) temp--;
    else temp = max_itemptr;
    scrollend_flag = 0;
    for (count = 0; count < (display_lines - 1) && !scrollend_flag; count++) {
        temp = goup_line(temp);
    }
    buffer_read_scrollptr = temp;
    scrollend_flag = 0;
    last_scrolladr = 0;
}

void Display_buffer_with_scroll::buffer_read(void) {
    if (buffer_read_scrollptr) {
        int index = 0;
	int linecount = 0;
	int charcount = 0;
	int full_buffer_pass = 0;
	if (scrollend_flag) full_buffer_pass = 1;
	for (;linecount < display_lines && 
	        (full_buffer_pass || buffer_read_scrollptr !=  itembuffer_backptr); index++) {
	  // (include full_buffer_pass in the last test so that a scroll to the front
	  // of a full buffer with scrollback() will not cause the loop to fail at
	  // the outset - with a full buffer itembuffer_backptr == itembuffer_baseptr)
	    if (*buffer_read_scrollptr == '\n' || charcount == (display_cols - 1)) {
	        linecount++;
		charcount = 0;
	    }
	    else charcount++;
	    buffer_read_ptr[index] = *buffer_read_scrollptr;
	    // now increment buffer_read_scrollptr
	    if (buffer_read_scrollptr < max_itemptr) buffer_read_scrollptr++;
	    else buffer_read_scrollptr = itembuffer;  // scroll buffer_read_scrollptr
	    full_buffer_pass = 0;
	}
	char* temp = buffer_read_scrollptr;
	if (temp > itembuffer) temp--;
	else temp = max_itemptr;
	if (linecount == display_lines && *temp == '\n') {// if last letter was \n
	    buffer_read_scrollptr = temp;          // then don't include it this time
	    index--;
	}
	if (buffer_read_scrollptr == itembuffer_backptr) last_scrolladr = 0;
	buffer_read_ptr[index] = 0;
    }
    else *buffer_read_ptr = 0;                     // indicate null string
}

char* Display_buffer_with_scroll::text(void) const {
    char* char_p = new char[itembuffer_length + 1];
    char* temp_p = itembuffer_baseptr;
    int count;
    for (count = 0; count < itemcount; count++) {
        char_p[count] = *temp_p;
        if (temp_p < max_itemptr) temp_p++;
	else temp_p = itembuffer;            // scroll temp_p
    }
    char_p[count] = 0;                       // null terminate the string
    return char_p;
}

void Display_buffer_with_curses::scrollup(WINDOW* display_window, scroll_mode mode) {
    if (!is_endofscroll()) {
        scrollback(mode);
	if (is_scrolling()) {
	    int x_pos, y_pos;
	    getsyx(y_pos, x_pos);
	    buffer_read();
	    char* temp = buffer_read_ptr;
	    wmove(display_window, 0, 0);
	    wnoutrefresh(display_window);
	    for (; *temp != 0; temp++) {
	        if (*temp != '\n') waddch(display_window, (uchar)*temp);
		else {
		    wclrtoeol(display_window);
		    waddch(display_window, '\n');
		    wnoutrefresh(display_window);
		}
	    }
	    wclrtobot(display_window);
	    wnoutrefresh(display_window);
	    setsyx(y_pos, x_pos);
	    doupdate();
	}
	else {
	    beep();
	    doupdate();
	}
    }
    else {
        beep();
	doupdate();
    }
}

void Display_buffer_with_curses::scrolldown(WINDOW* display_window, scroll_mode mode) {
    if (is_scrolling()) {
	int x_pos, y_pos;
	getsyx(y_pos, x_pos);
        scrollforward(mode);
	buffer_read();
	char* temp = buffer_read_ptr;
	wmove(display_window, 0, 0);
	wnoutrefresh(display_window);
	for (; *temp != 0; temp++) {
	    if (*temp != '\n') waddch(display_window, (uchar)*temp);
	    else {
	        wclrtoeol(display_window);
		waddch(display_window, '\n');
		wnoutrefresh(display_window);
	    }
	}
	wclrtobot(display_window);
	wnoutrefresh(display_window);
	setsyx(y_pos, x_pos);
	doupdate();
    }
    else {
	beep();
        doupdate();
    }
}

void Display_buffer_with_curses::scrollout(WINDOW* display_window) {
    if (is_scrolling()) redisplay(display_window);
    else {
        beep();
        doupdate();
    }
}

void Display_buffer_with_curses::redisplay(WINDOW* display_window) {
    int x_pos, y_pos;
    getsyx(y_pos, x_pos);
    exitscroll();
    buffer_read();
    char* temp = buffer_read_ptr;
    wmove(display_window, 0, 0);
    wnoutrefresh(display_window);
    for (; *temp != 0; temp++) {
        if (*temp != '\n') waddch(display_window, (uchar)*temp);
	else {
	    wclrtoeol(display_window);
	    waddch(display_window, '\n');
	    wnoutrefresh(display_window);
	}
    }
    wclrtobot(display_window);
    wnoutrefresh(display_window);
    setsyx(y_pos, x_pos);
    doupdate();
}

void Display_buffer_with_curses::display_first_page(WINDOW* display_window) {
    if (is_used()) {
        int full_screen_flag = 0;
        scrollback(page);
	if (is_scrolling()) full_screen_flag = 1; 
        buffer_read_scrollptr = last_scrolladr = itembuffer_baseptr;
	scrollend_flag = 1;
	int x_pos, y_pos;
	getsyx(y_pos, x_pos);
	buffer_read();
	char* temp = buffer_read_ptr;
	wmove(display_window, 0, 0);
	wnoutrefresh(display_window);
	for (; *temp != 0; temp++) {
	    if (*temp != '\n') waddch(display_window, (uchar)*temp);
	    else {
	        wclrtoeol(display_window);
		waddch(display_window, '\n');
		wnoutrefresh(display_window);
	    }
	}
	wclrtobot(display_window);
	wnoutrefresh(display_window);
	setsyx(y_pos, x_pos);
	doupdate();
	if (!full_screen_flag) {
	    scrollend_flag = 0;
	    last_scrolladr = 0;
	}
    }
}

void Display_buffer_with_curses::clear_display(WINDOW* display_window) {
    int x_pos, y_pos;
    getsyx(y_pos, x_pos);
    wmove(display_window, 0, 0);
    wnoutrefresh(display_window);
    wclrtobot(display_window);
    wnoutrefresh(display_window);
    setsyx(y_pos, x_pos);
    doupdate();
}

////////////////////////////////////////////////////////////////////////////////////////

int Transmit_buffer::add_letter(char letter) {
  // this method will add a letter to the transmit buffer, for later sending to the Kam.
  // It also keeps track of the number of letters entered in the current line
  // If the buffer is full so the letter can't be added, it returns -1
  // Otherwise it returns the letter added
    if (Qqueue<char>::add_item(letter) == -1) return -1;
    if (letter == '\n') {
        line_lettercount = 0;
	foundCMDinbuffer = FALSE;
    }
    else if (letter == 8 && line_lettercount) line_lettercount--; // we are to send a delete character
	                                                              // in pactor or gtor
    else if (letter == CMDinbuffer) {
        if (!foundCMDinbuffer) foundCMDinbuffer = TRUE;
	else foundCMDinbuffer = FALSE;
    }
    else if (!foundCMDinbuffer) line_lettercount++;
    return (uchar)letter;
}

int Transmit_buffer::extract_letter(void) {
  // this returns -1 if no letter is in the buffer to be extracted
  // otherwise it extracts and removes the letter at the front of the buffer
    if (Qqueue<char>::is_empty()) return -1;
    return (uchar)Qqueue<char>::front(Qqueue_enum::remove);
}

int Transmit_buffer::view_letter(void) {
  // this returns -1 if no letter is in the buffer to be viewed (all have been extracted)
  // otherwise it returns the letter most recently added to the buffer but does not remove it
    if (Qqueue<char>::is_empty()) return -1;
    return (uchar)Qqueue<char>::back(Qqueue_enum::view);
}

int Transmit_buffer::erase_letter(void) {
  // It returns 0 if we are at the beginning of a line so no letter can be erased, or if
  // all letters have been extracted from the buffer, but we are not using 
  // Pactor or Gtor so no delete can be sent to the station with which we are linked
  // (with Pactor or Gtor, can_transmit_delete_flag is TRUE to indicate a delete can
  // be validly transmitted).
  
  // Otherwise, in Pactor and Gtor it returns -1 if the back letter in the buffer
  // can't be removed from the buffer because it has already been extracted and passed
  // to the Kam TNC, and there is something to be deleted on the current line.

  // If a letter has been erased from the buffer in the normal way, it is returned and
  // line_lettercount is decremented

    int temp;
    if (!line_lettercount) temp = 0;
    else if (Qqueue<char>::is_empty()) {
        if (!can_transmit_delete_flag) temp = 0;
	else temp = -1;
    }
    else {
        temp = (uchar)Qqueue<char>::back(Qqueue_enum::remove);
	line_lettercount--;
    }
    return temp;
}

void Transmit_buffer::reset(void) {
    Qqueue<char>::reset();
    line_lettercount = 0;
}

