
/* yaesu.c - common functions for several yaesu programs.

   Copyright (c) 1997 Riku Kalinen, OH2LWO, oh2lwo@sral.fi

   RCS $Header: /home/mole/riku/CVS/yaesu/yaesu.c,v 1.2 1997/12/11 15:02:05 riku Exp $ */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>

#include "yaesu.h"

/* This is generic text-table to binary converter.  This is just shortcut
   because otherwise there would be N+1 almost similar functions. */

#define BIN(FUN, TAB, N) int FUN (char *cp) { \
int j; for (j = 0; j < N; j++) if (strcasecmp (TAB[j], cp) == 0) return j; \
return -1; }

/* This is generic binary to text-table converter, likewise. */

#define STRING(FUN, TAB, N, FMT, ERR) char * FUN (unsigned char f) { \
static char retval[256]; \
if (f >= N) strcpy (retval, ERR); \
else sprintf (retval, FMT, TAB[f]); \
return retval; }

#define BIN_STRING(FUNB, FUNS, TAB, N, FMT, ERR) \
  BIN(FUNB, TAB, N) STRING(FUNS, TAB, N, FMT, ERR)

/* This table maps Yaesu character generator to ASCII. */

static unsigned char yaesu_ascii[256] = {
  '0', '1', '2', '3', '4', '5', '6', '7', /* 0x00-0x07 */
  '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', /* 0x08-0x0f */
  'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', /* 0x10-0x17 */
  'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', /* 0x18-0x1f */
  'W', 'X', 'Y', 'Z', '_', '(', ')', '+', /* 0x20-0x27 */
  '-', '=', '*', '/', 'd', 'y', 's', '|', /* 0x28-0x2f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x30-0x37 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x38-0x3f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x40-0x47 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x48-0x4f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x50-0x57 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x58-0x5f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x60-0x67 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x68-0x6f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x70-0x77 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x78-0x7f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x80-0x87 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x88-0x8f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x90-0x97 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0x98-0x9f */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xa0-0xa7 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xa8-0xaf */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xb0-0xb7 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xb8-0xbf */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xc0-0xc7 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xc8-0xcf */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xd0-0xd7 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xd8-0xdf */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xe0-0xe7 */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xe8-0xef */
  '?', '?', '?', '?', '?', '?', '?', '?', /* 0xf0-0xf7 */
  '?', '?', '?', '?', '?', '?', '?', '?'  /* 0xf8-0xff */
};

/* This table maps ASCII to Yaesu character generator. */

static int ascii_yaesu[256] = {
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x00-0x07 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x08-0x0f */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x10-0x17 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x18-0x1f */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x20-0x27 */
  37, 38, 42, 39, -1, 40, -1, 43, /* 0x28-0x2f */
  0, 1, 2, 3, 4, 5, 6, 7, /* 0x30-0x37 */
  8, 9, -1, -1, -1, 41, -1, -1, /* 0x38-0x3f */
  -1, 10, 11, 12, 13, 14, 15, 16, /* 0x40-0x47 */
  17, 18, 19, 20, 21, 22, 23, 24, /* 0x48-0x4f */
  25, 26, 27, 28, 29, 30, 31, 32, /* 0x50-0x57 */
  33, 34, 35, -1, -1, -1, -1, 36, /* 0x58-0x5f */
  -1, -1, -1, -1, 44, -1, -1, -1, /* 0x60-0x67 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x68-0x6f */
  -1, -1, -1, 46, -1, -1, -1, -1, /* 0x70-0x77 */
  -1, 45, -1, -1, 47, -1, -1, -1, /* 0x78-0x7f */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x80-0x87 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x88-0x8f */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x90-0x97 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0x98-0x9f */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xa0-0xa7 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xa8-0xaf */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xb0-0xb7 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xb8-0xbf */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xc0-0xc7 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xc8-0xcf */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xd0-0xd7 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xd8-0xdf */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xe0-0xe7 */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xe8-0xef */
  -1, -1, -1, -1, -1, -1, -1, -1, /* 0xf0-0xf7 */
  -1, -1, -1, -1, -1, -1, -1, -1  /* 0xf8-0xff */
};

/* Dump Yaesu channel text. */

char *text_string (unsigned char *b)
{
  int i;
  static char retval[64];
  char *cp = retval;
  
  for (i = 0; i < 4; i ++) {
    *cp = yaesu_ascii[*b];
    b ++;
    cp ++;
  }
  *cp = 0;

  return retval;
}

/* Dump Yaesu CWID call sign. */

char *id_string (unsigned char *b)
{
  int i;
  static char retval[64];
  char *cp = retval;
  
  for (i = 0; i < 16; i ++) {
    if (*b == 0xff) break;
    *cp = yaesu_ascii[*b];
    b ++;
    cp ++;
  }
  *cp = 0;

  return retval;
}

/* Convert ASCII character to Yaesu charset. */

int char_bin (char c)
{
  return ascii_yaesu[(unsigned char)c];
}

/* This table maps power level index to text. */

#define N_TXPO 9
static char *txpo[N_TXPO] = {
  "L1", "?1", "L2", "?3", "L3", "?5", "?6", "?7", "HI"
};

BIN(txpo_bin, txpo, N_TXPO)

/* Return string corresponding power level. */

char *txpo_string (unsigned char f)
{
  static char retval[4];

  if (f >= N_TXPO)
    strcpy (retval, "??");
  else
    sprintf (retval, "%2.2s", txpo[f]);
  
  return retval;
  
}

/* This table maps channel step index to text. */

#define N_STEP 7
static char *step[N_STEP] = {
  "5", "10", "12.5", "15", "20", "25", "50"
};

BIN(step_bin, step, N_STEP)

/* Return string corresponding channel step. */

char *step_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_STEP)
    strcpy (retval, "????");
  else
    sprintf (retval, "%4.4s", step[f]);
  
  return retval;
  
}

#define N_MODE 4
static char *mode[N_MODE] = {
  "NFM", "AM", "WFM", "??3"
};

/* Return index to given mode string or -1 on error. */

BIN(mode_bin, mode, N_MODE)

char *mode_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_MODE)
    strcpy (retval, "???");
  else
    sprintf (retval, "%3.3s", mode[f]);
  
  return retval;
  
}

/* Convert frequency to BCD format. */

int freq_bin (unsigned char *base, char *freq_s)
{
  unsigned char d[6] = { 0, 0, 0, 0, 0, 0 };
  char tmp[32];
  char *cp;
  int len, j;

  len = strlen (freq_s);

  switch (len) {
  case 5: strcpy (tmp, "00"); strcat (tmp, freq_s); break;
  case 6: strcpy (tmp, "0"); strcat (tmp, freq_s); break;
  case 7: strcpy (tmp, freq_s); break;
  default: return -1;
  }

  for (cp = tmp, j = 0; *cp; cp ++, j++) {
    if (*cp == '.') {
      if (strlen (cp) != 4) return -1;
      j --;
      continue;
    }
    if (! isdigit (*cp)) return -1;
    d[j] = *cp - '0';
  }

  base[0] = ((d[0] & 0xf) << 4) | (d[1] & 0xf);
  base[1] = ((d[2] & 0xf) << 4) | (d[3] & 0xf);
  base[2] = ((d[4] & 0xf) << 4) | (d[5] & 0xf);

  return 0;
}

/* Dump BCD-coded frequency or offset.
   
   This is really kludge solution, more elaborate way would be converting
   BCD to integer and then to double, but this works as well (for this
   application anyway). --riku */

char *freq_string (unsigned char *b)
{
  static char retval[16];

  sprintf (retval, "%02x", b[0]);
  sprintf (retval + 2, "%02x", b[1]);
  sprintf (retval + 3, "%02x", b[1]);
  sprintf (retval + 5, "%02x", b[2]);
  retval[3] = '.';
  if (retval[0] == '0') {
    retval[0] = ' ';
    if (retval[1] == '0') retval[1] = ' ';
  }
  
  return retval;
}

/* This table maps TOT indices to text. */

#define N_TOT 5
static char *tot[N_TOT] = { "off", "1min", "2.5min", "5min", "10min" };

BIN(tot_bin, tot, N_TOT)

char *tot_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_TOT)
    strcpy (retval, "???");
  else
    strcpy (retval, tot[f]);

  return retval;
}

static char dtmf_char[16] = {
  '0', '1', '2', '3', '4', '5', '6', '7',
  '8', '9', 'A', 'B', 'C', 'D', '*', '#'
};

int dtmf_bin (char c)
{
  int j;

  for (j = 0; j < 16; j++)
    if (dtmf_char[j] == toupper (c))
      return j;
  
  return -1;
}

static char *carr5sec[2] = { "5sec", "carr" };
BIN(carr5sec_bin, carr5sec, 2)

static char *revhome[2] = { "home", "rev" };
BIN(revhome_bin, revhome, 2)

static char *montcal[2] = { "mon", "tcal" };
BIN(montcal_bin, montcal, 2)

/* This table maps Yaesu VX_1 character generator to ASCII. */

#define N_VX1_ASCII 60

static unsigned char vx1_ascii[N_VX1_ASCII] = {
  '0', '1', '2', '3', '4', '5', '6', '7', /* 0x00-0x07 */
  '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', /* 0x08-0x0f */
  'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', /* 0x10-0x17 */
  'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', /* 0x18-0x1f */
  'W', 'X', 'Y', 'Z', '_', '(', ')', '+', /* 0x20-0x27 */
  '-', '=', '*', '/', 'd', 'y', 's', '|', /* 0x28-0x2f */
  'i', '?', 'x', '&', 'u', '$', 'p', '\\', /* 0x30-0x37 */
  'l', '<', '>', 'o'
};

/* Convert ASCII character to Yaesu charset. */

int vx1_char_bin (char c)
{
  int i;

  for (i = 0; i < N_VX1_ASCII; i ++)
    if ((unsigned char) c == vx1_ascii[i])
      return i;

  return -1;
}

/* Dump Yaesu CWID call sign. */

char *vx1_id_string (unsigned char *b)
{
  int i;
  static char retval[64];
  char *cp = retval;
  
  for (i = 0; i < 9; i ++) {
    if (*b == 0x3d) break;
    *cp = vx1_ascii[*b];
    b ++;
    cp ++;
  }
  *cp = 0;

  return retval;
}

/* Dump Yaesu channel text. */

char *vx1_text_string (unsigned char *b)
{
  int i;
  static char retval[64];
  char *cp = retval;
  
  for (i = 0; i < 6; i ++) {
    if (*b == 0x3f)
      break;
    *cp = vx1_ascii[*b];
    b ++;
    cp ++;
  }
  *cp = 0;

  return retval;
}

/* Convert frequency to BCD format. */

int vx1_freq_bin (unsigned char *base, char *freq_s)
{
  unsigned char d[6] = { 0, 0, 0, 0, 0, 0 };
  char tmp[32];
  char *cp;
  int len, j;

  len = strlen (freq_s);

  switch (len) {
  case 5: strcpy (tmp, "00"); strcat (tmp, freq_s); break;
  case 6: strcpy (tmp, "0"); strcat (tmp, freq_s); break;
  case 7: strcpy (tmp, freq_s); break;
  default: return -1;
  }

  for (cp = tmp, j = 0; *cp; cp ++, j++) {
    if (*cp == '.') {
      if (strlen (cp) != 4) return -1;
      j --;
      continue;
    }
    if (! isdigit (*cp)) return -1;
    d[j] = *cp - '0';
  }

  base[2] = ((d[0] & 0xf) << 4) | (d[1] & 0xf);
  base[1] = ((d[2] & 0xf) << 4) | (d[3] & 0xf);
  base[0] = ((d[4] & 0xf) << 4) | (d[5] & 0xf);

  return 0;
}

/* Dump BCD-coded frequency or offset.
   
   This is really kludge solution, more elaborate way would be converting
   BCD to integer and then to double, but this works as well (for this
   application anyway). --riku */

char *vx1_freq_string (unsigned char *b)
{
  static char retval[16];

  sprintf (retval, "%02x", b[2]);
  sprintf (retval + 2, "%02x", b[1]);
  sprintf (retval + 3, "%02x", b[1]);
  sprintf (retval + 5, "%02x", b[0]);
  retval[3] = '.';
  if (retval[0] == '0') {
    retval[0] = ' ';
    if (retval[1] == '0') retval[1] = ' ';
  }
  
  return retval;
}

/* Broadcast frequency; 0x00 - 500 kHz to 0xfe - 1.7 MHz */

char *vx1_bcfreq_string (unsigned char *b)
{
  static char retval[16];
  double dstep = 1.2 / 254.0;
  double freq = 0.5 + (((double) (*b)) * dstep);

  sprintf (retval, "%5.3f", freq);
  
  return retval;
}

int vx1_bcfreq_bin (char *freq_s)
{
  int mhz, khz;
  double dstep = 1.2 / 254.0;
  int ret;

  mhz = atoi (freq_s);
  khz = atoi (freq_s + 2);

  khz += mhz * 1000;

  if ((khz < 500) || (khz > 1700)) return -1;

  khz -= 500;

  ret = (int) ((((double) khz / 1000.0) / dstep) + 0.5);

  return ret;

}

static char *onoff[2] = { "off", "on" };
BIN_STRING(onoff_bin, onoff_string, onoff, 2, "%s", "???")

#define VX1_N_SMTMD 2
static char *vx1_smtmd[VX1_N_SMTMD] = { "single", "conti" };
BIN_STRING(vx1_smtmd_bin, vx1_smtmd_string, vx1_smtmd, VX1_N_SMTMD,
	   "%s", "?")

#define VX1_N_RESUME 2
static char *vx1_resume[VX1_N_RESUME] = { "5sec", "carre" };
BIN_STRING(vx1_resume_bin, vx1_resume_string, vx1_resume, VX1_N_RESUME,
	   "%s", "?")

#define VX1_N_MONTC 2
static char *vx1_montc[VX1_N_MONTC] = { "moni", "tcall" };
BIN_STRING(vx1_montc_bin, vx1_montc_string, vx1_montc, VX1_N_MONTC,
	   "%s", "?")

#define VX1_N_SKIP 2
static char *vx1_skip[VX1_N_SKIP] = { "-", "S" };
BIN_STRING(vx1_skip_bin, vx1_skip_string, vx1_skip, VX1_N_SKIP, "%s", "?")

#define VX1_N_CLKSFT 2
static char *vx1_clksft[VX1_N_CLKSFT] = { "-", "C" };
BIN_STRING(vx1_clksft_bin, vx1_clksft_string, vx1_clksft, VX1_N_CLKSFT,
	   "%s", "?")

#define VX1_N_DTMFM 16
static char *vx1_dtmfm[VX1_N_DTMFM] = {
  "dtmf-1", "dtmf-2", "dtmf-3", "dtmf-4", "dtmf-5", "dtmf-6", "dtmf-7",
  "dtmf-8", "manu", "????9", "????a", "????b", "????c", "????d", "????e",
  "????f"
};
BIN_STRING(vx1_dtmfm_bin, vx1_dtmfm_string, vx1_dtmfm, VX1_N_DTMFM, "%s",
	   "?????")

#define VX1_N_DIALM 2
static char *vx1_dialm[VX1_N_DIALM] = { "dial", "volsql" };
BIN_STRING(vx1_dialm_bin, vx1_dialm_string, vx1_dialm, VX1_N_DIALM, "%s",
	   "????")

#define VX1_N_ATMD 2
static char *vx1_atmd[VX1_N_ATMD] = { "manual", "auto" };
BIN_STRING(vx1_atmd_bin, vx1_atmd_string, vx1_atmd, VX1_N_ATMD, "%s", "????")

#define VX1_N_MODE 3
static char *vx1_mode[VX1_N_MODE] = { "FM-N", "FM-W", "AM" };
BIN_STRING(vx1_mode_bin, vx1_mode_string, vx1_mode, VX1_N_MODE, "%4.4s",
	   "????")

#define VX1_N_LAMP 3
static char *vx1_lamp[VX1_N_LAMP] = { "key", "5sec", "tgl" };
BIN_STRING(vx1_lamp_bin, vx1_lamp_string, vx1_lamp, VX1_N_LAMP, "%s", "????")

#define VX1_N_GROUP 2
static char *vx1_group[VX1_N_GROUP] = {
  "grp-2", "grp-1"
};

BIN(vx1_group_bin, vx1_group, VX1_N_GROUP)

char *vx1_group_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_GROUP)
    strcpy (retval, "????");
  else
    sprintf (retval, "%s", vx1_group[f]);
  
  return retval;
  
}

/* This table maps CTCSS indices to corresponding tones. */

#define N_CTCSS 39
static char *ctcss[N_CTCSS] = {
  "67.0", "69.3", "71.9", "74.4", "77.0", "79.7", "82.5", "85.4",
  "88.5", "91.5", "94.8", "97.4", "100.0", "103.5", "107.2", "110.9",
  "114.8", "118.8", "123.0", "127.3", "131.8", "136.5", "141.3", "146.2",
  "151.4", "156.7", "162.2", "167.9", "173.8", "179.9", "186.2", "192.8",
  "203.5", "210.7", "218.1", "225.7", "233.6", "241.8", "250.3"
};

BIN(ctcss_bin, ctcss, N_CTCSS)

/* Return string corresponding ctcss index. */

char *ctcss_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_CTCSS)
    strcpy (retval, "?????");
  else
    sprintf (retval, "%5.5s", ctcss[f]);
  
  return retval;
  
}

/* This table maps DCS indices to corresponding tones. */

#define N_DCS 104
static int dcs[N_DCS] = {
  23, 25, 26, 31, 32, 36, 43, 47,
  51, 53, 54, 65, 71, 72, 73, 74,
  114, 115, 116, 122, 125, 131, 132, 134,
  143, 145, 152, 155, 156, 162, 165, 172,
  174, 205, 212, 223, 225, 226, 243, 244,
  245, 246, 251, 252, 255, 261, 263, 265,
  266, 271, 274, 306, 311, 315, 325, 331,
  332, 343, 346, 351, 356, 364, 365, 371,
  411, 412, 413, 423, 431, 432, 445, 446,
  452, 454, 455, 462, 464, 465, 466, 503,
  506, 516, 523, 526, 532, 546, 565, 606,
  612, 624, 627, 631, 632, 654, 662, 664,
  703, 712, 723, 731, 732, 734, 743, 754
};

/* Return index corresponding DCS string, or -1 on error */

int dcs_bin (char *cp)
{
  int i = atoi (cp);
  int j;

  if (i < 0) return -1;

  for (j = 0; j < N_DCS; j ++)
    if (dcs[j] == i)
      return j;

  return -1;
}

/* Return string corresponding dcs tone index. */

char *dcs_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_DCS)
    strcpy (retval, "???");
  else
    sprintf (retval, "%03d", dcs[f]);
  
  return retval;
  
}

/* This table maps selective calling index to text. */

#define N_SELCAL 4
static char *selcal[N_SELCAL] = {
  "-", "E", "E+D", "DCS"
};

BIN(selcal_bin, selcal, N_SELCAL)

/* Return string corresponding selective call type. */

char *selcal_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_SELCAL)
    strcpy (retval, "???");
  else
    sprintf (retval, "%3.3s", selcal[f]);
  
  return retval;
  
}

/* This table maps code memory index to text. */

#define N_SHIFT 4
static char *shift[N_SHIFT] = {
  "SI", "-R", "+R", "+-"
};

BIN(shift_bin, shift, N_SHIFT)

/* Return string corresponding repeater shift. */

char *shift_string (unsigned char f)
{
  static char retval[4];

  if (f >= N_SHIFT)
    strcpy (retval, "??");
  else
    sprintf (retval, "%2.2s", shift[f]);
  
  return retval;
  
}

/* This table maps code memory index to text. */

#define N_PAGING 4
static char *paging[N_PAGING] = {
  "-", "PG", "TP", "CO"
};

BIN(paging_bin, paging, N_PAGING)

/* Return string corresponding code mem. */

char *paging_string (unsigned char f)
{
  static char retval[4];

  if (f >= N_PAGING)
    strcpy (retval, "??");
  else
    sprintf (retval, "%2.2s", paging[f]);
  
  return retval;
  
}

/* This table maps code memory index to text. */

#define N_CODE 8
static char *code[N_CODE] = {
  "C", "P", "1", "2", "3", "4", "5", "6"
};

BIN(code_bin, code, N_CODE);

/* Return string corresponding code mem. */

char *code_string (unsigned char f)
{
  static char retval[4];

  if (f >= N_CODE)
    strcpy (retval, "?");
  else
    sprintf (retval, "%1.1s", code[f]);
  
  return retval;
  
}

/* This table maps band index to text. */

#define VX1_N_BAND 9
static char *vx1_band[VX1_N_BAND] = {
  "BCBAND", "FM", "AIR", "V-HAM", "VHF-TV", "ACT1", "U-HAM", "UHF-TV", "ACT2"
};

BIN(vx1_band_bin, vx1_band, VX1_N_BAND)

char *vx1_band_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_BAND)
    strcpy (retval, "????");
  else
    sprintf (retval, "%s", vx1_band[f]);
  
  return retval;
  
}

/* This table maps channel step index to text. */

#define VX1_N_STEP 8
static char *vx1_step[VX1_N_STEP] = {
  "5", "10", "12.5", "15", "20", "25", "50", "100"
};

BIN(vx1_step_bin, vx1_step, VX1_N_STEP)

/* Return string corresponding channel step. */

char *vx1_step_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_STEP)
    strcpy (retval, "????");
  else
    sprintf (retval, "%4.4s", vx1_step[f]);
  
  return retval;
  
}

/* This table maps power level index to text. */

#define VX1_N_TXPWR 2
static char *vx1_txpwr[VX1_N_TXPWR] = {
  "LO", "HI"
};

BIN(vx1_txpwr_bin, vx1_txpwr, VX1_N_TXPWR)

/* Return string corresponding power level. */

char *vx1_txpwr_string (unsigned char f)
{
  static char retval[4];

  if (f >= VX1_N_TXPWR)
    strcpy (retval, "??");
  else
    sprintf (retval, "%2.2s", vx1_txpwr[f]);
  
  return retval;
  
}

/* Name flag */

#define VX1_N_NAMEFLAG 2
static char *vx1_nameflag[VX1_N_NAMEFLAG] = { "FRE", "ALP" };
BIN_STRING(vx1_nameflag_bin, vx1_nameflag_string, vx1_nameflag,
	   VX1_N_NAMEFLAG, "%3.3s", "???")

/* This table maps channel flags to text. */

#define N_FLAGS 8
static char *flags[N_FLAGS] = { "-", "UM", "M", "U", "?4", "USM", "SM", "US" };

BIN(flag_bin, flags, N_FLAGS)

/* Return string corresponding channel flags. */

char *flag_string (unsigned char f)
{
  static char retval[4];
  
  if (f >= N_FLAGS)
    strcpy (retval, "??");
  else
    sprintf (retval, "%3.3s", flags[f]);
  
  return retval;
  
}

/* This maps SQL status to text. */

#define N_SQL 16
static char *sql[N_SQL] = {
  "0", "1", "2", "3", "4", "5", "6", "7", 
  "8", "9", "10", "11", "12", "13", "14", "15"
};

BIN(sql_bin, sql, N_SQL)

char *sql_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_SQL)
    strcpy (retval, "??");
  else
    strcpy (retval, sql[f]);

  return retval;
}

/* This maps SQL status to text. */

#define VX1_N_SQL 12
static char *vx1_sql[VX1_N_SQL] = {
  "aut", "opn", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10"
};

BIN(vx1_sql_bin, vx1_sql, VX1_N_SQL)

char *vx1_sql_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_SQL)
    strcpy (retval, "???");
  else
    strcpy (retval, vx1_sql[f]);

  return retval;
}

/* This maps VOL status to text. */

#define VX1_N_VOL 32
static char *vol[VX1_N_VOL] = {
  "mute", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13",
  "14", "15", "16", "17", "18", "19", "20", "21", "22", "23", "24", "25",
  "26", "27", "28", "29", "30", "31"
};

BIN(vx1_vol_bin, vol, VX1_N_VOL)

char *vx1_vol_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_VOL)
    strcpy (retval, "????");
  else
    strcpy (retval, vol[f]);

  return retval;
}

#define N_WSQL 16
static char *wsql[16] = {
  "0", "1", "2", "3", "4", "5", "6", "7", 
  "8", "9", "10", "11", "12", "13", "14", "15"
};

BIN(wsql_bin, wsql, N_WSQL)

char *wsql_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_WSQL)
    strcpy (retval, "??");
  else
    strcpy (retval, wsql[f]);

  return retval;
}

/* This maps ARTS status to text. */

#define N_ARTS 4
static char *arts[N_ARTS] = { "off", "rx", "tx", "trx" };

BIN(arts_bin, arts, N_ARTS)

char *arts_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_ARTS)
    strcpy (retval, "???");
  else
    strcpy (retval, arts[f]);

  return retval;
}

/* This maps ARTS/BEEP index to text. */

#define N_ARTS_BEEP 3
static char *arts_beep[N_ARTS_BEEP] = { "off", "rang", "all" };

BIN(arts_beep_bin, arts_beep, N_ARTS_BEEP)

char *arts_beep_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_ARTS_BEEP)
    strcpy (retval, "????");
  else
    strcpy (retval, arts_beep[f]);

  return retval;
}

/* This table maps PAGE-SPED indices to corresponding texts. */

#define N_PAGE_SPED 2
static char *page_sped[N_PAGE_SPED] = { "50ms", "100ms" };

BIN(page_sped_bin, page_sped, N_PAGE_SPED)

char *page_sped_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_PAGE_SPED)
    strcpy (retval, "???");
  else
    strcpy (retval, page_sped[f]);

  return retval;
}

/* This table maps PAGE-DLAY indices to texts. */

#define N_PAGE_DLAY 4
static char *page_dlay[N_PAGE_DLAY] = { "250ms", "450ms", "750ms", "1000ms" };

BIN(page_dlay_bin, page_dlay, N_PAGE_DLAY)

char *page_dlay_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_PAGE_DLAY)
    strcpy (retval, "????");
  else
    strcpy (retval, page_dlay[f]);

  return retval;
}

/* This table maps PAGE-BELL and BELL indices to texts. */

#define N_BELL 6
static char *bell[N_BELL] = { "off", "1", "3", "5", "8", "rpt" };

BIN(bell_bin, bell, N_BELL)

char *bell_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_BELL)
    strcpy (retval, "???");
  else
    strcpy (retval, bell[f]);

  return retval;
}

/* This table maps PAGE-ASBK indices to texts. */

#define N_PAGE_ASBK 3
static char *page_asbk[N_PAGE_ASBK] = { "off", "ans", "for" };

BIN(page_asbk_bin, page_asbk, N_PAGE_ASBK)

char *page_asbk_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_PAGE_ASBK)
    strcpy (retval, "???");
  else
    strcpy (retval, page_asbk[f]);

  return retval;
}

/* This table maps SUB indices to text. */

#define N_SUB 3
static char *sub[3] = { "off", "on", "dc" };

/* Return index to given sub display string or -1 on error. */

BIN(sub_bin, sub, N_SUB)

char *sub_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_SUB)
    strcpy (retval, "???");
  else
    strcpy (retval, sub[f]);

  return retval;
}

/* This table maps APO indices to text. */

#define N_APO 6
static char *apo[N_APO] = { "off", "0.5h", "1h", "3h", "5h", "8h" };

BIN(apo_bin, apo, N_APO)

char *apo_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_APO)
    strcpy (retval, "???");
  else
    strcpy (retval, apo[f]);

  return retval;
}

/* This table maps TOT indices to text. */

#define VX1_N_TOT 5
static char *vx1_tot[VX1_N_TOT] = { "off", "1min", "2min", "5min", "10min" };

BIN(vx1_tot_bin, vx1_tot, VX1_N_TOT)

char *vx1_tot_string (unsigned char f)
{
  static char retval[8];

  if (f >= VX1_N_TOT)
    strcpy (retval, "???");
  else
    strcpy (retval, vx1_tot[f]);

  return retval;
}

/* This table maps LOCK indices to text. */
#define N_LOCK 7
static char *lock[N_LOCK] = {
  "kl", "dl", "kl+dl", "pl", "kl+pl", "dl+pl", "kl+dl+pl"
};
BIN_STRING(lock_bin, lock_string, lock, N_LOCK, "%s", "????????")

/* This table maps RSAV indices to text. */
#define N_RSAV 6
static char *rsav[N_RSAV] = { "off", "0.2s", "0.3s", "0.5s", "1.0s", "2.0s" };
BIN_STRING(rsav_bin, rsav_string, rsav, N_RSAV, "%s", "???")

/* This table maps LAMP indices to text. */

#define N_LAMP 3
static char *lamp[3] = { "5sec", "key", "tgl" };

BIN(lamp_bin, lamp, N_LAMP)

char *lamp_string (unsigned char f)
{
  static char retval[8];

  if (f >= N_LAMP)
    strcpy (retval, "????");
  else
    strcpy (retval, lamp[f]);

  return retval;
}

int vx1_dtmf_bin (char c)
{
  int j;

  if (c == '-') return 0x28;
  
  for (j = 0; j < 16; j++)
    if (dtmf_char[j] == toupper (c))
      return j;
  
  return -1;
}

/* This table maps AD memory indices to memory names. */

char *admem[8] = {
  " AD1", " AD2", " AD3", " AD4", " AD5", " AD6", " AD7", " AD8"
};

/* This table maps DTMF memory indices to memory names. */

char *dtmfmem[8] = {
  "  DC", "  DP", "  D1", "  D2", "  D3", "  D4", "  D5", "  D6"
};

/* This table maps VFO indices to names. */

static char *vfo[10] = {
  "A145", "A220", "A380", "A430", "A800",
  "B145", "B220", "B380", "B430", "B800"
};

BIN(vfo_bin, vfo, 10)

char *vfo_string (int i)
{
  static char retval[8];

  if ((i >= 10) || (i < 0))
    strcpy (retval, "????");
  else
    strcpy (retval, vfo[i]);

  return retval;
}

/* This table maps channel indices (starting at 0) to channel names. */

char *channels[111] = {
  "M1", "M2", "M3", "M4", "M5", "M6", "M7", "M8", "M9",
  "M10", "M11", "M12", "M13", "M14", "M15", "M16", "M17", "M18",
  "M19", "M20", "M21", "M22", "M23", "M24", "M25", "M26", "M27",
  "M28", "M29", "M30", "M31", "M32", "M33", "M34", "M35", "M36",
  "M37", "M38", "M39", "M40", "M41", "M42", "M43", "M44", "M45",
  "M46", "M47", "M48", "M49", "M50", "M51", "M52", "M53", "M54",
  "M55", "M56", "M57", "M58", "M59", "M60", "M61", "M62", "M63",
  "M64", "M65", "M66", "M67", "M68", "M69", "M70", "M71", "M72",
  "M73", "M74", "M75", "M76", "M77", "M78", "M79", "M80", "M81",
  "M82", "M83", "M84", "M85", "M86", "M87", "M88", "M89", "M90",
  "M91", "M92", "M93", "M94", "M95", "M96", "M97", "M98", "M99",
  "M100", "ML1", "MU1", "ML2", "MU2", "ML3", "MU3", "ML4", "MU4",
  "ML5", "MU5", "????"
};

BIN(chname_bin, channels, 111)

/* VX-1 channel names */

#define VX1_N_CHANNELS_CG2 170
static char *vx1_channels_cg2[VX1_N_CHANNELS_CG2] = {
  "H-FM", "H-AIR", "H-V-HA", "H-V-TV", "H-ACT1", "H-U-HA", "H-U-TV", "H-ACT2",
  "1", "2", "3", "4", "5", "6", "7", "8", "9", "10",
  "11", "12", "13", "14", "15", "16", "17", "18", "19", "20",
  "21", "22", "23", "24", "25", "26", "27", "28", "29", "30",
  "31", "32", "33", "34", "35", "36", "37", "38", "39", "40",
  "41", "42", "43", "44", "45", "46", "47", "48", "49", "50",
  "51", "52", "53", "54", "55", "56", "57", "58", "59", "60",
  "61", "62", "63", "64", "65", "66", "67", "68", "69", "70",
  "71", "72", "73", "74", "75", "76", "77", "78", "79", "80",
  "81", "82", "83", "84", "85", "86", "87", "88", "89", "90",
  "91", "92", "93", "94", "95", "96", "97", "98", "99",
  "100", "101", "102", "103", "104", "105", "106", "107", "108", "109",
  "110", "111", "112", "113", "114", "115", "116", "117", "118", "119",
  "120", "121", "122", "123", "124", "125", "126", "127", "128", "129",
  "130", "131", "132", "133", "134", "135", "136", "137", "138", "139",
  "140", "141", "142",
  "L1", "U1", "L2", "U2", "L3", "U3", "L4", "U4", "L5", "U5", 
  "L6", "U6", "L7", "U7", "L8", "U8", "L9", "U9", "L10", "U10"
};
BIN_STRING(vx1_chan_cg2_bin, vx1_chan_cg2_string, vx1_channels_cg2,
	   VX1_N_CHANNELS_CG2, "%s", "???");

#define VX1_N_CHANNELS_CG1 80
static char *vx1_channels_cg1[VX1_N_CHANNELS_CG1] = {
  "H-FM", "H-AIR", "H-V-HA", "H-V-TV", "H-ACT1", "H-U-HA", "H-U-TV", "H-ACT2",
  "1", "2", "3", "4", "5", "6", "7", "8", "9", "10",
  "11", "12", "13", "14", "15", "16", "17", "18", "19", "20",
  "21", "22", "23", "24", "25", "26", "27", "28", "29", "30",
  "31", "32", "33", "34", "35", "36", "37", "38", "39", "40",
  "41", "42", "43", "44", "45", "46", "47", "48", "49", "50",
  "51", "52",
  "L1", "U1", "L2", "U2", "L3", "U3", "L4", "U4", "L5", "U5", 
  "L6", "U6", "L7", "U7", "L8", "U8", "L9", "U9", "L10", "U10"
};
BIN_STRING(vx1_chan_cg1_bin, vx1_chan_cg1_string, vx1_channels_cg1,
	   VX1_N_CHANNELS_CG1, "%s", "???");

#define VX1_N_CHANNELS_BC 10
static char *vx1_channels_bc[VX1_N_CHANNELS_BC] = {
  "H", "1", "2", "3", "4", "5", "6", "7", "8", "9"
};
BIN_STRING(vx1_chan_bc_bin, vx1_chan_bc_string, vx1_channels_bc,
	   VX1_N_CHANNELS_BC, "%s", "???");

int repeat_read (int fd, unsigned char *buf, size_t size, int incr)
{
  int total = 0;
  int zeroturns = 0;
  int ret;
  size_t now;

  for (;;) {
    now = size > incr ? incr : size;

    ret = read (fd, buf, now);
    
    if (ret < 0) {
      return -1;
    }
    
    if (ret == 0) {
      zeroturns ++;
      if (zeroturns > 10) {
#define DEBUG 0
#if DEBUG
	printf ("%d- ", total); fflush (stdout);
#endif
	return total;
      }
      continue;
    }

    zeroturns = 0;
    total += ret;
    size -= ret;
    if (size == 0) {
#if DEBUG
      printf ("%d+ ", total); fflush (stdout);
#endif
      return total;
    }
    buf += ret;

#if DEBUG
    printf ("%d ", total); fflush (stdout);
#endif
    
  } /* for */
}

int read_save_file (int fd, unsigned char *b1, unsigned char *b2,
		    unsigned char *b3, unsigned char *b4, unsigned char *b5,
		    unsigned char *b6, unsigned char *b7, unsigned char *b8)
{
  int ret;
  
  if ((ret = read (fd, b1, 10)) != 10) {
    perror ("read b1");
    printf ("read returned %d\n", ret);
    return -1;
  }
  if ((ret = read (fd, b2, 16)) != 16) {
    perror ("read b2");
    printf ("read returned %d\n", ret);
    return -2;
  }
  if ((ret = read (fd, b3, 112)) != 112) {
    perror ("read b3");
    printf ("read returned %d\n", ret);
    return -3;
  }
  if ((ret = read (fd, b4, 16)) != 16) {
    perror ("read b4");
    printf ("read returned %d\n", ret);
    return -4;
  }
  if ((ret = read (fd, b5, 16)) != 16) {
    perror ("read b5");
    printf ("read returned %d\n", ret);
    return -5;
  }
  if ((ret = read (fd, b6, 1776)) != 1776) {
    perror ("read b6");
    printf ("read returned %d\n", ret);
    return -6;
  }
  if ((ret = read (fd, b7, 1776)) != 1776) {
    perror ("read b7");
    printf ("read returned %d\n", ret);
    return -7;
  }
  if ((ret = read (fd, b8, 1)) != 1) {
    perror ("read b8");
    printf ("read returned %d\n", ret);
    return -8;
  }

  return 0;
}

int write_save_file (int fd, unsigned char *b1, unsigned char *b2,
		     unsigned char *b3, unsigned char *b4, unsigned char *b5,
		     unsigned char *b6, unsigned char *b7, unsigned char *b8)
{
  int ret;
  
  if ((ret = write (fd, b1, 10)) != 10) {
    perror ("write b1");
    printf ("write returned %d\n", ret);
    return -1;
  }
  if ((ret = write (fd, b2, 16)) != 16) {
    perror ("write b2");
    printf ("write returned %d\n", ret);
    return -2;
  }
  if ((ret = write (fd, b3, 112)) != 112) {
    perror ("write b3");
    printf ("write returned %d\n", ret);
    return -3;
  }
  if ((ret = write (fd, b4, 16)) != 16) {
    perror ("write b4");
    printf ("write returned %d\n", ret);
    return -4;
  }
  if ((ret = write (fd, b5, 16)) != 16) {
    perror ("write b5");
    printf ("write returned %d\n", ret);
    return -5;
  }
  if ((ret = write (fd, b6, 1776)) != 1776) {
    perror ("write b6");
    printf ("write returned %d\n", ret);
    return -6;
  }
  if ((ret = write (fd, b7, 1776)) != 1776) {
    perror ("write b7");
    printf ("write returned %d\n", ret);
    return -7;
  }
  if ((ret = write (fd, b8, 1)) != 1) {
    perror ("write b8");
    printf ("write returned %d\n", ret);
    return -8;
  }

  return 0;
}

int read_vx1_save_file (int fd, unsigned char *b1, int size)
{
  int ret;
  
  if ((ret = read (fd, b1, (size_t) size)) != size) {
    perror ("read b1");
    printf ("read returned %d\n", ret);
    return -1;
  }

  return 0;
}

int write_vx1_save_file (int fd, unsigned char *b1, int size)
{
  int ret;
  
  if ((ret = write (fd, b1, (size_t) size)) != size) {
    perror ("write b1");
    printf ("write returned %d\n", ret);
    return -1;
  }

  return 0;
}

unsigned char calculate_cksum (unsigned char *b1, unsigned char *b2,
			       unsigned char *b3, unsigned char *b4,
			       unsigned char *b5, unsigned char *b6,
			       unsigned char *b7)
{
  unsigned char cksum = 0;
  int i;
  
  for (i = 0; i < 10; i ++) cksum += b1[i];
  for (i = 0; i < 16; i ++) cksum += b2[i];
  for (i = 0; i < 112; i ++) cksum += b3[i];
  for (i = 0; i < 16; i ++) cksum += b4[i];
  for (i = 0; i < 16; i ++) cksum += b5[i];
  for (i = 0; i < 1776; i ++) cksum += b6[i];
  for (i = 0; i < 1776; i ++) cksum += b7[i];

  return cksum;
}

unsigned char calculate_vx1_cksum (unsigned char *b1, int size)
{
  unsigned char cksum = 0;
  int i;
  
  for (i = 0; i < size; i ++) cksum += b1[i];

  cksum = 0xf0 - cksum;

  return cksum;
}

unsigned char calculate_vx1_cksum_2 (unsigned char *b1, int size)
{
  unsigned char cksum = 0;
  int i;
  
  for (i = 0; i < size; i ++) cksum += b1[i];

  cksum = cksum + 0x10;

  return cksum;
}

/* Do standard hex dump, end it with \n */

void hexdump (unsigned char *b, int s)
{
  int i;
  for (i = 0; i < s; i ++) {
    printf ("%02x ", *b);
    b ++;
  }
  printf ("\n");
}

void hexdump_no_n (unsigned char *b, int s)
{
  int i;
  for (i = 0; i < s; i ++) {
    printf ("%02x ", *b);
    b ++;
  }
}

/* Do wide hex dump, end it with \n */

void wide_hexdump (unsigned char *b, int s)
{
  int i;
  for (i = 0; i < s; i ++) {
    printf ("%02x  ", *b);
    b ++;
  }
  printf ("\n");
}

/* Set one bit. */

void setbit (unsigned char *addr, int bit, int val)
{
  unsigned char uc = *addr;

  if (val == 1)
    uc |= 1 << bit;
  else
    uc &= ~ (1 << bit);

  *addr = uc;

  return;
}

