/******************************************************************************/
/*                                                                            */
/*  Title       : makeamsat.c                                                 */
/*  Author      : Manfred Bester                                              */
/*  Date        : 24Feb92                                                     */
/*  Last change : 15Mar95                                                     */
/*                                                                            */
/*  Synopsis    : This program reads a file with Keplerian elements in the    */
/*                NORAD 2-line format and generates one in the AMSAT format.  */
/*                                                                            */
/*  Input file  : tle.dat or sts-47.dat (for example)                         */
/*  Output file : amsat.dat                                                   */
/*                                                                            */
/*                                                                            */
/*  SatTrack is Copyright (c) 1992, 1993, 1994, 1995 by Manfred Bester.       */
/*  All Rights Reserved.                                                      */
/*                                                                            */
/*  Permission to use, copy, and distribute SatTrack and its documentation    */
/*  in its entirety for educational, research and non-profit purposes,        */
/*  without fee, and without a written agreement is hereby granted, provided  */
/*  that the above copyright notice and the following three paragraphs appear */
/*  in all copies. SatTrack may be modified for personal purposes, but        */
/*  modified versions may NOT be distributed without prior consent of the     */
/*  author.                                                                   */
/*                                                                            */
/*  Permission to incorporate this software into commercial products may be   */
/*  obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,    */
/*  Berkeley, CA 94709, USA. Note that distributing SatTrack 'bundled' in     */
/*  with ANY product is considered to be a 'commercial purpose'.              */
/*                                                                            */
/*  IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, */
/*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF   */
/*  THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED  */
/*  OF THE POSSIBILITY OF SUCH DAMAGE.                                        */
/*                                                                            */
/*  THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT      */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A   */
/*  PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"      */
/*  BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, */
/*  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                  */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef STDLIB
#include <stdlib.h>
#endif

#include "sattrack.h"

#ifdef HPTERM
#include "hpterm.h"
#else
#include "vt100.h"
#endif

extern double getElement();
extern void   mGets();

void main()

{
    double epochDay, decayRate, inclination, raan, eccentricity;
    double argPerigee, meanAnomaly, meanMotion, epochYear;

    int    lineNum, lineNum1, lineNum2, satNum, elementSet, orbitNum;
    int    i, nSat, checkValue, checkSum, checkSumTotal, ephemerisType;

    char   satName[100], line1[100], line2[100], data[100], tle[100];
    char   inputData[20], inputFile[100], outputFile[100], str[80], strng[10];
    char   sysComm[100];
    char   *strpHome, *getenv();

    FILE   *InputFile, *OutputFile;

#ifdef HOMEDIR
    strpHome = getenv("HOME");
#else
    strpHome = SATDIR;
#endif

    sprintf(data,"%s/%s",strpHome,DATA);
    sprintf(tle,"%s/%s",strpHome,TLE);

    nl();

    sprintf(sysComm,"cd %s; ls *.dat",tle);
    system(sysComm);

    printf("\nenter input data file       (e.g. tle.dat): ");
    mGets(inputData);
    sprintf(inputFile,"%s/%s",tle,inputData);

    if ((InputFile = fopen(inputFile,"r")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("%s not found\n",inputFile);
        normal(); nl();
        exit(-1);
    }

    sprintf(outputFile,"%s/amsat.dat",data);

    if ((OutputFile = fopen(outputFile,"w")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("can't write %s\n",outputFile);
        normal(); nl();
        exit(-1);
    }

    printf("\ncreating %s ....\n\n",outputFile);
    nSat = 0;

    while (fgets(satName,80,InputFile))
    {
        if (!strncmp(satName,TLEHEADER,10))
            fgets(satName,80,InputFile);

        fgets(line1,80,InputFile);
        fgets(line2,80,InputFile);

        sscanf(line1,"%1d",&lineNum1);
        sscanf(line2,"%1d",&lineNum2);

        if (lineNum1 != 1)
            printf("Line 1 not available for satellite %s",satName);

        if (lineNum2 != 2)
            printf("Line 2 not available for satellite %s",satName);

/******************************************************************************/
/*                                                                            */
/* calculate checksum                                                         */
/*                                                                            */
/******************************************************************************/

        if (lineNum1 == 1 && lineNum2 == 2)
        {
            checkSumTotal = 0;

            for (lineNum = 1; lineNum <=2; lineNum++)
            {
                checkSum = 0;

                if (lineNum == 1)
                    sprintf(str,"%s",line1);
                if (lineNum == 2)
                    sprintf(str,"%s",line2);

                for (i = 0; i < 68; i++)
                {
                    strng[0]   = str[i];
                    strng[1]   = '\0';
                    checkValue = atoi(strng);

                    if (!strcmp(strng,"-"))
                        checkValue = 1;      /* assign check sum value to '-' */

                    checkSum += checkValue;
                }

                strng[0] = str[68];
                strng[1] = '\0';

                if (checkSum % 10 != atoi(strng))
                {
                    doBeep(); reverseBlink();
                    printf("checksum error in line %d for satellite %s",
                            lineNum,satName);
                    normal();
                }

                checkSumTotal += checkSum;
            }

/******************************************************************************/
/*                                                                            */
/* get elements from lines 1 and 2                                            */
/*                                                                            */
/******************************************************************************/

            satNum        = getElement(line1, 3, 8);
            epochYear     = getElement(line1,19,20);
            epochDay      = getElement(line1,21,32);
            decayRate     = getElement(line1,34,43);
            ephemerisType = getElement(line1,63,63);
            elementSet    = getElement(line1,65,68);

            inclination   = getElement(line2, 9,16);
            raan          = getElement(line2,18,25);
            eccentricity  = getElement(line2,27,33);
            argPerigee    = getElement(line2,35,42);
            meanAnomaly   = getElement(line2,44,51);
            meanMotion    = getElement(line2,53,63);
            orbitNum      = getElement(line2,64,68);

            epochDay     += epochYear * 1000.0;
            eccentricity *= 1.0e-7;

            fprintf(OutputFile,"Satellite: %s",satName);
            fprintf(OutputFile,"Catalog number: %10d\n",satNum);
            fprintf(OutputFile,"Epoch time: %14.8f\n",epochDay);
            fprintf(OutputFile,"Element set: %13d\n",elementSet);
            fprintf(OutputFile,"Inclination: %13.4f deg\n",inclination);
            fprintf(OutputFile,"RA of node: %14.4f deg\n",raan);
            fprintf(OutputFile,"Eccentricity: %12.7f\n",eccentricity);
            fprintf(OutputFile,"Arg of perigee: %10.4f deg\n",argPerigee);
            fprintf(OutputFile,"Mean anomaly: %12.4f deg\n",meanAnomaly);
            fprintf(OutputFile,"Mean motion: %13.8f rev/day\n",meanMotion);
            fprintf(OutputFile,"Decay rate: %14.3e rev/day^2\n",decayRate);
            fprintf(OutputFile,"Epoch rev: %15d\n",orbitNum);
            fprintf(OutputFile,"Checksum: %16d\n",checkSumTotal);
            fprintf(OutputFile,"\n");

            nSat++;
            printf("%3d: %s",nSat,satName);
        }
    }

    fclose(InputFile);
    fclose(OutputFile);
    printf("\noutput file contains data for %d satellite",nSat);
    if (nSat > 1) printf("s");
    nl();
    nl();
}

/******************************************************************************/
/*                                                                            */
/* getElement: returns double of orbital element out of ASCII string          */
/*                                                                            */
/******************************************************************************/

double getElement(gstring,gstart,gstop)

int  gstart, gstop;
char gstring[80];

{
    int  k, glength;
    char gstr[80];

    glength = gstop - gstart + 1;

    for (k = 0; k <= glength; k++)
        gstr[k] = gstring[gstart+k-1];

    gstr[glength] = '\0';

    return((double) atof(gstr));
}

/******************************************************************************/
/*                                                                            */
/* mGets: Manfred's version of fgets (wipes out newline character)            */
/*                                                                            */
/******************************************************************************/

void mGets(string)

char *string;

{
    int i;
    fgets(string,80,stdin);
    i = (int) strlen(string);

    if (i > 0)
        string[i-1] = '\0';
    else
        string[0]   = '\0';

    return;
}

/******************************************************************************/
/*                                                                            */
/* End of program makeamsat.c                                                 */
/*                                                                            */
/******************************************************************************/
