/*
 * TRK - Satellite tracking program based on Norad SGP/SDP model with
 *       curses interface
 *
 *	by Lapo Pieri IK5NAX  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

/* Molte cose da rivedere:
3) Riunire time.c e t2.c
4) Usare isleap() dichiarata in /usr/include/bsd/tzfile.h al posto della
   leap() rifatta qui
5) Convertire tutte le routine in sgp.c al JD anziche' a JD-J2000
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <math.h>
#include <string.h>
#include "trk.h"

const double JD1970=2440587.5;
const char dow[7][4]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};


int leap(int);


/*--------------------------------------------------------------------------*/
/*       JDsyst:                                                            */
/*              IN: ---                                                     */
/*              OUT: giorno giuliano attuale derivato dall'orologio del     */
/*                   sustema che deve essere in UTC                         */
/*--------------------------------------------------------------------------*/
double JDsyst(void)
{
double jd;
struct timeb st;

  
ftime(&st);
    
jd=JD1970+st.time/86400.+st.millitm/86400e3;
  
/* Vanno bene come variabili globali ? */
TZsyst.tzmw=-st.timezone;
TZsyst.dst=st.dstflag;

return jd;
}

/*--------------------------------------------------------------------------*/
/*       leap:                                                              */
/*		IN:  anno                                                   */
/*              OUT: 1 se bisestile, 0 altrimenti                           */
/*                                                                          */
/*   !!! ATTENZIONE: valido solo per il calendario gregoriano !!!           */
/*--------------------------------------------------------------------------*/
int leap(int y)
{
return ((y%4==0)&&(y%100!=0))||((y%4==0)&&(y%100==0)&&(y%400==0));
}	  


/*--------------------------------------------------------------------------*/
/*       hr2jd:                                                             */
/*            IN: struttura con data e ora in forma leggibile               */
/*            OUT: giorno giuliano (JD)                                     */
/*--------------------------------------------------------------------------*/
double hr2jd(struct hr_time *hr){
int y, m, a, b;
double jd;

y=hr->y; m=hr->mo;

/* distinzione fra calendario giuliano o gregoriano */
 if(y<1582) b=0;
 else if(y<=1582) { 
   if(m<10) b=0; 
   else if(m<=10) { if(hr->dn<4) b=0; else if(hr->dn<=4) b=0; 
   } 
 }
 else { a=(int)(y/100); b=2-a+(int)(a/4); }

/* spostamento inizio anno il primo marzo */
if(m<=2) { y--; m+=12; }

/* calcolo del giorno giuliano */
jd=(double)((int)(365.25*(y+4716))+(int)(30.6001*(float)(m+1))+hr->dn+b-1524.5+
  (double)hr->h/24.+(double)hr->mi/1440.+(double)hr->s/86400.+
  (double)hr->ms/86.4e6);

return jd;
}


/*--------------------------------------------------------------------------*/
/*       jd2hr:                                                             */
/*            IN: giorno giuliano (JD)                                      */
/*            OUT: struttura con data e ora in forma leggibile              */
/*--------------------------------------------------------------------------*/
void jd2hr(double jd, struct hr_time *hr)
{
double z, F;
int a, A, B, C, D, E, Z;

F=modf(jd+0.5,&z); Z=(int)z;
if(Z<2299161) A=Z;
else {a=(int)((Z-1867216.25)/36524.25); A=Z+1+a-(int)(a/4);}
B=A+1524; C=(int)((B-122.1)/365.25); D=(int)(365.25*C); E=(int)((B-D)/30.6001);

hr->dn=B-D-(int)(30.6001*E);
if(E<14) hr->mo=E-1; else hr->mo=E-13;
if(hr->mo>2) hr->y=C-4716; else hr->y=C-4715;

hr->h=(int)(F*24); 
hr->mi=(int)((F-hr->h/24.)*1440.); 
hr->s=(int)((F-hr->h/24.-hr->mi/1440.)*86400.);
hr->ms=(int)((F-hr->h/24.-hr->mi/1440.-hr->s/86400.)*86400000.);

hr->dw=(int)fmod(jd+1.5,7.);
hr->dy=(int)(275.*hr->mo/9.)-(2-leap(hr->y))*(int)((hr->mo+9)/12.)+hr->dn-30;
}



/*--------------------------------------------------------------------------*/
/*       UTCstring:                                                         */
/*                 IN: giorno giuliano (JD), isZ=1 per UTC, 0 per locale    */
/*                 OUT: stringa che descrive data e ora UTC/locale          */
/*--------------------------------------------------------------------------*/
void UTCstring(double jd, char *utcs, int isZ)
{
struct hr_time hr;

if(isZ==0) jd+=(double)TZsyst.tzmw/1440.;
jd2hr(jd, &hr);

/*
sprintf(utcs, "%s  %.2d/%.2d/%.2d   %.2d:%.2d:%.2d.%.3d",
	dow[hr.dw], hr.dn, hr.mo, hr.y, hr.h, hr.mi, hr.s, hr.ms);
if(isZ==1) strstr(utcs, "Z");
*/
/* I millisecondi non sono molto utili... 
   Mettere magari un altro parametro che specifichi la precisione della 
   stringa
*/

sprintf(utcs, "%s  %.2d/%.2d/%.2d   %.2d:%.2d:%.2d",
	dow[hr.dw], hr.dn, hr.mo, hr.y, hr.h, hr.mi, hr.s);
if(isZ==1) strcat(utcs, "Z");
if(isZ==0) strcat(utcs, " ");
}

/*--------------------------------------------------------------------------*/
/*       DATE/TIMEstring:                                                   */
/*                        jd     - giorno giuliano (JD)                     */
/*                        *dts   - stringa contenente la data/ora           */
/*                        isDATE - 1 se si vuole la data                    */ 
/*                        isTIME - 1 se si vuole l'ora                      */ 
/*                        isZ    - 1 se l'ora deve essere UTC               */ 
/*                        isABS  - 1 se deve essere un tempo assoluto       */ 
/*                                 (in tutti gli altri casi is*=0           */ 
/*--------------------------------------------------------------------------*/

/* Il tempo relativo dovrebbe esserlo rispetto a un istante dato, percio`
   andrebbe modificato isABS in isREL e fare in modo che con 0 venga il tempo
   assoluto, con <0 il tempo relativo a JDsyst() e con >0 relativo a quel JD */
void DTstring(double jd, char *dts, int isDATE, int isTIME, int isZ, int isABS)
{
struct hr_time hr;
char b[16];

*dts='\0';

if(isABS) { 
  jd-=JDsyst(); hr.h=(int)(jd*24.); hr.mi=(int)(jd*1440.-hr.h*60.);
  hr.s=(int)(jd*86400-hr.h*3600.-hr.mi*60.);
  sprintf(dts, "+%.2d:%.2d:%.2d ", hr.h, hr.mi, hr.s);
  return;
}

else if(isZ==0) jd+=(double)TZsyst.tzmw/1440.+(double)TZsyst.dst/24.;

jd2hr(jd, &hr);

if(isDATE) { sprintf(b, "%s  %.2d/%.2d/%.2d  ", 
           dow[hr.dw], hr.dn, hr.mo, hr.y); strcat(dts, b); }

if(isTIME) { sprintf(b, " %.2d:%.2d:%.2d", hr.h, hr.mi, hr.s); 
	   strcat(dts, b); if(isZ&&!isABS) strcat(dts, "Z"); 
           else strcat(dts, " "); }

}



/*--------------------------------------------------------------------------*/
/*       tetha:                                                             */
/*              IN:  giorno giuliano (JD), longitudine [deg]                */
/*              OUT: tempo siderale [rad] (0-2Pi)                           */
/*--------------------------------------------------------------------------*/
double tetha(double jd, float lon)
{
double t, teg, th;
t=(jd-2451545.)/36525.;
/* valutare questo polinomio in modo un po' piu' efficiente */
teg=280.46061837+360.98564736629*(jd-2451545)+0.000387933*t*t-t*t*t/38710000;
th=(teg+lon)/180.*M_PI;

/* normalizzazione nell'intervallo 0-2*Pi */
th-=(int)(th/2/M_PI)*2.*M_PI;

return(th); 
}                        



/*--------------------------------------------------------------------------*/
/*       hms:                                                               */
/*              IN:  angolo [rad]                                           */
/*              OUT: puntatore a una stringa in ore, minuti, secondi        */
/*--------------------------------------------------------------------------*/
char *hms(double a){
static char b[30];
int h, m, s;
a=a*12./M_PI; while(a>24.) a-=24; while(a<0) a+=24.;
h=(int)a;
m=(int)((int)(a*60)-h*60);
s=(int)((int)(a*3600)-m*60-h*3600);

sprintf(b, "%2dh%02dm%02ds", h, m, s);
return b;
}
