/*
 * TRK - Satellite tracking program based on Norad SGP/SDP model with
 *       curses interface
 *
 *	by Lapo Pieri IK5NAX  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include <stdio.h>
#include <ncurses.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sgp.h"
#include "trk.h"


#define __VERSION "0.3.5"

int fdpoint, isROT=0, Zloc=1, ai=1, isGPT=0, gptfifochk, actobs=0;
cfg_file cfgf;

int main(int argc, char **argv)
{
FILE *fp, *fpdb;
SAT *sats[6], *localsat; 
OBS *obs[6], imdobs;
int e=1, roten=0, i, isFREEZ=0, horiz0[6]={0, 0, 0, 0, 0, 0}, multien=0;
int numobs, actobs;          /* numero totale osservatori (0 solo l'"Home"),
				e numero dell'osservatore attivo */
char b[80], imdqth[7];
satpos *asp, *sunp, *moonp;
float imdlon, imdlat;
                             /* 
				Le variabili che iniziano con "imd" sono 
				realtive all'introduzione al volo (immediata)
				di un nuovo osservatore
			     */
double T, tLOS, tAOS;

 
asp=(satpos*)malloc(sizeof(satpos));    /* Struttura puntamento satellite 
					   "attuale", in pratica struttura
					   di uso comune per tutti i satelliti
					*/
sunp=(satpos*)malloc(sizeof(satpos));   /* Struttura puntamento sole */
moonp=(satpos*)malloc(sizeof(satpos));  /* Struttura puntamento luna */


/* Ricerca dei file di configurazione e dati */
if(setup_config_file(argc, argv, &cfgf)==1) return 1;

/* Apertura e si lettura dei file di dati necessari */
 if((fp=fopen(cfgf.tlef, "r"))==NULL) { 
   fprintf(stderr, "Unable to open tle data file %s\n", cfgf.tlef); return 1; }
 if((sats[0]=sgp_sat_read(fp))== NULL) {
   fprintf(stderr, "No objects?\n"); return 1; }
 rewind(fp);


 if((obs[0]=getobs(0))==NULL){
   fprintf(stderr, "Unable to open obs.dat\n"); return 1; }
 while(obs!=NULL){
   if(strcmp(obs[0]->name, "Home")==0) break;
   obs[0]=obs[0]->next;
 }
 if(obs[0]==NULL){
   fprintf(stderr, "No \"Home\" observer found in obs.dat\n"); return 1; }


numobs=0; actobs=0;


 if((fpdb=fopen(cfgf.satdbf, "r"))==NULL){ 
   fprintf(stderr, "Unable to open satellite database file %s\n", 
	   cfgf.satdbf); return 1; }

 /* Inizializzazione delle ncurses */
initscr(); cbreak(); noecho(); timeout(0); start_color(); curs_set(0);
init_pair(1, COLOR_GREEN, COLOR_BLACK);
init_pair(2, COLOR_YELLOW, COLOR_BLACK);
init_pair(3, COLOR_WHITE, COLOR_BLACK);
init_pair(4, COLOR_BLACK, COLOR_CYAN);
init_pair(5, COLOR_RED, COLOR_BLACK); 
init_pair(6, COLOR_CYAN, COLOR_BLACK);

/* Scritte stabili della scheramata principale */
 attron(COLOR_PAIR(5)|A_BOLD); mvprintw(23,0,"TRK   v%s", __VERSION);
 attroff(A_BOLD); mvprintw(23, 15, "["); mvprintw(23, 40, "]"); 



 /* Riconoscimento servizi esterni */
 /* Fifo verso il rotore */
 if((fdpoint=open(cfgf.fiforot, O_RDWR|O_NONBLOCK))!=-1)
   {
     attron(COLOR_PAIR(5)|A_BOLD); 
     mvprintw(23,17,"R"); 
     attroff(A_BOLD);
     isROT=1;
   }

 /* Gnuplot */
 /* Non sapendo come far restituire al programma il risultato della system()
    questo mi sembra un po' arzigogoglato ma funzionante.
 */
 remove("gpttst.fifo");
 if((mkfifo("gpttst.fifo", S_IROTH|S_IWOTH)!=0)||
    (gptfifochk=open("gpttst.fifo", O_RDWR|O_NONBLOCK))==0)
   fprintf(stderr, "Unable to check if Gnuplot is present\n");
 else{
   sprintf(b, "gnuplot %s >/dev/null 2>./gpttst.fifo", cfgf.gpttf);
   system(b);
   read(gptfifochk, b, 40);
   if(strncmp("GNUPLOT", b, strlen("GNUPLOT"))==0){
     attron(COLOR_PAIR(5)|A_BOLD); 
     mvprintw(23,19,"G"); 
     attroff(A_BOLD);
     isGPT=1;
   }
 }     
 remove("gpttst.fifo");

 /* Ciclo principale */
 while(e) {

   /* 
      Innanzitutto il tempo (JD) a cui si faranno tutti i calcoli, ottenuto
      dall'orologio di sistema come tempo trascorso dall'"epoca".
    */
   T=JDsyst();

   /* E` l'unico posto in cui fermare se e` stato introdotto un istante fisso;
      non e` molto elegante...
   */
   if(isFREEZ==1) { timeout(-1); getch(); timeout(0); isFREEZ=0;}

   switch(getch()){
   case 27: { if(getch()==91) 
     switch(getch()){

     case 68: if(sats[actobs]->prev!=NULL) 
       sats[actobs]=sats[actobs]->prev;
     if(actobs==0) { roten=0; mvprintw(3, 15, "   "); }
     derivtle(T, sats[actobs], 0); /* Se si cambia sat/obs la finestra dei
				      dati orbitali derivati non e` piu` valida
				   */
     break;                                         /* curs sx */

     case 67: if(sats[actobs]->next!=NULL) 
       sats[actobs]=sats[actobs]->next;
     if(actobs==0) { roten=0; mvprintw(3, 15, "   "); }
     derivtle(T, sats[actobs], 0); /* Se si cambia sat/obs la finestra dei
				      dati orbitali derivati non e` piu` valida
				   */
     break;                                         /* curs dx */

     case 66: if(actobs<numobs) { actobs++; 
     derivtle(T, sats[actobs], 0); } /* Se si cambia sat/obs la finestra dei
				      dati orbitali derivati non e` piu` valida
				     */
     break;                                         /* curs up */

     case 65: if(actobs>0) { actobs--; 
     derivtle(T, sats[actobs], 0); } /* Se si cambia sat/obs la finestra dei
				      dati orbitali derivati non e` piu` valida
				     */
     break;                                         /* curs down */

     case 53: if(getch()!=126) break; 
       if(obs[actobs]->prev!=NULL && actobs!=0) 
	 obs[actobs]=obs[actobs]->prev; break;      /* Pg Up */

     case 54: if(getch()!=126) break; 
       if(obs[actobs]->next!=NULL && actobs!=0) 
	 obs[actobs]=obs[actobs]->next; break;      /* Pg Down */
     } 
   } 

    break;

   case 'q': e=0; break;

   case 's': sched(sats[actobs], obs[actobs]); break;

   case 'g': if(isGPT==1) graphel(sats[actobs], obs[actobs]); break;
   case 'r': if(isROT==1) roten^=0x01;
     if(roten==1) { 
       attron(COLOR_PAIR(5)); attron(A_BOLD); mvprintw(3, 15, "ROT"); }
     else mvprintw(3, 15, "   ");
     break;
   case ' ': timeout(-1); 
     DTstring(T, b, 1, 1, Zloc, 0); attroff(A_BOLD);
     attron(COLOR_PAIR(3)|A_REVERSE); mvprintw(0,50,"%s", b);
     getch(); timeout(0); attroff(A_REVERSE); isFREEZ=0; break;
   case 'z': Zloc^=0x01; break;
   case 'h': 
   case '?': help(); break;
   case 'a': ai^=0x01; break;
   case 't': T=getT(); if(T==0) { T=JDsyst(); break; } isFREEZ=1; break;
   case 'i': obsinfo(obs[actobs]); break;
   case 'd': derivtle(T, sats[actobs], 1); break;
   case '/': localsat=getsatname(sats[actobs]); 
     if(localsat!=NULL) sats[actobs]=localsat; break;
   case '0': horiz0[actobs]^=1; break;
   case 'm': multien^=1; break;

   /* Aggiunta di un osservatore */
   case '+': if(numobs<5) { numobs++; actobs=numobs; 
   sats[actobs]=sgp_sat_read(fp); rewind(fp); obs[actobs]=getobs(0); 
   derivtle(T, sats[actobs], 0); /* Se si cambia sat/obs la finestra dei
				    dati orbitali derivati non e` piu` valida
				 */
   } break;

   /* Eliminazione di un osservatore */
   /* in effetti andrebbe eliminato l'actobs e non l'ultimo... */
   case '-': if(numobs>0) { obs[numobs]=NULL; sats[numobs]=NULL; 
   numobs--; actobs=numobs; 
   derivtle(T, sats[actobs], 0); /* Se si cambia sat/obs la finestra dei
				    dati orbitali derivati non e` piu` valida
				 */
   }
   for(i=0; i<80; i++) mvprintw(4+numobs,i, " "); break;

   /* Introduzione nuovo osservatore al volo */
   case 'o': if(numobs<5) { numobs++; actobs=numobs;
   derivtle(T, sats[actobs], 0); /* Se si cambia sat/obs la finestra dei
				    dati orbitali derivati non e` piu` valida
				 */
   /* 
      Per il nuovo osservatore si crea tutta la struttura concatenata leggendo
      il file obs.dat, come con il comando "+" 
   */
   sats[actobs]=sgp_sat_read(fp); rewind(fp); obs[actobs]=getobs(0); 
   while(obs[actobs]->next!=NULL) obs[actobs]=obs[actobs]->next;
   /* 
      Il nuovo osservatore puo` essere specificato sia come lon, lat sia con
      il qth locator, il riconoscimento e` automatico, l'altezza e` posta
      uguale a zero, ma si potrebbe aggiungere
   */
   attron(COLOR_PAIR(3)|A_BOLD);
   curs_set(1); mvprintw(3+numobs, 2, "QTH or lon,lat: "); 
   timeout(-1); echo(); mvgetstr(3+numobs, 18, b); noecho(); 
   timeout(0); curs_set(0); 

   if((b[0]<='9' && b[0]>='0')||b[0]=='-'||b[0]=='+'){ 
     sscanf(b, "%f %f", &imdlon, &imdlat);
     if(lonlat2qthloc(imdlon, imdlat, imdqth)<0) {
       for(i=0; i<80; i++) mvprintw(3+numobs,i, " ");
       numobs--; actobs=numobs; break; }
     else {
       imdlon=imdlon/180.*M_PI; imdlat=imdlat/180.*M_PI; 
       imdqth[6]=0; imdobs.lat=imdlat; imdobs.lon=imdlon; imdobs.h0=0.;
       strcpy(imdobs.name, imdqth); strcpy(imdobs.qth, imdqth);
       strcpy(imdobs.descr, "On the fly supplied observer as lon,lat");
     }
   }
   /* Il nome dell'osservatore cosi` introdotto e` il qthlocator */
   else{
     sscanf(b, "%s", imdqth);
     if(qthloc2lonlat(imdqth, &imdlon, &imdlat)<0) { 
       for(i=0; i<80; i++) mvprintw(3+numobs,i, " ");
       numobs--; actobs=numobs; break;}
     else {
       imdlon=imdlon/180.*M_PI; imdlat=imdlat/180.*M_PI; 
       imdqth[6]=0; imdobs.lat=imdlat; imdobs.lon=imdlon; imdobs.h0=0.;
       strcpy(imdobs.name, imdqth); strcpy(imdobs.qth, imdqth);
       strcpy(imdobs.descr, "On the fly supplied observer as QTHloc");
     }
   }
   /* Questo e` il modo di concatenare le strutture, non ancora molto chiaro */
   imdobs.next=NULL; imdobs.prev=obs[actobs];
   obs[actobs]->next=(OBS *)malloc(sizeof(OBS));
   memcpy(obs[actobs]->next, &imdobs, sizeof(OBS));
   obs[actobs]=obs[actobs]->next;
   
   } break;


   default: break;     
   }
   
/* Visualizzazione ora del sistema */
   DTstring(T, b, 1, 1, Zloc, 0); attroff(A_BOLD); 
   if(isFREEZ==1) attron(COLOR_PAIR(3)|A_REVERSE);
   else attron(COLOR_PAIR(3)); mvprintw(0,50,"%s", b);

#ifdef DEBUG
mvprintw(1,50, "%f", T);
#endif

/* Calcolo posizione sole e luna */
/* 
   Vanno fatti qui e non insieme alla visualizzazione perche' questi dati 
   servono per il calcolo dell'illuminazione del satellite
*/
 sunpos(T, obs[0], sunp);
 moonpos(T, obs[0], moonp);

/* Gestione servizi esterni al programma */
   /* ... */

/* Gestione presentazione multisatellite */
 multisat(T, sats[actobs], obs[actobs], ai, multien);
                                         /* Se e` in corso la visualizzazione
					    multisatellite allora si presenta 
					    la visualizzazione normale solo per
					    l'osservatore Home
					 */
if(multien==1) { 
  attroff(A_BOLD); attron(COLOR_PAIR(3));
  mvprintw(0,0,"                       [%5d%c]",sats[0]->norad,
	   sats[0]->eclass);
  attron(COLOR_PAIR(6)|A_BOLD); 
  mvprintw(0,0,"%s", sats[0]->name);
  mvprintw(3, 0, "Home");
  single_sat_calc(T-J2000, sats[0], obs[0], asp);
  if(roten==1) rotor((float)asp->az, (float)asp->el);
  attroff(A_BOLD); attron(COLOR_PAIR(1));
  mvprintw(2,20, 
	   "AZ     EL     Range  dR/dt     Hz / MHz   AOS/LOS   I  mEl");
  if(asp->el>localhoriz(obs[0], asp->az, horiz0[0])*M_PI/180.) 
    attron(COLOR_PAIR(2)|A_BOLD);
  mvprintw(3,20,
	   "%5.1f  %5.1f %6.0f %6.3f% +7.0f / %-5.0f           %c %4.1f",
	   asp->az/M_PI*180+180, asp->el/M_PI*180, 
	   asp->range, asp->rangerate,
	   -asp->rangerate/3e5*sdb.beacon[0]*1e6, sdb.beacon[0],
	   sat_illum(*asp, *sunp),
	   localhoriz(obs[0], asp->az, horiz0[0]));   

  if(asp->el<0.){
    attroff(A_BOLD); attron(COLOR_PAIR(1));
    attron(COLOR_PAIR(3));
    if(AOS(sats[0], obs[0], T-J2000, &tAOS)==0)
      DTstring(tAOS+J2000, b, 0, 1, Zloc, ai);
    else
      strcpy(b, " -------- ");
    mvprintw(3, 61, "%s", b);
  }
  else { 
    attroff(A_BOLD); attron(COLOR_PAIR(1));
    attron(A_BOLD); attron(COLOR_PAIR(3));
    if(LOS(sats[0], obs[0], T-J2000, &tLOS)==0)
      DTstring(tLOS+J2000, b, 0, 1, Zloc, ai);
    else 
      strcpy(b, " -------- ");
    mvprintw(3, 61, "%s", b);
  }
  
}

                                          /* Se invece non e` in corso la
					     visualizzazione multisatellite
					     allora si presentano tutti gli
					     osservatori e le info aggiuntive
					  */
 else{
   /* Calcolo posizione satellite corrente e visualizzazione */
   for(i=0; i<=numobs; i++){
     read_db(fpdb, sats[i]); 
     if(i==0){
       attroff(A_BOLD); attron(COLOR_PAIR(3));
       mvprintw(0,0,"                       [%5d%c]",sats[i]->norad,
		sats[i]->eclass);
       if(i==actobs)  attron(COLOR_PAIR(6)|A_BOLD); 
       else { attroff(A_BOLD); attron(COLOR_PAIR(6)); }
       mvprintw(0,0,"%s", sats[i]->name);
       mvprintw(3, 0, "Home");
     }
     
     else{
       if(i==actobs)  attron(COLOR_PAIR(6)|A_BOLD); 
       else { attroff(A_BOLD); attron(COLOR_PAIR(6)); }
       mvprintw(3+i, 0, "                    ");
       mvprintw(3+i, 0, "%s", obs[i]->name);
       mvprintw(3+i, strlen(obs[i]->name), " %s", sats[i]->name);
       attroff(A_BOLD); attron(COLOR_PAIR(3));
       mvprintw(3+i, strlen(obs[i]->name), "-");
       
     }
     single_sat_calc(T-J2000, sats[i], obs[i], asp);
     if(i==0 && roten==1) rotor((float)asp->az, (float)asp->el);
     attroff(A_BOLD); attron(COLOR_PAIR(1));
     mvprintw(2,20, 
	      "AZ     EL     Range  dR/dt     Hz / MHz   AOS/LOS   I  mEl");
     if(asp->el>localhoriz(obs[i], asp->az, horiz0[i])*M_PI/180.) 
       attron(COLOR_PAIR(2)|A_BOLD);
     mvprintw(3+i,20,
	      "%5.1f  %5.1f %6.0f %6.3f% +7.0f / %-5.0f           %c %4.1f",
	      asp->az/M_PI*180+180, asp->el/M_PI*180, 
	      asp->range, asp->rangerate,
	      -asp->rangerate/3e5*sdb.beacon[0]*1e6, sdb.beacon[0],
	      sat_illum(*asp, *sunp),
	      localhoriz(obs[i], asp->az, horiz0[i]));   
     
     if(i==actobs){
       attroff(A_BOLD); attron(COLOR_PAIR(3));
       mvprintw(10,0, 
       "Orbit   Phase  RA         Dec      Lat  <--SSP-->  Lon       Alt");
       mvprintw(11,0, "%6d  %5.1f  %s %8.3f  %7.3f    %8.3f %9.2f", 
	sats[i]->rev, asp->phase/M_PI*128., hms(asp->ra), asp->dec/M_PI*180.,
		asp->ssplat/M_PI*180., asp->ssplon/M_PI*180., asp->height);
     }
     
     /* Calcolo AOS/LOS e visualizzazione */
     if(asp->el<0.){
       attroff(A_BOLD); attron(COLOR_PAIR(1));
       attron(COLOR_PAIR(3));
       if(AOS(sats[i], obs[i], T-J2000, &tAOS)==0)
	 DTstring(tAOS+J2000, b, 0, 1, Zloc, ai);
       else
	 strcpy(b, " -------- ");
       mvprintw(3+i, 61, "%s", b);
     }
     else { 
       attroff(A_BOLD); attron(COLOR_PAIR(1));
       attron(A_BOLD); attron(COLOR_PAIR(3));
       if(LOS(sats[i], obs[i], T-J2000, &tLOS)==0)
	 DTstring(tLOS+J2000, b, 0, 1, Zloc, ai);
       else
	 strcpy(b, " -------- ");
       mvprintw(3+i, 61, "%s", b);
     }
    
   }
 }

/* Visualizzazione posizione sole e luna */
 attroff(A_BOLD); attron(COLOR_PAIR(1));
 if(sunp->el>0.) attron(COLOR_PAIR(2)|A_BOLD);
 mvprintw(21,0, "Sun:  RA %s  Dec %7.3f  Range %9.0f  AZ %5.1f  EL %5.1f",
	  hms(sunp->ra), sunp->dec/M_PI*180., sunp->range,
	  sunp->az/M_PI*180., sunp->el/M_PI*180.);

 attroff(A_BOLD); attron(COLOR_PAIR(1));
 if(moonp->el>0.) attron(COLOR_PAIR(2)|A_BOLD);
 mvprintw(22,0, "Moon: RA %s  Dec %7.3f  Range %9.0f  AZ %5.1f  EL %5.1f",
	  hms(moonp->ra), moonp->dec/M_PI*180., moonp->range,
	  moonp->az/M_PI*180., moonp->el/M_PI*180.);
 



/* E` inutile fare i calcoli troppo spesso... */
 usleep(100000);
 }

/* All'uscita dal programma e` bene ripulire tutto */
 /* clear(); refresh(); */
 free(asp); free(sunp); free(moonp);
 endwin(); 
 if(isROT==1) close(fdpoint);
 fclose(fpdb);
 fclose(fp);
 remove("el.dat");
 return 0;
}



/* 
   La parte del database e delle frequenze e` tutta da rivedere!
*/
void read_db(FILE *fpdb, SAT *sats){
  char buff[80], satname[40], buff2[40];
  float f0;
  int i, n;

  rewind(fpdb);
   while(1) {
     buff[0]=0; satname[0]=0;
     if(fgets(buff, 80, fpdb)==NULL) break;
     if(buff[0]=='[') {
       i=0; while(buff[i]!=']') i++;
       i++;
       while(buff[i]==' ' || buff[i]=='\t') i++;
       n=strlen(buff+i);
       while(buff[n]=='\n' || buff[n]=='\r') n--;
       strncpy(satname, buff+i, n-1); satname[n-1]=0;
       
       if(strcmp(sats->name, satname)==0){
	 if(fgets(buff, 80, fpdb)==NULL) break;
	 if(fgets(buff, 80, fpdb)==NULL) break;
	 sscanf(buff, "%s %f", buff2, &f0);
	 sdb.beacon[0]=f0;
	 break;
       }
     }
     
   }

}


void rotor(float az, float el)
{  
char buff[80]; 
static double oldaz=0., oldel=0.; 
int i;  

if(el<0.) return;

 for(i=0; i<80; i++) buff[i]=0;
 if(fabs(az-oldaz)>0.5/180.*M_PI) {
   sprintf(buff, "AZ %.1f\n", az/M_PI*180.+180.);
   write(fdpoint, buff, strlen(buff));
   oldaz=az;
 }
 
 for(i=0; i<80; i++) buff[i]=0;
 if(fabs(el-oldel)>0.5/180.*M_PI) {
   sprintf(buff, "EL %.1f\n", el/M_PI*180.);
   write(fdpoint, buff, strlen(buff));
   oldel=el;
 }
 
}



void graphel(SAT *sats, OBS *obs){
FILE *fp;
double t, ti;
char b[80], c[10];

 if((fp=fopen("el.dat","w"))==0) 
   {printf("unable to open el.dat\n"); return;}

ti=JDsyst()-J2000;
 for(t=ti; t<ti+.9; t+=2.*M_PI/sats->n0/256.){
   UTCstring(t+J2000, b, Zloc); strncpy(c, b+3, 21); c[21]=0;
   fprintf(fp,"%s\t%4.1f\n", c, elev(t, sats, obs));
 }

 fclose(fp);

sprintf(b,"gnuplot %s >/dev/null 2>/dev/null", cfgf.gelf);
system(b);
}


double getT(void){
struct tm it;
struct hr_time hr;
char b[40], b1[40], b2[40];
int n;

 attron(COLOR_PAIR(3)|A_BOLD); curs_set(1); 
 mvprintw(15, 1,
	  "[Date &] Time {dd/mm/yyyy hh:mm:[ss]} {hh:mm:[ss]}: "); 
 attroff(A_BOLD);
 timeout(-1); echo(); mvgetstr(15, 53, b); noecho(); timeout(0); curs_set(0); 
 mvprintw(15, 1,
  "                                                                        "); 

 if(strlen(b)<1) return 0;
 if(strcmp(b, "now")==0) { UTCstring(JDsyst(), b, 0); strcpy(b, b+5); }
 n=sscanf(b, "%s %s", b1, b2);
 it.tm_sec=0; /* se non si introducono i secondi questa variabile resta non
		 inizializzata, con risultati impredicibili... */
 it.tm_min=0; it.tm_hour=0; /* tanto per stare tranquilli ... */
 switch(n){
 case 1: jd2hr(JDsyst(), &hr); strptime(b1, "%X", &it); 
   it.tm_year=hr.y; it.tm_mon=hr.mo-1; it.tm_mday=hr.dn;
   break;

 case 2: strcat (b1, " "); strcat(b1, b2); 
   strptime(b1, "%d/%m/%Y %T", &it); if(it.tm_year>=100) it.tm_year+=1900; 
   break;

 case 0: 
 default: return 0;
 }

hr.y=it.tm_year; hr.mo=it.tm_mon+1; hr.dn=it.tm_mday;
hr.h=it.tm_hour; hr.mi=it.tm_min; hr.s=it.tm_sec; hr.ms=1;

return (hr2jd(&hr));
}


/*
  Vengono presentati solo i primi 30 satelliti, per essere veramente utile
  o si mette uno scroll (ma mi sembra che si complichi troppo la struttura
  del programma) o si introducono i grupppi
*/
void multisat(double T, SAT *sats, OBS *obs, int ai, int endis){
int i, j;
satpos *msp, sun;
char b[20], nb[30];
double tALOS;
static int olden=0;

/* 
   Sarebbe stato piu` elegante, dal punto di vista della programmazione,
   usare una finestra delle ncurses, ma l'effetto dell'aggiornamento continuo
   della finestra principale e delle sotto-finestre e` molto fastidioso, cosi`
   l'effetto e` buono
*/

/* 
   Solo se c'e` stato un cambiamento fra multi e non multi allora si scancella
   la porzione di schermo condivisa fra le due modalita`, cancellarlo sempre,
   oltre che inutile da un effetto fastidioso nella visualizzazione
*/
 if(olden!=endis) { 
   olden=endis; for(i=4; i<20; i++) { move(i,0); deleteln(); insertln(); }
   refresh(); }

 if(endis==0) return;

 msp=(satpos*)malloc(sizeof(satpos));        /* Struttura contente la
						posizione del singolo satellite
						di volta in volta calcolata;
						in seguito bisognera` farne
						una per satellite visualizzato
						in modo da non dover rifare 
						tanti calcoli... (ampliare) */

 attron(A_BOLD); attron(COLOR_PAIR(5));
 mvprintw(4, 25, "Multi for obs: %s", obs->name); 

 /* Questo serve per calcolare l'illuminazione */
 sunpos(T, obs, &sun); 

 
 while(sats->prev!=NULL) sats=sats->prev;

 for(j=0; j<40; j+=39){
   for(i=0; i<15; i++) {
     /*     T=JDsyst(); */
     
     if(sats==NULL) break;
     single_sat_calc(T-J2000, sats, obs, msp);
     attroff(A_BOLD); attron(COLOR_PAIR(1));
     if(msp->el>0.) attron(COLOR_PAIR(2)|A_BOLD);
     
     if(msp->el<0.){
       if(AOS(sats, obs, T-J2000, &tALOS)==0)
	 DTstring(tALOS+J2000, b, 0, 1, Zloc, ai);
       else 
	 strcpy(b, " -------- ");
     }
     else { 
       if(LOS(sats, obs, T-J2000, &tALOS)==0)
	 DTstring(tALOS+J2000, b, 0, 1, Zloc, ai);
       else
	 strcpy(b, " -------- ");
     }
     

     strcpy(nb, sats->name); nb[5]=0; /* Il nome del satellite va troncato */
     mvprintw(5+i, j, "%5s %5.1f %5.1f %6.0f %c %10s    ", 
	      nb, msp->az*180./M_PI+180., msp->el*180./M_PI, msp->range,
	      sat_illum(*msp, sun), b);
     sats=sats->next;
     
   }
 }

free(msp);
}


/* 
   mode - 1: toggle
          0: elimina la  finestra
*/
void derivtle(double t, SAT *sats, int mode){
static WINDOW *dtw;
static int isshow=0;
double a, tper;
float x, y, z, rl;
satpos *apos;
OBS dummyobs;


 if(isshow==0&&mode==1){
 if((dtw=newwin(6, 79, 13, 0))==NULL) return;

 apos=(satpos*)malloc(sizeof(satpos));
 dummyobs.lon=0.; dummyobs.lat=0.; dummyobs.h0=0.; 

 single_sat_calc(t-J2000, sats, &dummyobs, apos);
 tper=(M_PI-apos->phase)/sats->n;
 tper=t-J2000+tper;
 single_sat_calc(tper, sats, &dummyobs, apos);

 wattron(dtw, COLOR_PAIR(3)); wattroff(dtw, A_BOLD);
 box(dtw, ACS_VLINE, ACS_HLINE);
 mvwprintw(dtw, 0,10, "Derived orbital element value at system time for %s",
	   sats->name);
 wattron(dtw, COLOR_PAIR(6)); wattroff(dtw, A_BOLD);
 mvwprintw(dtw, 1,72, "Using"); mvwprintw(dtw, 2,73, "SGP4"); 
 mvwprintw(dtw, 3,73, "SDP4"); mvwprintw(dtw, 4,72, "Model");


 wattron(dtw, COLOR_PAIR(3)); wattroff(dtw, A_BOLD);
 mvwprintw(dtw, 1,2, "Period %8.2f min", 1440.*2.*M_PI/sats->n);
 mvwprintw(dtw, 1,26, "TLEage %5.1f day", t-J2000-sats->t0);
 mvwprintw(dtw, 1,46, "Sat age (~)    %5.0f day", sats->rev/sats->n*2.*M_PI);

 a=pow(sqrt(mu)/2./M_PI*86400.*2*M_PI/sats->n, 2./3.);
 /* Quella di calcolare l'SSP e da quello il raggio locale terrestre per 
    sottrarlo nel calolo perigeo/apogeo e` una finezza, comunque si possono
    avere delle differenze al massimo di 21.4Km */
 dummyobs.lon=apos->ssplon; dummyobs.lat=apos->ssplat; dummyobs.h0=0.;
 obsijk(tetha(t, apos->ssplon), &dummyobs, &x, &y, &z);
 rl=sqrt(x*x+y*y+z*z); /* Siccome apogeo e perigeo hanno l'SSP alla latitudine
			  cambiata di segno (la longitudine non cambia), nel 
			  geoide di riferimento compete lo stesso raggio locale
		       */
 mvwprintw(dtw, 2,2, "S.M.A.  %8.1f Km", a);
 mvwprintw(dtw, 3,2, "Apogee  %8.1f Km", a*(1+sats->ecc)-rl);
 mvwprintw(dtw, 4,2, "Perigee %8.1f Km", a*(1-sats->ecc)-rl);

 mvwprintw(dtw, 2,26, "RAAN %7.3f deg", sats->node/M_PI*180.);
 mvwprintw(dtw, 3,26, "Incl %7.3f deg", sats->incl/M_PI*180.);
 mvwprintw(dtw, 4,26, "Ecc %8.5f", sats->ecc);

 mvwprintw(dtw, 2,46, "Arg Perigee %8.3f deg", sats->peri/M_PI*180.);

 mvwprintw(dtw, 3,46, "Lon Apogee  %8.3f deg", apos->ssplon/M_PI*180.);
 mvwprintw(dtw, 4,46, "Lat Apogee  %8.3f deg", apos->ssplat/M_PI*180.);
 
 wrefresh(dtw);
 isshow=1;
 }

 else if(isshow==1){
   wclear(dtw);  wrefresh(dtw); delwin(dtw); refresh();  isshow=0;
 }

}

SAT *getsatname(SAT *sats){
char b[40];
SAT *local;
int i;

 attron(COLOR_PAIR(3)); attroff(A_BOLD); curs_set(1); 
 mvprintw(15, 1, "Enter [part of] satellite name ");
 timeout(-1); echo(); mvgetstr(15, 32, b); noecho(); timeout(0); curs_set(0); 
 mvprintw(15, 1,
  "                                                                        "); 
 for(i=0; i<strlen(b); i++) b[i]=toupper(b[i]);

 local=sats;
 while(local->prev!=NULL) local=local->prev;
 while(local!=NULL) {
   /*   if(strncmp(b, local->name, sizeof(b))==0) return local; */
   /* in questo modo si cercherebbe la parte iniziale del nome del 
      satellite, ma mi sembra piu` utile cercare la sottostringa */
   if(strstr(local->name, b)!=NULL) return local;
   local=local->next;
 } 
return NULL;
}


/*
  Bisognerebbe cautelarsi che altre finestre non vengano sovrascritte a
  questa, o nel caso accada dovrebbe essere messo isshow=0 (meglio cosi`)
*/
void obsinfo(OBS *obsi){
static WINDOW *oiw;
static int isshow=0;

 if(isshow==0){
 if((oiw=newwin(6, 79, 13, 0))==NULL) return;
 attron(COLOR_PAIR(3)); attroff(A_BOLD);
 box(oiw, ACS_VLINE, ACS_HLINE);
 mvwprintw(oiw, 1,1, "[%10s]    Lat %8.3f    Lon %8.3f    Height %4.0f     \
QTH: %s", obsi->name, obsi->lat*180./M_PI, obsi->lon*180./M_PI
	   , obsi->h0, obsi->qth);
 mvwprintw(oiw, 2,2, "\"%s\"", obsi->descr);
 wmove(oiw, 3,2);
 wrefresh(oiw);
 isshow=1;
 }

 else{
   wclear(oiw);  wrefresh(oiw); delwin(oiw); refresh();  isshow=0;
 }
}


void sched(SAT *sats, OBS *obs)
{
double t, theta, rx, ry, rz, rS, rE, rZ, r, el, az, start, tstart, tinc, 
  tstop, tAOS, howlong;
float xo, yo, zo;
FILE *fp;
char b[40];
satpos sat, sun;

 /* Si sfrutta la getT() per ottenere il tempo assoluto di inizio */
 attron(COLOR_PAIR(3)); attroff(A_BOLD); curs_set(1); echo(); timeout(-1);
 mvprintw(14,1,"Starting"); start=getT()-J2000;
 /* La getT() restituisce zero se la stringa non e` valida */
 if(start==-J2000) { 
   mvprintw(14,1,"                                       "); return; }

 /* Gli altri tempi sono relativi, quindi si leggono cosi` */
 attron(COLOR_PAIR(3)); attroff(A_BOLD); curs_set(1); echo(); timeout(-1);
 mvprintw(14,1,"for [hours]: "); scanw("%lf", &howlong); howlong=howlong/24.;
 mvprintw(14,1,"                                       ");
 mvprintw(14,1,"deltaT [min]: "); scanw("%lf",&tinc); tinc/=1440.;
 mvprintw(14,1,"                                       ");
 mvprintw(14,1,"Calculating..."); refresh();
 noecho(); timeout(0); curs_set(0);

 /* Si inizia dal primo sorgere o dall'istante dato se e` gia` sorto */
 if(elev(start, sats, obs)<0.)
   AOS(sats, obs, start, &tstart);
   else tstart=start;


 if((fp=fopen("sched.dat","w"))==0)              /* Meglio aprire in append? */
   {printf("unable to open sched.dat\n"); return;}
fprintf(fp, "%s\t\t\tfor obs\t %s\n",sats->name, obs->name);

tstop=tstart+howlong;
/* Se al termine dello sched el>0 allora si allunga tstop fino al tramonto */
if(elev(tstop, sats, obs)>0.)  LOS(sats, obs, tstart+howlong, &tstop);
/* Ma se la LOS() non riesce a trovare un tramonto (e restituisce zero per
   questo), allora si usa il tstop impostato */
if(tstop==0) tstop=tstart+howlong;

for(t=tstart; t<=tstop; t+=tinc) {
theta=tetha(t+J2000, obs->lon*180./M_PI);
(void) (*sgp4)(sats, (double)t);
obsijk(theta, obs, &xo, &yo, &zo);

rx=sats->x[0]*R-xo; ry=sats->x[1]*R-yo; rz=sats->x[2]*R-zo; 
rS=sin(obs->lat)*cos(theta)*rx+sin(obs->lat)*sin(theta)*ry-cos(obs->lat)*rz;
rE=-sin(theta)*rx+cos(theta)*ry;
rZ=cos(obs->lat)*cos(theta)*rx+cos(obs->lat)*sin(theta)*ry+sin(obs->lat)*rz;

r=sqrt(rS*rS+rE*rE+rZ*rZ);
el=asin(rZ/r); az=atan2(-rE,rS);

 if(el<-0.1/180.*M_PI) {               /* Confrontando con 0. esatto si rischia
			         	  di perdere il primo passo, cosi` 
					  nella tabella la prima riga di ogni
					  passaggio iniziera` molto spesso con
					  EL= -0.0 */
   AOS(sats, obs, t, &tAOS); t=tAOS; 
   fprintf(fp, "-----\n");
 }

 else{
   DTstring(t+J2000, b, 1, 1, Zloc, 0);

   /* Questi servono per calcolare l'illuminazione */
   sunpos(t+J2000, obs, &sun); 
   sat.x=sats->x[0]*R; sat.y=sats->x[1]*R; sat.z=sats->x[2]*R;

   fprintf(fp, "%s AZ= %5.1f   EL= %5.1f   Range= %6.0f Illum: %c\n",
	   b, az/M_PI*180+180, el/M_PI*180, r, sat_illum(sat, sun));
 }
}

 mvprintw(14,1,"                    ");
 fclose(fp);
}


/* La chiamata a questa funzione non deve essere bloccante, perche' mentre
si legge l'help non si deve bloccare il controllo del rotore; piuttosto sara`
bene usare la 'h' come toggle per la hsw (controllare che durante il normale
funzionamento non vengano scritti altri dati in  quelle posizioni, altrimenti
disabilitare quest'ultime quando l'help e` attivo. */
void help(void){  
WINDOW *hsw;

 timeout(-1);
 if((hsw=newwin(7, 79, 13, 0))==NULL) return;
 box(hsw, ACS_VLINE, ACS_HLINE);

 mvwprintw(hsw, 0,29, " TRK v%s -  Help Screen ", __VERSION);
 mvwprintw(hsw, 1,2, "g - Graph EL display");
 mvwprintw(hsw, 2,2, "h - Help (this screen)");
 mvwprintw(hsw, 3,2, "r - Rotor (toggle)");
 mvwprintw(hsw, 4,2, "a - Abs/Rel time(toggle)");
 mvwprintw(hsw, 5,2, "Curs U/D - Set active obs");

 mvwprintw(hsw, 1,29, "q - Quit TRK");
 mvwprintw(hsw, 2,29, "s - Schedule");
 mvwprintw(hsw, 3,29, "z - UTC/Local (toggle)");
 mvwprintw(hsw, 4,29, "o - Set new observer");
 mvwprintw(hsw, 5,29, "Pg U/D - Prev/Next obs");

 mvwprintw(hsw, 1,53, "Space - Suspend");
 mvwprintw(hsw, 2,53, "Curs L/R - Prev/Next sat");
 mvwprintw(hsw, 3,53, "+/- - Add/Remove Obs");
 mvwprintw(hsw, 3,53, "t - Calc at fixed time");
 mvwprintw(hsw, 4,53, "m - Multisat (toggle)");
 mvwprintw(hsw, 5,53, "0 - Real horizon (toggle)");

 wrefresh(hsw);

 getch();
 timeout(0); wclear(hsw);  wrefresh(hsw); delwin(hsw); refresh();
}
