/*
 * TRK - Satellite tracking program based on Norad SGP/SDP model with
 *       curses interface
 *
 *	by Lapo Pieri IK5NAX  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include "sgp.h"

/*============================================================================
                           Definizioni e Costanti
 ============================================================================*/
#define R        6378.135
#define	J2000	2451545.0

/* Geodide terrestre di riferimento (di quale anno e`?) */
#define aT       6378.140
#define bT       6356.755  
#define eT       0.08181922


#define mu       3.986012e5  /* Km^3/s^2 */

/*============================================================================
                                   Strutture
 ============================================================================*/
/* Human Readable Time structure */
struct hr_time {
	int	y;	/* anno */
	int	mo;	/* mese */
	int	dn;	/* giorno del mese */
	int	dw;	/* giorno della settimana */
	int	dy;	/* giorno dell'anno */
	int	h;	/* ore */
	int	mi;	/* minuti */
	int	s;	/* secondi */
	int	ms;	/* millisecondi */
	};

struct gc_time{
	int	tzmw;	/* fuso orario in minuti a ovest di Greenwich */
	int	dst;	/* =1 se c'e' l'ora legale, 0 altrimenti */
	} TZsyst;

struct satdb {
  char name[40];          /* Nome convnzionale */
  char group[2];          /* Gruppo */
  long norad;              /* Designazione NORAD (senza classificazione) */ 
  float beacon[8];        /* Frequenza dei beacon (MHz) */
  float uplink[8];        /* Centro banda uplink (MHz) */
  float downlink[8];      /* Centro banda downlink (MHz) */
  char matrix[8][7][256]; /* Il primo indice e` la descrizione della matrice
			     IF o il modo operativo dipendente dal giorno 
			     della settimana (secondo indice) e della phase
			     (terzo indice) */
}sdb;

typedef struct satposition{
  char name[40];               /* Nome convenzionale (linea 0 tle)    */
  long norad;                  /* Numero di catalogo norad            */
  char class;                  /* Classificazione norad (sempre U)    */
  double x;                    /* Coordinate ECI, ripetute qui,       */
  double y;                    /* MA differenti come unita` di misura */
  double z;                    /* da quelle in SATS  (qui in Km)      */
  double az;                   /* Azimuth                             */
  double el;                   /* Elevazione                          */
  double range;                /* Distanza                            */
  double rangerate;            /* Modulo della velocita` rispetto 
				  all'osservatore                     */
  double phase;                /* Anomalia vera in 256-esimi          */
  double ra;                   /* Ascensione retta                    */
  double dec;                  /* Declinazione                        */
  double ssplon;               /* Longitudine del sub satellite point */
  double ssplat;               /* Longitudine del sub satellite point */
  double height;               /* Quota dalla superficie terrestre    */
  long revn;                   /* Numero dell'orbita                  */
} satpos;

typedef struct observer{
  char name[16];       /* Nome convenzionale */
  char descr[70];      /* Descrizione */

  struct observer *prev;   /* linked list */
  struct observer *next;   /* linked list */

  float lat;           /* Latitudine geodetica positiva verso nord [rad] */
  float lon;           /* Longitudine geodetica positiva verso est [rad] */
  float h0;            /* Altezza sul livello del mare [km] */
  char qth[6];         /* QTH locator 6 caratteri [maiuscoli] */
  float horiz[2][72];     /* Orizzonte locale, az,elmin, (al massimo un punto 
			  ogni 5gradi)*/
}OBS;

typedef struct configuration_file{
  char obsdatf[80];            /* Nome file osservatori                    */ 
  char tlef[80];               /* Nome file tle                            */
  char satdbf[80];             /* Nome file database satelliti             */
  char gelf[80];               /* Nome file comandi gpt grafico elevazione */
  char gpttf[80];              /* Nome file rilevamento presenza gpt       */
  char fiforot[80];            /* Nome fifo comandi verso il rotore (o Y)  */
}cfg_file;


/*============================================================================
                            Prototipi delle funzioni
 ============================================================================*/

/* NON tornano le suddivisioni fra i varii file! */

/* in trk2.c */
void single_sat_calc(double, SAT *, OBS *, satpos *), graphel(SAT *, OBS *obs);
void SSP(double, double, double, double, double *, double *, double *);
double elev(double, SAT *, OBS *), getT(void);
void rotor(float, float), help(void), obsinfo(OBS *),
  derivtle(double, SAT *, int);
SAT *getsatname(SAT *);
void read_db(FILE *, SAT *);
void sched(SAT *, OBS *), multisat(double, SAT *, OBS *, int, int);
char sat_illum(satpos, satpos);


/* in qthloc.c */
int lonlat2qthloc(float, float, char*), qthloc2lonlat(char*, float*, float*);


/* in coord.c */
void ijk2sez(double, double, double, double, double, 
	     double *, double *, double *);
void sez2ijk(double, double, double, double, double, 
	     double *, double *, double *);
void sez2azel(double , double , double ,
	      double *, double *, double *);
void radec2azel(double, double, double, double,
		double *, double *);
void elaz2radec(double, double, double, double,
		double *, double *);
void ijk2radec(double, double, double, double *, double *);
void radec2ijk(double, double, double *, double *, double *);


/* in sunmoon.c */
void sunpos(double, OBS *, satpos *);
void moonpos(double, OBS *, satpos *);


/* in t2.c */
void UTCstring(double, char *, int), jd2hr(double, struct hr_time *),
     DTstring(double, char *, int, int, int, int);
double JDsyst(void), tetha(double, float), hr2jd(struct hr_time *);
char *hms(double);


/* in obs.c */
void obsijk(double, OBS *, float *, float *, float *);
OBS *getobs(int);
double localhoriz(OBS *, double, int);


/* in predict.c */
int AOS(SAT *, OBS *, double, double *), 
  LOS(SAT *, OBS *, double, double *);


/* in file.c */
int setup_config_file(int, char **, cfg_file *);
