/*
 * User library for WP service access
 *
 * F1OAT 970831
 */

/* #include "wpdefs.h" */

#include <malloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/ioctl.h>

#include "wp.h"

#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "rsconfig.h"
#include "procutils.h"
#include "fpac.h"

static int wp_socket = -1;
int wp_debug = 0;

/*****************************************************************************
* Private internal functions section
*****************************************************************************/

static ax25_address *call_clean(ax25_address *call)
{
	/* Cleans the callsign */
	char *ptr = (char *)call;
	
	ptr[0] &= 0xfe;
	ptr[1] &= 0xfe;
	ptr[2] &= 0xfe;
	ptr[3] &= 0xfe;
	ptr[4] &= 0xfe;
	ptr[5] &= 0xfe;
	ptr[6] &= 0x1e;

	return(call);
}

static int wait_for_local_wp(int timeout)
{
	struct proc_rs *rp, *rlist;
	time_t timer = time(NULL) + timeout;

	while (time(NULL) <= timer) {
		if ((rlist = read_proc_rs()) == NULL) {
			if (errno) perror("do_links: read_proc_ax25");
			return -1;
		}

		for (rp = rlist; rp != NULL; rp = rp->next) {
			if (	(strcasecmp(rp->dest_addr, "*") == 0) && 
				(strcasecmp(rp->dest_call, "*") == 0) &&
				(strcasecmp(rp->src_call, "WP-0") == 0) &&				
				(strcasecmp(rp->dev, "rose0") == 0))
			{
				return 0;
				break;
			}
		}
		
		free_proc_rs(rlist);
		sleep(1);
	}
	
	return -1;
}

/*****************************************************************************
* WP inter-node and internal protocol functions 
*****************************************************************************/

int wp_send_pdu(int s, wp_pdu *pdu)
{
	int L = 0, i, rc;
	unsigned char p[128];
	
	p[L++] = pdu->type;
	
	switch (pdu->type) {
	case wp_type_set:
	case wp_type_get_response:
		*(unsigned long *)&p[L] = htonl(pdu->data.wp.date);
		L += 4;
		memcpy(&p[L], &pdu->data.wp.address.srose_addr, 5);
		L += 5;
		memcpy(&p[L], &pdu->data.wp.address.srose_call, 7);
		L += 7;
		p[L++] = (unsigned char)pdu->data.wp.address.srose_ndigis;
		for (i=0; i<pdu->data.wp.address.srose_ndigis; i++) {
			memcpy(&p[L], &pdu->data.wp.address.srose_addr, 7);
			L += 7;
		}
		memcpy(&p[L], &pdu->data.wp.name, 22);
		L += 22;
		memcpy(&p[L], &pdu->data.wp.city, 22);
		L += 22;
		memcpy(&p[L], &pdu->data.wp.qra_locator, 7);
		L += 7;			  
		break;		
	case wp_type_get:
		memcpy(&p[L], &pdu->data.call, 7);
		L += 7;		
		break;
	case wp_type_response:
		p[L++] = pdu->data.status;
		break;
	case wp_type_ascii:
		pdu->data.string[sizeof(pdu->data.string)-1] = 0;
		strncpy(&p[L], pdu->data.string, sizeof(pdu->data.string));
		L += strlen(pdu->data.string);
		break;	
	case wp_type_vector_request:
	case wp_type_vector_response:
		*(unsigned long *)&p[L] = htonl(pdu->data.vector.date_base);
		L += 4;
		*(unsigned short *)&p[L] = htons(pdu->data.vector.interval);
		L += 2;		
		*(unsigned short *)&p[L] = htons(pdu->data.vector.seed);
		L += 2;
		for (i=0; i<WP_VECTOR_SIZE; i++) {
			*(unsigned short *)&p[L] = htons(pdu->data.vector.crc[i]);
			L += 2;		
		}
		break;
	default: 
		return -1;
		break;		
	}
	
	rc = write(s, p, L);
	if (rc < 0) {
		perror("write");
		return rc;
	}
	return 0;
}

int wp_receive_pdu(int s, wp_pdu *pdu)
{
	int rc, L = 0, i;
	unsigned char p[128];
	fd_set rfds;
	struct timeval timeout;
	
	memset(p, 0, sizeof(p));
	memset(pdu, 0, sizeof(*pdu));
	
	FD_ZERO(&rfds);
	FD_SET(s, &rfds);
	timeout.tv_sec = WP_API_TIMEOUT;
	timeout.tv_usec = 0;
	
	rc = select(s+1, &rfds, NULL, NULL, &timeout);
	if (rc <= 0) {
		fprintf(stderr, "WP API timeout\n");
		return -1;
	}
	
	rc = read(s, p, sizeof(p));
	
	if (rc <= 0) return -1;	/* Disconnection or error */
	
	pdu->type = p[L++];
	
	switch (pdu->type) {
	case wp_type_set:
	case wp_type_get_response:
		pdu->data.wp.date = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		memcpy(&pdu->data.wp.address.srose_addr, &p[L], 5);
		L += 5;
		memcpy(&pdu->data.wp.address.srose_call, &p[L], 7);
		L += 7;
		pdu->data.wp.address.srose_ndigis = p[L++];
		for (i=0; i<pdu->data.wp.address.srose_ndigis; i++) {
			memcpy(&pdu->data.wp.address.srose_addr, &p[L], 7);
			L += 7;
		}
		memcpy(&pdu->data.wp.name, &p[L], 22);
		L += 22;
		memcpy(&pdu->data.wp.city, &p[L], 22);
		L += 22;
		memcpy(&pdu->data.wp.qra_locator, &p[L], 7);
		L += 7;			  
		break;		
	case wp_type_get:
		memcpy(&pdu->data.call, &p[L], 7);
		L += 7;		
		break;
	case wp_type_response:
		pdu->data.status = p[L++];
		break;
	case wp_type_ascii:
		memcpy(pdu->data.string, &p[L], MIN(rc-L, sizeof(pdu->data.string)));
		pdu->data.string[sizeof(pdu->data.string)-1] = 0;
		L = rc;
		break;
	case wp_type_vector_request:
	case wp_type_vector_response:
		pdu->data.vector.date_base = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.vector.interval = ntohs(*(unsigned short *)&p[L]);
		L += 2;		
		pdu->data.vector.seed = ntohs(*(unsigned short *)&p[L]);
		L += 2;
		for (i=0; i<WP_VECTOR_SIZE; i++) {
			pdu->data.vector.crc[i] = ntohs(*(unsigned short *)&p[L]);
			L += 2;		
		}
		break;
	default: 
		syslog(LOG_WARNING, "Received unknown pdu type %d\n", pdu->type);
		return -1;
		break;		
	}
	
	if (L != rc) {
		syslog(LOG_WARNING, "Received wp pdu bad length (type %d, want %d, received %d)\n", pdu->type, L, rc); 
	}
		
	return 0;
}

/*****************************************************************************
* Public API section
*****************************************************************************/

/*
 * Check if a callsign is valid or not.
 * 
 * Return 0 if correct or -1 if error
 */
 
int wp_check_call(const char *s)
{
	int len = 0;
	int nums = 0;
	int ssid = 0;
	char *p[1];

	if (s == NULL)
		return -1;
	while (*s && *s != '-') {
		if (!isalnum(*s))
			return -1;
		if (isdigit(*s))
			nums++;
		len++;
		s++;
	}
	if (*s == '-') {
		if (!isdigit(*++s))
			return -1;
		ssid = strtol(s, p, 10);
		if (**p)
			return -1;
	}
	if (len < 4 || len > 6 || !nums || nums > 2 || ssid < 0 || ssid > 15)
		return -1;
	return 0;
}

/*
 * Return the number of valid records in the database
 *
 * TO BE implemented using ROSE L3 communication
 *
 */
 
long wp_nb_records(void)
{
	long nb = 0L;
	wp_header wph;
	FILE *fptr = fopen(FPACWP, "r");
	
	if (fptr == NULL)
		return(0L);

	if (fread(&wph, sizeof(wph), 1, fptr))
		nb = (long)wph.nb_record;
		
	fclose(fptr);
	
	return(nb);
}

/*
 * Open a connection with the specified server. Connection
 * is open in ROSE mode with localhost.
 * 
 * If remote is NULL, this is a local call !
 *
 * Return the socket handle or -1 if error
 */
 
int wp_open_remote(char *source_call, struct sockaddr_rose *remote, int non_block)
{
	struct sockaddr_rose rose;
	int fd, rc;
	char *rs_addr = rs_get_addr(NULL);
	
	if (!rs_addr) return -1;
	
	fd = socket(AF_ROSE, SOCK_SEQPACKET, 0);
	if (fd < 0) return -1;
		
	rose.srose_family = AF_ROSE;
	rose.srose_ndigis = 0;
	convert_call_entry(source_call, rose.srose_call.ax25_call);
	convert_rose_address(rs_addr, rose.srose_addr.rose_addr);
	if (bind(fd, (struct sockaddr *)&rose, sizeof(struct sockaddr_rose)) == -1) {
		perror("wp_open: bind");
		close(fd);
		return -1;
	}

	
	if (non_block) {
		int flag = 1;
		rc = ioctl(fd, FIONBIO, &flag);
		if (rc) {
			perror("wp_open_remote:ioctl FIONBIO");
			close(fd);
			return -1;
		}
	}
	
	if (!remote) {	
		/* Wait for local WP server to be ready */
		if (wait_for_local_wp(60)) return -1;
		/* Reuse the rose structure : same X.121 address */
		convert_call_entry("WP", rose.srose_call.ax25_call);
		rc = connect(fd, (struct sockaddr *)&rose, sizeof(struct sockaddr_rose));		
	}
	else {
		rc = connect(fd, (struct sockaddr *)remote, sizeof(struct sockaddr_rose));		
	}
		
	if (rc && (errno != EINPROGRESS)) {
		close(fd);
		return -1;
	}
	return fd;
}

int wp_listen(void)
{
	struct sockaddr_rose rose;
	int fd;
	char *rs_addr;
			
	rose.srose_family = AF_ROSE;
	rose.srose_ndigis = 0;
	convert_call_entry("WP", rose.srose_call.ax25_call);
	if (!wp_debug) rs_addr = rs_get_addr(NULL);
	else rs_addr = rs_get_addr("rose1");

	if (!rs_addr) return -1;
	convert_rose_address(rs_addr, rose.srose_addr.rose_addr);
	
	fd = socket(AF_ROSE, SOCK_SEQPACKET, 0);
	if (fd < 0) return -1;
		
	if (bind(fd, (struct sockaddr *)&rose, sizeof(struct sockaddr_rose)) == -1) {
		perror("wp_listen: bind");
		close(fd);
		return -1;
	}

	listen(fd, 5);
	return fd;
}

/*
 * Open a connection with the specified server. Connection
 * is open in ROSE mode with localhost.
 *
 * Return 0 if sucessful
 */
 
int wp_open(char *client)
{	
	wp_socket = wp_open_remote(client, NULL, 0);
	if (wp_socket < 0) return -1;
	return 0;
}

/*
 * Close the currently selected WP connection
 *
 */
 
void wp_close()
{
	if (wp_socket >=0) {
		close(wp_socket);
		wp_socket = -1;
	}
}


/*
 * Update (or create) a WP record
 *
 * Return 0 if successful 
 */
 
int wp_update_addr(struct sockaddr_rose *addr)
{
	wp_t wp;

	memset(&wp, 0, sizeof(wp_t));		
	wp_get(&addr->srose_call, &wp);
	wp.date = time(NULL);
	wp.address = *addr;
	return wp_set(&wp);
}

/*
 * Search a WP record and return associated sockaddr_rose.
 *
 * Return 0 if found.
 */
 
int wp_search(ax25_address *call, struct sockaddr_rose *addr)
{
	wp_t wp;
	int rc;

	rc = wp_get(call, &wp);
	if (rc) return -1;
	
	*addr = wp.address;
	return 0;
}

/*
 * Search and return a WP record.
 *
 * Return 0 if found.
 */
 
int wp_get(ax25_address *call, wp_t *wp)
{
	wp_pdu pdu;
	int rc;
	
	call_clean(call);
	pdu.type = wp_type_get;
	pdu.data.call = *call;
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc) return -1;
	rc = wp_receive_pdu(wp_socket, &pdu);
	if (rc) return -1;
	if (pdu.type == wp_type_get_response) {
		*wp = pdu.data.wp;
		wp->address.srose_family = AF_ROSE;
		return 0;
	}
	
	return -1;
}

/*
 * Update (or create) a full WP record
 *
 * Return 0 if successful
 */
 
int wp_set(wp_t *wp)
{
	int rc;
	wp_pdu pdu;
	
	wp->date = time(NULL);
	call_clean(&wp->address.srose_call);
	if (wp->address.srose_ndigis)
		call_clean(&wp->address.srose_digi);
		
	pdu.type = wp_type_set;
	pdu.data.wp = *wp;		
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc) return -1;
	rc = wp_receive_pdu(wp_socket, &pdu);
	if (rc) return -1;
	if (pdu.type == wp_type_response && pdu.data.status == WP_OK) {
		return 0;
	}
				
	return -1;
}
