
/*
 *
 * fpacstat.c (from mon_ax25 & mheardd.c)
 *
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <linux/if_ether.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <netdb.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "config.h"
#include "axutils.h"
#include "axconfig.h"
#include "daemon.h"
#include "version.h"
#include "mheard.h"
#include "fpac.h"

#define	KISS_MASK	0x0F
#define	KISS_DATA	0x00

#define	PID_SEGMENT	0x08
#define	PID_ARP		0xCD
#define	PID_NETROM	0xCF
#define	PID_IP		0xCC
#define	PID_ROSE	0x01
#define	PID_TEXNET	0xC3
#define	PID_FLEXNET	0xCE
#define	PID_TEXT	0xF0
#define	PID_PSATFT	0xBB
#define	PID_PSATPB	0xBD

#define	I		0x00
#define	S		0x01
#define	RR		0x01
#define	RNR		0x05
#define	REJ		0x09
#define	U		0x03
#define	SABM	0x2F
#define	SABME	0x6F
#define	DISC	0x43
#define	DM		0x0F
#define	UA		0x63
#define	FRMR	0x87
#define	UI		0x03

#define	PF		0x10
#define	EPF		0x01

#define	MMASK		7

#define	HDLCAEB		0x01
#define	SSID		0x1E
#define	SSSID_SPARE	0x40
#define	ESSID_SPARE	0x20

#define	ALEN		6
#define	AXLEN		7

extern int errno;

#define I_MASK 1
#define U_MASK 2
#define S_MASK 4

/* Period to flush the data to disk */
#define PERIOD 60L

char node_call[10];
int logging = FALSE;
int cr = FALSE;
time_t start;

/* Configuration structure */
cfg_t cfg;

typedef struct count
{
	long size;
	long retsize;
	long uisize;
	long sabm;
	long sabme;
	long disc;
	long ua;
	long dm;
	long i;
	long rr;
	long rnr; 
	long rej; 
	long frmr;
	unsigned short nr;
	unsigned short ns;
	unsigned short frm[8];
} count_t;

typedef struct adj
{
	char call[10];
	char *port;
	count_t to;
	count_t fm;
	struct adj *next;
} adj_t;

adj_t *head = NULL;

static int ftype(unsigned char *, int *, int);

#define FPACLOCK "/var/ax25/fpac/fpacstat.lck"
#define FPACSTAT "/var/ax25/fpac/fpacstat.dat"

#define CR(fptr) \
{ \
	if (cr) fputc('\r', fptr); else fputc('\n', fptr); \
}

static void terminate(int sig)
{
	if (logging) 
	{
		syslog(LOG_INFO, "terminating on SIGTERM\n");
		closelog();
	}

	exit(0);
}

static void write_stats(void)
{
	int fd;
	adj_t *n;
	char *ptr;
	FILE *fptr;
	struct tm *tm;

	/* Creates the lock file if it does not exist */
	fd = open(FPACLOCK, O_RDWR|O_CREAT|O_EXCL, S_IREAD|S_IWRITE);
	if (fd == -1)
	{
		/* Locak file exists... Abort upodate */
		return;
	}
	close(fd);
		
	fptr = fopen(FPACSTAT, "w");
	if (fptr)
	{
		tm = localtime(&start);
		ptr = asctime(tm);
		ptr[strlen(ptr)-1] = '\0';
		fprintf(fptr,  "Starting date : %s", ptr);
		CR(fptr);CR(fptr);
		fprintf(fptr,  "   Adjacent       size retsize i-frame     rr  rnr  rej sabm disc   ua   dm");
		CR(fptr);

		for (n = head ; n ; n = n->next)
		{
			fprintf(fptr,  "to %-10s %8ld %7ld %7ld %6ld %4ld %4ld %4ld %4ld %4ld %4ld",
							n->call, n->to.size, n->to.retsize, n->to.i, n->to.rr, n->to.rnr, 
							n->to.rej, n->to.sabm, n->to.disc, n->to.ua, n->to.dm);
			CR(fptr);
			fprintf(fptr,  "fm %-10s %8ld         %7ld %6ld %4ld %4ld %4ld %4ld %4ld %4ld",
							n->call, n->fm.size, n->fm.i, n->fm.rr, n->fm.rnr, 
							n->fm.rej, n->fm.sabm, n->fm.disc, n->fm.ua, n->fm.dm);
			CR(fptr); CR(fptr);
		}
		fclose(fptr);
	}
	
	/* Deletes the lock file */	
	unlink(FPACLOCK);
}

static int ftype(unsigned char *data, int *type, int extseq)
{
	if (extseq) 
	{
		if ((*data & 0x01) == 0) 
		{	/* An I frame is an I-frame ... */
			*type = I;
			return 2;
		}
		if (*data & 0x02) 
		{
			*type = *data & ~PF;
			return 1;
		} 
		else 
		{
			*type = *data;
			return 2;
		}
	} 
	else 
	{
		if ((*data & 0x01) == 0) 
		{	/* An I frame is an I-frame ... */
			*type = I;
			return 1;
		}
		if (*data & 0x02) 
		{	/* U-frames use all except P/F bit for type */
			*type = *data & ~PF;
			return 1;
		} 
		else 
		{
			/* S-frames use low order 4 bits for type */
			*type = *data & 0x0F;
			return 1;
		}
	}
}

/* Mark acknowledged frames */
static void mark_ok(count_t *to, count_t *fm, unsigned short tst)
{
	unsigned short nr = fm->nr;

	while (nr != tst) {
		to->frm[nr] = 0;
		nr = (nr + 1) % 8;
	}
	return;
}

static int node(char *port, char *fm_call, char *to_call, count_t **fm, count_t **to)
{
	adj_t *n;
	
	*fm = *to = NULL;
	
	for (n = head ; n ; n = n->next)
	{
		if ((strcmp(n->call, fm_call) == 0) && (strcmp(cfg.callsign, to_call) == 0) && (strcmp(port, n->port) == 0))
		{
			*fm = &n->fm;
			*to = &n->to;
			return TRUE;
		}
		if ((strcmp(n->call, to_call) == 0) && (strcmp(cfg.callsign, fm_call) == 0) && (strcmp(port, n->port) == 0))
		{
			*fm = &n->to;
			*to = &n->fm;
			return TRUE;
		}
	}
	return FALSE;
}

void getax25(int s)
{
	unsigned char buffer[1500];
	unsigned char *data;
	unsigned int pid;
	char from_call[10];
	char to_call[120];
	char type_pk[80];
	int type;
	int nr;
	int pf;
	int size;
	char *port = NULL;
	char *ptr;
	int asize;
	int ctlen, end, extseq;
	struct sockaddr sa;
	count_t *fm, *to;


	asize = sizeof(sa);

	if ((size = recvfrom(s, buffer, sizeof(buffer), 0, &sa, &asize)) == -1) 
	{
		if (logging) 
		{
			syslog(LOG_ERR, "recv: %m");
			closelog();
		}
		return;
	}
	
	if ((port = ax25_config_get_name(sa.sa_data)) == NULL) 
	{
		if (logging)
			syslog(LOG_WARNING, "unknown port '%s'\n", sa.sa_data);
		return;
	}

	data = buffer;

	if ((*data & KISS_MASK) != KISS_DATA)
		return;

	data++;
	size--;

	if (size < (AXLEN + AXLEN + 1)) 
	{
		if (logging)
			syslog(LOG_WARNING, "packet too short\n");
		return;
	}

	if (!ax25validate(data + 0) || !ax25validate(data + AXLEN)) 
	{
		if (logging)
			syslog(LOG_WARNING, "invalid callsign on port %s\n", port);
		return;
	}

	strcpy(from_call, ax2asc((ax25_address *)(data + AXLEN)));
	ptr = strstr(from_call, "-0");
	if (ptr)
		*ptr = '\0';

	strcpy(to_call,   ax2asc((ax25_address *)data));
	ptr = strstr(to_call, "-0");
	if (ptr)
		*ptr = '\0';

	/* Check if the packet is a packet to an adjacent */
	if (!node(port, from_call, to_call, &fm, &to))
		return;
	
	extseq = ((data[AXLEN + ALEN] & SSSID_SPARE) != SSSID_SPARE);
	end    = (data[AXLEN + ALEN] & HDLCAEB);

	data += (AXLEN + AXLEN);
	size -= (AXLEN + AXLEN);

	while (!end) 
	{
		strcat(to_call, " ");
		strcat(to_call, ax2asc((ax25_address *)data));
		ptr = strstr(to_call, "-0");
		if (ptr)
			*ptr = '\0';

		end = (data[ALEN] & HDLCAEB);
		
		data += AXLEN;
		size -= AXLEN;
	}

	if (size == 0) 
	{
		if (logging)
			syslog(LOG_WARNING, "packet too short\n");
		return;
	}

	ctlen = ftype(data, &type, extseq);
	size -= ctlen;

	switch (type) 
	{
		case SABM:
			++fm->sabm;
			strcpy(type_pk, "SABM");
			break;
		case SABME:
			++fm->sabme;
			strcpy(type_pk, "SABME");
			break;
		case DISC:
			++fm->disc;
			strcpy(type_pk, "DISC");
			break;
		case UA:
			++fm->ua;
			strcpy(type_pk, "UA");
			break;
		case DM:
			++fm->dm;
			strcpy(type_pk, "DM");
			break;	
		case RR:
			++fm->rr;
			nr = (*data >> 5) & 7;
			mark_ok(to, fm, nr);
			fm->nr = nr;
			pf   = *data & PF;
			sprintf(type_pk, "RR%d", fm->nr);
			break;
		case RNR:
			++fm->rnr;
			nr = (*data >> 5) & 7;
			mark_ok(to, fm, nr);
			fm->nr = nr;
			pf   = *data & PF;
			sprintf(type_pk, "RNR%d", fm->nr);
			break;
		case REJ:
			++fm->rej;
			nr = (*data >> 5) & 7;
			mark_ok(to, fm, nr);
			fm->nr = nr;
			pf = *data & PF;
			sprintf(type_pk, "REJ%d", fm->nr);
			break;
		case FRMR:
			++fm->frmr;
			strcpy(type_pk, "FRMR");
			break;
		case I:
			++fm->i;
			fm->ns = (*data >> 1) & 7;
			nr = (*data >> 5) & 7;
			mark_ok(to, fm, nr);
			fm->nr = nr;
			pf = *data & EPF;
			fm->size += (size-1);
			if (fm->frm[fm->ns] == 1)	/* frame already sent */
				fm->retsize += (size-1);
			fm->frm[fm->ns] = 1;	/* frame sent */
			sprintf(type_pk, "I%d%d", fm->ns, fm->nr);
			break;
		case UI:
			strcpy(type_pk, "UI");
			fm->uisize += size;
			break;
		default:
			strcpy(type_pk, "??");
			break;
	}
	
	data += ctlen;

	if (type == I || type == UI) 
	{
		pid = (*data++);
		size--;
	}
	else
	{
		pid = 0;
	}
	
	/* 
	sprintf(str, "%s^%s^%s^%s^%02x^%d^", port, from_call, to_call, type_pk, pid, size);
	printf("%s\n", str);

	if (size > 0)
	{
		data[size] = '\0';
		printf("%s\n", data);
	}
	*/
}

int main(int argc, char **argv)
{
	int s;
	int nb;
	fd_set read_fdset;
	node_t *n;
	adj_t *h;
	time_t date;
	struct timeval timeval;
	
	while ((s = getopt(argc, argv, "lvc")) != -1) 
	{
		switch (s) {
			case 'c':
				cr = TRUE;
				break;
			case 'l':
				logging = TRUE;
				break;
			case 'v':
				printf("fpacstat: %s\n", version);
				return 0;
			case ':':
				fprintf(stderr, "fpacstat: option needs an argument\n");
				return 1;
			case '?':
				fprintf(stderr, "Usage: fpacstat [-l] [-a address] [-m port_mon] [-o I|U|S] [-p port] [-v]\n");
				return 1;
		}
	}

	signal(SIGTERM, terminate);

	if (ax25_config_load_ports() == 0) 
	{
		fprintf(stderr, "fpacstat: no AX.25 port data configured\n");
		return 1;
	}

	/* Open the configuration file */
	if (cfg_open(&cfg) != 0)
	{
		return(1);
	}

	for (n = cfg.node, h = head ; n ; n = n->next)
	{
		int i;
		adj_t *adj = calloc(1, sizeof(adj_t));
		if (adj == NULL)
			exit(1);
			
		for (i = 0 ; i < 9 ; i++)
		{
			if (!isgraph(n->call[i]))
				break;
				
			adj->call[i] = n->call[i];
		}
		adj->call[i] = '\0';
		
		adj->port = n->port;
		adj->next = NULL;
		
		if (h)
		{
			h->next = adj;
			h = h->next;
		}
		else
		{
			h = head = adj;
		}
	}
	
	/* Ax25 receive socket */
	if ((s = socket(AF_INET, SOCK_PACKET, htons(ETH_P_ALL))) == -1) 
	{
		perror("fpacstat: socket ax25");
		return 1;
	}

	if (!daemon_start(FALSE)) 
	{
		fprintf(stderr, "fpacstat: cannot become a daemon\n");
		return 1;
	}

	/* Use syslog for error messages rather than perror/fprintf */
	if (logging) 
	{
		openlog("fpacstat", LOG_PID, LOG_DAEMON);
		syslog(LOG_INFO, "starting");
	}

	start = time(NULL);
	date  = time(NULL);
	
	for (;;) 
	{

		FD_ZERO(&read_fdset);
		FD_SET(s, &read_fdset);

		timeval.tv_usec = 0;
		timeval.tv_sec  = 1;

		nb = select(s + 1, &read_fdset, NULL, NULL, &timeval);
		if (nb == -1)
		{
			printf("select %s\n", strerror(errno));
			break;
		}

		if (nb && (FD_ISSET(s, &read_fdset)))
		{
			getax25(s);
		}

		if (date < time(NULL))
		{
			write_stats();
			date += PERIOD;
		}		
	}
	return 0;
}
