/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.runtime.command;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;
import java.util.function.ToLongFunction;
import java.util.stream.Stream;
import org.apache.nifi.NiFiServer;
import org.apache.nifi.diagnostics.DiagnosticsDump;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiagnosticsCommand
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(DiagnosticsCommand.class);
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss");
    private static final String DIAGNOSTICS_FILE_FORMAT = "diagnostic-%s.log";
    private final NiFiProperties properties;
    private final NiFiServer server;

    public DiagnosticsCommand(NiFiProperties properties, NiFiServer server) {
        this.properties = Objects.requireNonNull(properties, "Properties required");
        this.server = Objects.requireNonNull(server, "Server required");
    }

    @Override
    public void run() {
        if (this.properties.isDiagnosticsOnShutdownEnabled()) {
            File diagnosticDirectory = new File(this.properties.getDiagnosticsOnShutdownDirectory());
            if (diagnosticDirectory.mkdir()) {
                logger.info("Diagnostics Directory created [{}]", (Object)diagnosticDirectory);
            }
            this.purgeOldestFiles(diagnosticDirectory);
            String localDateTime = DATE_TIME_FORMATTER.format(LocalDateTime.now());
            String diagnosticsFileName = DIAGNOSTICS_FILE_FORMAT.formatted(localDateTime);
            File diagnosticsPath = new File(diagnosticDirectory, diagnosticsFileName);
            this.writeDiagnostics(diagnosticsPath);
        }
    }

    private void purgeOldestFiles(File diagnosticDirectory) {
        long maxSize = DataUnit.parseDataSize((String)this.properties.getDiagnosticsOnShutdownDirectoryMaxSize(), (DataUnit)DataUnit.B).longValue();
        int maxFileCount = this.properties.getDiagnosticsOnShutdownMaxFileCount();
        while (this.isFileCountExceeded(diagnosticDirectory, maxFileCount) || this.isSizeExceeded(diagnosticDirectory, maxSize)) {
            try {
                Path oldestFile = this.getOldestFile(diagnosticDirectory);
                Files.delete(oldestFile);
            }
            catch (IOException e) {
                logger.warn("Delete oldest diagnostics failed", (Throwable)e);
            }
        }
    }

    private void writeDiagnostics(File diagnosticsPath) {
        DiagnosticsDump diagnosticsDump = this.server.getDiagnosticsFactory().create(this.properties.isDiagnosticsOnShutdownVerbose());
        try (FileOutputStream fileOutputStream = new FileOutputStream(diagnosticsPath);){
            diagnosticsDump.writeTo((OutputStream)fileOutputStream);
        }
        catch (IOException e) {
            logger.warn("Write Diagnostics failed [{}]", (Object)diagnosticsPath, (Object)e);
        }
    }

    private Path getOldestFile(File diagnosticDirectory) throws IOException {
        Optional<Path> oldestFile;
        Comparator<Path> lastModifiedComparator = Comparator.comparingLong(p -> p.toFile().lastModified());
        try (Stream<Path> paths = Files.walk(diagnosticDirectory.toPath(), new FileVisitOption[0]);){
            oldestFile = paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).min(lastModifiedComparator);
        }
        return oldestFile.orElseThrow(() -> new RuntimeException(String.format("Could not find oldest file in diagnostic directory: %s", diagnosticDirectory)));
    }

    private boolean isFileCountExceeded(File diagnosticDirectory, int maxFileCount) {
        String[] fileNames = diagnosticDirectory.list();
        if (fileNames == null) {
            logger.warn("Diagnostics Directory [{}] listing files failed", (Object)diagnosticDirectory);
            return false;
        }
        return fileNames.length >= maxFileCount;
    }

    private boolean isSizeExceeded(File diagnosticDirectory, long maxSizeInBytes) {
        return this.getDirectorySize(diagnosticDirectory.toPath()) >= maxSizeInBytes;
    }

    private long getDirectorySize(Path path) {
        long size = 0L;
        try (Stream<Path> walk = Files.walk(path, new FileVisitOption[0]);){
            size = walk.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).mapToLong(this.getFileSizeByPathFunction()).sum();
        }
        catch (IOException e) {
            logger.warn("Directory [{}] size calculation failed", (Object)path, (Object)e);
        }
        return size;
    }

    private ToLongFunction<Path> getFileSizeByPathFunction() {
        return path -> {
            try {
                return Files.size(path);
            }
            catch (IOException e) {
                logger.warn("Failed to get size of file {}", path, (Object)e);
                return 0L;
            }
        };
    }
}

