/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.examples.filestore.cli;

import com.beust.jcommander.Parameter;
import com.beust.jcommander.Parameters;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiFunction;
import org.apache.ratis.client.api.DataStreamOutput;
import org.apache.ratis.examples.filestore.FileStoreClient;
import org.apache.ratis.examples.filestore.cli.Client;
import org.apache.ratis.io.StandardWriteOption;
import org.apache.ratis.io.WriteOption;
import org.apache.ratis.protocol.DataStreamReply;
import org.apache.ratis.protocol.RoutingTable;
import org.apache.ratis.thirdparty.io.netty.buffer.ByteBuf;
import org.apache.ratis.thirdparty.io.netty.buffer.PooledByteBufAllocator;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;

@Parameters(commandDescription="Load Generator for FileStore DataStream")
public class DataStream
extends Client {
    private static final String DESCRIPTION = "[DirectByteBuffer, MappedByteBuffer, NettyFileRegion]";
    @Parameter(names={"--type"}, description="[DirectByteBuffer, MappedByteBuffer, NettyFileRegion]", required=true)
    private String dataStreamType;
    @Parameter(names={"--syncSize"}, description="Sync every syncSize, syncSize % bufferSize should be zero,-1 means on sync", required=true)
    private int syncSize;

    public DataStream() {
        String expected = Arrays.asList(Type.values()).toString();
        Preconditions.assertTrue(expected.equals(DESCRIPTION), () -> "Unexpected description: [DirectByteBuffer, MappedByteBuffer, NettyFileRegion] does not equal to the expected string " + expected);
        this.dataStreamType = Type.NettyFileRegion.name();
        this.syncSize = -1;
    }

    int getSyncSize() {
        return this.syncSize;
    }

    private boolean checkParam() {
        if (this.syncSize != -1 && this.syncSize % this.getBufferSizeInBytes() != 0) {
            System.err.println("Error: syncSize % bufferSize should be zero");
            return false;
        }
        if (Type.valueOfIgnoreCase(this.dataStreamType) == null) {
            System.err.println("Error: dataStreamType should be one of [DirectByteBuffer, MappedByteBuffer, NettyFileRegion]");
            return false;
        }
        return true;
    }

    @Override
    protected void operation(List<FileStoreClient> clients) throws IOException, ExecutionException, InterruptedException {
        if (!this.checkParam()) {
            this.close(clients);
            return;
        }
        ExecutorService executor = Executors.newFixedThreadPool(this.getNumThread());
        List<String> paths = this.generateFiles(executor);
        this.dropCache();
        System.out.println("Starting DataStream write now ");
        RoutingTable routingTable = this.getRoutingTable(Arrays.asList(this.getPeers()), this.getPrimary());
        long startTime = System.currentTimeMillis();
        long totalWrittenBytes = this.waitStreamFinish(this.streamWrite(paths, clients, routingTable, executor));
        long endTime = System.currentTimeMillis();
        System.out.println("Total files written: " + this.getNumFiles());
        System.out.println("Each files size: " + this.getFileSizeInBytes());
        System.out.println("Total data written: " + totalWrittenBytes + " bytes");
        System.out.println("Total time taken: " + (endTime - startTime) + " millis");
        this.close(clients);
    }

    private Map<String, CompletableFuture<List<CompletableFuture<DataStreamReply>>>> streamWrite(List<String> paths, List<FileStoreClient> clients, RoutingTable routingTable, ExecutorService executor) {
        HashMap<String, CompletableFuture<List<CompletableFuture<DataStreamReply>>>> fileMap = new HashMap<String, CompletableFuture<List<CompletableFuture<DataStreamReply>>>>();
        int clientIndex = 0;
        for (String path : paths) {
            CompletableFuture future = new CompletableFuture();
            FileStoreClient client = clients.get(clientIndex % clients.size());
            ++clientIndex;
            CompletableFuture.supplyAsync(() -> {
                File file = new File(path);
                long fileLength = file.length();
                Preconditions.assertTrue(fileLength == this.getFileSizeInBytes(), "Unexpected file size: expected size is " + this.getFileSizeInBytes() + " but actual size is " + fileLength);
                Type type = Optional.ofNullable(Type.valueOfIgnoreCase(this.dataStreamType)).orElseThrow(IllegalStateException::new);
                TransferType writer = type.getConstructor().apply(path, this);
                try {
                    future.complete(writer.transfer(client, routingTable));
                }
                catch (IOException e) {
                    future.completeExceptionally(e);
                }
                return future;
            }, executor);
            fileMap.put(path, future);
        }
        return fileMap;
    }

    private long waitStreamFinish(Map<String, CompletableFuture<List<CompletableFuture<DataStreamReply>>>> fileMap) throws ExecutionException, InterruptedException {
        long totalBytes = 0L;
        for (CompletableFuture<List<CompletableFuture<DataStreamReply>>> futures : fileMap.values()) {
            long writtenLen = 0L;
            for (CompletableFuture<DataStreamReply> future : futures.get()) {
                writtenLen += future.join().getBytesWritten();
            }
            if (writtenLen != this.getFileSizeInBytes()) {
                System.out.println("File written:" + writtenLen + " does not match expected:" + this.getFileSizeInBytes());
            }
            totalBytes += writtenLen;
        }
        return totalBytes;
    }

    static class NettyFileRegionType
    extends TransferType {
        NettyFileRegionType(String path, DataStream cli) {
            super(path, cli);
        }

        @Override
        long write(FileChannel in, DataStreamOutput out, long offset, List<CompletableFuture<DataStreamReply>> futures) {
            long packetSize = this.getPacketSize(offset);
            futures.add(this.isSync(offset + packetSize) ? out.writeAsync(this.getFile(), offset, packetSize, StandardWriteOption.SYNC) : out.writeAsync(this.getFile(), offset, packetSize, new WriteOption[0]));
            return packetSize;
        }
    }

    static class MappedByteBufferType
    extends TransferType {
        MappedByteBufferType(String path, DataStream cli) {
            super(path, cli);
        }

        @Override
        long write(FileChannel in, DataStreamOutput out, long offset, List<CompletableFuture<DataStreamReply>> futures) throws IOException {
            long packetSize = this.getPacketSize(offset);
            MappedByteBuffer mappedByteBuffer = in.map(FileChannel.MapMode.READ_ONLY, offset, packetSize);
            int remaining = mappedByteBuffer.remaining();
            futures.add(this.isSync(offset + (long)remaining) ? out.writeAsync((ByteBuffer)mappedByteBuffer, StandardWriteOption.SYNC) : out.writeAsync((ByteBuffer)mappedByteBuffer, new WriteOption[0]));
            return remaining;
        }
    }

    static class DirectByteBufferType
    extends TransferType {
        DirectByteBufferType(String path, DataStream cli) {
            super(path, cli);
        }

        @Override
        long write(FileChannel in, DataStreamOutput out, long offset, List<CompletableFuture<DataStreamReply>> futures) throws IOException {
            int bufferSize = this.getBufferSize();
            ByteBuf buf = PooledByteBufAllocator.DEFAULT.directBuffer(bufferSize);
            int bytesRead = buf.writeBytes(in, bufferSize);
            if (bytesRead < 0) {
                throw new IllegalStateException("Failed to read " + bufferSize + " byte(s) from " + this + ". The channel has reached end-of-stream at " + offset);
            }
            if (bytesRead > 0) {
                CompletableFuture<DataStreamReply> f = this.isSync(offset + (long)bytesRead) ? out.writeAsync(buf.nioBuffer(), StandardWriteOption.SYNC) : out.writeAsync(buf.nioBuffer(), new WriteOption[0]);
                f.thenRun(buf::release);
                futures.add(f);
            }
            return bytesRead;
        }
    }

    static abstract class TransferType {
        private final String path;
        private final File file;
        private final long fileSize;
        private final int bufferSize;
        private final long syncSize;
        private long syncPosition = 0L;

        TransferType(String path, DataStream cli) {
            this.path = path;
            this.file = new File(path);
            this.fileSize = cli.getFileSizeInBytes();
            this.bufferSize = cli.getBufferSizeInBytes();
            this.syncSize = cli.getSyncSize();
            long actualSize = this.file.length();
            Preconditions.assertTrue(actualSize == this.fileSize, () -> "Unexpected file size: expected size is " + this.fileSize + " but actual size is " + actualSize + ", path=" + path);
        }

        File getFile() {
            return this.file;
        }

        int getBufferSize() {
            return this.bufferSize;
        }

        long getPacketSize(long offset) {
            return Math.min((long)this.bufferSize, this.fileSize - offset);
        }

        boolean isSync(long position) {
            if (this.syncSize > 0L && (position >= this.fileSize || position - this.syncPosition >= this.syncSize)) {
                this.syncPosition = position;
                return true;
            }
            return false;
        }

        List<CompletableFuture<DataStreamReply>> transfer(FileStoreClient client, RoutingTable routingTable) throws IOException {
            if (this.fileSize <= 0L) {
                return Collections.emptyList();
            }
            ArrayList<CompletableFuture<DataStreamReply>> futures = new ArrayList<CompletableFuture<DataStreamReply>>();
            DataStreamOutput out = client.getStreamOutput(this.file.getName(), this.fileSize, routingTable);
            try (FileChannel in = FileUtils.newFileChannel(this.file, StandardOpenOption.READ);){
                for (long offset = 0L; offset < this.fileSize; offset += this.write(in, out, offset, futures)) {
                }
            }
            catch (Throwable e) {
                throw new IOException("Failed to transfer " + this.path);
            }
            finally {
                futures.add(out.closeAsync());
            }
            return futures;
        }

        abstract long write(FileChannel var1, DataStreamOutput var2, long var3, List<CompletableFuture<DataStreamReply>> var5) throws IOException;

        public String toString() {
            return JavaUtils.getClassSimpleName(this.getClass()) + "{" + this.path + ", size=" + this.fileSize + "}";
        }
    }

    static enum Type {
        DirectByteBuffer(DirectByteBufferType::new),
        MappedByteBuffer(MappedByteBufferType::new),
        NettyFileRegion(NettyFileRegionType::new);

        private final BiFunction<String, DataStream, TransferType> constructor;

        private Type(BiFunction<String, DataStream, TransferType> constructor) {
            this.constructor = constructor;
        }

        BiFunction<String, DataStream, TransferType> getConstructor() {
            return this.constructor;
        }

        static Type valueOfIgnoreCase(String s2) {
            for (Type type : Type.values()) {
                if (!type.name().equalsIgnoreCase(s2)) continue;
                return type;
            }
            return null;
        }
    }
}

