/* 
   Unix SMB/CIFS mplementation.
   LDAP protocol helper functions for SAMBA
   
   Copyright (C) Andrew Tridgell  2004
   Copyright (C) Volker Lendecke 2004
   Copyright (C) Stefan Metzmacher 2004
   Copyright (C) Simo Sorce 2004
    
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/

#include "includes.h"
#include "libcli/util/asn_1.h"
#include "dlinklist.h"
#include "lib/events/events.h"
#include "lib/socket/socket.h"
#include "libcli/ldap/ldap.h"
#include "libcli/ldap/ldap_client.h"
#include "libcli/composite/composite.h"
#include "lib/stream/packet.h"
#include "auth/gensec/gensec.h"


/*
  create a new ldap_connection stucture. The event context is optional
*/
struct ldap_connection *ldap_new_connection(TALLOC_CTX *mem_ctx, 
					    struct event_context *ev)
{
	struct ldap_connection *conn;

	conn = talloc_zero(mem_ctx, struct ldap_connection);
	if (conn == NULL) {
		return NULL;
	}

	if (ev == NULL) {
		ev = event_context_init(conn);
		if (ev == NULL) {
			talloc_free(conn);
			return NULL;
		}
	}

	conn->next_messageid  = 1;
	conn->event.event_ctx = ev;

	/* set a reasonable request timeout */
	conn->timeout = 60;

	return conn;
}


/*
  the connection is dead
*/
static void ldap_connection_dead(struct ldap_connection *conn)
{
	struct ldap_request *req;

	while (conn->pending) {
		req = conn->pending;
		DLIST_REMOVE(req->conn->pending, req);
		req->state = LDAP_REQUEST_DONE;
		req->status = NT_STATUS_UNEXPECTED_NETWORK_ERROR;
		if (req->async.fn) {
			req->async.fn(req);
		}
	}	

	talloc_free(conn->tls);
	conn->tls = NULL;
}

/*
  handle packet errors
*/
static void ldap_error_handler(void *private_data, NTSTATUS status)
{
	struct ldap_connection *conn = talloc_get_type(private_data, 
						       struct ldap_connection);
	ldap_connection_dead(conn);
}


/*
  match up with a pending message, adding to the replies list
*/
static void ldap_match_message(struct ldap_connection *conn, struct ldap_message *msg)
{
	struct ldap_request *req;

	for (req=conn->pending; req; req=req->next) {
		if (req->messageid == msg->messageid) break;
	}
	/* match a zero message id to the last request sent.
	   It seems that servers send 0 if unable to parse */
	if (req == NULL && msg->messageid == 0) {
		req = conn->pending;
	}
	if (req == NULL) {
		DEBUG(0,("ldap: no matching message id for %u\n",
			 msg->messageid));
		talloc_free(msg);
		return;
	}

	/* add to the list of replies received */
	talloc_steal(req, msg);
	req->replies = talloc_realloc(req, req->replies, 
				      struct ldap_message *, req->num_replies+1);
	if (req->replies == NULL) {
		req->status = NT_STATUS_NO_MEMORY;
		req->state = LDAP_REQUEST_DONE;
		DLIST_REMOVE(conn->pending, req);
		if (req->async.fn) {
			req->async.fn(req);
		}
		return;
	}

	req->replies[req->num_replies] = talloc_steal(req->replies, msg);
	req->num_replies++;

	if (msg->type != LDAP_TAG_SearchResultEntry &&
	    msg->type != LDAP_TAG_SearchResultReference) {
		/* currently only search results expect multiple
		   replies */
		req->state = LDAP_REQUEST_DONE;
		DLIST_REMOVE(conn->pending, req);
	}

	if (req->async.fn) {
		req->async.fn(req);
	}
}


/*
  check if a blob is a complete ldap packet
  handle wrapper or unwrapped connections
*/
NTSTATUS ldap_complete_packet(void *private_data, DATA_BLOB blob, size_t *size)
{
	struct ldap_connection *conn = talloc_get_type(private_data,
						       struct ldap_connection);
	if (conn->enable_wrap) {
		return packet_full_request_u32(private_data, blob, size);
	}
	return ldap_full_packet(private_data, blob, size);
}

/*
  decode/process plain data
*/
static NTSTATUS ldap_decode_plain(struct ldap_connection *conn, DATA_BLOB blob)
{
	struct asn1_data asn1;
	struct ldap_message *msg = talloc(conn, struct ldap_message);

	if (msg == NULL) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}

	if (!asn1_load(&asn1, blob)) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}
	
	if (!ldap_decode(&asn1, msg)) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}

	ldap_match_message(conn, msg);

	data_blob_free(&blob);
	asn1_free(&asn1);
	return NT_STATUS_OK;
}

/*
  decode/process wrapped data
*/
static NTSTATUS ldap_decode_wrapped(struct ldap_connection *conn, DATA_BLOB blob)
{
	DATA_BLOB wrapped, unwrapped;
	struct asn1_data asn1;
	struct ldap_message *msg = talloc(conn, struct ldap_message);
	NTSTATUS status;

	if (msg == NULL) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}

	wrapped = data_blob_const(blob.data+4, blob.length-4);

	status = gensec_unwrap(conn->gensec, msg, &wrapped, &unwrapped);
	if (!NT_STATUS_IS_OK(status)) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}

	data_blob_free(&blob);

	if (!asn1_load(&asn1, unwrapped)) {
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}

	while (ldap_decode(&asn1, msg)) {
		ldap_match_message(conn, msg);
		msg = talloc(conn, struct ldap_message);
	}
		
	talloc_free(msg);
	asn1_free(&asn1);

	return NT_STATUS_OK;
}


/*
  handle ldap recv events
*/
static NTSTATUS ldap_recv_handler(void *private_data, DATA_BLOB blob)
{
	struct ldap_connection *conn = talloc_get_type(private_data, 
						       struct ldap_connection);
	if (conn->enable_wrap) {
		return ldap_decode_wrapped(conn, blob);
	}

	return ldap_decode_plain(conn, blob);
}


/*
  handle ldap socket events
*/
static void ldap_io_handler(struct event_context *ev, struct fd_event *fde, 
			    uint16_t flags, void *private_data)
{
	struct ldap_connection *conn = talloc_get_type(private_data, 
						       struct ldap_connection);
	if (flags & EVENT_FD_WRITE) {
		packet_queue_run(conn->packet);
		if (conn->tls == NULL) return;
	}
	if (flags & EVENT_FD_READ) {
		packet_recv(conn->packet);
	}
}

/*
  parse a ldap URL
*/
static NTSTATUS ldap_parse_basic_url(TALLOC_CTX *mem_ctx, const char *url,
				     char **host, uint16_t *port, BOOL *ldaps)
{
	int tmp_port = 0;
	char protocol[11];
	char tmp_host[255];
	const char *p = url;
	int ret;

	/* skip leading "URL:" (if any) */
	if (strncasecmp(p, "URL:", 4) == 0) {
		p += 4;
	}

	/* Paranoia check */
	SMB_ASSERT(sizeof(protocol)>10 && sizeof(tmp_host)>254);
		
	ret = sscanf(p, "%10[^:]://%254[^:/]:%d", protocol, tmp_host, &tmp_port);
	if (ret < 2) {
		return NT_STATUS_INVALID_PARAMETER;
	}

	if (strequal(protocol, "ldap")) {
		*port = 389;
		*ldaps = False;
	} else if (strequal(protocol, "ldaps")) {
		*port = 636;
		*ldaps = True;
	} else {
		DEBUG(0, ("unrecognised ldap protocol (%s)!\n", protocol));
		return NT_STATUS_PROTOCOL_UNREACHABLE;
	}

	if (tmp_port != 0)
		*port = tmp_port;

	*host = talloc_strdup(mem_ctx, tmp_host);
	NT_STATUS_HAVE_NO_MEMORY(*host);

	return NT_STATUS_OK;
}

/*
  connect to a ldap server
*/

struct ldap_connect_state {
	struct composite_context *ctx;
	struct ldap_connection *conn;
};

static void ldap_connect_recv_conn(struct composite_context *ctx);

struct composite_context *ldap_connect_send(struct ldap_connection *conn,
					    const char *url)
{
	struct composite_context *result, *ctx;
	struct ldap_connect_state *state;

	result = talloc_zero(NULL, struct composite_context);
	if (result == NULL) goto failed;
	result->state = COMPOSITE_STATE_IN_PROGRESS;
	result->async.fn = NULL;
	result->event_ctx = conn->event.event_ctx;

	state = talloc(result, struct ldap_connect_state);
	if (state == NULL) goto failed;
	state->ctx = result;
	result->private_data = state;

	state->conn = conn;

	state->ctx->status = ldap_parse_basic_url(conn, url, &conn->host,
						  &conn->port, &conn->ldaps);
	if (!NT_STATUS_IS_OK(state->ctx->status)) {
		composite_error(state->ctx, state->ctx->status);
		return result;
	}

	ctx = socket_connect_multi_send(state, conn->host, 1, &conn->port,
					conn->event.event_ctx);
	if (ctx == NULL) goto failed;

	ctx->async.fn = ldap_connect_recv_conn;
	ctx->async.private_data = state;
	return result;

 failed:
	talloc_free(result);
	return NULL;
}

static void ldap_connect_recv_conn(struct composite_context *ctx)
{
	struct ldap_connect_state *state =
		talloc_get_type(ctx->async.private_data,
				struct ldap_connect_state);
	struct ldap_connection *conn = state->conn;
	uint16_t port;

	state->ctx->status = socket_connect_multi_recv(ctx, state, &conn->sock,
						       &port);
	if (!composite_is_ok(state->ctx)) return;

	/* setup a handler for events on this socket */
	conn->event.fde = event_add_fd(conn->event.event_ctx, conn->sock, 
				       socket_get_fd(conn->sock), 
				       EVENT_FD_READ, ldap_io_handler, conn);
	if (conn->event.fde == NULL) {
		composite_error(state->ctx, NT_STATUS_INTERNAL_ERROR);
		return;
	}

	conn->tls = tls_init_client(conn->sock, conn->event.fde, conn->ldaps);
	if (conn->tls == NULL) {
		talloc_free(conn->sock);
		return;
	}
	talloc_steal(conn, conn->tls);
	talloc_steal(conn->tls, conn->sock);

	conn->packet = packet_init(conn);
	if (conn->packet == NULL) {
		talloc_free(conn->sock);
		return;
	}
	packet_set_private(conn->packet, conn);
	packet_set_tls(conn->packet, conn->tls);
	packet_set_callback(conn->packet, ldap_recv_handler);
	packet_set_full_request(conn->packet, ldap_complete_packet);
	packet_set_error_handler(conn->packet, ldap_error_handler);
	packet_set_event_context(conn->packet, conn->event.event_ctx);
	packet_set_fde(conn->packet, conn->event.fde);
	packet_set_serialise(conn->packet);

	composite_done(state->ctx);

	return;
}

NTSTATUS ldap_connect_recv(struct composite_context *ctx)
{
	NTSTATUS status = composite_wait(ctx);
	talloc_free(ctx);
	return status;
}

NTSTATUS ldap_connect(struct ldap_connection *conn, const char *url)
{
	struct composite_context *ctx = ldap_connect_send(conn, url);
	return ldap_connect_recv(ctx);
}

/* destroy an open ldap request */
static int ldap_request_destructor(void *ptr)
{
	struct ldap_request *req = talloc_get_type(ptr, struct ldap_request);
	if (req->state == LDAP_REQUEST_PENDING) {
		DLIST_REMOVE(req->conn->pending, req);
	}
	return 0;
}

/*
  called on timeout of a ldap request
*/
static void ldap_request_timeout(struct event_context *ev, struct timed_event *te, 
				      struct timeval t, void *private_data)
{
	struct ldap_request *req = talloc_get_type(private_data, struct ldap_request);
	req->status = NT_STATUS_IO_TIMEOUT;
	if (req->state == LDAP_REQUEST_PENDING) {
		DLIST_REMOVE(req->conn->pending, req);
	}
	req->state = LDAP_REQUEST_DONE;
	if (req->async.fn) {
		req->async.fn(req);
	}
}


/*
  called on completion of a one-way ldap request
*/
static void ldap_request_complete(struct event_context *ev, struct timed_event *te, 
				  struct timeval t, void *private_data)
{
	struct ldap_request *req = talloc_get_type(private_data, struct ldap_request);
	if (req->async.fn) {
		req->async.fn(req);
	}
}

/*
  send a ldap message - async interface
*/
struct ldap_request *ldap_request_send(struct ldap_connection *conn,
				       struct ldap_message *msg)
{
	struct ldap_request *req;
	NTSTATUS status;

	if (conn->tls == NULL) {
		return NULL;
	}

	req = talloc_zero(conn, struct ldap_request);
	if (req == NULL) goto failed;

	req->state       = LDAP_REQUEST_SEND;
	req->conn        = conn;
	req->messageid   = conn->next_messageid++;
	if (conn->next_messageid == 0) {
		conn->next_messageid = 1;
	}
	req->type        = msg->type;
	if (req->messageid == -1) {
		goto failed;
	}

	talloc_set_destructor(req, ldap_request_destructor);

	msg->messageid = req->messageid;

	if (!ldap_encode(msg, &req->data, req)) {
		goto failed;		
	}

	/* possibly encrypt/sign the request */
	if (conn->enable_wrap) {
		DATA_BLOB wrapped;

		status = gensec_wrap(conn->gensec, req, &req->data, &wrapped);
		if (!NT_STATUS_IS_OK(status)) {
			goto failed;
		}
		data_blob_free(&req->data);
		req->data = data_blob_talloc(req, NULL, wrapped.length + 4);
		if (req->data.data == NULL) {
			goto failed;
		}
		RSIVAL(req->data.data, 0, wrapped.length);
		memcpy(req->data.data+4, wrapped.data, wrapped.length);
		data_blob_free(&wrapped);
	}

	status = packet_send(conn->packet, req->data);
	if (!NT_STATUS_IS_OK(status)) {
		goto failed;
	}

	/* some requests don't expect a reply, so don't add those to the
	   pending queue */
	if (req->type == LDAP_TAG_AbandonRequest ||
	    req->type == LDAP_TAG_UnbindRequest) {
		req->status = NT_STATUS_OK;
		req->state = LDAP_REQUEST_DONE;
		/* we can't call the async callback now, as it isn't setup, so
		   call it as next event */
		event_add_timed(conn->event.event_ctx, req, timeval_zero(),
				ldap_request_complete, req);
		return req;
	}

	req->state = LDAP_REQUEST_PENDING;
	DLIST_ADD(conn->pending, req);

	/* put a timeout on the request */
	req->time_event = event_add_timed(conn->event.event_ctx, req, 
					  timeval_current_ofs(conn->timeout, 0),
					  ldap_request_timeout, req);

	return req;

failed:
	talloc_free(req);
	return NULL;
}


/*
  wait for a request to complete
  note that this does not destroy the request
*/
NTSTATUS ldap_request_wait(struct ldap_request *req)
{
	while (req->state != LDAP_REQUEST_DONE) {
		if (event_loop_once(req->conn->event.event_ctx) != 0) {
			req->status = NT_STATUS_UNEXPECTED_NETWORK_ERROR;
			break;
		}
	}
	return req->status;
}


/*
  a mapping of ldap response code to strings
*/
static const struct {
	enum ldap_result_code code;
	const char *str;
} ldap_code_map[] = {
#define _LDAP_MAP_CODE(c) { c, #c }
	_LDAP_MAP_CODE(LDAP_SUCCESS),
	_LDAP_MAP_CODE(LDAP_OPERATIONS_ERROR),
	_LDAP_MAP_CODE(LDAP_PROTOCOL_ERROR),
	_LDAP_MAP_CODE(LDAP_TIME_LIMIT_EXCEEDED),
	_LDAP_MAP_CODE(LDAP_SIZE_LIMIT_EXCEEDED),
	_LDAP_MAP_CODE(LDAP_COMPARE_FALSE),
	_LDAP_MAP_CODE(LDAP_COMPARE_TRUE),
	_LDAP_MAP_CODE(LDAP_AUTH_METHOD_NOT_SUPPORTED),
	_LDAP_MAP_CODE(LDAP_STRONG_AUTH_REQUIRED),
	_LDAP_MAP_CODE(LDAP_REFERRAL),
	_LDAP_MAP_CODE(LDAP_ADMIN_LIMIT_EXCEEDED),
	_LDAP_MAP_CODE(LDAP_UNAVAILABLE_CRITICAL_EXTENSION),
	_LDAP_MAP_CODE(LDAP_CONFIDENTIALITY_REQUIRED),
	_LDAP_MAP_CODE(LDAP_SASL_BIND_IN_PROGRESS),
	_LDAP_MAP_CODE(LDAP_NO_SUCH_ATTRIBUTE),
	_LDAP_MAP_CODE(LDAP_UNDEFINED_ATTRIBUTE_TYPE),
	_LDAP_MAP_CODE(LDAP_INAPPROPRIATE_MATCHING),
	_LDAP_MAP_CODE(LDAP_CONSTRAINT_VIOLATION),
	_LDAP_MAP_CODE(LDAP_ATTRIBUTE_OR_VALUE_EXISTS),
	_LDAP_MAP_CODE(LDAP_INVALID_ATTRIBUTE_SYNTAX),
	_LDAP_MAP_CODE(LDAP_NO_SUCH_OBJECT),
	_LDAP_MAP_CODE(LDAP_ALIAS_PROBLEM),
	_LDAP_MAP_CODE(LDAP_INVALID_DN_SYNTAX),
	_LDAP_MAP_CODE(LDAP_ALIAS_DEREFERENCING_PROBLEM),
	_LDAP_MAP_CODE(LDAP_INAPPROPRIATE_AUTHENTICATION),
	_LDAP_MAP_CODE(LDAP_INVALID_CREDENTIALS),
	_LDAP_MAP_CODE(LDAP_INSUFFICIENT_ACCESS_RIGHTs),
	_LDAP_MAP_CODE(LDAP_BUSY),
	_LDAP_MAP_CODE(LDAP_UNAVAILABLE),
	_LDAP_MAP_CODE(LDAP_UNWILLING_TO_PERFORM),
	_LDAP_MAP_CODE(LDAP_LOOP_DETECT),
	_LDAP_MAP_CODE(LDAP_NAMING_VIOLATION),
	_LDAP_MAP_CODE(LDAP_OBJECT_CLASS_VIOLATION),
	_LDAP_MAP_CODE(LDAP_NOT_ALLOWED_ON_NON_LEAF),
	_LDAP_MAP_CODE(LDAP_NOT_ALLOWED_ON_RDN),
	_LDAP_MAP_CODE(LDAP_ENTRY_ALREADY_EXISTS),
	_LDAP_MAP_CODE(LDAP_OBJECT_CLASS_MODS_PROHIBITED),
	_LDAP_MAP_CODE(LDAP_AFFECTS_MULTIPLE_DSAS),
	_LDAP_MAP_CODE(LDAP_OTHER)
};

/*
  used to setup the status code from a ldap response
*/
NTSTATUS ldap_check_response(struct ldap_connection *conn, struct ldap_Result *r)
{
	int i;
	const char *codename = "unknown";

	if (r->resultcode == LDAP_SUCCESS) {
		return NT_STATUS_OK;
	}

	if (conn->last_error) {
		talloc_free(conn->last_error);
	}

	for (i=0;i<ARRAY_SIZE(ldap_code_map);i++) {
		if (r->resultcode == ldap_code_map[i].code) {
			codename = ldap_code_map[i].str;
			break;
		}
	}

	conn->last_error = talloc_asprintf(conn, "LDAP error %u %s - %s <%s> <%s>", 
					   r->resultcode,
					   codename,
					   r->dn?r->dn:"(NULL)", 
					   r->errormessage?r->errormessage:"", 
					   r->referral?r->referral:"");
	
	return NT_STATUS_LDAP(r->resultcode);
}

/*
  return error string representing the last error
*/
const char *ldap_errstr(struct ldap_connection *conn, NTSTATUS status)
{
	if (NT_STATUS_IS_LDAP(status) && conn->last_error != NULL) {
		return conn->last_error;
	}
	return nt_errstr(status);
}


/*
  return the Nth result message, waiting if necessary
*/
NTSTATUS ldap_result_n(struct ldap_request *req, int n, struct ldap_message **msg)
{
	*msg = NULL;

	NT_STATUS_HAVE_NO_MEMORY(req);

	while (req->state != LDAP_REQUEST_DONE && n >= req->num_replies) {
		if (event_loop_once(req->conn->event.event_ctx) != 0) {
			return NT_STATUS_UNEXPECTED_NETWORK_ERROR;
		}
	}

	if (n < req->num_replies) {
		*msg = req->replies[n];
		return NT_STATUS_OK;
	}

	if (!NT_STATUS_IS_OK(req->status)) {
		return req->status;
	}

	return NT_STATUS_NO_MORE_ENTRIES;
}


/*
  return a single result message, checking if it is of the expected LDAP type
*/
NTSTATUS ldap_result_one(struct ldap_request *req, struct ldap_message **msg, int type)
{
	NTSTATUS status;
	status = ldap_result_n(req, 0, msg);
	if (!NT_STATUS_IS_OK(status)) {
		return status;
	}
	if ((*msg)->type != type) {
		*msg = NULL;
		return NT_STATUS_UNEXPECTED_NETWORK_ERROR;
	}
	return status;
}

/*
  a simple ldap transaction, for single result requests that only need a status code
  this relies on single valued requests having the response type == request type + 1
*/
NTSTATUS ldap_transaction(struct ldap_connection *conn, struct ldap_message *msg)
{
	struct ldap_request *req = ldap_request_send(conn, msg);
	struct ldap_message *res;
	NTSTATUS status;
	status = ldap_result_n(req, 0, &res);
	if (!NT_STATUS_IS_OK(status)) {
		talloc_free(req);
		return status;
	}
	if (res->type != msg->type + 1) {
		talloc_free(req);
		return NT_STATUS_LDAP(LDAP_PROTOCOL_ERROR);
	}
	status = ldap_check_response(conn, &res->r.GeneralResult);
	talloc_free(req);
	return status;
}
