/**
 * Copyright (c) 2009, Opera Software ASA
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Opera Software ASA nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY OPERA SOFTWARE ASA ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL OPERA SOFTWARE ASA BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Rev: 5610 $
 *
 */

/* a window.languages array needs to be defined somewhere, like this:

window.languages = {
  'en' : 'locale/en.js',
  'de' : 'locale/de.js',
  'fr' : 'locale/fr.js'
};

*/

(function()
{
  var lang = 'en'; // default

  var langIsSet = false;

  /**
   * A function to change the currently used language dictionary
   *
   * @param {String}   newLang   Sets the new language
   * @param {Function} callback  Function to be called once the language is loaded
   * @return          String of the now-current language (not always the same as newLang)
   **/

  window.setLanguage = function(newLang, callback)
  {
    var oldLang = lang;

    lang = (newLang || navigator.language).toLowerCase();

    var i;
    var ele = null;

    while (true)
    {
      for (i in languages)
      {
        if (languages.hasOwnProperty(i) && i==lang)
        {
          // insert the script then remove it
          ele = document.createElement('script');
          ele.setAttribute('type','text/javascript');
          ele.setAttribute('src', languages[i]);
          ele.onload = function()
          {
            // evil closure, very small so shouldn't be a problem
            ele.parentNode.removeChild(ele); // cleanup
            if (callback)
            {
              callback(lang);
            }
            delete ele.onload;
          };
          (document.getElementsByTagName('head')[0] || document.documentElement).appendChild(ele);
          document.documentElement.lang = i;
          break;
        }
      }
      if (ele) // language was applied, (need double break to break double loop)
      {
        break;
      }
      if (lang.indexOf('-')+1) // eg, if "en-au" was tried then try just "en"
      {
        lang = lang.substr(0,lang.indexOf('-'));
      }
      else
      {
        if (lang==oldLang)
        {
          throw Error('Cannot set language');
        }
        lang = oldLang;
      }
    }
    langIsSet = true;
    return lang;
  }

  /**
   * This prototype function takes a string as the 'this' value and uses that
   * as the key for the translation.
   *
   * The 'replace' argument is optional and is overloaded.
   * - If 'replace' is a string then it will replace the substring %s before the
   *   return value is returned
   * - If 'replace' is an object then the key-value pairs will replace the substrings
   *   in the form of %(key)s
   *   So for example the replace of {name:"Ben",city:"Oslo"} will modify "Hello %(name)s in %(city)s."
   *   to become "Hello Ben in Oslo".
   *
   * @param replace   An optional string or object
   * @return          The translated string with the replacements made
   **/

  String.prototype.toLocale = function(replace)
  {
    if (!langIsSet) { window.setLanguage() }

    var str;
    if (window.localeStrings && (str = localeStrings[this]) !==undefined)
    {
      if (replace)
      {
        if (typeof replace == 'object')
        {
          str = str.replace(
            /%\(([^)]*)\)s/g,
            function(m0, m1)
            {
              return ''+(replace.hasOwnProperty(m1) ? replace[m1] : m0);
            }
          );
        }
        else
        {
          str = str.replace("%s",replace);
        }
      }
      return str;
    }
    else
    {
      return '#'+this;
    }
  };

  /**
   * Translates all children elements, applies to all Elements and document
   **/


  Node.prototype.toLocale = function()
  {
    var i = 0, ele = this.getElementsByTagName('*');

    // if we tranlate an element we should also localise the root element (this)
    if (this instanceof Element)
    {
      applyLocalisation(this);
    }

    for (i=0; i < ele.length; i++)  // loop every element
    {
      applyLocalisation(ele[i]);
    }
  };

  /**
   * Private function
   * Localises an element by reading the 'locale' attribute.
   *
   * @param {element} The element to localise
   **/

  // local function
  function applyLocalisation(e)
  {
    var result = '',
           key = null,
             i = 0;

    if (e.dataset)
    {
        result = e.dataset['locale'];
    }
    else
    {
        result = e.getAttribute('data-locale')
    }


    if (result) // yes, element is translatable
    {
      result = result.replace(/^\s+|\s+$/g,'').split(/\s*;+\s*/g);
      for (i=0; i<result.length; i++)
      {
        if (result)
        {
          // apply translation for each attribute
          key = result[i].split(/\s*:\s*/);

          if (key[1])
          {
            e[ key[0] ] = key[1].toLocale();
          }
          else
          {
            e.innerText = key[0].toLocale();
          }
        }
      }
    }
  }

})();